﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/basis/atkfnd_Time.h>

#include <nn/os/os_Tick.h>

namespace {
    int64_t NanoSecondsToTicks(int64_t value) NN_NOEXCEPT
    {
        nn::TimeSpan timeSpan = nn::TimeSpan::FromNanoSeconds(value);
        nn::os::Tick tick(timeSpan);
        return tick.GetInt64Value();
    }

    int64_t MicroSecondsToTicks(int64_t value) NN_NOEXCEPT
    {
        return NanoSecondsToTicks(value * 1000);
    }

    int64_t MilliSecondsToTicks(int64_t value) NN_NOEXCEPT
    {
        return NanoSecondsToTicks(value * 1000 * 1000);
    }

    int64_t TicksToNanoSeconds(int64_t ticks) NN_NOEXCEPT
    {
        nn::os::Tick tick(ticks);
        nn::TimeSpan timeSpan = tick.ToTimeSpan();
        return timeSpan.GetNanoSeconds();
    }

    int64_t TicksToMicroSeconds(int64_t ticks) NN_NOEXCEPT
    {
        nn::os::Tick tick(ticks);
        nn::TimeSpan timeSpan = tick.ToTimeSpan();
        return timeSpan.GetMicroSeconds();
    }

    int64_t TicksToMilliSeconds(int64_t ticks) NN_NOEXCEPT
    {
        nn::os::Tick tick(ticks);
        nn::TimeSpan timeSpan = tick.ToTimeSpan();
        return timeSpan.GetMilliSeconds();
    }
}

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

//----------------------------------------------------------
TimeSpan
Time::Current() NN_NOEXCEPT
{
    return TimeSpan(nn::os::GetSystemTick().GetInt64Value());
}

//----------------------------------------------------------
TimeSpan
TimeSpan::FromNanoSeconds(int64_t value) NN_NOEXCEPT
{

    return TimeSpan(NanoSecondsToTicks(value));
}

//----------------------------------------------------------
TimeSpan
TimeSpan::FromMicroSeconds(int64_t value) NN_NOEXCEPT
{
    return TimeSpan(MicroSecondsToTicks(value));
}

//----------------------------------------------------------
TimeSpan
TimeSpan::FromMilliSeconds(int64_t value) NN_NOEXCEPT
{
    return TimeSpan(MilliSecondsToTicks(value));
}

//----------------------------------------------------------
int64_t
TimeSpan::ToNanoSeconds() const NN_NOEXCEPT
{
    return TicksToNanoSeconds(m_TickSpan);
}

//----------------------------------------------------------
int64_t
TimeSpan::ToMicroSeconds() const NN_NOEXCEPT
{
    return TicksToMicroSeconds(m_TickSpan);
}

//----------------------------------------------------------
int64_t
TimeSpan::ToMilliSeconds() const NN_NOEXCEPT
{
    return TicksToMilliSeconds(m_TickSpan);
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
