﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/basis/atkfnd_Memory.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>

//#define NN_ATK_ENABLE_MEM_DEBUG

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

//---------------------------------------------------------------------------
NN_DEFINE_STATIC_CONSTANT( const int MemoryTraits::DefaultAlignment );

//---------------------------------------------------------------------------
void
FrameHeap::Initialize(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT(length > 0);

    m_Buffer = buffer;
    m_BufferLength = length;
}

//---------------------------------------------------------------------------
void
FrameHeap::Finalize() NN_NOEXCEPT
{
    m_Buffer = NULL;
    m_BufferLength = 0;
}

//---------------------------------------------------------------------------
size_t
FrameHeap::GetFreeLength(int32_t alignment/*= MemoryTraits::DefaultAlignment*/) const NN_NOEXCEPT
{
    if(!IsInitialized()) { return 0; }

    void* result = AddOffsetToPtr(m_Buffer, m_UsedLength, alignment);
    return m_BufferLength - util::BytePtr(m_Buffer).Distance(result);
}

//---------------------------------------------------------------------------
void*
FrameHeap::Alloc(size_t size, int32_t alignment/*= MemoryTraits::DefaultAlignment*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT(alignment > 0);

    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "FrameHeap is not initialized.\n");
        return NULL;
    }

    if(size == 0)
    {
        return NULL;
    }

    void*  result = AddOffsetToPtr(m_Buffer, m_UsedLength, alignment);
    size_t newUsedLength = util::BytePtr(m_Buffer).Distance(result) + size;

    if(newUsedLength > m_BufferLength)
    {
        NN_ATK_WARNING("out of memory.");
        return NULL;
    }

    m_UsedLength = newUsedLength;

    return result;
}

//---------------------------------------------------------------------------
void
UnitHeap::Initialize(
    void* buffer,
    size_t length,
    size_t unitLength,
    int unitCount,
    int unitAlignment/*= MemoryTraits::DefaultAlignment*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT(length > 0);
    NN_SDK_ASSERT(unitLength > 0);
    NN_SDK_ASSERT(unitLength < static_cast<size_t>((std::numeric_limits<int32_t>::max)()));
    NN_SDK_ASSERT(unitCount > 0);
    NN_SDK_ASSERT(m_AllocatedUnitCount == 0);

    void*  alignedBuffer = util::BytePtr(buffer).AlignUp(unitAlignment).Get();
    size_t alignedBufferLength = length - util::BytePtr(buffer).Distance(alignedBuffer);
    size_t alignedUnitLength = nn::util::align_up( unitLength, unitAlignment );

    if(alignedBufferLength < alignedUnitLength * unitCount)
    {
        NN_SDK_ASSERT(false, "out of memory.\n");
        return;
    }

    for(auto unitIndex = 0; unitIndex < unitCount; ++unitIndex)
    {
        void* unit = AddOffsetToPtr(alignedBuffer, unitIndex * alignedUnitLength, unitAlignment);
        if (unitIndex == unitCount - 1)
        {
            *reinterpret_cast<int32_t*>(unit) = 0;
        }
        else
        {
            *reinterpret_cast<int32_t*>(unit) = static_cast<int32_t>(alignedUnitLength);
        }

#if !defined(NN_SDK_BUILD_RELEASE)
        Memset(AddOffsetToPtr(unit, sizeof(int32_t)), 0xcd, alignedUnitLength - sizeof(int32_t));
#endif

    }

    m_Buffer = alignedBuffer;
    m_BufferLength = alignedBufferLength;
    m_UnitLength = alignedUnitLength;
    m_FirstFreeUnit = alignedBuffer;
}

//---------------------------------------------------------------------------
void
UnitHeap::Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_AllocatedUnitCount == 0);

    m_Buffer = NULL;
    m_BufferLength = 0;
    m_UnitLength = 0;
    m_FirstFreeUnit = NULL;
}

//---------------------------------------------------------------------------
void*
UnitHeap::Alloc(size_t size, int32_t /*alignment = MemoryTraits::DefaultAlignment*/) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "UnitHeap is not initialized.\n");
        return NULL;
    }

    if(m_FirstFreeUnit == NULL)
    {
        NN_ATK_WARNING("out of memory.");
        return NULL;
    }

    // １ユニットあたりのサイズを超えた場合は、失敗させます。
    if(m_UnitLength < size)
    {
        NN_SDK_ASSERT(false, "required size is too large.\n");
        return NULL;
    }

    void* result = m_FirstFreeUnit;
    m_FirstFreeUnit = GetNextFreeUnit(m_FirstFreeUnit);

    ++m_AllocatedUnitCount;

#if defined(NN_ATK_ENABLE_MEM_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndfnd] UnitMemoryAllocator::Alloc() : AllocatedUnitCount=%d, AllocatedBuffer=0x%8x, FirstFreeUnit=0x%8x.\n",
        m_AllocatedUnitCount,
        result,
        m_FirstFreeUnit);
#endif

    return result;
}

//---------------------------------------------------------------------------
void
UnitHeap::Free(void* ptr) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(ptr);

    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "UnitHeap is not initialized.\n");
        return;
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    Memset(ptr, 0xcd, m_UnitLength);
#endif

    ptrdiff_t offset = 0;

    if(m_FirstFreeUnit != NULL)
    {
        offset = util::BytePtr(ptr).Distance(m_FirstFreeUnit);
        NN_SDK_ASSERT(offset != 0);
    }

    *reinterpret_cast<int32_t*>(ptr) = static_cast<int32_t>(offset);
    m_FirstFreeUnit = ptr;

    --m_AllocatedUnitCount;

#if defined(NN_ATK_ENABLE_MEM_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndfnd] UnitMemoryAllocator::Free() : AllocatedUnitCount=%d, FreedBuffer=0x%8x, FirstFreeUnit=0x%8x.\n",
        m_AllocatedUnitCount,
        ptr,
        m_FirstFreeUnit);
#endif
}

//---------------------------------------------------------------------------
void*
UnitHeap::GetNextFreeUnit(void* unit) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(unit);

    int32_t offset = *reinterpret_cast<int32_t*>(unit);

    if(offset == 0)
    {
        return NULL;
    }

    return AddOffsetToPtr(unit, offset);
}

//---------------------------------------------------------------------------
NN_DEFINE_STATIC_CONSTANT( const size_t ExpandedHeap::BlockMdSize );
NN_DEFINE_STATIC_CONSTANT( const size_t ExpandedHeap::HeapMdSize );

//---------------------------------------------------------------------------
bool
ExpandedHeap::Initialize(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT(length > 0);

    m_HeapHandle = nn::lmem::CreateExpHeap(buffer, length, nn::lmem::CreationOption_NoOption);
    return IsInitialized();
}

//---------------------------------------------------------------------------
void
ExpandedHeap::Finalize() NN_NOEXCEPT
{
    if(m_HeapHandle != NULL)
    {
        nn::lmem::DestroyExpHeap(m_HeapHandle);
        m_HeapHandle = NULL;
    }
}

//---------------------------------------------------------------------------
void*
ExpandedHeap::Alloc(size_t size, int32_t alignment/*= MemoryTraits::DefaultAlignment*/) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "ExpandedHeap is not initialized.\n");
        return NULL;
    }

    if(size == 0)
    {
        return NULL;
    }

    return nn::lmem::AllocateFromExpHeap(m_HeapHandle, size, alignment);
}

//---------------------------------------------------------------------------
void
ExpandedHeap::Free(void* ptr) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(ptr);

    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "ExpandedHeap is not initialized.\n");
        return;
    }

    nn::lmem::FreeToExpHeap(m_HeapHandle, ptr);
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
