﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/basis/atkfnd_HeapBase.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>  // for GetOffsetFromPtr(), etc...
#include <cstring>

//---------------------------------------------------------------------------
//! @brief       特定のビット位置の値を取得します。
//!
//! @param[in]   data  取得するビットデータを含むデータ。
//! @param[in]   st    開始ビット(0から始まる)。
//! @param[in]   bits  ビット数(31以下とする)。
//!
//! @return      特定のビット位置の値を返します。
//---------------------------------------------------------------------------
#define     GetBitValue(data, st, bits) \
    (((data) >>(st)) & ((1 <<(bits)) - 1))

//---------------------------------------------------------------------------
//! @brief       特定のビット位置に値をセットします。
//!
//! @param[in]  data  セットするビットデータを格納する変数。
//! @param[in]  st    開始ビット(0から始まる)。
//! @param[in]  bits  ビット数(31以下とする)。
//! @param[in]  val   セットするビットデータ。
//!
//! @return      なし。
//---------------------------------------------------------------------------
#define     SetBitValue(data, st, bits, val)                        \
                do                                                          \
                {                                                           \
                    uint32_t maskBits = static_cast<uint32_t>((1 <<(bits)) - 1);                  \
                    uint32_t newVal = (val) & maskBits; /* 安全のためマスク */   \
                    (void)(maskBits <<= st);                                \
                    (data) &= ~maskBits; /* セットする領域をクリア */       \
                    (data) |= newVal <<(st);                                \
                } while((void)0, 0);


namespace nn { namespace atk { namespace detail { namespace fnd {

namespace {

// ルートのヒープリスト
HeapBase::HeapList sRootList;

// fill 値
uint32_t sFillVals[ HeapBase::FillType_Max ] =
{
    0xC3C3C3C3, // ヒープ作成時に埋める値
    0xF3F3F3F3, // メモリブロック確保時に埋める値
    0xD3D3D3D3, // メモリブロック解放時に埋める値
};

// ヒープを含有する親ヒープを検索し、その親ヒープのリストへのポインタを返します

#if 1
inline void DumpHeapList() NN_NOEXCEPT {}

#else

void DumpHeapList() NN_NOEXCEPT
{
    HeapBase* pHeapBase = NULL;
    int count = 0;

    OS_Printf("Dump HeapBase List\n");
    while ( NULL != ( pHeapBase = pHeap->GetNext() ) )
    {
        count += 1;
        OS_Printf("[%d] -> %p %08X\n", count, pHeap, pHeap->signature );
    }
}

#endif  // DumpHeapList

} // unnamed namespace

NN_DEFINE_STATIC_CONSTANT( const int HeapBase::DefaultAlignment );
NN_DEFINE_STATIC_CONSTANT( const uint32_t HeapBase::ExpHeapSignature );
NN_DEFINE_STATIC_CONSTANT( const uint32_t HeapBase::FrameHeapSignature );
NN_DEFINE_STATIC_CONSTANT( const uint32_t HeapBase::UnitHeapSignature );
NN_DEFINE_STATIC_CONSTANT( const int HeapBase::OptionZeroClear );
NN_DEFINE_STATIC_CONSTANT( const int HeapBase::OptionDebugFill );
NN_DEFINE_STATIC_CONSTANT( const int HeapBase::OptionThreadSafe );
NN_DEFINE_STATIC_CONSTANT( const int HeapBase::ErrorPrint );
NN_DEFINE_STATIC_CONSTANT( const int HeapBase::MIN_ALIGNMENT );

//---------------------------------------------------------------------------
//! @brief       ヒープを含有する親ヒープを検索し、その親ヒープのリストへの
//!              ポインタを返します。
//!
//! @param[out]  pHeapBase   検索対象のヒープへのポインタ。
//!
//! @return      指定したヒープを含有する親ヒープが見つかれば、
//!              親ヒープの子リストへのポインタを返します。
//!              親ヒープが見つからなければルートリストへのポインタが返ります。
//---------------------------------------------------------------------------
HeapBase::HeapList*
HeapBase::FindListContainHeap( HeapBase* pHeapBase ) NN_NOEXCEPT
{
    HeapBase::HeapList* pList = &sRootList;
    HeapBase* pContainHeapBase = HeapBase::FindContainHeap( &sRootList, pHeapBase );

    if ( pContainHeapBase )
    {
        pList = &pContainHeapBase->m_ChildList;
    }

    return pList;
}

//---------------------------------------------------------------------------
//! @brief       指定されたメモリブロックを含有するヒープをリストから再帰的に
//!              探し出します。
//!
//! @param[in]   pList      リストへのポインタ
//! @param[in]   memBlock   メモリブロックへのポインタ
//!
//! @return      指定されたメモリブロックを確保したヒープが見つかれば、
//!              そのヒープへのポインタを返します。
//!              見つからなかった時は NULL を返します。
//---------------------------------------------------------------------------
HeapBase*
HeapBase::FindContainHeap( HeapBase::HeapList* pList, const void* memBlock ) NN_NOEXCEPT
{
    uintptr_t memBlockAddress = reinterpret_cast<uintptr_t>(memBlock);

    for ( HeapList::iterator itr = pList->begin(); itr != pList->end();
        )
    {
        HeapList::iterator curItr = itr++;
        if ( reinterpret_cast<uintptr_t>(curItr->mHeapStart) <= memBlockAddress
          && reinterpret_cast<uintptr_t>(curItr->mHeapEnd) > memBlockAddress )
        {
            HeapBase* pChildHeapBase = FindContainHeap( &curItr->m_ChildList, memBlock );
            if ( pChildHeapBase )
            {
                return pChildHeapBase;
            }
            return &(*curItr);
        }
    }
    return NULL;
}

//---------------------------------------------------------------------------
//! @brief       メモリブロックを含有するヒープを検索します。
//!
//! @param[in]   memBlock   検索対象のメモリブロック。
//!
//! @return      指定したメモリブロックを含むヒープが見つかれば、
//!              そのヒープのハンドルを返します。
//!              見つからなければ、NULL が返ります。
//---------------------------------------------------------------------------
HeapBase*
HeapBase::FindContainHeap( const void* memBlock ) NN_NOEXCEPT
{
    return FindContainHeap( &sRootList, memBlock );
}

//---------------------------------------------------------------------------
//! @brief       親ヒープを検索します。
//!
//! @param[in]   heap   検索対象のヒープ
//!
//! @return      指定したメモリブロックを含むヒープが見つかれば、
//!              そのヒープのハンドルを返します。
//!              見つからなければ、NULL が返ります。
//---------------------------------------------------------------------------
HeapBase*
HeapBase::FindParentHeap( const HeapBase* heap ) NN_NOEXCEPT
{
    uint64_t heapAddress = reinterpret_cast<uint64_t>(heap);

    for ( HeapList::iterator itr = sRootList.begin(); itr != sRootList.end(); )
    {
        HeapList::iterator curItr = itr++;
        // ターゲットのアドレスが含まれているならば、子を検索する
        if ( reinterpret_cast<uint64_t>(curItr->mHeapStart) <= heapAddress
          && reinterpret_cast<uint64_t>(curItr->mHeapEnd) > heapAddress )
        {
            return FindContainHeap( &curItr->m_ChildList, &(*curItr) );
        }
    }

    return NULL;
}

#if 0
// ヒープアドレスを表示します
void HeapBase::Dump() NN_NOEXCEPT
{
    char str[64] = "[nn::atk ";

    switch ( mSignature )
    {
        case EXPHEAP_SIGNATURE: std::snprintf( str, 64, "%s Exp", str );    break;
        case FRMHEAP_SIGNATURE: std::snprintf( str, 64, "%s Frame", str );  break;
        case UNTHEAP_SIGNATURE: std::snprintf( str, 64, "%s Unit", str );   break;
        default:
            NN_SDK_ASSERT( false );
    }

    OS_Printf("%s Heap]\twhole [%p - %p)\n", str, this, mHeapEnd );
}
#endif

//---------------------------------------------------------------------------
//! @brief       ヒープの作成時やメモリブロックの確保・解放時にメモリに
//!              セットする値をセットします。
//!              この関数はデバッグ用の関数です。
//!              最終ROM版ライブラリでは常に0を返します。
//!
//! @param[in]   type   取得する値の種類
//! @param[in]   val    セットする値
//!
//! @return      以前の、メモリブロックの確保時にメモリにセットする値を返します。
//---------------------------------------------------------------------------
uint32_t
HeapBase::SetFillValue( FillType type, uint32_t val ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( type < FillType_Max );

    uint32_t oldVal = sFillVals[ type ];
    sFillVals[ type ] = val;
    return oldVal;
}

//---------------------------------------------------------------------------
//! @brief       ヒープの作成時やメモリブロックの確保・解放時にメモリに
//!              セットする値を取得します。
//!              この関数はデバッグ用の関数です。
//!              最終ROM版ライブラリでは常に0を返します。
//!
//! @param[in]   type   取得する値の種類
//!
//! @return      指定された種類のメモリにセットする値を返します。
//---------------------------------------------------------------------------
uint32_t
HeapBase::GetFillValue( FillType type ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( type < FillType_Max );
    return sFillVals[ type ];
}


//---------------------------------------------------------------------------
//! @brief       ヒープハンドルがどの種類のヒープであるかを取得します。
//!
//! @return      ヒープハンドルからヒープの種別を取得します。
//---------------------------------------------------------------------------
HeapBase::HeapType
HeapBase::GetHeapType() NN_NOEXCEPT
{
    switch ( m_Signature )
    {
        case ExpHeapSignature: return HeapType_Exp;
        case FrameHeapSignature: return HeapType_Frame;
        case UnitHeapSignature: return HeapType_Unit;
        default:                return HeapTyep_Unknown;
    }
}

//---------------------------------------------------------------------------
//! @brief       ヒープの初期化を行います。
//!
//! @param[in]   signature   シグネチャ。
//! @param[in]   heapStart   ヒープメモリの開始アドレス。
//! @param[in]   heapEnd     ヒープメモリの終了アドレス +1。
//! @param[in]   optFlag     ヒープオプション。
//---------------------------------------------------------------------------
void
HeapBase::Initialize( uint32_t signature, void* heapStart, void* heapEnd, uint16_t optFlag ) NN_NOEXCEPT
{
    m_Signature = signature;

    mHeapStart = heapStart;
    mHeapEnd = heapEnd;

    m_Attribute = 0;
    SetOptionFlag( optFlag );

    FillNoUseMemory( heapStart, static_cast<uint32_t>( GetOffsetFromPtr( heapStart, heapEnd ) ) );

    HeapList* pList = FindListContainHeap( this );
    pList->push_back( *this );
    DumpHeapList();
}

//---------------------------------------------------------------------------
//! @brief       ヒープ共通の後始末を行います。
//---------------------------------------------------------------------------
void
HeapBase::Finalize() NN_NOEXCEPT
{
    HeapList* pList = FindListContainHeap( this );
    pList->erase( pList->iterator_to( *this ) );
    m_Signature = 0;
    DumpHeapList();
}

//---------------------------------------------------------------------------
//! @brief       ヒープの排他制御をおこなうためのロック関数
//---------------------------------------------------------------------------
void
HeapBase::LockHeap() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
//! @brief       ヒープの排他制御をおこなうためのアンロック関数
//---------------------------------------------------------------------------
void
HeapBase::UnlockHeap() NN_NOEXCEPT
{
}


//---------------------------------------------------------------------------
//! @brief       フリー直後のメモリフィル
//!
//! @param[in]   address     アドレス
//! @param[in]   size        サイズ
//---------------------------------------------------------------------------
void
HeapBase::FillFreeMemory( void* address, size_t size ) NN_NOEXCEPT
{
    if ( GetOptionFlag() & OptionDebugFill )
    {
        std::memset(address, static_cast<int>(GetFillValue( FillType_Free )), size);
    }
}

//---------------------------------------------------------------------------
//! @brief       未使用メモリのメモリフィル
//!
//! @param[in]   address     アドレス
//! @param[in]   size        サイズ
//---------------------------------------------------------------------------
void
HeapBase::FillNoUseMemory( void* address, size_t size ) NN_NOEXCEPT
{
    if ( GetOptionFlag() & OptionDebugFill )
    {
        std::memset(address, static_cast<int>(GetFillValue( FillType_NoUse )), size);
    }
}

//---------------------------------------------------------------------------
//! @brief       アロケート直後のメモリフィル
//!
//! @param[in]   address     アドレス
//! @param[in]   size        サイズ
//---------------------------------------------------------------------------
void
HeapBase::FillAllocMemory( void* address, size_t size ) NN_NOEXCEPT
{
    if ( GetOptionFlag() & OptionZeroClear )
    {
        std::memset(address, 0, size);
    }
    else
    {
        if ( GetOptionFlag() & OptionDebugFill )
        {
            std::memset(address, static_cast<int>(GetFillValue( FillType_Alloc )), size);
        }
    }
}

//---------------------------------------------------------------------------
//! @brief       ヒープからオプションフラグを取得
//!
//! @return      オプションフラグの値
//---------------------------------------------------------------------------
uint16_t
HeapBase::GetOptionFlag() NN_NOEXCEPT
{
    return static_cast<uint16_t>( GetBitValue( m_Attribute, 0, 8 ) );
}

//---------------------------------------------------------------------------
//! @brief       ヒープへオプションフラグを設定
//!
//! @param[in]   optFlag  ヒープへのオプションです。
//---------------------------------------------------------------------------
void
HeapBase::SetOptionFlag( uint16_t optFlag ) NN_NOEXCEPT
{
    SetBitValue( m_Attribute, 0, 8, optFlag );
}

}}}} // namespace nn::atk::detail::fnd
