﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_EffectI3dl2Reverb.h>

#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/detail/atk_Macro.h>

namespace
{
int ConvertI3dl2ReverbChannelModeToInt(nn::atk::EffectI3dl2Reverb::ChannelMode channelMode) NN_NOEXCEPT
{
    switch(channelMode)
    {
    case nn::atk::EffectBase::ChannelMode_1Ch:
        return 1;
    case nn::atk::EffectBase::ChannelMode_2Ch:
        return 2;
    case nn::atk::EffectBase::ChannelMode_4Ch:
        return 4;
    case nn::atk::EffectBase::ChannelMode_6Ch:
        return 6;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool IsChannelModeSupported(nn::atk::EffectBase::ChannelMode channelMode) NN_NOEXCEPT
{
    return channelMode == nn::atk::EffectBase::ChannelMode_1Ch || channelMode == nn::atk::EffectBase::ChannelMode_2Ch
        || channelMode == nn::atk::EffectBase::ChannelMode_4Ch || channelMode == nn::atk::EffectBase::ChannelMode_6Ch;
}

int ConvertI3dl2ReverbSampleRateToInt(nn::atk::EffectI3dl2Reverb::SampleRate sampleRate) NN_NOEXCEPT
{
    switch(sampleRate)
    {
    case nn::atk::EffectBase::SampleRate_32000:
        return 32000;
    case nn::atk::EffectBase::SampleRate_48000:
        return 48000;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
}

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT(const int EffectI3dl2Reverb::ChannelSettingCountMax);

// 最大値・最小値の定義
const float EffectI3dl2Reverb::RoomGainMin = -10000.0f;
const float EffectI3dl2Reverb::RoomGainMax = 0.0f;
const float EffectI3dl2Reverb::RoomHfGainMin = -10000.0f;
const float EffectI3dl2Reverb::RoomHfGainMax = 0.0f;
const int64_t EffectI3dl2Reverb::LateReverbDecayTimeMilliSecondsMin = 100;
const int64_t EffectI3dl2Reverb::LateReverbDecayTimeMilliSecondsMax = 20000;
const float EffectI3dl2Reverb::LateReverbHfDecayRatioMin = 0.1f;
const float EffectI3dl2Reverb::LateReverbHfDecayRatioMax = 2.0f;
const float EffectI3dl2Reverb::ReflectionsGainMin = -10000.0f;
const float EffectI3dl2Reverb::ReflectionsGainMax = 1000.0f;
const int64_t EffectI3dl2Reverb::ReflectionDelayTimeMilliSecondsMin = 0;
const int64_t EffectI3dl2Reverb::ReflectionDelayTimeMilliSecondsMax = 300;
const float EffectI3dl2Reverb::ReverbGainMin = -10000.0f;
const float EffectI3dl2Reverb::ReverbGainMax = 2000.0f;
const int64_t EffectI3dl2Reverb::LateReverbDelayTimeMilliSecondsMin = 0;
const int64_t EffectI3dl2Reverb::LateReverbDelayTimeMilliSecondsMax = 100;
const float EffectI3dl2Reverb::LateReverbDiffusionMin = 0.0f;
const float EffectI3dl2Reverb::LateReverbDiffusionMax = 100.0f;
const float EffectI3dl2Reverb::LateReverbDensityMin = 0.0f;
const float EffectI3dl2Reverb::LateReverbDensityMax = 100.0f;
const float EffectI3dl2Reverb::HfReferenceMin = 20.0f;
const float EffectI3dl2Reverb::HfReferenceMax = 20000.0f;
const float EffectI3dl2Reverb::DryGainMin = 0.0f;
const float EffectI3dl2Reverb::DryGainMax = 1.0f;

// デフォルト値の定義
const float EffectI3dl2Reverb::DefaultRoomGain = -1000.0f;
const float EffectI3dl2Reverb::DefaultRoomHfGain = -1000.0f;
const int64_t EffectI3dl2Reverb::DefaultLateReverbDecayTimeMilliSeconds = 2000;
const float EffectI3dl2Reverb::DefaultLateReverbHfDecayRatio = 0.8f;
const float EffectI3dl2Reverb::DefaultReflectionsGain = -700.0f;
const int64_t EffectI3dl2Reverb::DefaultReflectionDelayTimeMilliSeconds = 5;
const float EffectI3dl2Reverb::DefaultReverbGain = -500.0f;
const int64_t EffectI3dl2Reverb::DefaultLateReverbDelayTimeMilliSeconds = 40;
const float EffectI3dl2Reverb::DefaultLateReverbDiffusion = 50.0f;
const float EffectI3dl2Reverb::DefaultLateReverbDensity = 3.0f;
const float EffectI3dl2Reverb::DefaultHfReference = 6000.0f;
const float EffectI3dl2Reverb::DefaultDryGain = 0.25f;

EffectI3dl2Reverb::EffectI3dl2Reverb() NN_NOEXCEPT
    : EffectBase()
    , m_AudioRendererUpdateCountWhenAddedI3d2lReverb(0)
    , m_ChannelMode(ChannelMode_6Ch)
{
    ResetChannelIndex();
}

EffectI3dl2Reverb::~EffectI3dl2Reverb() NN_NOEXCEPT
{
}

size_t EffectI3dl2Reverb::GetRequiredMemSize() const NN_NOEXCEPT
{
    int maxChannelCount = ConvertI3dl2ReverbChannelModeToInt(m_ChannelMode);
    int sampleRate = ConvertI3dl2ReverbSampleRateToInt(m_SampleRate);
    return nn::audio::GetRequiredBufferSizeForI3dl2Reverb(sampleRate, maxChannelCount);
}

nn::TimeSpan EffectI3dl2Reverb::GetLateReverbDecayTimeMin() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(LateReverbDecayTimeMilliSecondsMin);
}

nn::TimeSpan EffectI3dl2Reverb::GetLateReverbDecayTimeMax() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(LateReverbDecayTimeMilliSecondsMax);
}

nn::TimeSpan EffectI3dl2Reverb::GetReflectionDelayTimeMin() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(ReflectionDelayTimeMilliSecondsMin);
}

nn::TimeSpan EffectI3dl2Reverb::GetReflectionDelayTimeMax() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(ReflectionDelayTimeMilliSecondsMax);
}

nn::TimeSpan EffectI3dl2Reverb::GetLateReverbDelayTimeMin() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(LateReverbDelayTimeMilliSecondsMin);
}

nn::TimeSpan EffectI3dl2Reverb::GetLateReverbDelayTimeMax() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(LateReverbDelayTimeMilliSecondsMax);
}

nn::TimeSpan EffectI3dl2Reverb::GetDefaultLateReverbDecayTime() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(DefaultLateReverbDecayTimeMilliSeconds);
}

nn::TimeSpan EffectI3dl2Reverb::GetDefaultReflectionDelayTime() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(DefaultReflectionDelayTimeMilliSeconds);
}

nn::TimeSpan EffectI3dl2Reverb::GetDefaultLateReverbDelayTime() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(DefaultLateReverbDelayTimeMilliSeconds);
}

bool EffectI3dl2Reverb::AddEffect(nn::audio::AudioRendererConfig* pConfig, nn::atk::OutputMixer* pOutputMixer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pOutputMixer);
    if(m_IsActive)
    {
        return false;
    }

    int channelCount = ConvertI3dl2ReverbChannelModeToInt(m_ChannelMode);
    int sampleRate = ConvertI3dl2ReverbSampleRateToInt(m_SampleRate);
    NN_SDK_ASSERT_NOT_NULL(m_EffectBuffer);
    NN_SDK_REQUIRES_ALIGNED( m_EffectBuffer, nn::audio::BufferAlignSize );
    NN_SDK_ASSERT_GREATER_EQUAL(m_EffectBufferSize, nn::audio::GetRequiredBufferSizeForI3dl2Reverb(sampleRate, channelCount));
    NN_UNUSED(sampleRate);

    {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
        m_AudioRendererUpdateCountWhenAddedI3d2lReverb = detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount();

        nn::Result result;
        switch( pOutputMixer->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
            {
                result = nn::audio::AddI3dl2Reverb( pConfig, &m_I3dl2ReverbType, m_EffectBuffer, m_EffectBufferSize, reinterpret_cast<SubMix*>( pOutputMixer )->GetAudioSubMixInstance(), channelCount );
                break;
            }
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
            {
                result = nn::audio::AddI3dl2Reverb( pConfig, &m_I3dl2ReverbType, m_EffectBuffer, m_EffectBufferSize, reinterpret_cast<FinalMix*>( pOutputMixer )->GetAudioFinalMixInstance(), channelCount );
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        if( result.IsFailure() )
        {
            return false;
        }
    }

    m_IsActive = true;

    // 初期化後に I3dl2ReverbType へパラメータを反映させる
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbRoomGain(&m_I3dl2ReverbType, m_Param.roomGain);
        nn::audio::SetI3dl2ReverbRoomHfGain(&m_I3dl2ReverbType, m_Param.roomHfGain);
        nn::audio::SetI3dl2ReverbLateReverbDecayTime(&m_I3dl2ReverbType, m_Param.lateReverbDecayTime);
        nn::audio::SetI3dl2ReverbLateReverbHfDecayRatio(&m_I3dl2ReverbType, m_Param.lateReverbHfDecayRatio);
        nn::audio::SetI3dl2ReverbReflectionsGain(&m_I3dl2ReverbType, m_Param.reflectionsGain);
        nn::audio::SetI3dl2ReverbReflectionDelayTime(&m_I3dl2ReverbType, m_Param.reflectionDelayTime);
        nn::audio::SetI3dl2ReverbReverbGain(&m_I3dl2ReverbType, m_Param.reverbGain);
        nn::audio::SetI3dl2ReverbLateReverbDelayTime(&m_I3dl2ReverbType, m_Param.lateReverbDelayTime);
        nn::audio::SetI3dl2ReverbLateReverbDiffusion(&m_I3dl2ReverbType, m_Param.lateReverbDiffusion);
        nn::audio::SetI3dl2ReverbLateReverbDensity(&m_I3dl2ReverbType, m_Param.lateReverbDensity);
        nn::audio::SetI3dl2ReverbHfReference(&m_I3dl2ReverbType, m_Param.hfReference);
        nn::audio::SetI3dl2ReverbDryGain(&m_I3dl2ReverbType, m_Param.dryGain);
        nn::audio::SetI3dl2ReverbEnabled(&m_I3dl2ReverbType, m_Param.isEnabled);
    }

    return true;
}

void EffectI3dl2Reverb::SetEffectInputOutput(const int8_t* input, const int8_t* output, int inputCount, int outputCount) NN_NOEXCEPT
{
    // I3DL2 Reverb は入出力のチャンネル数が同じ
    NN_SDK_ASSERT_EQUAL(inputCount, outputCount);
    NN_SDK_ASSERT_NOT_NULL(input);
    NN_SDK_ASSERT_NOT_NULL(output);
    // I3DL2 Reverb が 2 ch, 4 ch, 6 ch のみの適用であるため 2 ch 以上であることを確認する
    NN_SDK_ASSERT_GREATER_EQUAL(inputCount, ConvertI3dl2ReverbChannelModeToInt(EffectI3dl2Reverb::ChannelMode_2Ch));

    int channelCount = ConvertI3dl2ReverbChannelModeToInt(m_ChannelMode);
    // 設定した I3DL2 Reverb のチャンネル数が与えられたバッファ数よりも少ないことを確認する
    NN_SDK_ASSERT_LESS_EQUAL(channelCount, inputCount);

    NN_UNUSED(inputCount);
    NN_UNUSED(outputCount);
    NN_UNUSED(output);

    // I3DL2 Reverb のチャンネル設定を適用する
    NN_SDK_ASSERT_LESS_EQUAL(channelCount, ChannelSettingCountMax);
    int8_t channelBus[ChannelSettingCountMax];
    for (int i = 0; i < channelCount; i++)
    {
        // m_ChannelSetting[i] が input の範囲外の配列を指していないか確認する
        NN_SDK_ASSERT_RANGE(m_ChannelSetting[i], 0, inputCount);
        channelBus[i] = input[m_ChannelSetting[i]];
    }

    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbInputOutput(&m_I3dl2ReverbType, channelBus, channelBus, channelCount);
    }
}

void EffectI3dl2Reverb::RemoveEffect(nn::audio::AudioRendererConfig* pConfig, nn::atk::OutputMixer* pOutputMixer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pOutputMixer);
    if(!m_IsActive)
    {
        return;
    }

    {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
        NN_SDK_ASSERT(nn::audio::IsI3dl2ReverbRemovable(&m_I3dl2ReverbType));

        void* returnedBuffer = nullptr;
        switch( pOutputMixer->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
            {
                returnedBuffer = nn::audio::RemoveI3dl2Reverb( pConfig, &m_I3dl2ReverbType, reinterpret_cast<SubMix*>( pOutputMixer )->GetAudioSubMixInstance() );
                break;
            }
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
            {
                returnedBuffer = nn::audio::RemoveI3dl2Reverb( pConfig, &m_I3dl2ReverbType, reinterpret_cast<FinalMix*>( pOutputMixer )->GetAudioFinalMixInstance() );
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        NN_SDK_ASSERT_EQUAL(returnedBuffer, m_EffectBuffer);
        NN_UNUSED(returnedBuffer);
    }

    m_IsActive = false;
}

bool EffectI3dl2Reverb::IsRemovable() const NN_NOEXCEPT
{
    if (!m_IsActive)
    {
        return false;
    }

    //  AddIsI3dl2Reverb 直後は IsI3dl2ReverbRemovable が true を返すため、
    //  1 回以上 RequestUpdateAudioRenderer が呼ばれてから IsI3dl2ReverbRemovable で確認するようにします

    if( !( m_AudioRendererUpdateCountWhenAddedI3d2lReverb < detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount() ) )
    {
        return false;
    }

    detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
    if( !nn::audio::IsI3dl2ReverbRemovable(&m_I3dl2ReverbType) )
    {
        return false;
    }

    //  RequestUpdateAudioRenderer の呼び出し回数を 2 度比較している理由はコミットログを参照
    return m_AudioRendererUpdateCountWhenAddedI3d2lReverb < detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount();
}

EffectI3dl2Reverb::ChannelMode EffectI3dl2Reverb::GetChannelMode() const NN_NOEXCEPT
{
    return m_ChannelMode;
}

bool EffectI3dl2Reverb::SetChannelMode(ChannelMode mode) NN_NOEXCEPT
{
    ChannelIndex defaultChannelSetting[ChannelModeCountMax];
    for (int i = 0; i < ChannelModeCountMax; i++)
    {
        defaultChannelSetting[i] = static_cast<ChannelIndex>(i);
    }
    return SetChannelIndex(defaultChannelSetting, mode);
}

void EffectI3dl2Reverb::GetChannelIndex(ChannelIndex* pChannel, int channelCount) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pChannel);
    int effectChannelCount = ConvertI3dl2ReverbChannelModeToInt(m_ChannelMode);
    NN_SDK_REQUIRES_GREATER_EQUAL(channelCount, effectChannelCount);
    NN_UNUSED(channelCount);
    for (int i = 0; i < effectChannelCount; i++)
    {
        pChannel[i] = m_ChannelSetting[i];
    }
}

bool EffectI3dl2Reverb::SetChannelIndex(const ChannelIndex* pChannel, ChannelMode channelMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pChannel);

    int channelCount = ConvertChannelModeToInt(channelMode);
    NN_SDK_ASSERT_LESS_EQUAL(channelCount, ChannelModeCountMax);
    for (int i = 0; i < channelCount; i++)
    {
        NN_SDK_REQUIRES_RANGE(pChannel[i], 0, nn::atk::ChannelIndex_Count);
        for (int j = i + 1; j < channelCount; j++)
        {
            NN_SDK_REQUIRES_NOT_EQUAL(pChannel[i], pChannel[j]);
        }
    }

    if(m_IsActive)
    {
        return false;
    }

    if(IsChannelModeSupported(channelMode))
    {
        m_ChannelMode = channelMode;
    }
    else
    {
        NN_ATK_WARNING("unsupported channel mode.");
        return false;
    }

    NN_SDK_ASSERT_LESS_EQUAL(channelCount, ChannelSettingCountMax);
    for (int i = 0; i < channelCount; i++)
    {
        m_ChannelSetting[i] = pChannel[i];
    }

    return true;
}

float EffectI3dl2Reverb::GetRoomGain() const NN_NOEXCEPT
{
    return m_Param.roomGain;
}

void EffectI3dl2Reverb::SetRoomGain( const float roomGain ) NN_NOEXCEPT
{
    m_Param.roomGain = nn::atk::detail::fnd::Clamp(roomGain, RoomGainMin, RoomGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbRoomGain(&m_I3dl2ReverbType, m_Param.roomGain);
    }
}

float EffectI3dl2Reverb::GetRoomHfGain() const NN_NOEXCEPT
{
    return m_Param.roomHfGain;
}

void EffectI3dl2Reverb::SetRoomHfGain( const float roomHfGain ) NN_NOEXCEPT
{
    m_Param.roomHfGain = nn::atk::detail::fnd::Clamp(roomHfGain, RoomHfGainMin, RoomHfGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbRoomHfGain(&m_I3dl2ReverbType, m_Param.roomHfGain);
    }
}

nn::TimeSpan EffectI3dl2Reverb::GetLateReverbDecayTime() const NN_NOEXCEPT
{
    return m_Param.lateReverbDecayTime;
}

void EffectI3dl2Reverb::SetLateReverbDecayTime( const nn::TimeSpan decayTime ) NN_NOEXCEPT
{
    m_Param.lateReverbDecayTime = nn::atk::detail::fnd::Clamp(decayTime, GetLateReverbDecayTimeMin(), GetLateReverbDecayTimeMax());
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbLateReverbDecayTime(&m_I3dl2ReverbType, m_Param.lateReverbDecayTime);
    }
}

float EffectI3dl2Reverb::GetLateReverbHfDecayRatio() const NN_NOEXCEPT
{
    return m_Param.lateReverbHfDecayRatio;
}

void EffectI3dl2Reverb::SetLateReverbHfDecayRatio( const float hfDecayRatio ) NN_NOEXCEPT
{
    m_Param.lateReverbHfDecayRatio = nn::atk::detail::fnd::Clamp(hfDecayRatio, LateReverbHfDecayRatioMin, LateReverbHfDecayRatioMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbLateReverbHfDecayRatio(&m_I3dl2ReverbType, m_Param.lateReverbHfDecayRatio);
    }
}

float EffectI3dl2Reverb::GetReflectionsGain() const NN_NOEXCEPT
{
    return m_Param.reflectionsGain;
}

void EffectI3dl2Reverb::SetReflectionsGain( const float reflectionsGain ) NN_NOEXCEPT
{
    m_Param.reflectionsGain = nn::atk::detail::fnd::Clamp(reflectionsGain, ReflectionsGainMin, ReflectionsGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbReflectionsGain(&m_I3dl2ReverbType, m_Param.reflectionsGain);
    }
}

nn::TimeSpan EffectI3dl2Reverb::GetReflectionDelayTime() const NN_NOEXCEPT
{
    return m_Param.reflectionDelayTime;
}

void EffectI3dl2Reverb::SetReflectionDelayTime( const nn::TimeSpan delayTime ) NN_NOEXCEPT
{
    m_Param.reflectionDelayTime = nn::atk::detail::fnd::Clamp(delayTime, GetReflectionDelayTimeMin(), GetReflectionDelayTimeMax());
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbReflectionDelayTime(&m_I3dl2ReverbType, m_Param.reflectionDelayTime);
    }
}

float EffectI3dl2Reverb::GetReverbGain() const NN_NOEXCEPT
{
    return m_Param.reverbGain;
}

void EffectI3dl2Reverb::SetReverbGain( const float reverbGain ) NN_NOEXCEPT
{
    m_Param.reverbGain = nn::atk::detail::fnd::Clamp(reverbGain, ReverbGainMin, ReverbGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbReverbGain(&m_I3dl2ReverbType, m_Param.reverbGain);
    }
}

nn::TimeSpan EffectI3dl2Reverb::GetLateReverbDelayTime() const NN_NOEXCEPT
{
    return m_Param.lateReverbDelayTime;
}

void EffectI3dl2Reverb::SetLateReverbDelayTime( const nn::TimeSpan delayTime ) NN_NOEXCEPT
{
    m_Param.lateReverbDelayTime = nn::atk::detail::fnd::Clamp(delayTime, GetLateReverbDelayTimeMin(), GetLateReverbDelayTimeMax());
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbLateReverbDelayTime(&m_I3dl2ReverbType, m_Param.lateReverbDelayTime);
    }
}

float EffectI3dl2Reverb::GetLateReverbDiffusion() const NN_NOEXCEPT
{
    return m_Param.lateReverbDiffusion;
}

void EffectI3dl2Reverb::SetLateReverbDiffusion( const float diffusion ) NN_NOEXCEPT
{
    m_Param.lateReverbDiffusion = nn::atk::detail::fnd::Clamp(diffusion, LateReverbDiffusionMin, LateReverbDiffusionMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbLateReverbDiffusion(&m_I3dl2ReverbType, m_Param.lateReverbDiffusion);
    }
}

float EffectI3dl2Reverb::GetLateReverbDensity() const NN_NOEXCEPT
{
    return m_Param.lateReverbDensity;
}

void EffectI3dl2Reverb::SetLateReverbDensity( const float density ) NN_NOEXCEPT
{
    m_Param.lateReverbDensity = nn::atk::detail::fnd::Clamp(density, LateReverbDensityMin, LateReverbDensityMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbLateReverbDensity(&m_I3dl2ReverbType, m_Param.lateReverbDensity);
    }
}

float EffectI3dl2Reverb::GetHfReference() const NN_NOEXCEPT
{
    return m_Param.hfReference;
}

void EffectI3dl2Reverb::SetHfReference( const float hfReference ) NN_NOEXCEPT
{
    m_Param.hfReference = nn::atk::detail::fnd::Clamp(hfReference, HfReferenceMin, HfReferenceMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbHfReference(&m_I3dl2ReverbType, m_Param.hfReference);
    }
}

float EffectI3dl2Reverb::GetDryGain() const NN_NOEXCEPT
{
    return m_Param.dryGain;
}

void EffectI3dl2Reverb::SetDryGain( const float dryGain ) NN_NOEXCEPT
{
    m_Param.dryGain = nn::atk::detail::fnd::Clamp(dryGain, DryGainMin, DryGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbDryGain(&m_I3dl2ReverbType, m_Param.dryGain);
    }
}

bool EffectI3dl2Reverb::IsEnabled() const NN_NOEXCEPT
{
    return m_Param.isEnabled;
}

void EffectI3dl2Reverb::SetEnabled( bool isEnabled ) NN_NOEXCEPT
{
    m_Param.isEnabled = isEnabled;
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetI3dl2ReverbEnabled(&m_I3dl2ReverbType, m_Param.isEnabled);
    }
}

void EffectI3dl2Reverb::ResetChannelIndex() NN_NOEXCEPT
{
    for(int i = 0; i < ChannelSettingCountMax; i++)
    {
        m_ChannelSetting[i] = static_cast<ChannelIndex>(i);
    }
}

EffectI3dl2Reverb::I3dl2ReverbParam::I3dl2ReverbParam() NN_NOEXCEPT
    : roomGain(DefaultRoomGain)
    , roomHfGain(DefaultRoomHfGain)
    , lateReverbDecayTime(GetDefaultLateReverbDecayTime())
    , lateReverbHfDecayRatio(DefaultLateReverbHfDecayRatio)
    , reflectionsGain(DefaultReflectionsGain)
    , reflectionDelayTime(GetDefaultReflectionDelayTime())
    , reverbGain(DefaultReverbGain)
    , lateReverbDelayTime(GetDefaultLateReverbDelayTime())
    , lateReverbDiffusion(DefaultLateReverbDiffusion)
    , lateReverbDensity(DefaultLateReverbDensity)
    , hfReference(DefaultHfReference)
    , dryGain(DefaultDryGain)
    , isEnabled(false)
{
}


}}
