﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WaveSoundRuntime.h>

#include <nn/atk/atk_SoundDataManager.h>
#include <nn/atk/atk_SoundPlayer.h>
#include <nn/atk/atk_SoundRuntimeUtility.h>
#include <nn/atk/atk_StartInfoReader.h>
#include <nn/atk/atk_WaveArchiveFileReader.h>

namespace
{
    const int MemAlignSize = nn::audio::BufferAlignSize;
}

namespace nn { namespace atk { namespace detail {

    WaveSoundRuntime::WaveSoundRuntime() NN_NOEXCEPT
    : m_pSoundArchiveFilesHook(NULL)
    {
    }

    WaveSoundRuntime::~WaveSoundRuntime() NN_NOEXCEPT
    {
    }

    bool WaveSoundRuntime::Initialize( int soundCount, void** pOutAllocatedAddr, const void* endAddr ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(*pOutAllocatedAddr), MemAlignSize) );
        if ( reinterpret_cast<uint64_t>(*pOutAllocatedAddr) % MemAlignSize != 0 )
        {
            return false;
        }

        // WaveSoundInstanceManager
        {
            const detail::SoundInstanceConfig config = nn::atk::SoundSystem::GetSoundInstanceConfig();
            size_t requireSize = nn::util::align_up(
                m_WaveSoundInstanceManager.GetRequiredMemSize(soundCount, config),
                MemAlignSize );

            void* estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();
            if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
            {
                return false;
            }

            unsigned long createdSoundCount = m_WaveSoundInstanceManager.Create( *pOutAllocatedAddr, requireSize, config );
#if defined(NN_SDK_BUILD_RELEASE)
            NN_UNUSED( createdSoundCount );
#endif
            NN_SDK_ASSERT( createdSoundCount == static_cast<unsigned long>(soundCount) );
            *pOutAllocatedAddr = estimateEndAddr;
        }

        // WaveSoundLoaderManager
        {
            size_t requireSize = nn::util::align_up(
                detail::driver::WaveSoundLoaderManager::GetRequiredMemSize(soundCount),
                MemAlignSize );

            void* estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();
            if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
            {
                return false;
            }

            uint32_t createdSoundLoaderCount = m_WaveSoundLoaderManager.Create( *pOutAllocatedAddr, requireSize );
#if defined(NN_SDK_BUILD_RELEASE)
            NN_UNUSED( createdSoundLoaderCount );
#endif
            NN_SDK_ASSERT( static_cast<int32_t>(createdSoundLoaderCount) == soundCount,
                    "WaveSoundLoaderManager::Create createdSoundLoaderCount(%d) but soundCount(%d)",
                    createdSoundLoaderCount, soundCount);
            *pOutAllocatedAddr = estimateEndAddr;
        }

        // 各 WaveSound に WaveSoundLoaderManager をセット
        detail::WaveSoundInstanceManager::PriorityList& list =
            m_WaveSoundInstanceManager.GetFreeList();
        for (detail::WaveSoundInstanceManager::PriorityList::iterator itr = list.begin();
                itr != list.end();
                ++itr)
        {
            itr->SetLoaderManager(m_WaveSoundLoaderManager);
        }
        detail::driver::SoundThread::GetInstance().RegisterSoundFrameCallback(&m_WaveSoundLoaderManager);

        return true;
    }

    void WaveSoundRuntime::Finalize() NN_NOEXCEPT
    {
        m_WaveSoundInstanceManager.Destroy();
        m_WaveSoundLoaderManager.Destroy();
        detail::driver::SoundThread::GetInstance().UnregisterSoundFrameCallback(&m_WaveSoundLoaderManager);
    }

    size_t WaveSoundRuntime::GetRequiredMemorySize(const SoundArchive::SoundArchivePlayerInfo& soundArchivePlayerInfo, int alignment) NN_NOEXCEPT
    {
        size_t size = 0;

        size += nn::util::align_up(
            WaveSoundInstanceManager::GetRequiredMemSize(soundArchivePlayerInfo.waveSoundCount, nn::atk::SoundSystem::GetSoundInstanceConfig()),
            alignment );
        // NN_ATK_INIT_LOG("[WSD] instance: max(%2d) ALIGN(%d) => %8d\n",
        //         soundArchivePlayerInfo.waveSoundCount, alignment, size);

        size += nn::util::align_up(
            detail::driver::WaveSoundLoaderManager::GetRequiredMemSize(soundArchivePlayerInfo.waveSoundCount),
            alignment );
        // NN_ATK_INIT_LOG("[WSD] loader:   max(%2d) ALIGN(%d) => %8d\n",
        //         soundArchivePlayerInfo.waveSoundCount, alignment, size);

        return size;
    }

    int WaveSoundRuntime::GetActiveCount() const NN_NOEXCEPT
    {
        return m_WaveSoundInstanceManager.GetActiveCount();
    }

    int WaveSoundRuntime::GetFreeWaveSoundCount() const NN_NOEXCEPT
    {
        return m_WaveSoundInstanceManager.GetFreeCount();
    }

    void WaveSoundRuntime::SetupUserParam(void** startAddr, size_t adjustSize) NN_NOEXCEPT
    {
        void* curAddr = *startAddr;

        detail::WaveSoundInstanceManager::PriorityList& list = m_WaveSoundInstanceManager.GetFreeList();
        for (detail::WaveSoundInstanceManager::PriorityList::iterator itr = list.begin();
                itr != list.end();
                ++itr)
        {
            itr->SetUserParamBuffer(curAddr, adjustSize);

            curAddr = util::BytePtr(curAddr, adjustSize).Get();
        }

        *startAddr = curAddr;
    }

    void WaveSoundRuntime::Update() NN_NOEXCEPT
    {
        m_WaveSoundInstanceManager.SortPriorityList();
    }

    WaveSound* WaveSoundRuntime::AllocSound(
        SoundArchive::ItemId soundId,
        int priority,
        int ambientPriority,
        detail::BasicSound::AmbientInfo* ambientArgInfo,
        OutputReceiver* pOutputReceiver
    ) NN_NOEXCEPT
    {
        return SoundRuntimeUtility::AllocSound<WaveSound>(
            &m_WaveSoundInstanceManager,
            soundId,
            priority,
            ambientPriority,
            ambientArgInfo,
            pOutputReceiver
        );
    }

    SoundStartable::StartResult WaveSoundRuntime::PrepareImpl(
        const SoundArchive* pSoundArchive,
        const SoundDataManager* pSoundDataManager,
        SoundArchive::ItemId soundId,
        detail::WaveSound* sound,
        const SoundArchive::SoundInfo* commonInfo,
        const StartInfoReader& startInfoReader
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( sound );
        NN_SDK_ASSERT_NOT_NULL( pSoundArchive );
        NN_SDK_ASSERT_NOT_NULL( pSoundDataManager );

        // ウェーブサウンド情報の取得
        SoundArchive::WaveSoundInfo info;

        if ( ! pSoundArchive->detail_ReadWaveSoundInfo( soundId, &info ) )
        {
            SoundStartable::StartResult
                result( SoundStartable::StartResult::ResultCode_ErrorInvalidSoundId );
            return result;
        }

        const SoundStartable::StartInfo::WaveSoundInfo* externalWsdInfo = startInfoReader.GetWsdInfo();

        bool canUsePlayerHeap = sound->GetSoundPlayer()->detail_CanUsePlayerHeap();
        bool isRegisterDataLoadTask = false;
        detail::LoadItemInfo loadWsd;

        // データの取得
        const void* wsdFile = NULL;
        const void* waveFile = NULL;

        // // ウェーブサウンド、波形アーカイブファイルアクセスのフック
        if(m_pSoundArchiveFilesHook != NULL)
        {
            const char* soundLabel = pSoundArchive->GetItemLabel(sound->GetId());

            if(soundLabel != NULL)
            {
                wsdFile = m_pSoundArchiveFilesHook->GetFileAddress(
                    soundLabel,
                    detail::SoundArchiveFilesHook::ItemTypeWaveSound,
                    detail::SoundArchiveFilesHook::FileTypeWaveSoundBinary);

                const void* warcFile = m_pSoundArchiveFilesHook->GetFileAddress(
                    soundLabel,
                    detail::SoundArchiveFilesHook::ItemTypeWaveSound,
                    detail::SoundArchiveFilesHook::FileTypeWaveArchiveBinary);

                if(wsdFile != NULL && warcFile != NULL)
                {
                    detail::WaveArchiveFileReader warcReader( warcFile, false );
                    waveFile = warcReader.GetWaveFile( info.index );
                }

                // 完全に読み込めなかった場合は、通常動作を試みます。
                if(waveFile == NULL)
                {
                    wsdFile = NULL;
                }
            }
        }

        if ( wsdFile == NULL )
        {
            wsdFile = pSoundDataManager->detail_GetFileAddress( commonInfo->fileId );
        }

        int release = static_cast<int>( nn::atk::AdshrMax );
        bool isContextCalculationSkipMode = false;
        int waveSoundParameterFlag = 0;

        loadWsd.itemId = sound->GetId();
        loadWsd.address = wsdFile;
        if ( wsdFile == NULL )
        {
            if (canUsePlayerHeap == false)
            {
                SoundStartable::StartResult result(
                        SoundStartable::StartResult::ResultCode_ErrorNotWsdLoaded );
                return result;
            }
            isRegisterDataLoadTask = true;
        }
        else if(waveFile == NULL)
        {
            if (externalWsdInfo != NULL)
            {
                waveFile = externalWsdInfo->waveAddress;
                release = nn::atk::detail::fnd::Clamp(externalWsdInfo->release, nn::atk::AdshrMin, nn::atk::AdshrMax);
                isContextCalculationSkipMode = externalWsdInfo->isContextCalculationSkipMode;
                waveSoundParameterFlag = externalWsdInfo->enableParameterFlag;
            }
            if( waveFile == NULL )
            {
                waveFile = detail::Util::GetWaveFileOfWaveSound(
                        wsdFile,
                        info.index,
                        *pSoundArchive,
                        *pSoundDataManager );
            }
            if ( waveFile == NULL )
            {
                if (canUsePlayerHeap == false)
                {
                    SoundStartable::StartResult result(
                            SoundStartable::StartResult::ResultCode_ErrorNotWarcLoaded );
                    return result;
                }
                isRegisterDataLoadTask = true;
            }
        }

        detail::driver::WaveSoundPlayer::StartOffsetType wsdStartOffsetType;
        int startOffset = startInfoReader.GetStartOffset();
        switch ( startInfoReader.GetStartOffsetType() )
        {
        case SoundStartable::StartInfo::StartOffsetType_MilliSeconds:
            wsdStartOffsetType = detail::driver::WaveSoundPlayer::StartOffsetType_Millisec;
            break;
        case SoundStartable::StartInfo::StartOffsetType_Tick:
            wsdStartOffsetType = detail::driver::WaveSoundPlayer::StartOffsetType_Sample;
            startOffset = 0;
            break;
        case SoundStartable::StartInfo::StartOffsetType_Sample:
            wsdStartOffsetType = detail::driver::WaveSoundPlayer::StartOffsetType_Sample;
            break;
        default:
            wsdStartOffsetType = detail::driver::WaveSoundPlayer::StartOffsetType_Sample;
            startOffset = 0;
            break;
        }

        auto loopInfo = startInfoReader.GetLoopInfo();
        detail::driver::WaveSoundPlayer::LoopInfo playerLoopInfo =
        {
            0,
            false,
            0,
            0
        };
        if (loopInfo != nullptr)
        {
            playerLoopInfo.loopFlagBit = loopInfo->enableParameterFlag;
            playerLoopInfo.isLoopEnabled = loopInfo->isLoopEnabled;
        }

        detail::driver::WaveSoundPlayer::StartInfo startInfo =
        {
            static_cast<int32_t>( info.index ),
            wsdStartOffsetType,
            startOffset,
            startInfoReader.GetDelayTime(),
            startInfoReader.GetDelayCount(),
            waveSoundParameterFlag,
            release,
            isContextCalculationSkipMode,
            playerLoopInfo,
            startInfoReader.GetUpdateType()
        };

        if ( isRegisterDataLoadTask )
        {
            // 必要なデータが欠けている場合は、プレイヤーヒープへのロードタスクを投げる
            // (プリペアは、タスク完了後のコールバック内で処理される)
            detail::driver::WaveSoundLoader::LoadInfo loadInfo(
                pSoundArchive,
                pSoundDataManager,
                &loadWsd,
                sound->GetSoundPlayer()
            );
            sound->RegisterDataLoadTask( loadInfo, startInfo );
        }
        else
        {
            int8_t waveType = WaveType_Nwwav;
            if (externalWsdInfo != NULL && externalWsdInfo->waveAddress != NULL )
            {
                waveType = externalWsdInfo->waveType;
            }
            // 必要なデータが全部揃っている場合は、プリペア処理を行う
            sound->Prepare( wsdFile, waveFile, startInfo, waveType );
        }

        sound->InitializeChannelParam(info.channelPriority, info.isReleasePriorityFix);

        SoundStartable::StartResult startResult(
                SoundStartable::StartResult::ResultCode_Success );
        return startResult;
    } // NOLINT(impl/function_size)

    void WaveSoundRuntime::DumpMemory(const SoundArchive* pSoundArchive) const NN_NOEXCEPT
    {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
        NN_SDK_ASSERT_NOT_NULL( pSoundArchive );

        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( pSoundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            size_t instanceManagerSize = m_WaveSoundInstanceManager.GetRequiredMemSize( soundArchivePlayerInfo.waveSoundCount, nn::atk::SoundSystem::GetSoundInstanceConfig() );
            size_t loaderManagerSize   = detail::driver::WaveSoundLoaderManager::GetRequiredMemSize( soundArchivePlayerInfo.waveSoundCount );
            NN_DETAIL_ATK_INFO("    waveSound\n"
                       "      instanceManager        %8zu bytes\n"
                       "      loaderManager          %8zu bytes\n"
                       , instanceManagerSize, loaderManagerSize );
        }
#else
        NN_UNUSED(pSoundArchive);
#endif
    }
}}} // namespace nn::atk::detail
