﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WaveSoundFileReader.h>
#include <nn/atk/fnd/binary/atkfnd_Binary.h>

namespace nn {
namespace atk {
namespace detail {

namespace
{

const uint32_t SignatureInfoBlockWsd = NN_UTIL_CREATE_SIGNATURE_4( 'I', 'N', 'F', 'O' );

const uint32_t SupportedFileVersionWsd   = NN_ATK_MAKE_VERSION( 0, 1, 0, 0 );   // ライブラリがサポートする最低バージョン
const uint32_t CurrentFileVersionWsd     = NN_ATK_MAKE_VERSION( 0, 1, 1, 0 );   // ライブラリがサポートする最新バージョン
const uint32_t FilterSupportedVersionWsd = NN_ATK_MAKE_VERSION( 0, 1, 1, 0 );

bool IsValidFileHeaderWsd( const void* waveSoundFile ) NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( waveSoundFile );

    // シグニチャ確認
    NN_SDK_ASSERT( header.signature == WaveSoundFileReader::SignatureFile,
            "invalid file signature." );
    if ( header.signature != WaveSoundFileReader::SignatureFile )
    {
        return false;
    }

    // バイトオーダーマーク確認
    NN_SDK_ASSERT( header.byteOrder == BinaryFileHeader::ValidByteOrderMark,
            "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", BinaryFileHeader::ValidByteOrderMark, header.byteOrder );
    if ( header.byteOrder != BinaryFileHeader::ValidByteOrderMark )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersionWsd <= header.version) &&
                                   (header.version <= CurrentFileVersionWsd) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfwsd file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersionWsd, CurrentFileVersionWsd, header.version
    );
    return isSupportedVersion;
}

} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const uint32_t WaveSoundFileReader::SignatureFile );

WaveSoundFileReader::WaveSoundFileReader( const void* waveSoundFile ) NN_NOEXCEPT
: m_pHeader( NULL )
, m_pInfoBlockBody( NULL )
{
    NN_SDK_ASSERT_NOT_NULL( waveSoundFile );

    if ( ! IsValidFileHeaderWsd( waveSoundFile ) ) return;

    m_pHeader = reinterpret_cast<const WaveSoundFile::FileHeader*>( waveSoundFile );

    // INFO ブロックチェック
    const WaveSoundFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();

    NN_SDK_ASSERT_NOT_NULL( infoBlock );
    if ( infoBlock == NULL ) return;

    // シグニチャチェック
    NN_SDK_ASSERT( infoBlock->header.kind == SignatureInfoBlockWsd, "invalid block signature." );
    if ( infoBlock->header.kind != SignatureInfoBlockWsd )
    {
        return;
    }
    m_pInfoBlockBody = &infoBlock->body;
}

uint32_t WaveSoundFileReader::GetWaveSoundCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetWaveSoundCount();
}

uint32_t WaveSoundFileReader::GetNoteInfoCount( uint32_t index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT( index < GetWaveSoundCount() );

    const WaveSoundFile::WaveSoundData& wsdData =
        m_pInfoBlockBody->GetWaveSoundData( index );
    return wsdData.GetNoteCount();
}

uint32_t WaveSoundFileReader::GetTrackInfoCount( uint32_t index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT( index < GetWaveSoundCount() );

    const WaveSoundFile::WaveSoundData& wsdData =
        m_pInfoBlockBody->GetWaveSoundData( index );
    return wsdData.GetTrackCount();
}

bool WaveSoundFileReader::ReadWaveSoundInfo( WaveSoundInfo* dst, uint32_t index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( dst );

    const WaveSoundFile::WaveSoundInfo& src =
        m_pInfoBlockBody->GetWaveSoundData( index ).GetWaveSoundInfo();

    dst->pitch = src.GetPitch();
    dst->pan = src.GetPan();
    dst->surroundPan = src.GetSurroundPan();
    src.GetSendValue( &dst->mainSend, dst->fxSend, AuxBus_Count );
    dst->adshr = src.GetAdshrCurve();
    if ( IsFilterSupportedVersion() )
    {
        dst->lpfFreq = src.GetLpfFreq();
        dst->biquadType = src.GetBiquadType();
        dst->biquadValue = src.GetBiquadValue();
    }
    else
    {
        dst->lpfFreq = 64;
        dst->biquadType = 0;
        dst->biquadValue = 0;
    }

    return true;
}

bool WaveSoundFileReader::ReadNoteInfo(
        WaveSoundNoteInfo* dst,
        uint32_t index,
        uint32_t noteIndex ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( dst );

    const WaveSoundFile::NoteInfo& src =
        m_pInfoBlockBody->GetWaveSoundData( index ).GetNoteInfo( noteIndex );

    const Util::WaveId* pWaveId =
        m_pInfoBlockBody->GetWaveIdTable().GetWaveId( src.waveIdTableIndex );

    if ( pWaveId == NULL )
    {
        return false;
    }

    dst->waveArchiveId = pWaveId->waveArchiveId;
    dst->waveIndex = pWaveId->waveIndex;
    dst->pitch = src.GetPitch();
    dst->adshr = src.GetAdshrCurve();
    dst->originalKey = src.GetOriginalKey();
    dst->pan = src.GetPan();
    dst->surroundPan = src.GetSurroundPan();
    dst->volume = src.GetVolume();
    return true;
}

bool WaveSoundFileReader::IsFilterSupportedVersion() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( m_pHeader );
    if (header.version >= FilterSupportedVersionWsd)
    {
        return true;
    }
    return false;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
