﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_MultiVoice.h>

#include <cstring>              // std::memcpy
#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_Config.h>
#include <nn/atk/atk_MultiVoiceManager.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/atk/atk_DecodeAdpcm.h>

// #define NN_ATK_DUMP_DECODE_DSPADPCM
// ↑↑↑ 有効な場合、「途中再生に必要な DSP ADPCM のデコード」の結果を、
//        CAFE_SAVE_DIR/_cafe_decodeDspAdpcm.raw に書き出します。

#if defined(NN_ATK_DUMP_DECODE_DSPADPCM)
static FSClient   g_fsClient;
static FSCmdBlock g_fsCmdBlock;
#endif

namespace nn {
namespace atk {
namespace detail {
namespace driver {

namespace
{

size_t DspAdpcmFrameToNibbleAddress( position_t frame ) NN_NOEXCEPT
{
    size_t mod = frame % 14;
    size_t nibbleAddress = ( frame / 14 ) * 16;
    if ( mod != 0 )
    {
        nibbleAddress += ( mod + 2 );
    }

    return nibbleAddress;
}

inline void PanCurveToPanInfo( Util::PanInfo& panInfo, PanCurve curve ) NN_NOEXCEPT
{
    switch ( curve )
    {
    case PanCurve_Sqrt:
        panInfo.curve = Util::PanCurve_Sqrt;
        break;
    case PanCurve_Sqrt0Db:
        panInfo.curve = Util::PanCurve_Sqrt;
        panInfo.centerZeroFlag = true;
        break;
    case PanCurve_Sqrt0DbClamp:
        panInfo.curve = Util::PanCurve_Sqrt;
        panInfo.centerZeroFlag = true;
        panInfo.zeroClampFlag = true;
        break;
    case PanCurve_Sincos:
        panInfo.curve = Util::PanCurve_Sincos;
        break;
    case PanCurve_Sincos0Db:
        panInfo.curve = Util::PanCurve_Sincos;
        panInfo.centerZeroFlag = true;
        break;
    case PanCurve_Sincos0DbClamp:
        panInfo.curve = Util::PanCurve_Sincos;
        panInfo.centerZeroFlag = true;
        panInfo.zeroClampFlag = true;
        break;
    case PanCurve_Linear:
        panInfo.curve = Util::PanCurve_Linear;
        break;
    case PanCurve_Linear0Db:
        panInfo.curve = Util::PanCurve_Linear;
        panInfo.centerZeroFlag = true;
        break;
    case PanCurve_Linear0DbClamp:
        panInfo.curve = Util::PanCurve_Linear;
        panInfo.centerZeroFlag = true;
        panInfo.zeroClampFlag = true;
        break;
    default:
        panInfo.curve = Util::PanCurve_Sqrt;
        break;
    }
}

// モノラル・バランス or デュアル用パン計算
inline void CalcPanForMono( float* left, float* right, const Util::PanInfo& panInfo ) NN_NOEXCEPT
{
    *left = *right = Util::CalcPanRatio( MultiVoice::PanCenter, panInfo, OutputMode_Monaural );
}

// ステレオ or サラウンド・バランス用パン計算
inline void CalcBarancePanForStereo(
        float* left, float* right, const float& pan, int channelIndex, const Util::PanInfo& panInfo, OutputMode mode ) NN_NOEXCEPT
{
    if ( channelIndex == 0 )
    {
        *left = Util::CalcPanRatio( pan, panInfo, mode );
        *right = 0.0f;
    }
    else if ( channelIndex == 1 )
    {
        *left = 0.0f;
        *right = Util::CalcPanRatio( MultiVoice::PanCenter - pan, panInfo, mode );
    }
}

// ステレオ or サラウンド用デュアルパン計算
inline void CalcDualPanForStereo(
        float* left, float* right, const float& pan, const Util::PanInfo& panInfo, OutputMode mode ) NN_NOEXCEPT
{
    *left = Util::CalcPanRatio( pan, panInfo, mode );
    *right = Util::CalcPanRatio( MultiVoice::PanCenter - pan, panInfo, mode );
}

// モノラル or ステレオ用サラウンドパン計算
inline void CalcSurroundPanForMono(
        float* front, float* rear, const Util::PanInfo& panInfo ) NN_NOEXCEPT
{
    *front = Util::CalcSurroundPanRatio( MultiVoice::SpanFront, panInfo );
    *rear = Util::CalcSurroundPanRatio( MultiVoice::SpanRear, panInfo );
}

// サラウンド用サラウンドパン計算
inline void CalcSurroundPanForSurround(
        float* front, float* rear, const float& span, const Util::PanInfo& panInfo ) NN_NOEXCEPT
{
    *front = Util::CalcSurroundPanRatio( span, panInfo );
    *rear = Util::CalcSurroundPanRatio( MultiVoice::SpanRear - span, panInfo );
}

inline int GetOutputReceiverMixBufferIndex(const nn::atk::OutputReceiver* pOutputReceiver, int channel, int bus)
{
    return pOutputReceiver->GetChannelCount() * bus + channel;
}

} // anonymous namespace


// ------------------------------------------------------------------------
// 定数
NN_DEFINE_STATIC_CONSTANT( const int MultiVoice::UpdateStart );
NN_DEFINE_STATIC_CONSTANT( const int MultiVoice::UpdatePause );
NN_DEFINE_STATIC_CONSTANT( const int MultiVoice::UpdateSrc );
NN_DEFINE_STATIC_CONSTANT( const int MultiVoice::UpdateMix );
NN_DEFINE_STATIC_CONSTANT( const int MultiVoice::UpdateLpf );
NN_DEFINE_STATIC_CONSTANT( const int MultiVoice::UpdateBiquad );
NN_DEFINE_STATIC_CONSTANT( const int MultiVoice::UpdateVe );
NN_DEFINE_STATIC_CONSTANT( const uint32_t MultiVoice::PriorityNoDrop );

const float MultiVoice::VolumeMin           = 0.0f;
const float MultiVoice::VolumeDefault       = 1.0f;
const float MultiVoice::VolumeMax           = 2.0f;

const float MultiVoice::PanLeft             = -1.0f;
const float MultiVoice::PanCenter           = 0.0f;
const float MultiVoice::PanRight            = 1.0f;

const float MultiVoice::SpanFront           = 0.0f;
const float MultiVoice::SpanCenter          = 1.0f;
const float MultiVoice::SpanRear            = 2.0f;

const float MultiVoice::CutoffFreqMin      = 0.0f;
const float MultiVoice::CutoffFreqMax      = 1.0f;

const float MultiVoice::BiquadValueMin     = 0.0f;
const float MultiVoice::BiquadValueMax     = 1.0f;

const float MultiVoice::SendMin             = 0.0f;
const float MultiVoice::SendMax             = 1.0f;

/* ========================================================================
        inline function
   ======================================================================== */

inline uint16_t CalcMixVolume( float volume ) NN_NOEXCEPT
{
    if ( volume <= 0.f ) return 0UL;

    return static_cast<uint16_t>(
        std::min(
            0x0000ffffUL,
            static_cast<unsigned long>( volume * 0x8000 )
        )
    );
}


MultiVoice::MultiVoice() NN_NOEXCEPT
: m_ChannelCount( 0 )
, m_Callback( NULL )
, m_IsActive( false )
, m_IsStart( false )
, m_IsStarted( false )
, m_IsPause( false )
, m_SyncFlag( 0 )
, m_pTvAdditionalParam( nullptr )
, m_UpdateType( UpdateType_AudioFrame )
, m_pOutputReceiver( nullptr )
{
}

MultiVoice::MultiVoice(OutputAdditionalParam* pAdditionalParam) NN_NOEXCEPT
: m_ChannelCount( 0 )
, m_Callback( NULL )
, m_IsActive( false )
, m_IsStart( false )
, m_IsStarted( false )
, m_IsPause( false )
, m_SyncFlag( 0 )
, m_pTvAdditionalParam( pAdditionalParam )
, m_UpdateType( UpdateType_AudioFrame )
, m_pOutputReceiver( nullptr )
{
}

MultiVoice::~MultiVoice() NN_NOEXCEPT
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[ channelIndex ].Free();
    }
}

bool MultiVoice::Alloc(
    int channelCount,
    int priority,
    MultiVoice::VoiceCallback callback,
    void* callbackData
) NN_NOEXCEPT
{
    NN_SDK_ASSERT( channelCount >= 1 && channelCount <= WaveChannelMax );
    channelCount = nn::atk::detail::fnd::Clamp( channelCount, 1, static_cast<int>(WaveChannelMax) );
    m_ChannelCount = 0;

    NN_SDK_ASSERT( ! m_IsActive );

    // ボイス取得
    int allocCount = 0;
    for ( int i = 0 ; i < channelCount; i++ )
    {
        if ( ! m_Voice[i].AllocVoice(priority) )
        {
            break;
        }
        allocCount++;
    }
    if ( allocCount != channelCount )
    {
        for ( int i = 0 ; i < allocCount; i++ )
        {
            m_Voice[i].Free();
        }
        return false;
    }

    m_ChannelCount = channelCount;

    // パラメータ初期化
    InitParam( callback, callbackData );
    m_IsActive = true;

    return true;
}


void MultiVoice::Free() NN_NOEXCEPT
{
    if ( ! m_IsActive ) return;

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
            m_Voice[ channelIndex ].Free();
    }
    m_ChannelCount = 0;

    MultiVoiceManager::GetInstance().FreeVoice( this );

    m_IsActive = false;
}

void MultiVoice::Start() NN_NOEXCEPT
{
    m_IsStart = true;
    m_IsPause = false;

    m_SyncFlag |= UpdateStart;
}

void MultiVoice::Stop() NN_NOEXCEPT
{
    if ( m_IsStarted )
    {
        StopAllSdkVoice();
        m_IsStarted = false;
    }
    m_IsPausing = false;
    m_IsPause = false;
    m_IsStart = false;
}

void MultiVoice::UpdateVoiceStatus() NN_NOEXCEPT
{
    if ( ! m_IsActive )
    {
        return;
    }

    // 再生停止チェック
    if ( ! m_IsStarted )
    {
        return;
    }

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].UpdateVoiceStatus();
    }
#endif

    if ( IsPlayFinished() )
    {
        // 波形停止
        if ( m_Callback != NULL )
        {
            m_Callback( this, VoiceCallbackStatus_FinishWave, m_pCallbackData );
        }
        m_IsStarted = false;
        m_IsStart = false;
    }
    else
    {
        // ボイスドロップチェック
        bool dropFlag = false;
        for ( int ch = 0 ; ch < m_ChannelCount; ch++ )
        {
            if ( ! m_Voice[ch].IsAvailable() ) {
                dropFlag = true;
                break;
            }
        }
        if ( dropFlag )
        {
            for ( int ch = 0 ; ch < m_ChannelCount; ch++ )
            {
                m_Voice[ch].Free();
            }
            m_ChannelCount = 0;
            m_IsActive = false;

            // TODO: FreeVoiceのタイミングは適切か？
            MultiVoiceManager::GetInstance().FreeVoice( this );
            if ( m_Callback != NULL )
            {
                m_Callback( this, VoiceCallbackStatus_DropDsp, m_pCallbackData );
            }

            m_IsStarted = false;
            m_IsStart = false;
        }
    }
}

void MultiVoice::Pause( bool flag ) NN_NOEXCEPT
{
    if ( m_IsPause == flag ) return;

    m_IsPause = flag;
    m_SyncFlag |= UpdatePause;
}

void MultiVoice::Calc() NN_NOEXCEPT
{
    if ( m_IsStart )
    {
        // 再生レート
        if ( m_SyncFlag & UpdateSrc )
        {
            CalcSrc( false );
            m_SyncFlag &= ~UpdateSrc;
        }

        // ボリューム
        if ( m_SyncFlag & UpdateVe )
        {
            CalcVe();
            m_SyncFlag &= ~UpdateVe;
        }

        // ミックス
        if ( m_SyncFlag & UpdateMix )
        {
            CalcMix();
            m_SyncFlag &= ~UpdateMix;
        }

        // ローパスフィルタ
        if ( m_SyncFlag & UpdateLpf )
        {
            CalcLpf();
            m_SyncFlag &= ~UpdateLpf;
        }

        // Biquadフィルタ
        if ( m_SyncFlag & UpdateBiquad )
        {
            CalcBiquadFilter();
            m_SyncFlag &= ~UpdateBiquad;
        }
    }
}

void MultiVoice::Update() NN_NOEXCEPT
{
    if ( ! m_IsActive ) return;

    // ボイスの再生・停止処理をあとでまとめて行うためのフラグ
    enum { NONE, RUN, PAUSE } runFlag = NONE;

    // 再生開始
    if ( m_SyncFlag & UpdateStart )
    {
        if ( m_IsStart && !m_IsStarted )
        {
            CalcSrc( true );
            CalcMix();
            CalcVe();   // マスターボリューム反映
            runFlag = RUN;
            m_IsStarted = true;

            m_SyncFlag &= ~UpdateStart;
            m_SyncFlag &= ~UpdateSrc;
            m_SyncFlag &= ~UpdateMix;
        }
    }

    if ( m_IsStarted )
    {
        // 一時停止
        if ( m_SyncFlag & UpdatePause )
        {
            if ( m_IsStart )
            {
                if ( m_IsPause )
                {
                    runFlag = PAUSE;
                    m_IsPausing = true;
                }
                else
                {
                    runFlag = RUN;
                    m_IsPausing = false;
                }

                m_SyncFlag &= ~UpdatePause;
            }
        }
    }

    for ( int i = 0; i < m_ChannelCount; i++ )
    {
        m_Voice[i].UpdateParam();
    }

    // ボイスの再生・停止処理をまとめて行う
    // 同一フレームで、まだ再生されていないボイスに対して
    // 再生→停止の処理を行うとデポップが発生するため
    switch ( runFlag )
    {
    case RUN:
        RunAllSdkVoice();
        break;
#if 0 // 未使用のコードであるため無効化。
    case STOP:
        StopAllSdkVoice();
        break;
#endif
    case PAUSE:
        PauseAllSdkVoice();
    case NONE:
    default:
        break;
    }
}

void MultiVoice::SetSampleFormat( SampleFormat format ) NN_NOEXCEPT
{
    m_Format = format;
    for ( int i = 0; i < m_ChannelCount; i++ )
    {
        m_Voice[i].SetSampleFormat( m_Format );
    }
}

void MultiVoice::SetSampleRate( int sampleRate ) NN_NOEXCEPT
{
    for ( int i = 0; i < m_ChannelCount; i++ )
    {
        m_Voice[i].SetSampleRate( sampleRate );
    }
}

void MultiVoice::SetVolume( float volume ) NN_NOEXCEPT
{
    if ( volume < VolumeMin ) volume = VolumeMin;

    if ( volume != m_Volume )
    {
        m_Volume = volume;
        m_SyncFlag |= UpdateVe;
    }
}

void MultiVoice::SetPitch( float pitch ) NN_NOEXCEPT
{
    if ( pitch != m_Pitch )
    {
        m_Pitch = pitch;
        m_SyncFlag |= UpdateSrc;
    }
}

void MultiVoice::SetPanMode( PanMode panMode ) NN_NOEXCEPT
{
    if ( panMode != m_PanMode )
    {
        m_PanMode = panMode;
        m_SyncFlag |= UpdateMix;
    }
}

void MultiVoice::SetPanCurve( PanCurve panCurve ) NN_NOEXCEPT
{
    if ( panCurve != m_PanCurve )
    {
        m_PanCurve = panCurve;
        m_SyncFlag |= UpdateMix;
    }
}


void MultiVoice::SetLpfFreq( float lpfFreq ) NN_NOEXCEPT
{
    if ( lpfFreq != m_LpfFreq )
    {
        m_LpfFreq = lpfFreq;
        m_SyncFlag |= UpdateLpf;
    }
}

void MultiVoice::SetBiquadFilter( int type, float value ) NN_NOEXCEPT
{
    // ここに来るまでに、INHERITの処理は済んでいるはずなので、BiquadFilterType_DataMinが最小になるはずです。
    NN_SDK_ASSERT( type >= BiquadFilterType_DataMin && type <= BiquadFilterType_Max );
    value = nn::atk::detail::fnd::Clamp( value, BiquadValueMin, BiquadValueMax );

    bool isUpdate = false;

    if ( type != m_BiquadType )
    {
        m_BiquadType = static_cast<uint8_t>( type );
        isUpdate = true;
    }

    if ( value != m_BiquadValue )
    {
        m_BiquadValue = value;
        isUpdate = true;
    }

    if ( isUpdate )
    {
        m_SyncFlag |= UpdateBiquad;
    }
}

void MultiVoice::SetPriority( int priority ) NN_NOEXCEPT
{
    m_Priority = priority;
    MultiVoiceManager::GetInstance().ChangeVoicePriority( this );

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetPriority( m_Priority );
    }
}

void MultiVoice::SetOutputLine( uint32_t lineFlag ) NN_NOEXCEPT
{
    if ( lineFlag != m_OutputLineFlag )
    {
        m_OutputLineFlag = lineFlag;
        m_SyncFlag |= UpdateMix;
    }
}

void MultiVoice::SetOutputParamImpl( const OutputParam& in, OutputParam& out ) NN_NOEXCEPT
{
    // volume
    {
        float volume = in.volume;
        if ( volume < VolumeMin )
        {
            volume = VolumeMin;
        }
        if ( volume != out.volume )
        {
            out.volume = volume;
            m_SyncFlag |= UpdateMix;
        }
    }

    // mixMode
    if ( in.mixMode != out.mixMode )
    {
        out.mixMode = in.mixMode;
        m_SyncFlag |= UpdateMix;
    }

    // mixParameter
    if ( out.mixMode == MixMode_Mixparameter )
    {
        for ( int i = 0; i < WaveChannelMax; i++ )
        {
            for ( int j = 0; j < ChannelIndex_Count; j++ )
            {
                if ( in.mixParameter[i].ch[j] != out.mixParameter[i].ch[j] )
                {
                    out.mixParameter[i].ch[j] = in.mixParameter[i].ch[j];
                    m_SyncFlag |= UpdateMix;
                }
            }
        }
    }

    // pan, span
    if ( out.mixMode == MixMode_Pan )
    {
        if ( in.pan != out.pan )
        {
            out.pan = in.pan;
            m_SyncFlag |= UpdateMix;
        }

        if ( in.span != out.span )
        {
            out.span = in.span;
            m_SyncFlag |= UpdateMix;
        }
    }

    // send
    for ( int i = 0; i < DefaultBusCount; i++ )
    {
        float send = in.send[i];
        if ( send != out.send[i] )
        {
            out.send[i] = send;
            m_SyncFlag |= UpdateMix;
        }
    }
}

void MultiVoice::SetOutputAdditionalParamImpl(const SendArray* const pAdditionalSend, const BusMixVolumePacket* const pBusMixVolumePacket, const OutputBusMixVolume* const pBusMixVolume, const VolumeThroughModePacket* const pVolumeThroughModePacket) NN_NOEXCEPT
{
    if ( m_pTvAdditionalParam == nullptr )
    {
        return;
    }

    // Additional Send
    if(m_pTvAdditionalParam->IsAdditionalSendEnabled() && pAdditionalSend != nullptr)
    {
        const int BusCount = m_pOutputReceiver->GetBusCount();
        for( int i = DefaultBusCount; i < BusCount; ++i )
        {
            float send = pAdditionalSend->TryGetValue(Util::GetAdditionalSendIndex(i));
            if( send != m_pTvAdditionalParam->TryGetAdditionalSend(i) )
            {
                m_pTvAdditionalParam->TrySetAdditionalSend(i, send);
                m_SyncFlag |= UpdateMix;
            }
        }
    }

    // BusMixVolume
    if (m_pTvAdditionalParam->IsBusMixVolumeEnabled() && pBusMixVolumePacket != nullptr && pBusMixVolume != nullptr)
    {
        SetOutputBusMixVolumeImpl( *pBusMixVolumePacket, *pBusMixVolume, *m_pTvAdditionalParam->GetBusMixVolumePacketAddr() );
    }

    // VolumeThroughModePacket
    if (m_pTvAdditionalParam->IsVolumeThroughModeEnabled() && pVolumeThroughModePacket != nullptr)
    {
        SetOutputVolumeThroughModePacketImpl( *pVolumeThroughModePacket, *m_pTvAdditionalParam->GetVolumeThroughModePacketAddr() );
    }
}

void MultiVoice::SetOutputBusMixVolumeImpl( const BusMixVolumePacket& in, const OutputBusMixVolume& busMixVolume, BusMixVolumePacket& out) NN_NOEXCEPT
{
    if ( in.IsUsed() != out.IsUsed() )
    {
        out.SetUsed( in.IsUsed() );
        m_SyncFlag |= UpdateMix;
    }

    // busMixVolume
    if ( out.IsUsed() )
    {
        const int BusCount = std::min(in.GetBusCount(), out.GetBusCount());
        for ( int i = 0; i < BusCount; ++i )
        {
            const bool IsEnabled = in.IsEnabled(i);
            if( IsEnabled != out.IsEnabled(i) )
            {
                out.SetEnabled( i, IsEnabled );
                m_SyncFlag |= UpdateMix;
            }
        }

        const int MixBufferCount = m_pOutputReceiver->GetBusCount() * m_pOutputReceiver->GetChannelCount();
        for ( int i = 0; i < WaveChannelMax; i++ )
        {
            for ( int j = 0; j < MixBufferCount; j++ )
            {
                if ( busMixVolume.volume[i][j] != out.GetBusMixVolume(i, j) )
                {
                    out.SetBusMixVolume(i, j, busMixVolume.volume[i][j]);
                    m_SyncFlag |= UpdateMix;
                }
            }
        }
    }
}

void MultiVoice::SetOutputVolumeThroughModePacketImpl(const VolumeThroughModePacket& in, VolumeThroughModePacket& out) NN_NOEXCEPT
{
    // Used
    bool isChanged = false;
    const bool volumeThroughModeUsed = in.IsVolumeThroughModeUsed();
    if( volumeThroughModeUsed != out.IsVolumeThroughModeUsed() )
    {
        out.SetVolumeThroughModeUsed(volumeThroughModeUsed);
        isChanged = true;
    }
    if( !out.IsVolumeThroughModeUsed() )
    {
        if(isChanged)
        {
            m_SyncFlag |= UpdateMix;
            m_SyncFlag |= UpdateVe;
            return;
        }
    }

    // BinaryVolume
    float volume = in.GetBinaryVolume();
    if( volume < VolumeMin )
    {
        volume = VolumeMin;
    }
    if( volume != out.GetBinaryVolume() )
    {
        out.SetBinaryVolume(volume);
        m_SyncFlag |= UpdateMix;
        m_SyncFlag |= UpdateVe;
    }

    // VolumeThroughMode
    const int BusCount = m_pOutputReceiver->GetBusCount();
    for( int i = 0; i < BusCount; ++i )
    {
        uint8_t volumeThroughMode = in.TryGetVolumeThroughMode(i);
        if( volumeThroughMode != out.TryGetVolumeThroughMode(i) )
        {
            out.TrySetVolumeThroughMode(i,volumeThroughMode);
            m_SyncFlag |= UpdateMix;
        }
    }
}

void MultiVoice::SetTvParam( const OutputParam& param ) NN_NOEXCEPT
{
    SetOutputParamImpl( param, m_TvParam );
}

void MultiVoice::SetTvAdditionalParam(const OutputAdditionalParam& param) NN_NOEXCEPT
{
    if(param.IsBusMixVolumeEnabled())
    {
        SetOutputAdditionalParamImpl(param.GetAdditionalSendAddr(), param.GetBusMixVolumePacketAddr(), &param.GetBusMixVolume(), param.GetVolumeThroughModePacketAddr());
    }
    else
    {
        SetOutputAdditionalParamImpl(param.GetAdditionalSendAddr(), param.GetBusMixVolumePacketAddr(), nullptr, param.GetVolumeThroughModePacketAddr());
    }
}

void MultiVoice::SetTvAdditionalParam(const SendArray* const pAdditionalSend, const BusMixVolumePacket* const pBusMixVolumePacket, const OutputBusMixVolume* const pBusMixVolume, const VolumeThroughModePacket* const pVolumeThroughModePacket) NN_NOEXCEPT
{
    SetOutputAdditionalParamImpl(pAdditionalSend, pBusMixVolumePacket, pBusMixVolume, pVolumeThroughModePacket);
}

void MultiVoice::SetOutputReceiver( OutputReceiver* pOutputReceiver ) NN_NOEXCEPT
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetOutputReceiver( pOutputReceiver );
        m_pOutputReceiver = pOutputReceiver;
    }
}

void MultiVoice::InitParam(
    MultiVoice::VoiceCallback callback,
    void* callbackData
) NN_NOEXCEPT
{
    // Init Flag
    m_SyncFlag  = 0;
    m_IsPause   = false;
    m_IsPausing = false;
    m_IsStart   = false;
    m_IsStarted = false;

    // Init Param
    m_pLastWaveBuffer = NULL;
    m_Callback        = callback;
    m_pCallbackData   = callbackData;
    m_Volume          = VolumeDefault;
    m_Pitch           = 1.0f;
    m_PanMode         = PanMode_Dual;
    m_PanCurve        = PanCurve_Sqrt;
    m_LpfFreq         = 1.0f;
    m_BiquadType      = BiquadFilterType_None;
    m_BiquadValue     = 0.0f;
    m_IsEnableFrontBypass = false;

    m_TvParam.Initialize();
    if(m_pTvAdditionalParam != nullptr)
    {
        m_pTvAdditionalParam->Reset();
    }
}

void MultiVoice::CalcSrc( bool initialUpdate ) NN_NOEXCEPT
{
    NN_UNUSED( initialUpdate );

    float ratio = m_Pitch;

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetPitch( ratio );
    }
}

void MultiVoice::CalcVe() NN_NOEXCEPT
{
    float volume = m_Volume;
    volume *= HardwareManager::GetInstance().GetOutputVolume();

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetVolume( volume );
    }
}

void MultiVoice::CalcMix() NN_NOEXCEPT
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        Voice& voice = m_Voice[ channelIndex ];

        PreMixVolume preMix;
        // TV
        {
            CalcPreMixVolume( &preMix, m_TvParam, m_pTvAdditionalParam, channelIndex, OutputDevice_Main );
            OutputMix tvMix;
            tvMix.Initialize();
            CalcTvMix( &tvMix, preMix );
            voice.SetTvMix( tvMix );
        }
    }
}

void MultiVoice::CalcLpf() NN_NOEXCEPT
{
    uint16_t freq = Util::CalcLpfFreq( m_LpfFreq );

    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        Voice& voice = m_Voice[channelIndex];

        if ( freq >= 16000 )
        {
            voice.SetMonoFilter( false );
        }
        else
        {
            voice.SetMonoFilter( true, freq );
        }

    }
}

void MultiVoice::CalcBiquadFilter() NN_NOEXCEPT
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        Voice& voice = m_Voice[channelIndex];

        const BiquadFilterCallback* cb =
            HardwareManager::GetInstance().GetBiquadFilterCallback( m_BiquadType );
        if ( cb == NULL )
        {
            voice.SetBiquadFilter( false );
        }
        else
        {
            if ( m_BiquadValue <= 0.0f )
            {
                voice.SetBiquadFilter( false );
            }
            else
            {
                BiquadFilterCallback::Coefficients coef;
                cb->GetCoefficients( &coef, m_BiquadType, m_BiquadValue );
                voice.SetBiquadFilter( true, &coef );
            }
        }
    }
}

void MultiVoice::CalcPreMixVolume(
        PreMixVolume* mix,
        const OutputParam& param,
        const OutputAdditionalParam* const pAdditionalParam,
        int channelIndex,
        OutputDevice device ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( mix );

    // NW OutputMode に応じたミックスパラメータの計算
    const OutputMode nwMode = HardwareManager::GetInstance().GetOutputMode(device);
    float fL = 0.0f; float fR = 0.0f; float rL = 0.0f; float rR = 0.0f;
    float fC = 0.0f; float lfe = 0.0f;
    if ( param.mixMode == MixMode_Pan )
    {
        float left, right, front, rear;
        left = right = front = rear = 0.0f;

        Util::PanInfo panInfo;
        PanCurveToPanInfo( panInfo, m_PanCurve );


        // left, right の計算
        switch ( nwMode )
        {
        case OutputMode_Monaural:
            CalcPanForMono( &left, &right, panInfo );
            break;
        case OutputMode_Stereo:
        case OutputMode_Surround:
            if ( ( m_ChannelCount > 1 ) && ( m_PanMode == PanMode_Balance ) )
            {
                CalcBarancePanForStereo(
                    &left, &right, param.pan, channelIndex, panInfo, nwMode );
            }
            else
            {
                float voicePan = param.pan;
                if ( m_ChannelCount == 2 )
                {
                    if ( channelIndex == 0 ) { voicePan -= 1.0f; }
                    if ( channelIndex == 1 ) { voicePan += 1.0f; }
                }
                CalcDualPanForStereo( &left, &right, voicePan, panInfo, nwMode );
            }
            break;
        default:
            NN_SDK_ASSERT(false, "[%s] nwMode(%d) is invalid\n", __FUNCTION__, nwMode);
            break;
        }

        // front, rear の計算
        switch ( nwMode )
        {
        case OutputMode_Monaural:
        case OutputMode_Stereo:
            CalcSurroundPanForMono( &front, &rear, panInfo );
            break;
        case OutputMode_Surround:
            CalcSurroundPanForSurround( &front, &rear, param.span, panInfo );
            break;
        default:
            break;
        }

        fL = front * left;
        fR = front * right;
        rL = rear  * left;
        rR = rear  * right;
    }
    else
    {
        fL  = param.mixParameter[channelIndex].ch[ChannelIndex_FrontLeft];
        fR  = param.mixParameter[channelIndex].ch[ChannelIndex_FrontRight];
        rL  = param.mixParameter[channelIndex].ch[ChannelIndex_RearLeft];
        rR  = param.mixParameter[channelIndex].ch[ChannelIndex_RearRight];
        fC  = param.mixParameter[channelIndex].ch[ChannelIndex_FrontCenter];
        lfe = param.mixParameter[channelIndex].ch[ChannelIndex_Lfe];
    }
    // NN_DETAIL_ATK_INFO("[0] fL(%5.2f) fR(%5.2f) rL(%5.2f) rR(%5.2f)\n", fL, fR, rL, rR);

    // EndUserOutputMode に応じたダウンミックス
    // -----------------------------------------------------
    // EndUserOutput   |            NW mode
    // mode            | MONO   STEREO      SURROUND
    // -----------------------------------------------------
    // MONO       fL   |  M     L+R/2    (fL+rL + fR+rR)/2
    //            fR   |  M     L+R/2    (fL+rL + fR+rR)/2
    //            rL   |  0      0            0
    //            rR   |  0      0            0
    //            C    |  0      0            0
    //            LFE  |  0      0            0
    // -----------------------------------------------------
    // STEREO     fL   |  M      L          fL+rL
    //            fR   |  M      R          fR+rR
    //            rL   |  0      0            0
    //            rR   |  0      0            0
    //            C    |  0      0            0
    //            LFE  |  0      0            0
    // -----------------------------------------------------
    // SURROUND   fL   |  M      L           fL
    //            fR   |  M      R           fR
    //            rL   |  0      0           rL
    //            rR   |  0      0           rR
    //            C    |  0      0            0
    //            LFE  |  0      0            0
    // -----------------------------------------------------
    const OutputMode endUserMode =
        HardwareManager::GetInstance().GetEndUserOutputMode(device);
    switch (endUserMode)
    {
    case OutputMode_Monaural:
        switch (nwMode)
        {
            // case OutputMode_Mono:
            //     // do nothing
            //     break;
        case OutputMode_Stereo:
        {
            float mono = (fL + fR) * 0.5f;
            fL = fR = mono;
        }
        break;
        case OutputMode_Surround:
        {
            float mono = ((fL + rL) + (fR + rR)) * 0.5f;
            fL = fR = mono;
            rL = rR = 0.0f;
        }
        break;
        default:
            break;
        }
        break;
    case OutputMode_Stereo:
        switch (nwMode)
        {
            // case OutputMode_Mono:
            //     // do nothing
            //     break;
            // case OutputMode_Stereo:
            //     // do nothing
            //     break;
        case OutputMode_Surround:
        {
            float left = fL + rL;
            float right = fR + rR;
            fL = left;
            fR = right;
            rL = rR = 0.0f;
        }
        break;
        default:
            break;
        }
        break;
    case OutputMode_Surround:
        // switch (nwMode)
        // {
        // case OutputMode_Mono:
        //     // do nothing
        //     break;
        // case OutputMode_Stereo:
        //     // do nothing
        //     break;
        // case OutputMode_Surround:
        //     // do nothing
        //     break;
        // }
        break;
    default:
        NN_SDK_ASSERT(false, "[%s] endUserMode(%d) is invalid\n", __FUNCTION__, endUserMode);
        break;
    }
    // NN_DETAIL_ATK_INFO("[1] fL(%5.2f) fR(%5.2f) rL(%5.2f) rR(%5.2f)\n", fL, fR, rL, rR);

    // 値の格納
    {
        for(int i = 0; i < OutputMix::ChannelCountMax; i++)
        {
            const int BusIndex = i / m_pOutputReceiver->GetChannelCount();

            if( BusIndex >= m_pOutputReceiver->GetBusCount() )
            {
                mix->volume[i] = 0.0f;
                continue;
            }

            if(pAdditionalParam != nullptr && pAdditionalParam->IsBusMixVolumeEnabled() )
            {
                if( pAdditionalParam->IsBusMixVolumeUsed() && pAdditionalParam->IsBusMixVolumeEnabledForBus(BusIndex) )
                {
                    mix->volume[i] = pAdditionalParam->GetBusMixVolume(channelIndex, i);
                    continue;
                }
            }

            const int ChannelIndex = i % m_pOutputReceiver->GetChannelCount();

            switch( ChannelIndex )
            {
            case ChannelIndex_FrontLeft:
                mix->volume[i] = fL;
                break;
            case ChannelIndex_FrontRight:
                mix->volume[i] = fR;
                break;
            case ChannelIndex_RearLeft:
                mix->volume[i] = rL;
                break;
            case ChannelIndex_RearRight:
                mix->volume[i] = rR;
                break;
            case ChannelIndex_FrontCenter:
                mix->volume[i] = fC;
                break;
            case ChannelIndex_Lfe:
                mix->volume[i] = lfe;
                break;
            default:
                mix->volume[i] = 0.0f;
                break;
            }
        }
    }
} // NOLINT(impl/function_size)

void MultiVoice::CalcTvMix( OutputMix* mix, const PreMixVolume& preMix ) NN_NOEXCEPT
{
    CalcMixImpl( mix, OutputDeviceIndex_Main, m_TvParam, m_pTvAdditionalParam, preMix );
}

void MultiVoice::CalcMixImpl(
        OutputMix* mix,
        uint32_t outputDeviceIndex,
        const OutputParam& param,
        const OutputAdditionalParam* const pAdditionalParam,
        const PreMixVolume& pre) NN_NOEXCEPT
{
    // send の初期化
    float sendVolume[OutputReceiver::BusCountMax];
    const int BusCount = m_pOutputReceiver->GetBusCount();
    for(int i = 0; i < BusCount; i++)
    {
        sendVolume[i] = 0.0f;
    }

    uint32_t outputDeviceFlag = 0;
    for ( int i = 0; i < OutputLineIndex_Max; i++ )
    {
        if ( ( m_OutputLineFlag >> i ) & 1 )
        {
            outputDeviceFlag |= HardwareManager::GetInstance().GetOutputDeviceFlag( i );
        }
    }

    // send 量の設定
    if ( outputDeviceFlag & ( 1 << outputDeviceIndex ) )
    {
        for ( int i = 0; i < BusCount; i++ )
        {
            if (i < DefaultBusCount)
            {
                const float actualSend = (i == Util::GetSubMixBusFromMainBus()) ? param.send[i] + 1.0f : param.send[i];
                if(m_pOutputReceiver->IsSoundSendClampEnabled(i))
                {
                    sendVolume[i] = nn::atk::detail::fnd::Clamp( actualSend, 0.0f, 1.0f );
                }
                else
                {
                    sendVolume[i] = std::max( actualSend, 0.0f );
                }
            }
            else
            {
                if(pAdditionalParam == nullptr || !pAdditionalParam->IsAdditionalSendEnabled())
                {
                    break;
                }
                if(m_pOutputReceiver->IsSoundSendClampEnabled(i))
                {
                    sendVolume[i] = nn::atk::detail::fnd::Clamp( pAdditionalParam->TryGetAdditionalSend(i), 0.0f, 1.0f );
                }
                else
                {
                    sendVolume[i] = std::max( pAdditionalParam->TryGetAdditionalSend(i), 0.0f );
                }
            }
        }
    }

    // Voice の各チャンネルに送る mix の計算
    for (int bus = 0; bus < BusCount; bus++)
    {
        for (int voiceChannel = 0; voiceChannel < std::min(static_cast<int>(ChannelIndex_Count), m_pOutputReceiver->GetChannelCount()); voiceChannel++)
        {
            const int MixBufferIndex = GetOutputReceiverMixBufferIndex(m_pOutputReceiver, voiceChannel, bus);

            float volume = param.volume * sendVolume[bus] * pre.volume[MixBufferIndex];

            if( pAdditionalParam != nullptr && pAdditionalParam->IsVolumeThroughModeUsed())
            {
                const bool IsThroughBinaryVolume = pAdditionalParam->TryGetVolumeThroughMode(bus) & VolumeThroughMode_Binary;
                if( !IsThroughBinaryVolume )
                {
                    volume *= pAdditionalParam->GetBinaryVolume();
                }
            }

            mix->channelGain[MixBufferIndex] = volume;
        }
    }
}

void MultiVoice::RunAllSdkVoice() NN_NOEXCEPT
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetState( VoiceState_Play );
    }
}

void MultiVoice::StopAllSdkVoice() NN_NOEXCEPT
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetState( VoiceState_Stop );
    }
}

void MultiVoice::PauseAllSdkVoice() NN_NOEXCEPT
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetState( VoiceState_Pause );
    }
}

const Voice& MultiVoice::detail_GetSdkVoice( int channelIndex ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( channelIndex < WaveChannelMax );
    return m_Voice[ channelIndex ];
}

position_t MultiVoice::GetCurrentPlayingSample() const NN_NOEXCEPT
{
    if ( ! IsActive() ) return 0;
    return m_Voice[0].GetPlayPosition();
}


nn::atk::SampleFormat MultiVoice::GetFormat() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsActive() );
    return m_Format;
}

bool MultiVoice::IsRun() const NN_NOEXCEPT
{
    if ( IsActive() )
    {
        return ( m_Voice[0].GetState() == VoiceState_Play );
    }
    return false;
}


bool MultiVoice::IsPlayFinished() const NN_NOEXCEPT
{
    if ( ! IsActive() ) return false;
    if ( m_pLastWaveBuffer == NULL ) return false;
    if ( m_pLastWaveBuffer->status == WaveBuffer::Status_Wait || m_pLastWaveBuffer->status == WaveBuffer::Status_Play ) return false;

    return true;
}

void MultiVoice::SetInterpolationType( uint8_t interpolationType ) NN_NOEXCEPT
{
    for ( int channelIndex = 0; channelIndex < m_ChannelCount; channelIndex++ )
    {
        m_Voice[channelIndex].SetInterpolationType( interpolationType );
    }
}

void MultiVoice::AppendWaveBuffer(
    int channelIndex,
    WaveBuffer* pBuffer,
    bool lastFlag
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pBuffer );
    NN_SDK_ASSERT_RANGE( channelIndex, 0, static_cast<int>(WaveChannelMax) );

    m_Voice[channelIndex].AppendWaveBuffer( pBuffer );

#if 0 // DEBUG: ADPCM コンテキストの確認
    if ( pBuffer->pAdpcmContext )
    {
        NN_DETAIL_ATK_INFO("MultiVoice[%p]  bufferPS(%04x) ps(%04x) yn1(%6d) yn2(%6d)\n",
                this,
                reinterpret_cast<const uint8_t*>(pBuffer->bufferAddress)[0],
                pBuffer->pAdpcmContext->pred_scale,
                pBuffer->pAdpcmContext->yn1,
                pBuffer->pAdpcmContext->yn2);
    }
    else
    {
        NN_DETAIL_ATK_INFO("MultiVoice[%p]  bufferPS(%04x)\n", this,
                reinterpret_cast<const uint8_t*>(pBuffer->bufferAddress)[0] );
    }
#endif

    if ( lastFlag )
    {
        m_pLastWaveBuffer = pBuffer;
    }
}

void MultiVoice::SetAdpcmParam(
        int channelIndex,
        const AdpcmParam& param ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( channelIndex, 0, static_cast<int>(WaveChannelMax) );

    m_Voice[channelIndex].SetAdpcmParam( param );
}

size_t MultiVoice::FrameToByte( position_t sample, nn::atk::SampleFormat format ) NN_NOEXCEPT
{
    size_t result = 0;

    switch ( format )
    {
    case nn::atk::SampleFormat_DspAdpcm:
        result = DspAdpcmFrameToNibbleAddress( sample ) >> 1;
        break;
    case nn::atk::SampleFormat_PcmS8:
        result = sample;
        break;
    case nn::atk::SampleFormat_PcmS16:
        result = ( sample << 1 );
        break;
    default:
        NN_SDK_ASSERT( false, "Invalid format\n" );
    }

    return result;
}

void MultiVoice::CalcOffsetAdpcmParam(
    nn::atk::AdpcmContext* context, // in/out
    const nn::atk::AdpcmParam& param,
    position_t offsetSamples,
    const void* dataAddress
) NN_NOEXCEPT
{
#ifdef NN_ATK_DUMP_DECODE_DSPADPCM
    NN_DETAIL_ATK_INFO("[%s] ctx:ps(%5d) yn1(%d) yn2(%d) offset:(%d) adr:(%p)\n",
            __FUNCTION__, context->pred_scale, context->yn1, context->yn2,
            offsetSamples, dataAddress);

    FSAddClient(&g_fsClient, FS_RET_ALL_ERROR);
    FSInitCmdBlock(&g_fsCmdBlock);

    FSAStatus status;
    status = FSChangeDir( &g_fsClient, &g_fsCmdBlock, "", FS_RET_ALL_ERROR );
    NN_DETAIL_ATK_INFO("[FS] ChangeDir(%d)\n", status);

    FSFileHandle fileHandle;
    status = FSOpenFile(
        &g_fsClient,
        &g_fsCmdBlock,
        "_cafe_decodeDspAdpcm.raw",
        "w",
        &fileHandle,
        FS_RET_ALL_ERROR);
    NN_DETAIL_ATK_INFO("[FS] OpenFile(%d)\n", status );

    const uint32_t DumpBufferSize = 4 * 1024 + 256;
    int16_t dumpBuffer_[ DumpBufferSize ];
    int16_t* dumpBuffer = reinterpret_cast<int16_t*>(util::BytePtr( dumpBuffer_ ).AlignUp( 256 ).Get());
    int16_t dumpBufferCount = 0;
    NN_DETAIL_ATK_INFO("  offsetSamples(%d)\n", offsetSamples );
#endif

    // int32_t AXDecodeAdpcmData(
    //     uint8_t* input,
    //     DSPADPCM* info,
    //     int32_t       samples,
    //     int16_t*      output);

#if 0 // 参考用 : NW4F 実装
    uint32_t currentSample = 0;
    const uint8_t* InputAddress = reinterpret_cast<const uint8_t*>(dataAddress);
    const uint8_t* PredScale = reinterpret_cast<const uint8_t*>(dataAddress);
    const uint32_t DecodeSampleCount = 14; // 14 サンプル単位でデコードする
    const uint32_t DecodeByteSize = 8;     // 8 バイト (=14サンプル) 単位でデコードする
    NN_SDK_ASSERT( offsetSamples % DecodeSampleCount == 0 );

    DSPADPCM adpcm = {0};
    for ( int i = 0; i < 16; i++ )
    {
        adpcm.coef[i] = param.coefficients[i];
    }
    adpcm.ps = context->predScale;
    adpcm.yn1 = static_cast<int16_t>(context->history[0]);
    adpcm.yn2 = static_cast<int16_t>(context->history[1]);
    adpcm.num_samples = DecodeSampleCount;
    adpcm.num_adpcm_nibbles = DecodeByteSize * 2;
    adpcm.sa = 2;
    adpcm.ea = 15;
    adpcm.ca = 2;

    int16_t output[DecodeSampleCount] = {0};
    while ( currentSample < offsetSamples )
    {
        AXDecodeAdpcmData(
            InputAddress,
            &adpcm,
            DecodeSampleCount,
            output );

    #ifdef NN_ATK_DUMP_DECODE_DSPADPCM
        for ( int i = 0; i < DecodeSampleCount; i++ )
        {
            dumpBuffer[dumpBufferCount] = output[i];
            dumpBufferCount++;
            if ( dumpBufferCount == DumpBufferSize )
            {
                status = FSWriteFile(
                    &g_fsClient,
                    &g_fsCmdBlock,
                    dumpBuffer,
                    sizeof(int16_t) * DumpBufferSize,
                    1,
                    fileHandle,
                    0,
                    FS_RET_ALL_ERROR);
                NN_DETAIL_ATK_INFO("[FS] Write(%d)\n", status );
                dumpBufferCount = 0;
            }
        }
    #endif
        currentSample += DecodeSampleCount;
        PredScale =
            reinterpret_cast<const uint8_t*>( util::ConstBytePtr( PredScale, DecodeByteSize ).Get() );

        // ps, yn1, yn2 更新
        adpcm.ps = static_cast<uint16_t>(PredScale[0]);
        adpcm.yn1 = static_cast<uint16_t>(output[DecodeSampleCount - 1]);
        adpcm.yn2 = static_cast<uint16_t>(output[DecodeSampleCount - 2]);
        adpcm.sa += 16;
        adpcm.ea += 16;
        adpcm.ca += 16;
    }

    // コンテキストの更新
    {
        context->predScale  = adpcm.ps;
        context->history[0] = output[DecodeSampleCount - 1];
        context->history[1] = output[DecodeSampleCount - 2];
    }
#else
    position_t currentSample = 0;
    const uint8_t* InputAddress = reinterpret_cast<const uint8_t*>(dataAddress);
    const uint32_t DecodeSampleCount = 14; // 14 サンプル単位でデコードする
    NN_SDK_ASSERT( offsetSamples % DecodeSampleCount == 0 );

    int16_t output[DecodeSampleCount] = {0};
    while ( currentSample < offsetSamples )
    {
        DecodeDspAdpcm(currentSample, *context, param, InputAddress, DecodeSampleCount, output);

#ifdef NN_ATK_DUMP_DECODE_DSPADPCM
        for ( int i = 0; i < DecodeSampleCount; i++ )
        {
            dumpBuffer[dumpBufferCount] = output[i];
            dumpBufferCount++;
            if ( dumpBufferCount == DumpBufferSize )
            {
                status = FSWriteFile(
                    &g_fsClient,
                    &g_fsCmdBlock,
                    dumpBuffer,
                    sizeof(int16_t) * DumpBufferSize,
                    1,
                    fileHandle,
                    0,
                    FS_RET_ALL_ERROR);
                NN_DETAIL_ATK_INFO("[FS] Write(%d)\n", status );
                dumpBufferCount = 0;
            }
        }
#endif
        currentSample += DecodeSampleCount;
    }
#endif

#ifdef NN_ATK_DUMP_DECODE_DSPADPCM
    if ( dumpBufferCount > 0 )
    {
        status = FSWriteFile(
                &g_fsClient,
                &g_fsCmdBlock,
                dumpBuffer,
                sizeof(int16_t) * dumpBufferCount,
                1,
                fileHandle,
                0,
                FS_RET_ALL_ERROR);
        NN_DETAIL_ATK_INFO("[FS] Write(%d)\n", status );
    }
    status = FSCloseFile( &g_fsClient, &g_fsCmdBlock, fileHandle, FS_RET_ALL_ERROR );
    NN_DETAIL_ATK_INFO("[FS] Close(%d)\n", status);

    status = FSDelClient( &g_fsClient, FS_RET_ALL_ERROR );
    NN_DETAIL_ATK_INFO("[FS] DelClient(%d)\n", status );
#endif
} // NOLINT(impl/function_size)

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
