﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_FrameHeap.h>

#include <nn/atk/atk_BankFileReader.h>
#include <nn/atk/atk_GroupFileReader.h>
#include <nn/atk/atk_SequenceSoundFileReader.h>
#include <nn/atk/atk_WaveArchiveFileReader.h>
#include <nn/atk/atk_WaveFileReader.h>
#include <nn/atk/atk_WaveSoundFileReader.h>
#include <nn/atk/atk_Util.h>
#include <new>

namespace nn {
namespace atk {
namespace detail {

namespace
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
uint32_t GetWaveArchiveId( uint32_t fileId, const SoundArchive& arc ) NN_NOEXCEPT
{
    for ( uint32_t j = 0; j < arc.GetWaveArchiveCount(); j++ )
    {
        SoundArchive::WaveArchiveInfo info;
        SoundArchive::ItemId id =
            Util::GetMaskedItemId( j, ItemType_WaveArchive );
        if ( arc.ReadWaveArchiveInfo( id, &info ) )
        {
            if ( info.fileId == fileId )
            {
                return id;
            }
        }
    }
    return SoundArchive::InvalidId;
}
#endif // NN_SDK_BUILD_DEBUG || NN_SDK_BUILD_DEVELOP

} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const int FrameHeap::HeapAlign );

/* ========================================================================
        member function
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         FrameHeap

  Description:  コンストラクタ

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
FrameHeap::FrameHeap() NN_NOEXCEPT
: m_pHeap( NULL )
{
}

/*--------------------------------------------------------------------------------*
  Name:         ~FrameHeap

  Description:  デストラクタ

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
FrameHeap::~FrameHeap() NN_NOEXCEPT
{
    if ( IsValid() )
    {
        Destroy();
    }
}

/*--------------------------------------------------------------------------------*
  Name:         Create

  Description:  ヒープを作成

  Arguments:    startAddress - 開始アドレス
                size         - メモリサイズ

  Returns:      ヒープハンドル
 *--------------------------------------------------------------------------------*/
bool FrameHeap::Create( void* startAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( startAddress );

    if ( IsValid() )
    {
        Destroy();
    }

    void* endAddress = static_cast<uint8_t*>( startAddress ) + size;
    startAddress = util::BytePtr( startAddress ).AlignUp( 4 ).Get(); // Heap align

    if ( startAddress > endAddress )
    {
        return false;
    }

    size = static_cast<size_t>( static_cast<uint8_t*>( endAddress ) - static_cast<uint8_t*>( startAddress ) );

    m_pHeap = fnd::FrameHeapImpl::Create( startAddress, size );
    if ( m_pHeap == NULL )
    {
        return false;
    }

    // ベースセクションの作成
    if ( ! NewSection() )
    {
        return false;
    }

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         Destroy

  Description:  ヒープを破棄します

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void FrameHeap::Destroy() NN_NOEXCEPT
{
    if ( IsValid() )
    {
        // セクションの破棄
        ClearSection();

        // ヒープのクリア
        m_pHeap->Free( fnd::FrameHeapImpl::FreeAllMode );

        // ヒープの破棄
        m_pHeap->Destroy();

        m_pHeap = NULL;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         Clear

  Description:  ヒープを作成時の状態に戻す

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void FrameHeap::Clear() NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    // セクションの破棄
    ClearSection();

    // ヒープのクリア
    m_pHeap->Free( fnd::FrameHeapImpl::FreeAllMode );

    // ベースセクションの作成
    bool result = NewSection();
#if defined(NN_SDK_BUILD_RELEASE)
    NN_UNUSED( result );
#endif
    NN_SDK_ASSERT( result, "FrameHeap::Clear(): NewSection is Failed");
}

/*--------------------------------------------------------------------------------*
  Name:         Alloc

  Description:  ヒープからメモリを確保

  Arguments:    size     - メモリサイズ
                callback - メモリが破棄されたときに呼びだされるコールバック関数
                callbackArg - コールバック引数

  Returns:      確保したメモリへのポインタ
 *--------------------------------------------------------------------------------*/
void* FrameHeap::Alloc( size_t size, FrameHeap::DisposeCallback callback, void* callbackArg, FrameHeap::HeapCallback heapCallback, void* heapCallbackArg ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    const size_t blockSize = nn::util::align_up( sizeof(Block), HeapAlign );
    const size_t allocSize = blockSize + nn::util::align_up( size, HeapAlign );
    void* mem = m_pHeap->Alloc( allocSize, HeapAlign );
    if ( mem == NULL )
    {
        return NULL;
    }
    void* buffer = util::BytePtr( mem, blockSize ).Get();
    NN_SDK_ASSERT( ( reinterpret_cast<intptr_t>(buffer) & 0x1f ) == 0, "FrameHeap::Alloc: Internal Error" );

    Block* block = new ( mem ) Block( buffer, size, callback, callbackArg, heapCallback, heapCallbackArg );

    m_SectionList.back().AppendBlock( block );
    // NN_DETAIL_ATK_INFO("AppendBlock block(%p) allocSize(%d) callback(%p) arg(%p) heapCallback(%p) heapArg(%p)\n",
    //         block, allocSize, callback, callbackArg, heapCallback, heapCallbackArg );

    return buffer;
}

/*--------------------------------------------------------------------------------*
  Name:         SaveState

  Description:  ヒープの状態を保存

  Arguments:    None.

  Returns:      保存した階層レベルを返す
                失敗時には、-1
 *--------------------------------------------------------------------------------*/
int FrameHeap::SaveState() NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    if ( ! m_pHeap->RecordState( m_SectionList.size() ) )
    {
        return -1;
    }

    if ( ! NewSection() )
    {
        int result = m_pHeap->FreeByState( 0 );
#if defined(NN_SDK_BUILD_RELEASE)
        NN_UNUSED( result );
#endif
        NN_SDK_ASSERT( result, "FrameHeap::SaveState(): nn::atk::detail::fnd::FrameHeapImpl::FreeByState is Failed");
        return -1;
    }

    return static_cast<int>( m_SectionList.size() ) - 1;
}

// 「指定レベルまでに含まれているすべての Block」で、
// 設定されているコールバック・コールバック引数が同じ場合は、
// 当該コールバックをまとめて呼び出します。
// 同じでない場合は、コールバックは呼び出さず、false を返します。
bool FrameHeap::ProcessCallback( int level ) NN_NOEXCEPT
{
    void* begin = NULL;
    void* end = NULL;
    DisposeCallback callback = NULL;
    const void* callbackArg = NULL;
    HeapCallback heapCallback = nullptr;
    const void* heapCallbackArg = nullptr;
    bool isSameCallback = true;
    bool useCallback = false;

    int sectionCount = m_SectionList.size();
    for ( SectionList::iterator itr = m_SectionList.end();
            itr != m_SectionList.begin(); )
    {
        sectionCount--;
        SectionList::iterator curItr = --itr;

        BlockList& list = curItr->GetBlockList();
        // NN_DETAIL_ATK_INFO("[%d](%p) blockSize(%d) isEmpty(%d)\n",
        //         sectionCount, &*curItr, list.GetSize(), list.empty());

        // end の決定、callback* の初期化
        if ( end == NULL && list.size() > 0 )
        {
            const Block& last = list.back();
            end = const_cast<void*>(util::ConstBytePtr(
                        last.GetBufferAddr(), last.GetBufferSize() ).Get() );

            callback = last.GetDisposeCallback();
            callbackArg = last.GetDisposeCallbackArg();
            heapCallback = last.GetHeapCallback();
            heapCallbackArg = last.GetHeapCallbackArg();
        }

        // 各ブロックで callback* に変更が無いか？
        for ( BlockList::iterator blockItr = list.begin();
                blockItr != list.end(); )
        {
            BlockList::iterator curBlockItr = blockItr++;
            DisposeCallback cb = curBlockItr->GetDisposeCallback();
            const void* cbArg = curBlockItr->GetDisposeCallbackArg();
            HeapCallback hcb = curBlockItr->GetHeapCallback();
            const void* hcbArg = curBlockItr->GetHeapCallbackArg();

            if ( callback != cb || callbackArg != cbArg || heapCallback != hcb || heapCallbackArg != hcbArg )
            {
                isSameCallback = false;
                break;
            }
        }

        // begin の設定
        if ( sectionCount == level )
        {
            begin = &*curItr;
            break;
        }
    }
    if ( end == NULL )
    {
        isSameCallback = false;
    }

    // NN_DETAIL_ATK_INFO("begin(%p) end(%p) isSameCallback(%d)\n", begin, end, isSameCallback );
    if ( isSameCallback )
    {
        if ( callback )
        {
            callback( begin, util::BytePtr(begin).Distance(end),
                    const_cast<void*>(callbackArg) );
        }
        useCallback = true;
    }

    return useCallback;
}

/*--------------------------------------------------------------------------------*
  Name:         LoadState

  Description:  ヒープの状態を戻す

  Arguments:    level - 階層レベル

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void FrameHeap::LoadState( int level ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );
    NN_SDK_ASSERT_RANGE( level, 0, static_cast<int>( m_SectionList.size() ) );

    if ( level == 0 )
    {
        Clear();
        return;
    }

    bool alreadyUseCallback = ProcessCallback( level );

    // セクションの整理
    while ( level < static_cast<int>( m_SectionList.size() ) )
    {
        // NN_DETAIL_ATK_INFO("LoadState(%d) / size(%d)\n", level, m_SectionList.GetSize());

        // get latest section
        Section& section = m_SectionList.back();

        if ( alreadyUseCallback )
        {
            section.SetUseCallback( false );
        }

        // call dispose callback
        section.~Section();

        // セクションリストからの削除
        m_SectionList.erase( m_SectionList.iterator_to( section ) );
    }

    // ヒープ状態を復元
    int result = m_pHeap->FreeByState( static_cast<uint32_t>( level ) );
    NN_UNUSED( result );
    NN_SDK_ASSERT( result, "FrameHeap::LoadState(): nn::atk::detail::fnd::FrameHeapImpl::FreeByState is Failed");

    // 再度記録
    result =  m_pHeap->RecordState( m_SectionList.size() );
    NN_SDK_ASSERT( result, "FrameHeap::LoadState(): nn::atk::detail::fnd::FrameHeapImpl::RecordState is Failed");

    // セクションの作成
    bool result2 = NewSection();
#if defined(NN_SDK_BUILD_RELEASE)
    NN_UNUSED( result2 );
#endif
    NN_SDK_ASSERT( result2, "FrameHeap::LoadState(): NewSection is Failed");
}

/*--------------------------------------------------------------------------------*
  Name:         GetCurrentLevel

  Description:  ヒープの現在の階層レベルを取得

  Arguments:    None.

  Returns:      現在の階層レベル
 *--------------------------------------------------------------------------------*/
int FrameHeap::GetCurrentLevel() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    return static_cast<int>( m_SectionList.size() ) - 1;
}

/*--------------------------------------------------------------------------------*
  Name:         GetSize

  Description:  ヒープの容量を取得

  Arguments:    None.

  Returns:      ヒープの容量
 *--------------------------------------------------------------------------------*/
size_t FrameHeap::GetSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    return static_cast<size_t>(
        static_cast<uint8_t*>( m_pHeap->GetHeapEndAddress() ) -
        static_cast<uint8_t*>( m_pHeap->GetHeapStartAddress() )
    );
}

/*--------------------------------------------------------------------------------*
  Name:         GetFreeSize

  Description:  ヒープの空き容量を取得

  Arguments:    None.

  Returns:      空き容量
 *--------------------------------------------------------------------------------*/
size_t FrameHeap::GetFreeSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsValid() );

    size_t size = m_pHeap->GetAllocatableSize( HeapAlign );

    if ( size < sizeof( Block ) )
    {
        return 0;
    }
    size -= sizeof( Block );

    size &= ~(HeapAlign - 1);

    return size;
}

/*--------------------------------------------------------------------------------*
  Name:         NewSection

  Description:  新しいセクションを作成

  Arguments:    None.

  Returns:      成功したかどうか
 *--------------------------------------------------------------------------------*/
bool FrameHeap::NewSection() NN_NOEXCEPT
{
    // new HeapSection
    void* buffer = m_pHeap->Alloc( sizeof( Section ) );
    if ( buffer == NULL )
    {
        return false;
    }
    std::memset( buffer, 0x0, sizeof(Section) );

    Section* section = new( buffer ) Section();
    m_SectionList.push_back( *section );
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         ClearSection

  Description:  セクションを全て破棄する

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void FrameHeap::ClearSection() NN_NOEXCEPT
{
    bool alreadyUseCallback = ProcessCallback( 0 );

    // セクションの破棄
    while ( !m_SectionList.empty() )
    {
        Section& section = m_SectionList.back();

        if ( alreadyUseCallback )
        {
            section.SetUseCallback( false );
        }

        // コールバックの呼び出し
        section.~Section();

        // セクションリストからの削除
        m_SectionList.erase( m_SectionList.iterator_to( section ) );
    }
}

void FrameHeap::Dump(
        const nn::atk::SoundDataManager& mgr, const nn::atk::SoundArchive& arc ) const NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
    int i = 0;
    for ( SectionList::const_iterator itr = m_SectionList.begin();
            itr != m_SectionList.end(); )
    {
        SectionList::const_iterator curItr = itr++;
        NN_DETAIL_ATK_INFO("section[%d]\n", i++ );
        curItr->Dump( mgr, arc );
    }
#else
    (void)mgr;
    (void)arc;
#endif // NN_SDK_BUILD_DEBUG || NN_SDK_BUILD_DEVELOP
}

/* ========================================================================
        FrameHeap::Section class member function
   ======================================================================== */

FrameHeap::Section::Section() NN_NOEXCEPT
: m_UseCallback(true)
{
    m_BlockList.clear();
}

FrameHeap::Section::~Section() NN_NOEXCEPT
{
    if ( m_UseCallback )
    {
        BlockList::iterator itr = m_BlockList.end();
        while ( itr != m_BlockList.begin() )
        {
            (void)--itr;
            itr->~Block();
        }
     }
     m_BlockList.clear();
}

void FrameHeap::Section::AppendBlock( Block* block ) NN_NOEXCEPT
{
    m_BlockList.push_back( *block );
}

void FrameHeap::Section::Dump(
        const nn::atk::SoundDataManager& mgr, const nn::atk::SoundArchive& arc ) const NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
    int i = 0;
    for ( BlockList::const_iterator itr = m_BlockList.begin();
            itr != m_BlockList.end(); )
    {
        BlockList::const_iterator curItr = itr++;
        const void* ptr = curItr->GetBufferAddr();

        uint32_t sign = *reinterpret_cast<const uint32_t*>(ptr);

        char signature[5];
        signature[4] = '\0';
        std::memcpy( signature, &sign, 4 );

        uint32_t fileId = mgr.detail_GetFileIdFromTable( ptr );

        SoundArchive::ItemId itemId = SoundArchive::InvalidId;
        uint32_t waveIndex = 0xffffffff;
        switch ( sign )
        {
        case BankFileReader::SignatureFile:
            for ( uint32_t j = 0; j < arc.GetBankCount(); j++ )
            {
                SoundArchive::BankInfo info;
                SoundArchive::ItemId id =
                    Util::GetMaskedItemId( j, ItemType_Bank );
                if ( arc.ReadBankInfo( &info, id ) )
                {
                    if ( info.fileId == fileId )
                    {
                        itemId = id;
                        break;
                    }
                }
            }
            break;
        case GroupFileReader::SignatureFile:
            for ( uint32_t j = 0; j < arc.GetGroupCount(); j++ )
            {
                SoundArchive::GroupInfo info;
                SoundArchive::ItemId id =
                    Util::GetMaskedItemId( j, ItemType_Group );
                if ( arc.ReadGroupInfo( &info, id ) )
                {
                    if ( info.fileId == fileId )
                    {
                        itemId = id;
                        break;
                    }
                }
            }
            break;
        case SequenceSoundFileReader::SignatureFile:
        case WaveSoundFileReader::SignatureFile:
            for ( uint32_t j = 0; j < arc.GetSoundCount(); j++ )
            {
                SoundArchive::SoundInfo info;
                SoundArchive::ItemId id =
                    Util::GetMaskedItemId( j, ItemType_Sound );
                if ( arc.ReadSoundInfo( &info, id ) )
                {
                    if ( info.fileId == fileId )
                    {
                        itemId = id;
                        break;
                    }
                }
            }
            break;
        case WaveArchiveFileReader::SignatureFile:
            itemId = GetWaveArchiveId( fileId, arc );
            break;
        // 個別ロードの波形
        case SoundDataManager::SignatureIndividualWave:
            {
                const uint32_t* pU32Array = reinterpret_cast<const uint32_t*>(ptr);
                fileId = pU32Array[1];
                waveIndex = pU32Array[2];
                itemId = GetWaveArchiveId( fileId, arc );
            }
            break;
        default:
            itemId = SoundArchive::InvalidId;
            break;
        }

        const char* pItemLabel = arc.GetItemLabel( itemId );
        if ( pItemLabel != NULL )
        {
            NN_DETAIL_ATK_INFO("  block[%d] 0x%08X [%s] fileId(%6d) itemId(%08X) [%s]",
                    i++, ptr, signature, fileId, itemId, pItemLabel );
        }
        else
        {
            NN_DETAIL_ATK_INFO("  block[%d] 0x%08X [%s] fileId(%6d) itemId(%08X) [(anonymous)]",
                    i++, ptr, signature, fileId, itemId );
        }

        if ( waveIndex != 0xffffffff )
        {
            NN_DETAIL_ATK_INFO("(%d)\n", waveIndex );
        }
        else
        {
            NN_DETAIL_ATK_INFO("\n");
        }
    }
#else
    (void)mgr;
    (void)arc;
#endif // NN_SDK_BUILD_DEBUG || NN_SDK_BUILD_DEVELOP
} // NOLINT(impl/function_size)

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

