﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_StreamBufferPool.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {
namespace detail {
namespace driver {

NN_DEFINE_STATIC_CONSTANT( const int StreamBufferPool::BlockMax );
NN_DEFINE_STATIC_CONSTANT( const int StreamBufferPool::BitPerByte );

void StreamBufferPool::Initialize( void* buffer, size_t size, int blockCount ) NN_NOEXCEPT
{
    if ( blockCount == 0 )
    {
        return;
    }

    if (!atk::detail::Util::IsValidMemoryForDsp( buffer, size ))
    {
        NN_ATK_WARNING(
            "the memory area (0x%08x - 0x%08x %dbyte) provided cross a 512 MB segment.",
            buffer,
            util::BytePtr( buffer, size ).Get(),
            size );
    }

    // ストリームバッファのアドレス (全チャンネル分)
    m_Buffer = buffer;

    // 全チャンネル分のストリームバッファサイズ
    m_BufferSize = size;

    // 何チャンネル分か？ (bfsar のストリームチャンネル数)
    m_BlockCount = blockCount;

    // 1 チャンネルあたりのバッファサイズ
    // (各バッファ頭が BufferAlignSize バイト整列するように BufferAlignSize バイトの倍数にする)
    m_BlockSize = nn::util::align_down( size / blockCount, nn::audio::BufferAlignSize );

    m_AllocCount = 0;
    std::memset( m_AllocFlags, 0, sizeof(m_AllocFlags) );

    NN_SDK_ASSERT( m_BlockCount <= BlockMax, "Too large stream buffer size." );
}

void StreamBufferPool::Finalize() NN_NOEXCEPT
{
    m_Buffer = NULL;
    m_BufferSize = 0;
    m_BlockSize = 0;
    m_BlockCount = 0;
}

void* StreamBufferPool::Alloc() NN_NOEXCEPT
{
    if ( m_AllocCount >= m_BlockCount ) return NULL;

    const int availableByte = nn::util::align_up( m_BlockCount, BitPerByte ) / BitPerByte;

    for( int byteIndex = 0 ; byteIndex < availableByte ; byteIndex++ )
    {
        const uint8_t byte = m_AllocFlags[ byteIndex ];
        if ( byte == 0xff ) continue;

        uint8_t mask = ( 1 << 0 );
        for( int bitIndex = 0 ; bitIndex < BitPerByte ; bitIndex++, mask<<=1 )
        {
            if ( ( byte & mask ) == 0 )
            {
                m_AllocFlags[ byteIndex ] |= mask;
                m_AllocCount++;

                const int totalIndex = byteIndex * BitPerByte + bitIndex;
                void* buffer = util::BytePtr( m_Buffer, m_BlockSize * totalIndex ).Get();
                // NN_DETAIL_ATK_INFO("[StreamBufferPool::%s] m_BlockSize(%p) return(%p)\n",
                //         __FUNCTION__, m_BlockSize, buffer);
                return buffer;
            }
        }
    }

    return NULL;
}

void StreamBufferPool::Free( void* pPtr ) NN_NOEXCEPT
{
    ptrdiff_t offset = util::BytePtr( m_Buffer ).Distance( pPtr );

    const unsigned long totalIndex = static_cast<unsigned long>( offset / m_BlockSize );
    NN_SDK_ASSERT( totalIndex < BlockMax );
    const unsigned long byteIndex = ( totalIndex / BitPerByte );
    const unsigned long bitIndex  = ( totalIndex % BitPerByte );

    const int mask = ( 1 << bitIndex );
    NN_SDK_ASSERT( ( m_AllocFlags[ byteIndex ] & mask ) != 0 );
    m_AllocFlags[ byteIndex ] &= ~mask;
    m_AllocCount--;
    NN_SDK_ASSERT( m_AllocCount >= 0 );
}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

