﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>
#include <nn/atk/atk_BankFile.h>
#include <nn/atk/atk_ElementType.h>
#include <nn/atk/atk_CurveAdshr.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {
namespace detail {

namespace
{

const uint8_t DefaultOriginalKey       = 60; // cn4
const uint8_t DefaultVolume            = 127;
const uint8_t DefaultPan               = 64;   // 中央
const float DefaultPitch               = 1.0f;
const bool DefaultIgnoreNoteOff        = false;
const uint8_t DefaultKeyGroup          = 0;
const uint8_t DefaultInterpolationType = 0;    // 0: ４点補間, 1: 線形補間, 2: 補間なし
const AdshrCurve DefaultAdshrCurve(
    127,    // uint8_t attack
    127,    // uint8_t decay
    127,    // uint8_t sustain
    127,    // uint8_t hold
    127     // uint8_t release
);

enum VelocityRegionBitFlag
{
    VelocityRegionBitFlag_Key                     = 0x00,
    VelocityRegionBitFlag_Volume,
    VelocityRegionBitFlag_Pan,
    VelocityRegionBitFlag_Pitch,
    VelocityRegionBitFlag_InstrumentNoteParam,
    VelocityRegionBitFlag_Sends                   = 0x08, // 欠番（予約）
    VelocityRegionBitFlag_Envelope,
    VelocityRegionBitFlag_Randomizer,             // 未実装
    VelocityRegionBitFlag_Lfo,                    // 未実装
    VelocityRegionBitFlag_BasicParamFlag =
        ( 1 << VelocityRegionBitFlag_Key )
      | ( 1 << VelocityRegionBitFlag_Volume )
      | ( 1 << VelocityRegionBitFlag_Pan )
      | ( 1 << VelocityRegionBitFlag_Pitch )
      | ( 1 << VelocityRegionBitFlag_InstrumentNoteParam )
      | ( 1 << VelocityRegionBitFlag_Envelope )   // ベロシティリージョンの基本パラメータ
                                                  // (NW4C-1.2.0 現在、このパラメータが必ず入っている)
};



struct DirectChunk
{
    Util::Reference toRegion;
    const void* GetRegion() const NN_NOEXCEPT
    {
        return util::ConstBytePtr( this, toRegion.offset ).Get();
    }
};

struct RangeChunk
{
    // データ
    Util::Table<uint8_t> borderTable;
        // borderTable の最終 item のうしろに 4 バイト境界に、
        // RegionTable が配置されている。
        // (この RegionTable は Util::Table と異なり、count を持たない)

    // アクセサ
    const Util::Reference& GetRegionTableAddress( int index ) const NN_NOEXCEPT
    {
        return *reinterpret_cast<const Util::Reference*>(
                util::ConstBytePtr( this,
                sizeof(borderTable.count) + nn::util::align_up( borderTable.count, 4 )
                + sizeof(Util::Reference) * index ).Get() );
    }
    const void* GetRegion( uint32_t index ) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT( index <= 127 );

        bool isFoundRangeChunkIndex = false;
        uint32_t regionTableIndex = 0;
        for ( uint32_t i = 0; i < borderTable.count; i++ )
        {
            if ( index <= borderTable.item[i] )
            {
                regionTableIndex = i;
                isFoundRangeChunkIndex = true;
                break;
            }
        }
        if ( isFoundRangeChunkIndex == false )
        {
            return NULL;
        }
        const Util::Reference& ref = GetRegionTableAddress( regionTableIndex );
        return util::ConstBytePtr( this, ref.offset ).Get();
    }
};

struct IndexChunk
{
    // データ
    uint8_t min;
    uint8_t max;
    uint8_t reserved[2];
    Util::Reference toRegion[1];

    // アクセサ
    const void* GetRegion( uint32_t index ) const NN_NOEXCEPT
    {
        if ( index < min )
        {
            NN_ATK_WARNING("out of region value[%d] < min[%d]", index, min );
            return NULL;
        }
        else if ( index > max )
        {
            NN_ATK_WARNING("out of region value[%d] > max[%d]", index, max );
            return NULL;
        }
        return util::ConstBytePtr( this, toRegion[index - min].offset ).Get();
    }
};

enum RegionType
{
    RegionType_Direct,         // リージョンの要素は 1 個だけ
    RegionType_Range,          // リージョンの要素は 10 個まで
    RegionType_Index,          // リージョンの要素は 127 個
    RegionType_Unknown
};

RegionType GetRegionType( uint16_t typeId ) NN_NOEXCEPT
{
    switch ( typeId )
    {
        case ElementType_BankFile_DirectReferenceTable:
            return RegionType_Direct;
        case ElementType_BankFile_RangeReferenceTable:
            return RegionType_Range;
        case ElementType_BankFile_IndexReferenceTable:
            return RegionType_Index;
        default:
            return RegionType_Unknown;
    }
}

inline const void* GetDirectChunk( const void* regionChunk ) NN_NOEXCEPT
{
    const DirectChunk& directChunk =
        *reinterpret_cast<const DirectChunk*>( regionChunk );
    return directChunk.GetRegion();
}

inline const void* GetRangeChunk( const void* regionChunk, uint32_t index ) NN_NOEXCEPT
{
    const RangeChunk& rangeChunk =
        *reinterpret_cast<const RangeChunk*>( regionChunk );
    return rangeChunk.GetRegion( index );
}

inline const void* GetIndexChunk( const void* regionChunk, uint32_t index ) NN_NOEXCEPT
{
    const IndexChunk& indexChunk =
        *reinterpret_cast<const IndexChunk*>( regionChunk );
    return indexChunk.GetRegion( index );
}

const void* GetRegion( const void* startPtr, uint16_t typeId, uint32_t offset, uint32_t index ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( startPtr );

    const void* regionChunk = util::ConstBytePtr( startPtr, offset ).Get();
    const void* region = NULL;

    switch ( GetRegionType( typeId ) )
    {
        case RegionType_Direct:
            region = GetDirectChunk( regionChunk );
            break;
        case RegionType_Range:
            region = GetRangeChunk( regionChunk, index );
            break;
        case RegionType_Index:
            region = GetIndexChunk( regionChunk, index );
            break;
        case RegionType_Unknown:
        default:
            region = NULL;
    }
    return region;
}

} // anonymous namespace


//
// BankFile::FileHeader
//
const BankFile::InfoBlock* BankFile::FileHeader::GetInfoBlock() const NN_NOEXCEPT
{
    return reinterpret_cast<const InfoBlock*>( GetBlock( ElementType_BankFile_InfoBlock ) );
}


//
// BankFile::InfoBlockBody
//
const Util::WaveIdTable&
BankFile::InfoBlockBody::GetWaveIdTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::WaveIdTable*>(
            util::ConstBytePtr( this, toWaveIdTable.offset ).Get() );
}

const Util::ReferenceTable&
BankFile::InfoBlockBody::GetInstrumentReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toInstrumentReferenceTable.offset ).Get() );
}

const BankFile::Instrument*
BankFile::InfoBlockBody::GetInstrument( int programNo ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( programNo < GetInstrumentCount() );
    const Util::ReferenceTable& table = GetInstrumentReferenceTable();
    const Util::Reference& ref = table.item[ programNo ];

    switch ( ref.typeId )
    {
        case ElementType_BankFile_InstrumentInfo:
            break;
        case ElementType_BankFile_NullInfo:
            return NULL;
        default:
            NN_SDK_ASSERT( 0, "cannot support Bank::InstRef::TypeId" );
            return NULL;
    }

    return reinterpret_cast<const BankFile::Instrument*>(
            table.GetReferedItem( programNo ) );
}

const BankFile::KeyRegion*
BankFile::Instrument::GetKeyRegion( uint32_t key ) const NN_NOEXCEPT
{
    // toKeyRegionChunk の typeId によって、
    // offset 先にあるオブジェクトの型が DirectChunk,
    // RangeChunk, IndexChunk のいずれかになる
    return reinterpret_cast<const BankFile::KeyRegion*>(
            GetRegion(
                this,
                toKeyRegionChunk.typeId,
                toKeyRegionChunk.offset,
                key )
            );
}

const BankFile::VelocityRegion*
BankFile::KeyRegion::GetVelocityRegion( uint32_t velocity ) const NN_NOEXCEPT
{
    // Instrument::GetKeyRegion と同様の構成になっている
    return reinterpret_cast<const BankFile::VelocityRegion*>(
            GetRegion(
                this,
                toVelocityRegionChunk.typeId,
                toVelocityRegionChunk.offset,
                velocity )
            );
}


//
// BankFile::VelocityRegion
//
uint8_t BankFile::VelocityRegion::GetOriginalKey() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, VelocityRegionBitFlag_Key );
    if ( result == false ) return DefaultOriginalKey;
    return Util::DevideBy8bit( value, 0 );
}

uint8_t BankFile::VelocityRegion::GetVolume() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, VelocityRegionBitFlag_Volume );
    if ( result == false ) return DefaultVolume;
    return Util::DevideBy8bit( value, 0 );
}

uint8_t BankFile::VelocityRegion::GetPan() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, VelocityRegionBitFlag_Pan );
    if ( result == false ) return DefaultPan;
    return Util::DevideBy8bit( value, 0 );
}

float BankFile::VelocityRegion::GetPitch() const NN_NOEXCEPT
{
    float value;
    bool result = optionParameter.GetValueF32( &value, VelocityRegionBitFlag_Pitch );
    if ( result == false ) return DefaultPitch;
    return value;
}

bool BankFile::VelocityRegion::IsIgnoreNoteOff() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, VelocityRegionBitFlag_InstrumentNoteParam );
    if ( result == false ) return DefaultIgnoreNoteOff;
    return Util::DevideBy8bit( value, 0 ) > 0;
}

uint8_t BankFile::VelocityRegion::GetKeyGroup() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, VelocityRegionBitFlag_InstrumentNoteParam );
    if ( result == false ) return DefaultKeyGroup;
    return Util::DevideBy8bit( value, 1 );
}

uint8_t BankFile::VelocityRegion::GetInterpolationType() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, VelocityRegionBitFlag_InstrumentNoteParam );
    if ( result == false ) return DefaultInterpolationType;
    return Util::DevideBy8bit( value, 2 );
}

const AdshrCurve& BankFile::VelocityRegion::GetAdshrCurve() const NN_NOEXCEPT
{
    uint32_t offsetToReference;
    bool result = optionParameter.GetValue( &offsetToReference, VelocityRegionBitFlag_Envelope );
    if ( result == false ) return DefaultAdshrCurve;
    const Util::Reference& ref = *reinterpret_cast<const Util::Reference*>(
            util::ConstBytePtr( this, offsetToReference ).Get() );
    return *reinterpret_cast<const AdshrCurve*>(
            util::ConstBytePtr( &ref, ref.offset ).Get() );
}

const BankFile::RegionParameter* BankFile::VelocityRegion::GetRegionParameter() const NN_NOEXCEPT
{
    if ( optionParameter.bitFlag != VelocityRegionBitFlag_BasicParamFlag )
    {
        return NULL;
    }

    return reinterpret_cast<const RegionParameter*>(
            util::ConstBytePtr( this, sizeof(VelocityRegion) ).Get() );
}



} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

