﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundArchiveLoader.h>
#include <nn/atk/atk_SoundMemoryAllocatable.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_WaveArchiveFile.h>
#include <nn/atk/atk_WaveArchiveFileReader.h>
#include <nn/atk/atk_WaveSoundFileReader.h>
#include <nn/atk/atk_BankFileReader.h>
#include <nn/atk/atk_GroupFileReader.h>
#include <nn/atk/atk_HardwareManager.h>

// #define NN_ATK_DEBUG_PRINT_ENABLE

namespace nn {
namespace atk {
namespace detail {

namespace
{

const uint32_t RequiredSizeForLoadflagWarcFileHeader = sizeof( detail::WaveArchiveFile::FileHeader );

class FileStreamHandle
{
public:
    explicit FileStreamHandle( nn::atk::detail::fnd::FileStream* stream ) NN_NOEXCEPT : m_pStream( stream ) { }
    ~FileStreamHandle() NN_NOEXCEPT
    {
        m_pStream = NULL;
    }

    nn::atk::detail::fnd::FileStream* operator->() NN_NOEXCEPT
    {
        return m_pStream;
    }
    NN_IMPLICIT operator bool() const
    {
        return m_pStream != NULL;
    }

private:
    nn::atk::detail::fnd::FileStream* m_pStream;
};

} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchiveLoader::SignatureIndividualWave );
NN_DEFINE_STATIC_CONSTANT( const int SoundArchiveLoader::WaveBufferAlignSize );

/*--------------------------------------------------------------------------------*
  Name:         SoundArchiveLoader

  Description:  コンストラクタ

  Arguments:    arc - ローダで使用するサウンドアーカイブ

  Returns:      なし
 *--------------------------------------------------------------------------------*/
SoundArchiveLoader::SoundArchiveLoader() NN_NOEXCEPT
: m_pSoundArchive( NULL )
{
}

/*--------------------------------------------------------------------------------*
  Name:         SoundArchiveLoader

  Description:  デストラクタ

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
SoundArchiveLoader::~SoundArchiveLoader() NN_NOEXCEPT
{
    m_pSoundArchive = NULL;
}

void SoundArchiveLoader::SetSoundArchive( const SoundArchive* arc ) NN_NOEXCEPT
{
    m_pSoundArchive = arc;
}


/*--------------------------------------------------------------------------------*
  Name:         IsAvailable

  Description:  サウンドアーカイブプレイヤーが利用可能かどうかを調べる

  Arguments:    無し

  Returns:      利用可能かどうかを返す
 *--------------------------------------------------------------------------------*/
bool SoundArchiveLoader::IsAvailable() const NN_NOEXCEPT
{
    if ( m_pSoundArchive == NULL )
    {
        return false;
    }
    if ( ! m_pSoundArchive->IsAvailable() )
    {
        return false;
    }

    return true;
}

bool SoundArchiveLoader::LoadData(
    SoundArchive::ItemId itemId,
    SoundMemoryAllocatable* pAllocator,
    uint32_t loadFlag,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    if ( ! IsAvailable() )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadData Failed: not IsAvailable\n");
        return false;
    }
    if ( itemId == SoundArchive::InvalidId )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadData Failed: InvalidId\n");
        return false;
    }
    if ( pAllocator == NULL )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadData Failed: allocater is NULL\n");
        return false;
    }

    m_pSoundArchive->FileAccessBegin();

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( !m_pSoundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        return false;
    }

    bool result = false;
    switch ( detail::Util::GetItemType( itemId ) )
    {
    case detail::ItemType_Sound:
        switch ( m_pSoundArchive->GetSoundType( itemId ) )
        {
        case SoundArchive::SoundType_Sequence:
            result = LoadSequenceSound( itemId, pAllocator, loadFlag, loadBlockSize );
            break;
        case SoundArchive::SoundType_Wave:
            if ( soundArchivePlayerInfo.isAdvancedWaveSoundEnabled )
            {
                result = LoadAdvancedWaveSound( itemId, pAllocator, loadFlag, loadBlockSize );
            }
            else
            {
                result = LoadWaveSound( itemId, pAllocator, loadFlag, loadBlockSize );
            }
            break;
        case SoundArchive::SoundType_Stream:
            result = LoadStreamSoundPrefetch( itemId, pAllocator, loadBlockSize );
            break;
        default:
            result = false;
            break;
        }
        break;
    case detail::ItemType_Bank:
        result = LoadBank( itemId, pAllocator, loadFlag, loadBlockSize );
        break;
    case detail::ItemType_WaveArchive:
        result = LoadWaveArchive( itemId, pAllocator, loadFlag, loadBlockSize );
        break;
    case detail::ItemType_Group:
        result = LoadGroup( itemId, pAllocator, loadBlockSize );
        break;
    case detail::ItemType_SoundGroup:
        result = LoadSoundGroup( itemId, pAllocator, loadFlag, loadBlockSize );
        break;
    case detail::ItemType_Player:
    default:
        result = false;
        break;
    }

    m_pSoundArchive->FileAccessEnd();

    return result;
}

bool SoundArchiveLoader::LoadData(
    const char* pItemName,
    SoundMemoryAllocatable* pAllocator,
    uint32_t loadFlag,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pSoundArchive );

    SoundArchive::ItemId id = m_pSoundArchive->GetItemId( pItemName );
    return LoadData( id, pAllocator, loadFlag, loadBlockSize );
}


const void* SoundArchiveLoader::LoadImpl(
    SoundArchive::FileId fileId,
    SoundMemoryAllocatable* pAllocator,
    size_t loadBlockSize,
    bool needMemoryPool ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pSoundArchive );
    NN_UNUSED( loadBlockSize );

    const void* fileAddress = GetFileAddressImpl( fileId );
    // 未ロードの場合
    if ( fileAddress == NULL )
    {
        fileAddress = LoadFile( fileId, pAllocator, loadBlockSize, needMemoryPool );
        if ( fileAddress == NULL )
        {
            NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadImpl Failed: LoadFile is Failed\n");
            return NULL;    // ロード失敗
        }
        else
        {
            SetFileAddressToTable( fileId, fileAddress );
            return fileAddress;
        }
    }
    // ロード済みだった場合
    else
    {
        return fileAddress;
    }
}

bool SoundArchiveLoader::LoadSequenceSound(
    SoundArchive::ItemId soundId,
    SoundMemoryAllocatable* pAllocator,
    uint32_t loadFlag,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    if ( loadFlag & LoadFlag_Seq )
    {
        uint32_t fileId = m_pSoundArchive->GetItemFileId( soundId );
        const void* pFile = LoadImpl( fileId, pAllocator, loadBlockSize );
        if ( pFile == NULL )
        {
            return false;   // ロード失敗
        }
    }

    if ( loadFlag & LoadFlag_Bank || loadFlag & LoadFlag_Warc )
    {
        SoundArchive::SequenceSoundInfo info;
        if ( ! m_pSoundArchive->ReadSequenceSoundInfo( &info, soundId ) )
        {
            return false;       // INFO ブロックからシーケンスサウンド情報が得られなかった
        }

        for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
        {
            uint32_t bankId = info.bankIds[ i ];
            if ( bankId != SoundArchive::InvalidId )
            {
                if ( ! LoadBank( bankId, pAllocator, loadFlag, loadBlockSize ) )
                {
                    return false;
                }
            }
        }
    }

    return true;
}

bool SoundArchiveLoader::LoadAdvancedWaveSound(
    SoundArchive::ItemId soundId,
    SoundMemoryAllocatable* pAllocator,
    uint32_t loadFlag,
    size_t loadBlockSize) NN_NOEXCEPT
{
    uint32_t awsdFileId = m_pSoundArchive->GetItemFileId( soundId );
    if ( loadFlag & LoadFlag_Wsd )
    {
        const void* pFile = LoadImpl( awsdFileId, pAllocator, loadBlockSize );
        if ( pFile == NULL )
        {
            NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadAdvancedWaveSound Failed: LoadImpl is Failed\n");
            return false;   // ロード失敗
        }
    }

    if ( loadFlag & LoadFlag_Warc )
    {
        const void* pAwsdFile = GetFileAddressImpl( awsdFileId );

        if ( pAwsdFile == NULL )
        {
            return false;
        }

        // 一括ロードのみ行う
        SoundArchive::AdvancedWaveSoundInfo info;
        if ( ! m_pSoundArchive->detail_ReadAdvancedWaveSoundInfo( soundId, &info ) )
        {
            NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadAdvancedWaveSound Failed: detail_ReadAdvancedWaveSoundInfo is Failed\n");
            return false;   // AdvancedWaveSoundInfo 取得失敗
        }

        if ( ! LoadWaveArchive(
                    info.waveArchiveId,
                    pAllocator,
                    loadFlag,
                    loadBlockSize ) )
        {
            NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadWaveSound Failed: LoadWaveArchive is Failed\n");
            return false;
        }
    }
    return true;
}

bool SoundArchiveLoader::LoadWaveSound(
    SoundArchive::ItemId soundId,
    SoundMemoryAllocatable* pAllocator,
    uint32_t loadFlag,
    size_t loadBlockSize,
    SoundArchive::ItemId waveSoundSetId ) NN_NOEXCEPT
{
    uint32_t wsdFileId = m_pSoundArchive->GetItemFileId( soundId );
    if ( loadFlag & LoadFlag_Wsd )
    {
        const void* pFile = LoadImpl( wsdFileId, pAllocator, loadBlockSize );
        if ( pFile == NULL )
        {
            NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadWaveSound Failed: LoadImpl is Failed\n");
            return false;   // ロード失敗
        }
    }

    if ( loadFlag & LoadFlag_Warc )
    {
        const void* pWsdFile = GetFileAddressImpl( wsdFileId );

        // ウェーブサウンドセットファイルがロード済みなら、
        // 必要に応じて個別ロードを行う
        if ( pWsdFile != NULL )
        {
            // bfwsd 内でのインデックス取得
            uint32_t index;
            {
                SoundArchive::WaveSoundInfo info;
                if ( ! m_pSoundArchive->detail_ReadWaveSoundInfo( soundId, &info ) )
                {
                    NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadWaveSound Failed: detail_ReadWaveSoundInfo is Failed\n");
                    return false;   // WaveSoundInfo 取得失敗
                }
                index = info.index;
            }
            // 関連する波形アーカイブ ID を取得
            uint32_t warcId = SoundArchive::InvalidId;
            uint32_t waveIndex;
            {
                detail::WaveSoundFileReader reader( pWsdFile );
                detail::WaveSoundNoteInfo info;
                if ( ! reader.ReadNoteInfo( &info, index, 0 ) )
                {
                    NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadWaveSound Failed: WaveSoundFileReader::ReadNoteInfo is Failed\n");
                    return false;   // NoteInfo 取得失敗
                }
                warcId = info.waveArchiveId;
                waveIndex = info.waveIndex;
            }
            // ロード
            if ( ! LoadWaveArchiveImpl(
                        warcId,
                        waveIndex,
                        pAllocator,
                        loadFlag,
                        loadBlockSize ) )
            {
                NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadWaveSound Failed: LoadWaveArchiveImpl is Failed\n");
                return false;
            }
        }
        // 未ロードの場合は、一括ロードのみ行う
        else
        {
            SoundArchive::ItemId itemId;
            if ( waveSoundSetId != SoundArchive::InvalidId )
            {
                itemId = waveSoundSetId;
                /* ウェーブサウンド ID → ウェーブサウンドセット ID への変換は
                   時間がかかるので、あらかじめ waveSoundSetId を渡してもらえる場合は
                   これを利用する */
            }
            else
            {
                itemId = soundId;
            }
            const detail::Util::Table<uint32_t>* pWarcIdTable =
                m_pSoundArchive->detail_GetWaveArchiveIdTable( itemId );

            NN_SDK_ASSERT_NOT_NULL( pWarcIdTable );

            for ( uint32_t i = 0; i < pWarcIdTable->count; i++ )
            {
                if ( ! LoadWaveArchive(
                            pWarcIdTable->item[ i ],
                            pAllocator,
                            loadFlag,
                            loadBlockSize ) )
                {
                    NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadWaveSound Failed: LoadWaveArchive is Failed\n");
                    return false;
                }
            }
        }
    }
    return true;
}

bool SoundArchiveLoader::LoadStreamSoundPrefetch(
    SoundArchive::ItemId soundId,
    SoundMemoryAllocatable* pAllocator,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    uint32_t prefetchFileId = m_pSoundArchive->GetItemPrefetchFileId( soundId );
    const void* pFile = LoadImpl( prefetchFileId, pAllocator, loadBlockSize, true );
    if ( pFile == NULL )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadStreamSoundPrefetch Failed: LoadImpl is Failed\n");
        return false;   // ロード失敗
    }

    return true;
}

bool SoundArchiveLoader::LoadBank(
    SoundArchive::ItemId bankId,
    SoundMemoryAllocatable* pAllocator,
    uint32_t loadFlag,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    uint32_t bankFileId = m_pSoundArchive->GetItemFileId( bankId );
    if ( loadFlag & LoadFlag_Bank )
    {
        const void* pFile = LoadImpl( bankFileId, pAllocator, loadBlockSize );
        if ( pFile == NULL )
        {
            return false;   // ロード失敗
        }
    }

    if ( loadFlag & LoadFlag_Warc )
    {
        const void* pFile = GetFileAddressImpl( bankFileId );

        // バンクがロード済みなら、必要に応じて個別ロードを行う
        if ( pFile != NULL )
        {
            detail::BankFileReader reader( pFile );
            const detail::Util::WaveIdTable* table = reader.GetWaveIdTable();
            if ( table == NULL )
            {
                return false;
            }

            for ( uint32_t i = 0; i < table->GetCount(); i++ )
            {
                const detail::Util::WaveId* pWaveId = table->GetWaveId( i );

                if ( pWaveId == NULL )
                {
                    return false;
                }

                if ( ! LoadWaveArchiveImpl(
                            pWaveId->waveArchiveId,
                            pWaveId->waveIndex,
                            pAllocator,
                            loadFlag,
                            loadBlockSize ) )
                {
                    return false;
                }
            }
        }
        // バンクが未ロードの場合は、一括ロードのみ行う
        else
        {
            const detail::Util::Table<uint32_t>* pWarcIdTable =
                m_pSoundArchive->detail_GetWaveArchiveIdTable( bankId );

            NN_SDK_ASSERT_NOT_NULL( pWarcIdTable );

            for ( uint32_t i = 0; i < pWarcIdTable->count; i++ )
            {
                if ( ! LoadWaveArchive(
                            pWarcIdTable->item[ i ],
                            pAllocator,
                            loadFlag,
                            loadBlockSize ) )
                {
                    return false;
                }
            }
        }
    }
    return true;
}

bool SoundArchiveLoader::LoadWaveArchiveImpl(
    SoundArchive::ItemId warcId,
    uint32_t waveIndex,
    SoundMemoryAllocatable* pAllocator,
    uint32_t loadFlag,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    SoundArchive::WaveArchiveInfo info;
    if ( ! m_pSoundArchive->ReadWaveArchiveInfo( warcId, &info ) )
    {
        return false;
    }
    if ( info.isLoadIndividual )
    {
        // 波形のロード
        if ( ! LoadIndividualWave(
                    warcId,
                    waveIndex,
                    pAllocator,
                    loadBlockSize ) )
        {
            return false;
        }
    }
    else
    {
        if ( ! LoadWaveArchive(
                    warcId,
                    pAllocator,
                    loadFlag,
                    loadBlockSize ) )
        {
            return false;
        }
    }
    return true;
}

// 波形アーカイブの一括ロード
bool SoundArchiveLoader::LoadWaveArchive(
    SoundArchive::ItemId warcId,
    SoundMemoryAllocatable* pAllocator,
    uint32_t loadFlag,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    if ( loadFlag & LoadFlag_Warc )
    {
        uint32_t fileId = m_pSoundArchive->GetItemFileId( warcId );
        // 要メモリプール
        const void* pFile = LoadImpl( fileId, pAllocator, loadBlockSize, true );
        if ( pFile == NULL )
        {
            return false;   // ロード失敗
        }
    }
    return true;
}

const void* SoundArchiveLoader::LoadWaveArchiveTable(
    SoundArchive::ItemId warcId,
    SoundMemoryAllocatable* pAllocator,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    uint32_t fileId = m_pSoundArchive->GetItemFileId( warcId );
    const void* pWaveArchiveFile = GetFileAddressImpl( fileId );
    if ( pWaveArchiveFile != NULL )
    {
        return pWaveArchiveFile;
    }

    // 波形ファイル数
    uint32_t waveCount;
    {
        SoundArchive::WaveArchiveInfo info;
        if ( ! m_pSoundArchive->ReadWaveArchiveInfo( warcId, &info ) )
        {
            return NULL;
        }
        if ( info.waveCount <= 0 )
        {
            return NULL;
        }
        waveCount = info.waveCount;
    }

    // ファイルブロック手前までのオフセット
    position_t fileBlockOffset;
    {
        char pBuffer[ RequiredSizeForLoadflagWarcFileHeader ];
        size_t readSize = ReadFile(
                fileId,
                pBuffer,
                sizeof(detail::WaveArchiveFile::FileHeader),
                0,
                loadBlockSize );
        if ( readSize != sizeof(detail::WaveArchiveFile::FileHeader) )
        {
            return NULL;
        }
        const detail::WaveArchiveFile::FileHeader* pHeader =
            reinterpret_cast<const detail::WaveArchiveFile::FileHeader*>( pBuffer );
        fileBlockOffset = static_cast<position_t>(pHeader->GetFileBlockOffset());
        position_t infoBlockOffset = static_cast<position_t>(pHeader->GetInfoBlockOffset());
        if ( infoBlockOffset > fileBlockOffset )
        {
            return NULL;    // INFO ブロック → FILE ブロックの順に並んでいる必要がある
        }
    }

    // ロード用テーブルを含めたヒープサイズ
    const size_t RequiredSize =
        fileBlockOffset +
        sizeof( detail::WaveArchiveFileReader::SignatureWarcTable ) +
        waveCount * sizeof(void*); // 個別ロードした波形のアドレスの管理領域;

    void* buffer = pAllocator->Allocate( RequiredSize );
    if (buffer == NULL)
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::%s Failed: Not enough memory(%d byte required)\n", __FUNCTION__, RequiredSize);
        return NULL;
    }

    {
        size_t readSize = ReadFile( fileId, buffer, static_cast<size_t>(fileBlockOffset), 0, loadBlockSize );
        if ( readSize != static_cast<size_t>(fileBlockOffset) )
        {
            return NULL;
        }
    }

    // テーブル頭につけるシグニチャ
    std::memcpy(
            util::BytePtr( buffer, fileBlockOffset ).Get(),
            &detail::WaveArchiveFileReader::SignatureWarcTable,
            sizeof( detail::WaveArchiveFileReader::SignatureWarcTable ) );
    detail::WaveArchiveFileReader reader( buffer, true );
    reader.InitializeFileTable();

    SetFileAddressToTable( fileId, buffer );

    return buffer;
}

bool SoundArchiveLoader::LoadIndividualWave(
    SoundArchive::ItemId warcId,
    uint32_t waveIndex,
    SoundMemoryAllocatable* pAllocator,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    NN_UNUSED( loadBlockSize );

    // (ロードされている) 波形アーカイブファイル取得
    uint32_t fileId = m_pSoundArchive->GetItemFileId( warcId );
    const void* pWaveArchiveFile = GetFileAddressImpl( fileId );
    if ( pWaveArchiveFile == NULL )
    {
        pWaveArchiveFile = LoadWaveArchiveTable( warcId, pAllocator, loadBlockSize );
        if ( pWaveArchiveFile == NULL )
        {
            return false;
        }
    }

    detail::WaveArchiveFileReader reader( pWaveArchiveFile, true );

    // TODO: reader に対して IsLoaded のようなメソッドを呼ぶのは、少し気持ち悪い。
    //       波形アーカイブファイルのロード周りの機能は、
    //       WaveArchiveFileReader から切り分けたほうがいいかもしれない。
    if ( reader.IsLoaded( waveIndex ) )
    {
        return true;
    }

    const size_t WaveFileSize = reader.GetWaveFileSize( waveIndex );
    const size_t RequiredSize = nn::util::align_up( WaveFileSize + sizeof(IndividualWaveInfo) + WaveBufferAlignSize, nn::audio::MemoryPoolType::SizeGranularity );

    void* buffer = pAllocator->Allocate( RequiredSize );
    if (buffer == NULL)
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::%s Failed: Not enough memory(%d byte required)\n", __FUNCTION__, RequiredSize);
        return false;
    }
    uint8_t* pAlignedBuffer = reinterpret_cast<uint8_t*>( util::BytePtr( buffer ).AlignUp( WaveBufferAlignSize ).Get() );

    // 個別ロード用の情報書き込み
    {
        IndividualWaveInfo iWavInfo( fileId, waveIndex );
        std::memcpy( pAlignedBuffer, &iWavInfo, sizeof(IndividualWaveInfo) );
    }

    // 個別ロード波形のロード
    {
        // sizeof(IndividualWaveInfo) が 64 バイトなので、 WaveBufferAlignSize のアライメントに沿っている
        void* loadingAddress =
            util::BytePtr( pAlignedBuffer, sizeof(IndividualWaveInfo) ).Get();
        size_t readSize = ReadFile(
                fileId,
                loadingAddress,
                WaveFileSize,
                reader.GetWaveFileOffsetFromFileHead( waveIndex ),
                loadBlockSize );
        if ( readSize != WaveFileSize )
        {
            return false;
        }
        // NN_DETAIL_ATK_INFO("[%s] warc[%08X](%s) wIdx(%d)\n",
        //         __FUNCTION__,
        //         warcId,
        //         m_pSoundArchive->GetItemLabel(warcId),
        //         waveIndex );
        reader.SetWaveFile( waveIndex, loadingAddress );

        nn::atk::detail::driver::HardwareManager::FlushDataCache(loadingAddress, WaveFileSize);
    }

    return true;
}

// グループ内の埋め込みアイテムをロード済みファイルテーブルに登録し、
// リンクアイテムをロードする。
bool SoundArchiveLoader::PostProcessForLoadedGroupFile(
            const void* pGroupFile,
            SoundMemoryAllocatable* pAllocator,
            size_t loadBlockSize ) NN_NOEXCEPT
{
    detail::GroupFileReader reader( pGroupFile );
    bool isLinkGroup = false;
    uint32_t groupItemCount = reader.GetGroupItemCount();
    for ( uint32_t i = 0; i < groupItemCount; i++ )
    {
        detail::GroupItemLocationInfo info;
        if ( ! reader.ReadGroupItemLocationInfo( &info, i ) )
        {
            return false;   // ファイル位置情報取得失敗
        }

        // 埋め込みアイテム
        if ( info.address != NULL )
        {
            SetFileAddressToTable( info.fileId, info.address );
        }
        else
        {
            // 0 個目のアイテムが埋め込みかどうかで、グループ全体が埋め込みかどうかを判断
            if ( i == 0 )
            {
                isLinkGroup = true;
                break;
            }
        }
    }
    if ( isLinkGroup )
    {
        uint32_t groupItemInfoExCount = reader.GetGroupItemExCount();
        for ( uint32_t i = 0; i < groupItemInfoExCount; i++ )
        {
            GroupFile::GroupItemInfoEx infoEx;
            if ( reader.ReadGroupItemInfoEx( &infoEx, i ) )
            {
                if ( ! LoadData( infoEx.itemId, pAllocator, infoEx.loadFlag, loadBlockSize ) )
                {
                    return false;
                }
            }
        }
    }
    // 埋め込みかつ、個別ロードの波形アーカイブが入っている場合は、
    // 波形アーカイブテーブルを生成しないと、当該波形アーカイブの波形を使ったサウンド再生ができない
    else
    {
        uint32_t groupItemInfoExCount = reader.GetGroupItemExCount();
        for ( uint32_t i = 0; i < groupItemInfoExCount; i++ )
        {
            GroupFile::GroupItemInfoEx infoEx;
            if ( reader.ReadGroupItemInfoEx( &infoEx, i ) )
            {
                // infoEx.itemId - infoEx.loadFlag の組み合わせから波形アーカイブを見つける
                if ( ! ( infoEx.loadFlag & LoadFlag_Warc ) )
                {
                    continue;
                }

                switch ( Util::GetItemType( infoEx.itemId ) )
                {
                case ItemType_Sound:
                    switch ( m_pSoundArchive->GetSoundType( infoEx.itemId ) )
                    {
                        case SoundArchive::SoundType_Sequence:
                            SetWaveArchiveTableWithSeqInEmbeddedGroup(
                                    infoEx.itemId, pAllocator );
                            break;
                    #if 0 // 現状、グループのアイテムとして、直接ウェーブサウンドを指定できない
                        case SoundArchive::SoundType_Wave:
                            SetWaveArchiveTableWithWsdInEmbeddedGroup(
                                    infoEx.itemId, pAllocator );
                            break;
                    #endif
                        default:
                            break;
                    }
                    break;
                case ItemType_Bank:
                    SetWaveArchiveTableWithBankInEmbeddedGroup( infoEx.itemId, pAllocator );
                    break;
                case ItemType_SoundGroup:
                    {
                        SoundArchive::SoundGroupInfo info;
                        if ( ! m_pSoundArchive->detail_ReadSoundGroupInfo(
                                    infoEx.itemId, &info ) )
                        {
                            continue;
                        }
                        if ( info.startId == SoundArchive::InvalidId )
                        {
                            continue;   // この {SEQ,WSD}SET には、サウンドが含まれていない
                        }
                        switch ( m_pSoundArchive->GetSoundType( info.startId ) )
                        {
                            case SoundArchive::SoundType_Sequence:
                                for ( uint32_t id = info.startId; id <= info.endId; id++ )
                                {
                                    SetWaveArchiveTableWithSeqInEmbeddedGroup(
                                            id, pAllocator );
                                }
                                break;
                            case SoundArchive::SoundType_Wave:
                                // ウェーブサウンドはすべて同じ波形アーカイブを指すため、
                                // startId のもののみ評価する
                                SetWaveArchiveTableWithWsdInEmbeddedGroup(
                                        info.startId, pAllocator );
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                default:
                    break;
                }
            }
        }
    }

    return true;
} // NOLINT(impl/function_size)

// 埋め込みグループ内の個別ロード波形アーカイブに対して、テーブルを作成する
void SoundArchiveLoader::SetWaveArchiveTableWithSeqInEmbeddedGroup(
        SoundArchive::ItemId seqId, SoundMemoryAllocatable* pAllocator ) NN_NOEXCEPT
{
    SoundArchive::SequenceSoundInfo info;
    if ( ! m_pSoundArchive->ReadSequenceSoundInfo( &info, seqId ) )
    {
        return;
    }
    for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
    {
        SetWaveArchiveTableWithBankInEmbeddedGroup(
                info.bankIds[i], pAllocator );
    }
}

void SoundArchiveLoader::SetWaveArchiveTableWithBankInEmbeddedGroup(
        SoundArchive::ItemId bankId, SoundMemoryAllocatable* pAllocator ) NN_NOEXCEPT
{
    if ( bankId == SoundArchive::InvalidId )
    {
        return;
    }

    // 波形アーカイブ ID を調べる
    SoundArchive::BankInfo bankInfo;
    if ( ! m_pSoundArchive->ReadBankInfo( &bankInfo, bankId ) )
    {
        return;
    }
    const void* bankFile = GetFileAddressImpl( bankInfo.fileId );
    if ( bankFile == NULL )
    {
        return;
    }

    BankFileReader bankReader( bankFile );
    const Util::WaveIdTable* table = bankReader.GetWaveIdTable();
    if ( table == NULL )
    {
        return;
    }

    if ( table->GetCount() <= 0 )
    {
        return;
    }

    const Util::WaveId* pWaveId = table->GetWaveId( 0U );
    if ( pWaveId == NULL )
    {
        return;
    }
    SoundArchive::ItemId warcId = pWaveId->waveArchiveId;
    SetWaveArchiveTableInEmbeddedGroupImpl( warcId, pAllocator );
}

void SoundArchiveLoader::SetWaveArchiveTableWithWsdInEmbeddedGroup(
        SoundArchive::ItemId wsdId, SoundMemoryAllocatable* pAllocator ) NN_NOEXCEPT
{
    if ( wsdId == SoundArchive::InvalidId )
    {
        return;
    }

    // 波形アーカイブ ID を調べる
    SoundArchive::SoundInfo soundInfo;
    if ( ! m_pSoundArchive->ReadSoundInfo( &soundInfo, wsdId ) )
    {
        return;
    }
    SoundArchive::WaveSoundInfo wsdInfo;
    if ( ! m_pSoundArchive->detail_ReadWaveSoundInfo( wsdId, &wsdInfo ) )
    {
        return;
    }

    const void* wsdFile = GetFileAddressImpl( soundInfo.fileId );
    if ( wsdFile == NULL )
    {
        return;
    }

    WaveSoundFileReader reader( wsdFile );
    WaveSoundNoteInfo noteInfo;
    if ( ! reader.ReadNoteInfo( &noteInfo, wsdInfo.index, 0 ) )
    {
        return;
    }

    SetWaveArchiveTableInEmbeddedGroupImpl( noteInfo.waveArchiveId, pAllocator );
}

void SoundArchiveLoader::SetWaveArchiveTableInEmbeddedGroupImpl(
        SoundArchive::ItemId warcId, SoundMemoryAllocatable* pAllocator ) NN_NOEXCEPT
{
    // 当該波形アーカイブが個別ロード OFF なら処理しない
    SoundArchive::WaveArchiveInfo info;
    if ( ! m_pSoundArchive->ReadWaveArchiveInfo( warcId, &info ) )
    {
        return;
    }
    if ( ! info.isLoadIndividual )
    {
        return;
    }

    const void* warcFile = GetFileAddressImpl( info.fileId );
    WaveArchiveFileReader loadedFileReader( warcFile, false );

    // すでにテーブル作成済みの場合は、処理しない
    if ( loadedFileReader.HasIndividualLoadTable() )
    {
        return;
    }

    // ファイルブロック手前までのオフセット取得
    uint32_t fileBlockOffset = reinterpret_cast<const WaveArchiveFile::FileHeader*>(
            warcFile )->GetFileBlockOffset();
    const uint32_t RequiredTableSize =
        fileBlockOffset + info.waveCount * sizeof(uint32_t) +
        sizeof( WaveArchiveFileReader::SignatureWarcTable );


    // ファイルテーブル用のバッファを確保
    void* buffer = pAllocator->Allocate( RequiredTableSize );
    if ( buffer == NULL )
    {
        return;
    }

    // ロード済みの波形アーカイブヘッダーをコピー
    std::memcpy( buffer, warcFile, fileBlockOffset );

    // テーブルのシグニチャ書き込み
    std::memcpy( util::BytePtr( buffer, fileBlockOffset ).Get(),
            &WaveArchiveFileReader::SignatureWarcTable,
            sizeof( WaveArchiveFileReader::SignatureWarcTable ) );

    WaveArchiveFileReader reader( buffer, true );
    reader.InitializeFileTable();

    // 波形アーカイブ中のすべての波形エントリを、テーブル上に書き出す
    for ( uint32_t i = 0; i < info.waveCount; i++ )
    {
        reader.SetWaveFile( i, loadedFileReader.GetWaveFile( i ) );
    }

    // ロードテーブル上のエントリは今回作ったテーブルのものに差し替える
    SetFileAddressToTable( info.fileId, buffer );
}

bool SoundArchiveLoader::LoadGroup(
    SoundArchive::ItemId groupId,
    SoundMemoryAllocatable* pAllocator,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    // グループファイルのロード
    const void* pGroupFile = NULL;
    {
        uint32_t fileId = m_pSoundArchive->GetItemFileId( groupId );

        // NOTE: グループ内の埋め込みアイテムが、メモリプールにアタッチされるべきかどうかは判断できない。
        //       そのため、つねに LoadImpl の bool needMemoryPool は true にしておく。
        pGroupFile = LoadImpl( fileId, pAllocator, loadBlockSize, true );
        if ( pGroupFile == NULL )
        {
            return false;   // ロード失敗
        }

    #ifdef NN_ATK_DEBUG_PRINT_ENABLE
        {
            GroupFileReader reader( pGroupFile );

            NN_DETAIL_ATK_INFO("*** GroupItemLocationInfo ***\n");
            for ( uint32_t i = 0; i < reader.GetGroupItemCount(); i++ )
            {
                GroupItemLocationInfo info;
                if ( reader.ReadGroupItemLocationInfo( &info, i ) )
                {
                    NN_DETAIL_ATK_INFO("  [%3d] fileId(%08x) addr(%p)\n", i, info.fileId, info.address );
                }
            }

            NN_DETAIL_ATK_INFO("*** GroupItemInfoEx ***\n");
            for ( uint32_t i = 0; i < reader.GetGroupItemExCount(); i++ )
            {
                GroupFile::GroupItemInfoEx infoEx;
                if ( reader.ReadGroupItemInfoEx( &infoEx, i ) )
                {
                    NN_DETAIL_ATK_INFO("  [%3d] itemId(%08x) loadFlag(%08x)\n",
                            i, infoEx.itemId, infoEx.loadFlag );
                }
            }
        }
    #endif
    }
    return PostProcessForLoadedGroupFile( pGroupFile, pAllocator, loadBlockSize );
}

bool SoundArchiveLoader::LoadSoundGroup(
    SoundArchive::ItemId soundGroupId,
    SoundMemoryAllocatable* pAllocator,
    uint32_t loadFlag,
    size_t loadBlockSize ) NN_NOEXCEPT
{
    // SoundGroupInfo 取得
    SoundArchive::SoundGroupInfo info;
    if ( ! m_pSoundArchive->detail_ReadSoundGroupInfo( soundGroupId, &info ) )
    {
        return false;
    }
    if ( info.startId == SoundArchive::InvalidId )
    {
        return true;    // この {SEQ,WSD}SET には、サウンドが含まれていない。
                        // ヒープが足りない or 該当する {SEQ,WSD}SET が無いというわけではないので
                        // 関数自体は true を返す。
    }

    // info.startId と info.endId の SoundType は同一であることが保障されている。
    switch ( m_pSoundArchive->GetSoundType( info.startId ) )
    {
    case SoundArchive::SoundType_Sequence:
        for ( uint32_t id = info.startId; id <= info.endId; id++ )
        {
            if ( ! LoadSequenceSound( id, pAllocator, loadFlag, loadBlockSize ) )
            {
                return false;
            }
        }
        break;
    case SoundArchive::SoundType_Wave:
        for ( uint32_t id = info.startId; id <= info.endId; id++ )
        {
            if ( ! LoadWaveSound( id, pAllocator, loadFlag, loadBlockSize, soundGroupId ) )
            {
                return false;
            }
        }
        break;
    default:
        break;
    }

#if 0
    // TODO: パフォーマンスが気になる場合は、loadFlag と照らし合わせて、
    //       下記のようにする。
    //
    if ( ( loadFlag == LoadFlag_Seq ) || ( loadFlag == LoadFlag_Wsd ) )
    {
        // SoundGroupInfo の fileIds を順番にロード
        // ...
    }
#endif

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         ReadFile

  Description:  サウンドアーカイブ内のファイルの一部をメモリ上に読み出します。
                ID が fileId のファイルの offset バイト目から size だけ読み出します。

  Arguments:    fileId - サウンドアーカイブ内のファイル ID です。
                buffer - ロードするバッファの先頭アドレスです。
                size   - 読み出すサイズです。
                offset - ファイル「ファイル ID」の先頭からのオフセットです。

  Returns:      読み込んだサイズを返します。
                失敗したときは、0 を返します。
 *--------------------------------------------------------------------------------*/
size_t SoundArchiveLoader::ReadFile(
        SoundArchive::FileId fileId,
        void* buffer,
        size_t size,
        int offset,
        size_t loadBlockSize ) NN_NOEXCEPT
{
    FileStreamHandle stream(const_cast<SoundArchive*>(m_pSoundArchive)->detail_OpenFileStream(
            fileId, m_StreamArea, sizeof(m_StreamArea), NULL, 0 ));
    if ( ! stream )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::ReadFile Failed: detail_OpenFileStream is Failed\n");
        return 0;
    }
    if ( ! stream->CanSeek() || ! stream->CanRead() )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::ReadFile Failed: not CanSeek not CanRead\n");
        return 0;
    }

    stream->Seek( offset, fnd::Stream::SeekOrigin::SeekOrigin_Begin );

    if ( loadBlockSize == 0 )
    {
        // 一括ロード
        fnd::FndResult result;
        stream->Read( buffer, size, &result );
        if ( result.IsFailed() )
        {
            NN_DETAIL_ATK_INFO("SoundArchiveLoader::ReadFile Failed: stream->Read\n");
            return 0;
        }
    }
    else
    {
        // 分割ロード
        uint8_t* ptr = reinterpret_cast<uint8_t*>( buffer );
        size_t restSize = size;
        while ( restSize > 0 )
        {
            fnd::FndResult result;
            size_t curReadingSize = std::min( loadBlockSize, restSize );
            size_t readByte = stream->Read( ptr, curReadingSize, &result );
            if ( result.IsFailed() )
            {
                NN_DETAIL_ATK_INFO("SoundArchiveLoader::ReadFile Failed: stream->Read\n");
                return 0;
            }
            if ( restSize > readByte )
            {
                restSize -= readByte;
                ptr += readByte;
            }
            else
            {
                restSize = 0;
            }
        }
    }

    return size;
}

/*--------------------------------------------------------------------------------*
  Name:         LoadFile

  Description:  サウンドアーカイブ内のファイルをロードします。

  Arguments:    fileId   - サウンドアーカイブ内のファイル ID です。
                allocator - ファイルを格納するメモリを確保するアロケータです。
                loadBlockSize - 分割ロード時、１回の Read で読み取るサイズ。
                                0 だと一括ロードする。
                needMemoryPool - メモリプールへのアタッチが必要かどうかのフラグです。

  Returns:      ロードしたメモリアドレス
                ロード失敗時には NULL を返す
 *--------------------------------------------------------------------------------*/
void* SoundArchiveLoader::LoadFile(
    SoundArchive::FileId fileId,
    SoundMemoryAllocatable* allocator,
    size_t loadBlockSize,
    bool needMemoryPool ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( allocator );
    NN_SDK_ASSERT_NOT_NULL( m_pSoundArchive );

    SoundArchive::FileInfo fileInfo;
    if ( ! m_pSoundArchive->detail_ReadFileInfo( fileId, &fileInfo ) )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadFile Failed: detail_ReadFileInfo is Failed\n");
        return NULL;
    }
    uint32_t fileSize = fileInfo.fileSize;
    if ( fileSize == 0 || fileSize == SoundArchive::FileInfo::InvalidSize )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadFile Failed: fileSize == 0\n");
        return NULL;
    }

    void* buffer = allocator->Allocate( allocator->GetAllocateSize(fileSize, needMemoryPool) );

    if ( buffer == NULL )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::%s Failed: Not enough memory(%d byte required)\n", __FUNCTION__, fileSize);
        return NULL;
    }

    if ( ReadFile( fileId, buffer, (int32_t)fileSize, 0, loadBlockSize ) != static_cast<int32_t>(fileSize) )
    {
        NN_DETAIL_ATK_INFO("SoundArchiveLoader::LoadFile Failed: ReadFile is Failed\n");
        return NULL;
    }

    nn::atk::detail::driver::HardwareManager::FlushDataCache(buffer,fileSize);

    return buffer;
}

bool SoundArchiveLoader::IsDataLoaded( const char* pItemName, uint32_t loadFlag ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pSoundArchive );

    SoundArchive::ItemId id = m_pSoundArchive->GetItemId( pItemName );
    return IsDataLoaded( id, loadFlag );
}

bool SoundArchiveLoader::IsDataLoaded( SoundArchive::ItemId itemId, uint32_t loadFlag ) const NN_NOEXCEPT
{
    if ( ! IsAvailable() )
    {
        return false;
    }
    if ( itemId == SoundArchive::InvalidId )
    {
        return false;
    }

    switch ( detail::Util::GetItemType( itemId ) )
    {
    case detail::ItemType_Sound:
        switch ( m_pSoundArchive->GetSoundType( itemId ) )
        {
        case SoundArchive::SoundType_Sequence:
            return IsSequenceSoundDataLoaded( itemId, loadFlag );
        case SoundArchive::SoundType_Wave:
            return IsWaveSoundDataLoaded( itemId, loadFlag );
        case SoundArchive::SoundType_Stream:
        default:
            return false;
        }
    case detail::ItemType_Bank:
        return IsBankDataLoaded( itemId, loadFlag );
    case detail::ItemType_WaveArchive:
        return IsWaveArchiveDataLoaded( itemId, SoundArchive::InvalidId );
    case detail::ItemType_Group:
        return IsGroupDataLoaded( itemId );
    case detail::ItemType_SoundGroup:
        return IsSoundGroupDataLoaded( itemId, loadFlag );
    case detail::ItemType_Player:
    default:
        return false;
    }
}

bool SoundArchiveLoader::IsSequenceSoundDataLoaded(
        SoundArchive::ItemId itemId, uint32_t loadFlag ) const NN_NOEXCEPT
{
    if ( loadFlag & LoadFlag_Seq )
    {
        uint32_t fileId = m_pSoundArchive->GetItemFileId( itemId );
        if ( GetFileAddressImpl( fileId ) == NULL )
        {
            return false;
        }
    }

    if ( loadFlag & LoadFlag_Bank || loadFlag & LoadFlag_Warc )
    {
        SoundArchive::SequenceSoundInfo info;
        if ( ! m_pSoundArchive->ReadSequenceSoundInfo( &info, itemId ) )
        {
            return false;
        }

        for ( int i = 0; i < SoundArchive::SequenceBankMax; i++ )
        {
            uint32_t bankId = info.bankIds[i];
            if ( bankId != SoundArchive::InvalidId )
            {
                if ( ! IsBankDataLoaded( bankId, loadFlag ) )
                {
                    return false;
                }
            }
        }
    }
    return true;
}

bool SoundArchiveLoader::IsWaveSoundDataLoaded( SoundArchive::ItemId itemId, uint32_t loadFlag ) const NN_NOEXCEPT
{
    const void* pWsdFile = NULL;
    {
        uint32_t wsdFileId = m_pSoundArchive->GetItemFileId( itemId );
        pWsdFile = GetFileAddressImpl( wsdFileId );
    }

    if ( (loadFlag & LoadFlag_Wsd) || (loadFlag & LoadFlag_Warc) )
    {
        if ( pWsdFile == NULL )
        {
            return false;
        }
    }

    if ( loadFlag & LoadFlag_Warc )
    {
        uint32_t index;  // bfwsd 内のインデックス
        {
            SoundArchive::WaveSoundInfo info;
            if ( ! m_pSoundArchive->detail_ReadWaveSoundInfo( itemId, &info ) )
            {
                return false;
            }
            index = info.index;
        }
        uint32_t warcId;
        uint32_t waveIndex;
        {
            detail::WaveSoundFileReader reader( pWsdFile );
            detail::WaveSoundNoteInfo info;
            if ( ! reader.ReadNoteInfo( &info, index, 0 ) )
            {
                return false;
            }
            warcId = info.waveArchiveId;
            waveIndex = info.waveIndex;
        }
        if ( ! IsWaveArchiveDataLoaded( warcId, waveIndex ) )
        {
            return false;
        }
    }
    return true;
}

bool SoundArchiveLoader::IsBankDataLoaded( SoundArchive::ItemId itemId, uint32_t loadFlag ) const NN_NOEXCEPT
{
    const void* pBankFile = NULL;
    {
        uint32_t fileId = m_pSoundArchive->GetItemFileId( itemId );
        pBankFile = GetFileAddressImpl( fileId );
    }
    if ( (loadFlag & LoadFlag_Bank) || (loadFlag & LoadFlag_Warc) )
    {
        if ( pBankFile == NULL )
        {
            return false;
        }
    }

    if ( loadFlag & LoadFlag_Warc )
    {
        detail::BankFileReader reader( pBankFile );
        const detail::Util::WaveIdTable* table = reader.GetWaveIdTable();
        if ( table == NULL )
        {
            return false;
        }

        for ( uint32_t i = 0; i < table->GetCount(); i++ )
        {
            const detail::Util::WaveId* pWaveId = table->GetWaveId( i );
            if ( pWaveId == NULL )
            {
                return false;
            }
            uint32_t warcId    = pWaveId->waveArchiveId;
            uint32_t waveIndex = pWaveId->waveIndex;
            if ( ! IsWaveArchiveDataLoaded( warcId, waveIndex ) )
            {
                return false;
            }
        }
    }
    return true;
}

// LoadFlag_Warc が有効なだけ呼ばれるので、uint32_t loadFlag は必要ない。
bool SoundArchiveLoader::IsWaveArchiveDataLoaded(
        SoundArchive::ItemId itemId, uint32_t waveIndex ) const NN_NOEXCEPT
{
    const void* pWarcFile = NULL;
    {
        uint32_t fileId = m_pSoundArchive->GetItemFileId( itemId );
        pWarcFile = GetFileAddressImpl( fileId );
    }
    if ( pWarcFile == NULL )
    {
        return false;
    }

    SoundArchive::WaveArchiveInfo info;
    if ( ! m_pSoundArchive->ReadWaveArchiveInfo( itemId, &info ) )
    {
        return false;
    }
    if ( info.isLoadIndividual == true )
    {
        detail::WaveArchiveFileReader reader( pWarcFile, true );
        if ( waveIndex != SoundArchive::InvalidId )
        {
            if ( ! reader.IsLoaded( waveIndex ) )
            {
                return false;
            }
        }
        else
        {
            // 波形アーカイブのみのロード確認 and 個別ロード時は、全部確認
            uint32_t waveCount = reader.GetWaveFileCount();
            for ( uint32_t i = 0; i < waveCount; i++ )
            {
                if ( ! reader.IsLoaded( i ) )
                {
                    return false;
                }
            }
        }
    }
    return true;
}

// 中身がなにか分からないので uint32_t loadFlag は必要ない
bool SoundArchiveLoader::IsGroupDataLoaded( SoundArchive::ItemId itemId ) const NN_NOEXCEPT
{
    uint32_t fileId = m_pSoundArchive->GetItemFileId( itemId );
    if ( GetFileAddressImpl( fileId ) == NULL )
    {
        return false;
    }
    return true;
}

bool SoundArchiveLoader::IsSoundGroupDataLoaded(
        SoundArchive::ItemId itemId, uint32_t loadFlag ) const NN_NOEXCEPT
{
    SoundArchive::SoundGroupInfo info;
    if ( ! m_pSoundArchive->detail_ReadSoundGroupInfo( itemId, &info ) )
    {
        return false;
    }

    switch ( m_pSoundArchive->GetSoundType( info.startId ) )
    {
    case SoundArchive::SoundType_Sequence:
        for ( uint32_t id = info.startId; id <= info.endId; id++ )
        {
            if ( ! IsSequenceSoundDataLoaded( id, loadFlag ) )
            {
                return false;
            }
        }
        break;
    case SoundArchive::SoundType_Wave:
        for ( uint32_t id = info.startId; id <= info.endId; id++ )
        {
            if ( ! IsWaveSoundDataLoaded( id, loadFlag ) )
            {
                return false;
            }
        }
        break;
    default:
        break;
    }
    return true;
}

const void* SoundArchiveLoader::GetFileAddressFromSoundArchive(
        SoundArchive::FileId fileId ) const NN_NOEXCEPT
{
    if ( m_pSoundArchive == NULL )
    {
        return NULL;
    }
    return m_pSoundArchive->detail_GetFileAddress( fileId );
}

const void* SoundArchiveLoader::detail_GetFileAddressByItemId(
        SoundArchive::ItemId itemId ) const NN_NOEXCEPT
{
    if ( m_pSoundArchive == NULL )
    {
        return NULL;
    }

    SoundArchive::FileId fileId = SoundArchive::InvalidId;
    const void* result = NULL;

    switch ( Util::GetItemType( itemId ) )
    {
    case ItemType_Sound:
        {
            SoundArchive::SoundInfo info;
            m_pSoundArchive->ReadSoundInfo( &info, itemId );
            fileId = info.fileId;
        }
        break;
    case ItemType_Bank:
        {
            SoundArchive::BankInfo info;
            m_pSoundArchive->ReadBankInfo( &info, itemId );
            fileId = info.fileId;
        }
        break;
    case ItemType_WaveArchive:
        {
            SoundArchive::WaveArchiveInfo info;
            m_pSoundArchive->ReadWaveArchiveInfo( itemId, &info );
            fileId = info.fileId;
        }
        break;
    case ItemType_Group:
        {
            SoundArchive::GroupInfo info;
            m_pSoundArchive->ReadGroupInfo( &info, itemId );
            fileId = info.fileId;
        }
        break;
    default:
        return NULL;
    }

    if ( fileId != SoundArchive::InvalidId )
    {
        result = GetFileAddressImpl( fileId );
    }
    return result;
}

// 当該バンクに必要な波形をロード
bool SoundArchiveLoader::detail_LoadWaveArchiveByBankFile(
        const void* bankFile,
        SoundMemoryAllocatable* pAllocator ) NN_NOEXCEPT
{
    if ( bankFile == NULL )
    {
        return false;
    }

    // 当該バンクに必要な波形のみロード
    detail::BankFileReader reader( bankFile );
    const detail::Util::WaveIdTable* table = reader.GetWaveIdTable();
    if ( table == NULL )
    {
        return false;
    }

    for ( uint32_t i = 0; i < table->GetCount(); i++ )
    {
        const detail::Util::WaveId* pWaveId = table->GetWaveId( i );
        if ( pWaveId == NULL )
        {
            return false;
        }
        if ( ! LoadWaveArchiveImpl(
                    pWaveId->waveArchiveId,
                    pWaveId->waveIndex,
                    pAllocator,
                    LoadFlag_Warc ) )
        {
            return false;
        }
    }
    return true;
}

// 当該ウェーブサウンド１つに必要な波形をロード
bool SoundArchiveLoader::detail_LoadWaveArchiveByWaveSoundFile(
        const void* wsdFile,
        int32_t wsdIndex,
        SoundMemoryAllocatable* pAllocator ) NN_NOEXCEPT
{
    if ( wsdFile == NULL )
    {
        return false;
    }

    // wsdFile, wsdIndex から必要な波形アーカイブ ID および、波形インデックスを取得
    uint32_t warcId = SoundArchive::InvalidId;
    uint32_t waveIndex;
    {
        detail::WaveSoundFileReader reader( wsdFile );
        detail::WaveSoundNoteInfo info;
        if ( ! reader.ReadNoteInfo( &info, wsdIndex, 0 ) )
        {
            return false;
        }
        warcId = info.waveArchiveId;
        waveIndex = info.waveIndex;
    }
    if ( ! LoadWaveArchiveImpl(     // 波形アーカイブの個別ロードフラグが ON なら、個別ロードされる
                warcId,
                waveIndex,
                pAllocator,
                LoadFlag_Warc ) )
    {
        return false;
    }
    return true;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

