﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundArchiveFile.h>

#include <cstring>                          // std::strcmp
#include <nn/atk/atk_ElementType.h>
#include <nn/atk/atk_ItemType.h>
#include <nn/util/util_BytePtr.h>

namespace nn {
namespace atk {
namespace detail {

namespace
{
const uint32_t DefaultStringId             = SoundArchive::InvalidId;
const PanMode  DefaultPanMode              = PanMode_Dual;
const PanCurve DefaultPanCurve             = PanCurve_Sqrt;
const SinglePlayType DefaultSinglePlayType = SinglePlayType_None;
const uint16_t DefaultSinglePlayEffectiveDuration = 0xffff;
const uint8_t  DefaultPlayerPriority       = 64;
const uint8_t  DefaultChannelPriority      = 64;
const uint8_t  DefaultActorPlayerId        = 0;
const uint8_t  DefaultIsReleasePriorityFix = 0;
const bool     DefaultIsFrontBypass        = false;
const uint32_t DefaultUserParam            = 0xffffffff;  // 無効値だとわかるように
const uint32_t DefaultSeqStartOffset       = 0;
const uint32_t DefaultWarcWaveCount        = 0;
const uint32_t DefaultPlayerHeapSize       = 0;

const int      UserParamCount              = 4;

// optionParameter の各ビットの意味
enum SoundInfoBitFlag
{
    SoundInfoBitFlag_StringId = 0x00,
    SoundInfoBitFlag_PanParam,           // uint8_t PanMode, uint8_t PanCurve
    SoundInfoBitFlag_PlayerParam,        // uint8_t PlayerPriority, uint8_t ActorPlayerId
    SoundInfoBitFlag_SinglePlayParam,    // uint8_t SinglePlayType, uint8_t reserved, uint16_t duration

    SoundInfoBitFlag_OffsetTo3dParam = 0x08,
    SoundInfoBitFlag_OffsetToSendParam,
    SoundInfoBitFlag_OffsetToModParam,

    SoundInfoBitFlag_OffsetToRvlParam = 0x10,
    SoundInfoBitFlag_OffsetToCtrParam,
    SoundInfoBitFlag_OffsetToCafeParam,

    SoundInfoBitFlag_UserParam3 = 0x1c,  // 28
    SoundInfoBitFlag_UserParam2 = 0x1d,  // 29
    SoundInfoBitFlag_UserParam1 = 0x1e,  // 30
    SoundInfoBitFlag_UserParam = 0x1f    // 31
};

const int UserParamIndex[UserParamCount] =
{
    SoundInfoBitFlag_UserParam,
    SoundInfoBitFlag_UserParam1,
    SoundInfoBitFlag_UserParam2,
    SoundInfoBitFlag_UserParam3
};

enum StreamSoundInfoBitFlag
{
    StreamSoundInfoBitFlag_Pitch = 0x00,
    StreamSoundInfoBitFlag_Send  = 0x08
};

enum WaveSoundInfoBitFlag
{
    WaveSoundInfoBitFlag_Priority = 0x00
};

enum SequenceSoundInfoBitFlag
{
    SequenceSoundInfoBitFlag_StartOffset = 0x00,
    SequenceSoundInfoBitFlag_Priority
};

enum BankInfoBitFlag
{
    BankInfoBitFlag_StringId = 0x00
};

enum PlayerInfoBitFlag
{
    PlayerInfoBitFlag_StringId = 0x00,
    PlayerInfoBitFlag_HeapSize               // プレイヤーヒープサイズ
};

enum SoundGroupInfoBitFlag
{
    SoundGroupInfoBitFlag_StringId = 0x00
};

enum GroupInfoBitFlag
{
    GroupInfoBitFlag_StringId = 0x00
};

enum WaveArchiveInfoBitFlag
{
    WaveArchiveInfoBitFlag_StringId = 0x00,
    WaveArchiveInfoBitFlag_WaveCount
};

ItemType GetItemTypeEnum( uint32_t id ) NN_NOEXCEPT
{
    return static_cast<ItemType>( Util::GetItemType(id) );
}

struct SendValue
{
    uint8_t mainSend;
    Util::Table<uint8_t,uint8_t> fxSend;
};
} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const int SoundArchiveFile::BlockCount );

//
// SoundArchiveFile::FileHeader
//
const Util::ReferenceWithSize*
SoundArchiveFile::FileHeader::GetReferenceBy( uint16_t typeId ) const NN_NOEXCEPT
{
    for ( int i = 0; i < BlockCount; i++ )
    {
        if ( toBlocks[ i ].typeId == typeId )
        {
            return &toBlocks[ i ];
        }
    }

    NN_SDK_ASSERT(false); // should not reach here
    return NULL;
}

#if 0
const SoundArchiveFile::StringBlock*
SoundArchiveFile::FileHeader::GetStringBlock() const
{
    // return reinterpret_cast<const StringBlock*>( GetBlock( SoundArchiveFile_StringBlock ) );
    return NULL;
}

const SoundArchiveFile::InfoBlock*
SoundArchiveFile::FileHeader::GetInfoBlock() const
{
    // return reinterpret_cast<const InfoBlock*>( GetBlock( SoundArchiveFile_InfoBlock ) );
    return NULL;
}

const SoundArchiveFile::FileBlock*
SoundArchiveFile::FileHeader::GetFileBlock() const
{
    // return reinterpret_cast<const FileBlock*>( GetBlock( SoundArchiveFile_FileBlock ) );
    return NULL;
}
#endif

// ブロックサイズの取得
uint32_t SoundArchiveFile::FileHeader::GetStringBlockSize() const NN_NOEXCEPT
{
    return GetReferenceBy( ElementType_SoundArchiveFile_StringBlock )->size;
}
uint32_t SoundArchiveFile::FileHeader::GetInfoBlockSize() const NN_NOEXCEPT
{
    return GetReferenceBy( ElementType_SoundArchiveFile_InfoBlock )->size;
}
uint32_t SoundArchiveFile::FileHeader::GetFileBlockSize() const NN_NOEXCEPT
{
    return GetReferenceBy( ElementType_SoundArchiveFile_FileBlock )->size;
}

// ブロックへのオフセットの取得
int SoundArchiveFile::FileHeader::GetStringBlockOffset() const NN_NOEXCEPT
{
    return GetReferenceBy( ElementType_SoundArchiveFile_StringBlock )->offset;
}
int SoundArchiveFile::FileHeader::GetInfoBlockOffset() const NN_NOEXCEPT
{
    return GetReferenceBy( ElementType_SoundArchiveFile_InfoBlock )->offset;
}
int SoundArchiveFile::FileHeader::GetFileBlockOffset() const NN_NOEXCEPT
{
    return GetReferenceBy( ElementType_SoundArchiveFile_FileBlock )->offset;
}

//
// SoundArchiveFile::StringBlock
//
const void*
SoundArchiveFile::StringBlockBody::GetSection( Sections section ) const NN_NOEXCEPT
{
    if ( section > Sections_Max ) return NULL;
    return util::ConstBytePtr( this, toSection[section].offset ).Get();
}

const char*
SoundArchiveFile::StringBlockBody::GetString( SoundArchive::StringId stringId ) const NN_NOEXCEPT
{
    if ( stringId == SoundArchive::InvalidId )
    {
        return NULL;
    }
    const StringTable* table = GetStringTable();
    if ( table == NULL ) return NULL;

    return table->GetString( stringId );
}

uint32_t SoundArchiveFile::StringBlockBody::GetItemIdImpl(
        Sections section, const char* str ) const NN_NOEXCEPT
{
    const PatriciaTree* tree = GetPatriciaTree( section );
    const PatriciaTree::NodeData* nodeData = tree->GetNodeDataBy( str );
    if ( nodeData == NULL )
    {
        return SoundArchive::InvalidId;
    }
    const char* nodeDataStr = GetString( nodeData->stringId );
    NN_SDK_ASSERT_NOT_NULL(nodeDataStr);
    if ( std::strcmp( str, nodeDataStr ) == 0 )
    {
        return nodeData->itemId;
    }
    return SoundArchive::InvalidId;
}

void SoundArchiveFile::StringBlockBody::DumpTree() const NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
    for ( int section = Sections_PatriciaTree; section <= Sections_Max; section++ )
    {
        const PatriciaTree* tree = GetPatriciaTree( (Sections)section );

        NN_DETAIL_ATK_INFO("Section[%d] rootIdx(%d) count(%d)\n",
                section, tree->rootIdx, tree->nodeTable.count );
        for ( uint32_t i = 0; i < tree->nodeTable.count; i++ )
        {
            const PatriciaTree::Node* node = &tree->nodeTable.item[i];
            const PatriciaTree::NodeData* data = &node->nodeData;
            NN_DETAIL_ATK_INFO("  idx(%4d) str(%4d) itemId(0x%08X) left(%4d) right(%4d)\n",
                    i, data->stringId, data->itemId,
                    node->leftIdx, node->rightIdx );
        }
    }
#endif // NN_SDK_BUILD_DEBUG || NN_SDK_BUILD_DEVELOP
}

//
//  SoundArchiveFile::PatriciaTree
//
NN_DEFINE_STATIC_CONSTANT( const uint16_t SoundArchiveFile::PatriciaTree::Node::FlagLeaf );

const SoundArchiveFile::PatriciaTree::NodeData*
SoundArchiveFile::PatriciaTree::GetNodeDataBy( const char* str, std::size_t len ) const NN_NOEXCEPT
{
    if ( rootIdx >= nodeTable.count )
    {
        return NULL;
    }

    const Node* node = &nodeTable.item[ rootIdx ];
    if ( len == 0 )
    {
        len = std::strlen( str );    // TODO: strnlen ?
    }

    while( ( node->flags & Node::FlagLeaf ) == 0 )
    {
        const int pos = ( node->bit >> 3 );
        const int bit = ( node->bit & 7 );
        uint32_t nodeIdx;
        if ( pos < static_cast<int>(len) && str[pos] & ( 1 << ( 7 - bit ) ) )
        {
            nodeIdx = node->rightIdx;
        }
        else
        {
            nodeIdx = node->leftIdx;
        }
        node = &nodeTable.item[ nodeIdx ];
    }
    return &node->nodeData;
}


//
// SoundArchiveFile::InfoBlockBody
//
const SoundArchiveFile::SoundInfo*
SoundArchiveFile::InfoBlockBody::GetSoundInfo( SoundArchive::ItemId itemId ) const NN_NOEXCEPT
{
    if ( GetItemTypeEnum( itemId ) != ItemType_Sound )
    {
        return NULL;
    }

    uint32_t index = Util::GetItemIndex( itemId );
    const Util::ReferenceTable& table = GetSoundInfoReferenceTable();
    if ( index >= table.count )
    {
        return NULL;
    }
    return reinterpret_cast<const SoundInfo*>( table.GetReferedItem( index ) );
}

const SoundArchiveFile::BankInfo*
SoundArchiveFile::InfoBlockBody::GetBankInfo( SoundArchive::ItemId itemId ) const NN_NOEXCEPT
{
    if ( GetItemTypeEnum( itemId ) != ItemType_Bank )
    {
        return NULL;
    }

    uint32_t index = Util::GetItemIndex( itemId );
    const Util::ReferenceTable& table = GetBankInfoReferenceTable();
    if ( index >= table.count )
    {
        return NULL;
    }
    return reinterpret_cast<const BankInfo*>( table.GetReferedItem( index ) );
}

const SoundArchiveFile::PlayerInfo*
SoundArchiveFile::InfoBlockBody::GetPlayerInfo( SoundArchive::ItemId itemId ) const NN_NOEXCEPT
{
    if ( GetItemTypeEnum( itemId ) != ItemType_Player )
    {
        return NULL;
    }

    uint32_t index = Util::GetItemIndex( itemId );
    const Util::ReferenceTable& table = GetPlayerInfoReferenceTable();
    if ( index >= table.count )
    {
        return NULL;
    }
    return reinterpret_cast<const PlayerInfo*>( table.GetReferedItem( index ) );
}

const SoundArchiveFile::SoundGroupInfo*
SoundArchiveFile::InfoBlockBody::GetSoundGroupInfo( SoundArchive::ItemId itemId ) const NN_NOEXCEPT
{
    if ( GetItemTypeEnum( itemId ) != ItemType_SoundGroup )
    {
        return NULL;
    }

    uint32_t index = Util::GetItemIndex( itemId );
    const Util::ReferenceTable& table = GetSoundGroupInfoReferenceTable();
    if ( index >= table.count )
    {
        return NULL;
    }
    return reinterpret_cast<const SoundGroupInfo*>( table.GetReferedItem( index ) );
}

const SoundArchiveFile::GroupInfo*
SoundArchiveFile::InfoBlockBody::GetGroupInfo( SoundArchive::ItemId itemId ) const NN_NOEXCEPT
{
    if ( GetItemTypeEnum( itemId ) != ItemType_Group )
    {
        return NULL;
    }

    uint32_t index = Util::GetItemIndex( itemId );
    const Util::ReferenceTable& table = GetGroupInfoReferenceTable();
    if ( index >= table.count )
    {
        return NULL;
    }
    return reinterpret_cast<const GroupInfo*>( table.GetReferedItem( index ) );
}

const SoundArchiveFile::WaveArchiveInfo*
SoundArchiveFile::InfoBlockBody::GetWaveArchiveInfo( SoundArchive::ItemId itemId ) const NN_NOEXCEPT
{
    if ( GetItemTypeEnum( itemId ) != ItemType_WaveArchive )
    {
        return NULL;
    }

    uint32_t index = Util::GetItemIndex( itemId );
    const Util::ReferenceTable& table = GetWaveArchiveInfoReferenceTable();
    if ( index >= table.count )
    {
        return NULL;
    }
    return reinterpret_cast<const WaveArchiveInfo*>( table.GetReferedItem( index ) );
}

const SoundArchiveFile::FileInfo*
SoundArchiveFile::InfoBlockBody::GetFileInfo( SoundArchive::FileId itemId ) const NN_NOEXCEPT
{
    uint32_t index = Util::GetItemIndex( itemId );
    const Util::ReferenceTable& table = GetFileInfoReferenceTable();
    if ( index >= table.count )
    {
        return NULL;
    }
    return reinterpret_cast<const FileInfo*>( table.GetReferedItem( index ) );
}

SoundArchive::FileId
SoundArchiveFile::InfoBlockBody::GetItemFileId( SoundArchive::ItemId id ) const NN_NOEXCEPT
{
    SoundArchive::FileId fileId = SoundArchive::InvalidId;

    switch ( Util::GetItemType( id ) )
    {
    case ItemType_Sound:
        {
            const SoundInfo* info = GetSoundInfo( id );
            if ( info != NULL )
            {
                fileId = info->fileId;
            }
        }
        break;
    case ItemType_Bank:
        {
            const BankInfo* info = GetBankInfo( id );
            if ( info != NULL )
            {
                fileId = info->fileId;
            }
        }
        break;
    case ItemType_WaveArchive:
        {
            const WaveArchiveInfo* info = GetWaveArchiveInfo( id );
            if ( info != NULL )
            {
                fileId = info->fileId;
            }
        }
        break;
    case ItemType_Group:
        {
            const GroupInfo* info = GetGroupInfo( id );
            if ( info != NULL )
            {
                fileId = info->fileId;
            }
        }
        break;
    case ItemType_SoundGroup:
        {
            const SoundGroupInfo* info = GetSoundGroupInfo( id );
            if ( info != NULL )
            {
                // WSDSET or SEQSET のうち、一番若い ItemId と関連のある fileId を返す。
                SoundArchive::ItemId soundId = info->startId;
                const SoundInfo* soundInfo = GetSoundInfo( soundId );
                if ( soundInfo != NULL )
                {
                    fileId = soundInfo->fileId;
                }
            }
        }
        break;
    case ItemType_Player:
        // ファイル ID は無い
        break;
    default:
        break;
    }

    return fileId;
}

SoundArchive::FileId
SoundArchiveFile::InfoBlockBody::GetItemPrefetchFileId( SoundArchive::ItemId id ) const NN_NOEXCEPT
{
    SoundArchive::FileId fileId = SoundArchive::InvalidId;

    switch ( Util::GetItemType( id ) )
    {
    case ItemType_Sound:
        {
            const SoundInfo* info = GetSoundInfo(id);
            if (info != NULL && info->GetSoundType() == SoundArchive::SoundType_Stream)
            {
                const SoundArchiveFile::StreamSoundInfo& streamSoundInfo = info->GetStreamSoundInfo();
                fileId = streamSoundInfo.prefetchFileId;
            }
        }
        break;
    default:
        break;
    }

    return fileId;
}

SoundArchive::StringId
SoundArchiveFile::InfoBlockBody::GetItemStringId( SoundArchive::ItemId id ) const NN_NOEXCEPT
{
    SoundArchive::StringId stringId = SoundArchive::InvalidId;

    switch ( Util::GetItemType( id ) )
    {
    case ItemType_Sound:
        {
            const SoundInfo* info = GetSoundInfo( id );
            if ( info != NULL )
            {
                stringId = info->GetStringId();
            }
        }
        break;
    case ItemType_Bank:
        {
            const BankInfo* info = GetBankInfo( id );
            if ( info != NULL )
            {
                stringId = info->GetStringId();
            }
        }
        break;
    case ItemType_WaveArchive:
        {
            const WaveArchiveInfo* info = GetWaveArchiveInfo( id );
            if ( info != NULL )
            {
                stringId = info->GetStringId();
            }
        }
        break;
    case ItemType_SoundGroup:
        {
            const SoundGroupInfo* info = GetSoundGroupInfo( id );
            if ( info != NULL )
            {
                stringId = info->GetStringId();
            }
        }
        break;
    case ItemType_Group:
        {
            const GroupInfo* info = GetGroupInfo( id );
            if ( info != NULL )
            {
                stringId = info->GetStringId();
            }
        }
        break;
    case ItemType_Player:
        {
            const PlayerInfo* info = GetPlayerInfo( id );
            if ( info != NULL )
            {
                stringId = info->GetStringId();
            }
        }
        break;
    default:
        break;
    }

    return stringId;
}

const SoundArchiveFile::SoundArchivePlayerInfo*
SoundArchiveFile::InfoBlockBody::GetSoundArchivePlayerInfo() const NN_NOEXCEPT
{
    return reinterpret_cast<const SoundArchivePlayerInfo*>(
            util::ConstBytePtr( this, toSoundArchivePlayerInfo.offset ).Get() );
}


const Util::ReferenceTable&
SoundArchiveFile::InfoBlockBody::GetSoundInfoReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toSoundInfoReferenceTable.offset ).Get() );
}

const Util::ReferenceTable&
SoundArchiveFile::InfoBlockBody::GetBankInfoReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toBankInfoReferenceTable.offset ).Get() );
}

const Util::ReferenceTable&
SoundArchiveFile::InfoBlockBody::GetPlayerInfoReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toPlayerInfoReferenceTable.offset ).Get() );
}

const Util::ReferenceTable&
SoundArchiveFile::InfoBlockBody::GetSoundGroupInfoReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toSoundGroupInfoReferenceTable.offset ).Get() );
}

const Util::ReferenceTable&
SoundArchiveFile::InfoBlockBody::GetGroupInfoReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toGroupInfoReferenceTable.offset ).Get() );
}

const Util::ReferenceTable&
SoundArchiveFile::InfoBlockBody::GetWaveArchiveInfoReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toWaveArchiveInfoReferenceTable.offset ).Get() );
}

const Util::ReferenceTable&
SoundArchiveFile::InfoBlockBody::GetFileInfoReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toFileInfoReferenceTable.offset ).Get() );
}


//
// SoundArchiveFile::SoundInfo
//

SoundArchive::SoundType SoundArchiveFile::SoundInfo::GetSoundType() const NN_NOEXCEPT
{
    switch ( toDetailSoundInfo.typeId )
    {
    case ElementType_SoundArchiveFile_StreamSoundInfo:
        return SoundArchive::SoundType_Stream;
    case ElementType_SoundArchiveFile_WaveSoundInfo:
        return SoundArchive::SoundType_Wave;
    case ElementType_SoundArchiveFile_SequenceSoundInfo:
        return SoundArchive::SoundType_Sequence;
    default:
        return SoundArchive::SoundType_Invalid;
    }
}

const SoundArchiveFile::StreamSoundInfo&
SoundArchiveFile::SoundInfo::GetStreamSoundInfo() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( toDetailSoundInfo.typeId ==
            ElementType_SoundArchiveFile_StreamSoundInfo );

    return *reinterpret_cast<const StreamSoundInfo*>(
            util::ConstBytePtr( this, toDetailSoundInfo.offset ).Get() );
}

const SoundArchiveFile::WaveSoundInfo&
SoundArchiveFile::SoundInfo::GetWaveSoundInfo() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( toDetailSoundInfo.typeId ==
            ElementType_SoundArchiveFile_WaveSoundInfo );

    return *reinterpret_cast<const WaveSoundInfo*>(
            util::ConstBytePtr( this, toDetailSoundInfo.offset ).Get() );
}

const SoundArchiveFile::AdvancedWaveSoundInfo&
SoundArchiveFile::SoundInfo::GetAdvancedWaveSoundInfo() const NN_NOEXCEPT
{
    // 将来的に従来のWaveSound もカバーするので、同じタイプを使用
    NN_SDK_ASSERT( toDetailSoundInfo.typeId ==
            ElementType_SoundArchiveFile_WaveSoundInfo );

    nn::util::ConstBytePtr bytePtr(this);
    bytePtr += toDetailSoundInfo.offset;
    return *reinterpret_cast<const AdvancedWaveSoundInfo*>(bytePtr.Get());
}

const SoundArchiveFile::SequenceSoundInfo&
SoundArchiveFile::SoundInfo::GetSequenceSoundInfo() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( toDetailSoundInfo.typeId ==
            ElementType_SoundArchiveFile_SequenceSoundInfo );

    return *reinterpret_cast<const SequenceSoundInfo*>(
            util::ConstBytePtr( this, toDetailSoundInfo.offset ).Get() );
}

const SoundArchiveFile::Sound3DInfo*
SoundArchiveFile::SoundInfo::GetSound3DInfo() const NN_NOEXCEPT
{
    uint32_t offset;
    bool result = optionParameter.GetValue( &offset, SoundInfoBitFlag_OffsetTo3dParam );
    if ( result == false )
    {
        return NULL;    // 3D パラメータが省かれている (格納されていない)
    }

    return reinterpret_cast<const Sound3DInfo*>(
            util::ConstBytePtr( this, offset ).Get() );
}

uint32_t SoundArchiveFile::SoundInfo::GetStringId() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundInfoBitFlag_StringId );
    if ( result == false )
    {
        return DefaultStringId;
    }
    return value;
}

PanMode SoundArchiveFile::SoundInfo::GetPanMode() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundInfoBitFlag_PanParam );
    if ( result == false )
    {
        return DefaultPanMode;
    }
    return static_cast<PanMode>( Util::DevideBy8bit( value, 0 ) );
}

PanCurve SoundArchiveFile::SoundInfo::GetPanCurve() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundInfoBitFlag_PanParam );
    if ( result == false )
    {
        return DefaultPanCurve;
    }
    return static_cast<PanCurve>( Util::DevideBy8bit( value, 1 ) );
}

SinglePlayType SoundArchiveFile::SoundInfo::GetSinglePlayType() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundInfoBitFlag_SinglePlayParam );
    if ( result == false )
    {
        return DefaultSinglePlayType;
    }
    return static_cast<SinglePlayType>( Util::DevideBy8bit( value, 0 ) );
}

uint16_t SoundArchiveFile::SoundInfo::GetSinglePlayEffectiveDuration() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundInfoBitFlag_SinglePlayParam );
    if ( result == false )
    {
        return DefaultSinglePlayEffectiveDuration;
    }
    return Util::DevideBy16bit( value, 1 );
}

uint8_t SoundArchiveFile::SoundInfo::GetPlayerPriority() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundInfoBitFlag_PlayerParam );
    if ( result == false )
    {
        return DefaultPlayerPriority;
    }
    return Util::DevideBy8bit( value, 0 );
}

uint8_t SoundArchiveFile::SoundInfo::GetActorPlayerId() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundInfoBitFlag_PlayerParam );
    if ( result == false )
    {
        return DefaultActorPlayerId;
    }
    return Util::DevideBy8bit( value, 1 );
}

uint32_t SoundArchiveFile::SoundInfo::GetUserParam() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundInfoBitFlag_UserParam );
    if ( result == false )
    {
        return DefaultUserParam;
    }
    return value;
}

bool SoundArchiveFile::SoundInfo::ReadUserParam( uint32_t* pOutValue, int index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);
    NN_SDK_ASSERT_RANGE( index, 0, UserParamCount );
    return optionParameter.GetValue( pOutValue, UserParamIndex[index] );
}

bool SoundArchiveFile::SoundInfo::IsFrontBypass() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundInfoBitFlag_OffsetToCtrParam );
    if ( result == false )
    {
        return DefaultIsFrontBypass;
    }
    return ( value & ( 1 << 0 ) );
}


//
// SoundArchiveFile::StreamSoundInfo
//
const SoundArchiveFile::StreamTrackInfoTable*
SoundArchiveFile::StreamSoundInfo::GetTrackInfoTable() const NN_NOEXCEPT
{
    if ( toTrackInfoTable.typeId != ElementType_Table_ReferenceTable )
    {
        return NULL;
    }
    return static_cast<const StreamTrackInfoTable*>(
            util::ConstBytePtr( this, toTrackInfoTable.offset ).Get() );
}

const SoundArchiveFile::StreamSoundExtension*
SoundArchiveFile::StreamSoundInfo::GetStreamSoundExtension() const NN_NOEXCEPT
{
    if ( !toStreamSoundExtension.IsValidOffset() )
    {
        return NULL;
    }

    if ( toStreamSoundExtension.typeId != ElementType_SoundArchiveFile_StreamSoundExtensionInfo )
    {
        return NULL;
    }
    return reinterpret_cast<const SoundArchiveFile::StreamSoundExtension*>(
            util::ConstBytePtr( this, toStreamSoundExtension.offset ).Get() );
}

const SoundArchiveFile::SendValue&
SoundArchiveFile::StreamSoundInfo::GetSendValue() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( toSendValue.typeId ==
            nn::atk::detail::ElementType_SoundArchiveFile_SendInfo );
    return *reinterpret_cast<const SoundArchiveFile::SendValue*>(
            util::ConstBytePtr( this, toSendValue.offset ).Get() );
}

//
// SoundArchiveFile::WaveSoundInfo
//

uint8_t SoundArchiveFile::WaveSoundInfo::GetChannelPriority() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveSoundInfoBitFlag_Priority );
    if ( result == false )
    {
        return DefaultChannelPriority;
    }
    return Util::DevideBy8bit( value, 0 );
}

uint8_t SoundArchiveFile::WaveSoundInfo::GetIsReleasePriorityFix() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveSoundInfoBitFlag_Priority );
    if ( result == false )
    {
        return DefaultIsReleasePriorityFix;
    }
    return Util::DevideBy8bit( value, 1 );
}


//
// SoundArchiveFile::SequenceSoundInfo
//
const Util::Table<uint32_t>&
SoundArchiveFile::SequenceSoundInfo::GetBankIdTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::Table<uint32_t>*>(
            util::ConstBytePtr( this, toBankIdTable.offset ).Get() );
}
void SoundArchiveFile::SequenceSoundInfo::GetBankIds( uint32_t* bankIds ) const NN_NOEXCEPT
{
    const Util::Table<uint32_t>& table = GetBankIdTable();
    for ( uint32_t i = 0; i < SoundArchive::SequenceBankMax; i++ )
    {
        if ( i >= table.count )
        {
            bankIds[ i ] = SoundArchive::InvalidId;
        }
        else
        {
            bankIds[ i ] = table.item[ i ];
        }
    }
}
uint32_t SoundArchiveFile::SequenceSoundInfo::GetStartOffset() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SequenceSoundInfoBitFlag_StartOffset );
    if ( result == false )
    {
        return DefaultSeqStartOffset;
    }
    return value;
}
uint8_t SoundArchiveFile::SequenceSoundInfo::GetChannelPriority() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SequenceSoundInfoBitFlag_Priority );
    if ( result == false )
    {
        return DefaultChannelPriority;
    }
    return Util::DevideBy8bit( value, 0 );
}
bool SoundArchiveFile::SequenceSoundInfo::IsReleasePriorityFix() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SequenceSoundInfoBitFlag_Priority );
    if ( result == false )
    {
        return DefaultChannelPriority;
    }
    if ( Util::DevideBy8bit( value, 1 ) > 0 ) return true;
    return false;
}

//
// SoundArchiveFile::BankInfo
//
uint32_t SoundArchiveFile::BankInfo::GetStringId() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, BankInfoBitFlag_StringId );
    if ( result == false )
    {
        return DefaultStringId;
    }
    return value;
}

//
// SoundArchiveFile::PlayerInfo
//
uint32_t SoundArchiveFile::PlayerInfo::GetStringId() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, PlayerInfoBitFlag_StringId );
    if ( result == false )
    {
        return DefaultStringId;
    }
    return value;
}
uint32_t SoundArchiveFile::PlayerInfo::GetPlayerHeapSize() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, PlayerInfoBitFlag_HeapSize );
    if ( result == false )
    {
        return DefaultPlayerHeapSize;
    }
    return value;
}

//
// SoundArchiveFile::SoundGroupInfo
//
uint32_t SoundArchiveFile::SoundGroupInfo::GetStringId() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, SoundGroupInfoBitFlag_StringId );
    if ( result == false )
    {
        return DefaultStringId;
    }
    return value;
}

//
// SoundArchiveFile::GroupInfo
//
uint32_t SoundArchiveFile::GroupInfo::GetStringId() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, GroupInfoBitFlag_StringId );
    if ( result == false )
    {
        return DefaultStringId;
    }
    return value;
}

//
// SoundArchiveFile::WaveArchiveInfo
//
uint32_t SoundArchiveFile::WaveArchiveInfo::GetStringId() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveArchiveInfoBitFlag_StringId );
    if ( result == false )
    {
        return DefaultStringId;
    }
    return value;
}
uint32_t SoundArchiveFile::WaveArchiveInfo::GetWaveCount() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveArchiveInfoBitFlag_WaveCount );
    if ( result == false )
    {
        return DefaultWarcWaveCount;
    }
    return value;
}

//
// SoundArchiveFile::FileInfo
//
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchiveFile::InternalFileInfo::InvalidOffset );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchiveFile::InternalFileInfo::InvalidSize );

SoundArchiveFile::FileLocationType SoundArchiveFile::FileInfo::GetFileLocationType() const NN_NOEXCEPT
{
    switch ( toFileLocation.typeId )
    {
    case ElementType_SoundArchiveFile_InternalFileInfo:
        return FileLocationType_Internal;
    case ElementType_SoundArchiveFile_ExternalFileInfo:
        return FileLocationType_External;
    case 0:
        return FileLocationType_None;
    default:
        NN_SDK_ASSERT( false, "invalid file location type");
        return FileLocationType_None;
    }
}

const SoundArchiveFile::InternalFileInfo*
SoundArchiveFile::FileInfo::GetInternalFileInfo() const NN_NOEXCEPT
{
    if ( GetFileLocationType() != FileLocationType_Internal )
    {
        return NULL;
    }

    return reinterpret_cast<const InternalFileInfo*>(
            util::ConstBytePtr( this, toFileLocation.offset ).Get() );
}

const SoundArchiveFile::ExternalFileInfo*
SoundArchiveFile::FileInfo::GetExternalFileInfo() const NN_NOEXCEPT
{
    if ( GetFileLocationType() != FileLocationType_External )
    {
        return NULL;
    }

    return reinterpret_cast<const ExternalFileInfo*>(
            util::ConstBytePtr( this, toFileLocation.offset ).Get() );
}

#if 0
const void*
SoundArchiveFile::FileInfo::GetFileAddress( const void* dataBlockBodyAddress ) const NN_NOEXCEPT
{
    // TODO: toFileImage.typeId には、
    //  (1) FILE ブロックに入っていることを示す ID ((* この関数で扱う *))
    //  (2) 外部ファイルパスを示す ID [サウンドアーカイブファイル外]
    //  (3) 「グループ埋め込みファイル」テーブルを示す ID がありえる。
    //
    //  1-3 のうち、1/2/3 とも同時に存在しうる。
    //  ([1,3] と [2] は、同時に存在しない気もする。同時に存在する意味がないから。 )
    //
    //  TODO としては、typeId で 1 でないなら、この関数は NULL を返す必要がある
    return reinterpret_cast<const void*>(
            util::ConstBytePtr( dataBlockBodyAddress, toFileImage.offset ).Get() );
}
#endif

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
