﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/detail/atk_RegionManager.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/atk_StreamSoundLoader.h>

#include <nn/util/util_StringUtil.h>

namespace nn {
namespace atk {
namespace detail {

void RegionManager::Initialize() NN_NOEXCEPT
{
    m_IsRegionInfoEnabled = false;
    m_IsRegionIndexCheckEnabled = false;
    m_IsRegionInitialized = false;

    m_StreamRegionCallbackFunc = nullptr;
    m_StreamRegionCallbackArg = nullptr;

    m_AdpcmContextForStartOffsetFrame = 0xffffffff;
}


bool RegionManager::InitializeRegion( IRegionInfoReadable* pRegionReader, StreamDataInfoDetail* pStreamDataInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pRegionReader);
    NN_SDK_ASSERT_NOT_NULL(pStreamDataInfo);

    if (m_IsRegionInitialized)
    {
        return true;
    }

    m_CurrentRegionNo = 0;
    m_pCurrentRegionName = nullptr;
    m_IsCurrentRegionNameEnabled = false;

    StreamSoundFile::RegionInfo regionInfo;
    m_IsRegionInfoEnabled = pRegionReader->ReadRegionInfo(&regionInfo, 0);
    m_IsRegionIndexCheckEnabled = pStreamDataInfo->isRegionIndexCheckEnabled;

    if ( IsPreparedForRegionJump() )
    {
        NN_SDK_ASSERT(pStreamDataInfo->loopFlag == false, "Cannot use Loop and StreamJump at the same time.");

        if (!ChangeRegion( 0, pRegionReader, pStreamDataInfo ))
        {
            return false;
        }
    }
    else
    {
        if ( m_StreamRegionCallbackFunc != nullptr )
        {
            NN_ATK_WARNING("Cannot use regionCallback without regionInfo.");
        }
        m_CurrentRegion.begin = 0;
        m_CurrentRegion.end = pStreamDataInfo->sampleCount;
    }

    m_CurrentRegion.current = m_CurrentRegion.begin;

    m_IsRegionInitialized = true;

    return true;
}

bool RegionManager::ChangeRegion( int currentRegionNo, IRegionInfoReadable* pRegionReader, StreamDataInfoDetail* pStreamDataInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pRegionReader );
    NN_SDK_ASSERT_NOT_NULL( pStreamDataInfo );

    // ユーザーに次のリージョンを決定してもらう
    StreamRegionCallbackParam param;
    param.regionNo = currentRegionNo;
    param.regionCount = pStreamDataInfo->regionCount;
    param.pRegionInfoReader = pRegionReader;
    if (m_pCurrentRegionName != nullptr)
    {
        nn::util::Strlcpy(param.regionName, m_pCurrentRegionName, RegionNameLengthMax + 1);
    }
    else
    {
        std::memset(param.regionName, 0, RegionNameLengthMax + 1);
    }
    param.isRegionNameEnabled = m_IsCurrentRegionNameEnabled;
    if (m_StreamRegionCallbackFunc(&param, m_StreamRegionCallbackArg) != StreamRegionCallbackResult_Finish)
    {
        // 次のリージョンインデックスを取得
        if (param.isRegionNameEnabled)
        {
            // TODO: 高速化
            int targetRegionIndex = InvalidStreamJumpRegionIndex;
            for (auto i = 0; i < pStreamDataInfo->regionCount; ++i)
            {
                StreamSoundFile::RegionInfo regionInfo;
                pRegionReader->ReadRegionInfo(&regionInfo, i);

                if (nn::util::Strncmp(regionInfo.regionName, param.regionName, RegionNameLengthMax + 1) == 0)
                {
                    targetRegionIndex = i;
                    break;
                }
            }

            if (targetRegionIndex != InvalidStreamJumpRegionIndex)
            {
                m_CurrentRegionNo = targetRegionIndex;

                nn::util::Strlcpy(m_CurrentRegionName, param.regionName, RegionNameLengthMax + 1);
                m_pCurrentRegionName = m_CurrentRegionName;
                m_IsCurrentRegionNameEnabled = param.isRegionNameEnabled;
            }
            else
            {
                m_CurrentRegionNo = param.regionNo;
                m_pCurrentRegionName = nullptr;
                m_IsCurrentRegionNameEnabled = false;
            }
        }
        else
        {
            m_CurrentRegionNo = param.regionNo;
            m_pCurrentRegionName = nullptr;
            m_IsCurrentRegionNameEnabled = false;
        }
        NN_SDK_ASSERT(m_CurrentRegionNo != InvalidStreamJumpRegionIndex);
        NN_SDK_ASSERT_RANGE(m_CurrentRegionNo, 0, param.regionCount);

        // リージョン情報を取得し設定
        StreamSoundFile::RegionInfo regionInfo;
        bool result = pRegionReader->ReadRegionInfo( &regionInfo, m_CurrentRegionNo );
        if (result)
        {
            SetRegionInfo( &regionInfo, pStreamDataInfo );
            if (m_IsRegionIndexCheckEnabled && !regionInfo.isEnabled)
            {
                NN_ATK_WARNING("Invalid region index(%d) is selected.", m_CurrentRegionNo);
            }
        }
        else
        {
            SetRegionInfo( nullptr, pStreamDataInfo );
        }

        return true;
    }
    else
    {
        return false;
    }
}

bool RegionManager::TryMoveNextRegion( IRegionInfoReadable* pRegionReader, StreamDataInfoDetail* pStreamDataInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pRegionReader );
    NN_SDK_ASSERT_NOT_NULL( pStreamDataInfo );

    if ( pStreamDataInfo->loopFlag )
    {
        m_CurrentRegion.begin = pStreamDataInfo->loopStart;
        m_CurrentRegion.end = pStreamDataInfo->sampleCount;
        m_CurrentRegion.current = m_CurrentRegion.begin;

        return true;
    }

    if ( IsPreparedForRegionJump() )
    {
        if (ChangeRegion( m_CurrentRegionNo, pRegionReader, pStreamDataInfo ))
        {
            m_CurrentRegion.current = m_CurrentRegion.begin;
            return true;
        }
    }

    m_CurrentRegion.current = m_CurrentRegion.end;

    return false;
}


void RegionManager::SetPosition( position_t position ) NN_NOEXCEPT
{
    m_CurrentRegion.current = position;
}

void RegionManager::AddPosition( position_t position ) NN_NOEXCEPT
{
    m_CurrentRegion.current += position;
}

bool RegionManager::IsInFirstRegion() const NN_NOEXCEPT
{
    return m_CurrentRegionNo == 0;
}

bool RegionManager::IsPreparedForRegionJump() const NN_NOEXCEPT
{
    return (m_IsRegionInfoEnabled && m_StreamRegionCallbackFunc != nullptr);
}

void RegionManager::SetRegionInfo( const StreamSoundFile::RegionInfo* pRegionInfo, const StreamDataInfoDetail* pStreamDataInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pStreamDataInfo );

    if (pRegionInfo == nullptr)
    {
        // リージョン情報がない場合は全体を１つのリージョンとみなす
        m_CurrentRegion.begin = 0;
        m_CurrentRegion.end = pStreamDataInfo->sampleCount;
        m_CurrentRegion.isEnabled = true;
    }
    else
    {
        m_CurrentRegion.begin = pRegionInfo->start;
        m_CurrentRegion.end = pRegionInfo->end;
        m_CurrentRegion.isEnabled = m_IsRegionIndexCheckEnabled ? pRegionInfo->isEnabled : true;

        if ( pStreamDataInfo->sampleFormat == SampleFormat_DspAdpcm )
        {
            for ( auto ch = 0; ch < pStreamDataInfo->channelCount; ch++ )
            {
                m_AdpcmContextForStartOffset[ch].audioAdpcmContext.predScale = pRegionInfo->adpcmContext[ch].loopPredScale;
                m_AdpcmContextForStartOffset[ch].audioAdpcmContext.history[0] = pRegionInfo->adpcmContext[ch].loopYn1;
                m_AdpcmContextForStartOffset[ch].audioAdpcmContext.history[1] = pRegionInfo->adpcmContext[ch].loopYn2;
            }
            m_AdpcmContextForStartOffsetFrame = pRegionInfo->start;
        }
    }
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
