﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_PlayerHeapDataManager.h>

namespace nn {
namespace atk {
namespace detail {

NN_DEFINE_STATIC_CONSTANT( const int PlayerHeapDataManager::FileAddressCount );

PlayerHeapDataManager::PlayerHeapDataManager() NN_NOEXCEPT
: m_IsInitialized( false )
, m_IsFinalized ( true )
{
}

PlayerHeapDataManager::~PlayerHeapDataManager() NN_NOEXCEPT
{
    Finalize();
}

void PlayerHeapDataManager::Initialize( const SoundArchive* arc ) NN_NOEXCEPT
{
    if ( m_IsInitialized )
    {
        return;
    }
    m_IsInitialized = true;
    m_IsFinalized = false;

    for ( int i = 0; i < FileAddressCount; i++ )
    {
        m_FileAddress[i].fileId = SoundArchive::InvalidId;
        m_FileAddress[i].address = NULL;
    }
    SetSoundArchive( arc );
}

void PlayerHeapDataManager::Finalize() NN_NOEXCEPT
{
    if ( m_IsFinalized )
    {
        return;
    }
    m_IsInitialized = false;
    m_IsFinalized = true;
    SetSoundArchive( NULL );
}

const void* PlayerHeapDataManager::SetFileAddress(
        SoundArchive::FileId fileId, const void* address ) NN_NOEXCEPT
{
    return SetFileAddressToTable( fileId, address );
}
const void* PlayerHeapDataManager::GetFileAddress( SoundArchive::FileId fileId ) const NN_NOEXCEPT
{
    // ファイルの検索順は、SoundDataManager → PlayerHeapDataManager。
    // サウンドアーカイブ内からの検索は、SoundDataManager で行っているので、
    // ここでは、PlayerHeapDataManager 内の検索だけでよい。
    return GetFileAddressFromTable( fileId );
}

void PlayerHeapDataManager::InvalidateData( const void* /*start*/, const void* /*end*/ ) NN_NOEXCEPT
{
    // ファイルアドレステーブルから破棄
    for ( int i = 0; i < FileAddressCount; i++ )
    {
        m_FileAddress[i].fileId = SoundArchive::InvalidId;
        m_FileAddress[i].address = NULL;
    }
}

const void* PlayerHeapDataManager::SetFileAddressToTable(
        SoundArchive::FileId fileId, const void* address ) NN_NOEXCEPT
{
    // 同じ fileId がある場合
    for ( int i = 0; i< FileAddressCount; i++ )
    {
        if ( m_FileAddress[i].fileId == fileId )
        {
            const void* prev = m_FileAddress[i].address;
            m_FileAddress[i].address = address;
            return prev;
        }
    }

    // ない場合
    for ( int i = 0; i< FileAddressCount; i++ )
    {
        if ( m_FileAddress[i].fileId == SoundArchive::InvalidId )
        {
            m_FileAddress[i].fileId = fileId;
            m_FileAddress[i].address = address;
            return NULL;
        }
    }

    // ここにくることは無いはず
    NN_SDK_ASSERT( false, "invalid PlayerHeap" );
    return NULL;
}

const void* PlayerHeapDataManager::GetFileAddressFromTable( SoundArchive::FileId fileId ) const NN_NOEXCEPT
{
    for ( int i = 0; i< FileAddressCount; i++ )
    {
        if ( m_FileAddress[i].fileId == fileId )
        {
            return m_FileAddress[i].address;
        }
    }
    return NULL;
}

const void* PlayerHeapDataManager::GetFileAddressImpl( SoundArchive::FileId fileId ) const NN_NOEXCEPT
{
    return GetFileAddressFromTable( fileId );
}


} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

