﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundArchivePlayer.h>

#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_SoundDataManager.h>
#include <nn/atk/atk_SoundPlayer.h>
#include <nn/atk/atk_SoundHandle.h>
#include <nn/atk/atk_SoundActor.h>
#include <nn/atk/atk_SoundArchiveFilesHook.h>
#include <nn/atk/atk_SoundArchiveParametersHook.h>
#include <nn/atk/atk_Bank.h>
#include <nn/atk/atk_StreamSoundFile.h>
#include <nn/atk/atk_StreamSoundFileReader.h>
#include <nn/atk/atk_SequenceSoundFileReader.h>
#include <nn/atk/atk_WaveSoundFileReader.h>
#include <nn/atk/atk_WaveArchiveFileReader.h>
#include <nn/atk/atk_WaveFileReader.h>
#include <nn/atk/atk_Debug.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_PlayerHeap.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_Config.h>
#include <nn/atk/atk_StartInfoReader.h>
#include <nn/atk/atk_VoiceCommand.h>
#include <nn/atk/atk_Result.h>

#include <nn/atk/detail/atk_AddonSoundArchiveContainer.h>
#include <nn/atk/detail/atk_Macro.h>

#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/atk/fnd/io/atkfnd_FileStream.h>
#include <nn/atk/fnd/io/atkfnd_FileStreamImpl.h>
#include <nn/atk/fnd/string/atkfnd_String.h>
#include <nn/util/util_StringUtil.h>


// #define NN_ATK_DEBUG_PRINT_ENABLE
#ifdef NN_ATK_DEBUG_PRINT_ENABLE
    #define NN_ATK_INIT_LOG NN_DETAIL_ATK_INFO
#else
    #define NN_ATK_INIT_LOG(...)
#endif

namespace
{
    const int MemAlignSize = nn::audio::BufferAlignSize;
    const size_t FileLoadAlignSize = 256; //  StreamSoundLoader::g_LoadBuffer のアラインに合わせます

    size_t GetRequiredSoundArchivePlayerMemSizeWithoutStreamSoundInstance(const nn::atk::SoundArchive* arc, size_t userParamSizePerSound, int addonSoundArchiveCount) NN_NOEXCEPT
    {
        size_t result = nn::atk::SoundArchivePlayer::GetRequiredMemSize( arc, userParamSizePerSound, addonSoundArchiveCount );

        const size_t alignedStreamInstanceSize = nn::util::align_up( nn::atk::SoundArchivePlayer::GetRequiredStreamInstanceSize( arc ), nn::audio::MemoryPoolType::SizeGranularity );
        NN_SDK_ASSERT_GREATER_EQUAL( result, alignedStreamInstanceSize );
        result -= alignedStreamInstanceSize;

        return result;
    }
    nn::atk::OutputReceiver* GetOutputReceiver(const nn::atk::detail::StartInfoReader& reader, nn::atk::OutputReceiver* pDefaultOutputReceiver) NN_NOEXCEPT
    {
        nn::atk::OutputReceiver* pOutputReceiver = nullptr;
        auto& hardwareManager = nn::atk::detail::driver::HardwareManager().GetInstance();

        if( hardwareManager.IsPresetSubMixEnabled() )
        {
            pOutputReceiver = &hardwareManager.GetSubMix( reader.GetSubMixIndex() );
        }
        else
        {
            pOutputReceiver = reader.GetOutputReceiver();
            if( pOutputReceiver == nullptr )
            {
                pOutputReceiver = (pDefaultOutputReceiver != nullptr) ? pDefaultOutputReceiver : &hardwareManager.GetFinalMix();
            }
        }

        return pOutputReceiver;
    }
    inline const nn::atk::SoundArchive* GetTargetSoundArchive(const nn::atk::detail::SoundArchiveManager& soundArchiveManager, const char* soundArchiveName) NN_NOEXCEPT
    {
        if( soundArchiveName == nullptr )
        {
            return soundArchiveManager.GetMainSoundArchive();
        }
        else
        {
            return soundArchiveManager.GetAddonSoundArchive( soundArchiveName );
        }
    }
    inline const nn::atk::SoundDataManager* GetTargetSoundDataManager(const nn::atk::detail::SoundArchiveManager& soundArchiveManager, const char* soundArchiveName) NN_NOEXCEPT
    {
        if( soundArchiveName == nullptr )
        {
            return soundArchiveManager.GetMainSoundDataManager();
        }
        else
        {
            return soundArchiveManager.GetAddonSoundDataManager( soundArchiveName );
        }
    }
    inline size_t GetSoundPlayerInstanceSize(const nn::atk::detail::SoundInstanceConfig& config) NN_NOEXCEPT
    {
        size_t result = sizeof( nn::atk::SoundPlayer );
        const size_t AdditionalParamBufferSize = nn::atk::detail::OutputAdditionalParam::GetRequiredMemSize(config);
        if(AdditionalParamBufferSize > 0)
        {
            result += sizeof( nn::atk::detail::OutputAdditionalParam );
            result += AdditionalParamBufferSize;
        }

        return result;
    }
    inline void SetupSoundInstanceConfigForSoundPlayer(nn::atk::detail::SoundInstanceConfig& config) NN_NOEXCEPT
    {
        // SoundPlayer は BusMixVolume や VolumeThroughMode を BasicSound に伝えない
        config.isBusMixVolumeEnabled = false;
        config.isVolumeThroughModeEnabled = false;
    }
}

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchivePlayer::BufferAlignSize );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchivePlayer::StreamBufferTimesMax );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchivePlayer::UserParamBoundary );
NN_DEFINE_STATIC_CONSTANT( const int SoundArchivePlayer::AddonSoundArchiveNameLengthMax );

/*--------------------------------------------------------------------------------*
  Name:         SoundArchivePlayer

  Description:  コンストラクタ

  Arguments:    無し

  Returns:      無し
 *--------------------------------------------------------------------------------*/
SoundArchivePlayer::SoundArchivePlayer() NN_NOEXCEPT
: m_SoundPlayerCount( 0 )
, m_pSoundPlayers( NULL )
, m_SoundUserParamSize( 0 )
, m_ArchiveContainerCount( 0 )
, m_pArchiveContainers( nullptr )
, m_IsMemoryPoolForStreamInstanceAttached( false )
, m_IsMemoryPoolForPlayerHeapAttached( false )
, m_pSoundArchiveFilesHook( NULL )
, m_IsEnableWarningPrint(false)
, m_IsInitialized(false)
, m_IsAdvancedWaveSoundEnabled(false)
, m_pDefaultOutputReceiver( nullptr )
{
}

/*--------------------------------------------------------------------------------*
  Name:         SoundArchivePlayer

  Description:  デストラクタ

  Arguments:    無し

  Returns:      無し
 *--------------------------------------------------------------------------------*/
SoundArchivePlayer::~SoundArchivePlayer() NN_NOEXCEPT
{
}

/*--------------------------------------------------------------------------------*
  Name:         IsAvailable

  Description:  サウンドアーカイブプレイヤーが利用可能かどうかを調べる

  Arguments:    無し

  Returns:      利用可能かどうかを返す
 *--------------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsAvailable() const NN_NOEXCEPT
{
    return m_SoundArchiveManager.IsAvailable();
}

/*--------------------------------------------------------------------------------*
  Name:         Initialize

  Description:  サウンドアーカイブ中のプレイヤー情報にしたがって
                プレイヤーをセットアップする

  Arguments:    arc - サウンドアーカイブ
                buffer - セットアップに使用するメモリ
                size - メモリサイズ
                aramBuffer - セットアップに使用するＡメモリ
                aramSize - メモリサイズ

  Returns:      セットアップに成功したらtrue
 *--------------------------------------------------------------------------------*/
bool SoundArchivePlayer::Initialize(
    const SoundArchive* arc,
    const SoundDataManager* manager,
    void*   buffer,
    size_t  size,
    void*   strmBuffer,
    size_t  strmBufferSize,
    size_t  userParamSizePerSound
) NN_NOEXCEPT
{
    InitializeParam param;
    param.pSoundArchive = arc;
    param.pSoundDataManager = manager;
    param.pSetupBuffer = buffer;
    param.setupBufferSize = size;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmBufferSize;
    param.userParamSizePerSound = userParamSizePerSound;

    return Initialize(param);
}

/*--------------------------------------------------------------------------------*
  Name:         Finalize

  Description:  サウンドアーカイブプレイヤーの機能を停止させる

  Arguments:    無し

  Returns:      無し
 *--------------------------------------------------------------------------------*/
void SoundArchivePlayer::Finalize() NN_NOEXCEPT
{
    if (!m_IsInitialized)
    {
        return;
    }

    StopAllSound(0, true);

#if defined(NN_ATK_CONFIG_ENABLE_VOICE_COMMAND)
    // VoiceCommand 版の場合、SoundThread::FrameProcess() が実行されないので、
    // Release 処理中のチャンネルが止められなくなるため、このタイミングで停止を待つ。
    while (nn::atk::SoundSystem::GetVoiceCount() != 0)
    {
        nn::atk::SoundSystem::VoiceCommandProcess(1);
        if ( !nn::atk::SoundSystem::detail_IsSoundThreadEnabled() )
        {
            nn::atk::SoundSystem::VoiceCommandUpdate(); // SoundThread 無効時は手動でオーディオフレーム更新を行う
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(detail::driver::HardwareManager::SoundFrameIntervalMsec));
    }

    // DriverCommand の停止完了待ちの際に VoiceCommand を Push する関係で、
    // VoiceCommand にはまだ nn::audio の停止コマンドなどが Flush されずに残っている可能性があるため、
    // Flush して SoundThread::UpdateLowLevelVoices() で処理が行われるのを待つ。
    detail::LowLevelVoiceCommand& lowLevelVoiceCmdmgr = detail::LowLevelVoiceCommand::GetInstance();
    lowLevelVoiceCmdmgr.FlushCommand(true, false);
    while (lowLevelVoiceCmdmgr.GetCommandListCount() != 0)
    {
        if ( !nn::atk::SoundSystem::detail_IsSoundThreadEnabled() )
        {
            nn::atk::SoundSystem::VoiceCommandUpdate(); // SoundThread 無効時は手動でオーディオフレーム更新を行う
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(detail::driver::HardwareManager::SoundFrameIntervalMsec));
    }

    {
        detail::driver::SoundThreadLock lock;
        // VoiceCommand 版の場合、このタイミングで全ボイスの停止と開放処理が終わっている想定なので、それをチェックします
        if (detail::VirtualVoiceManager::GetInstance().GetUnreleasedLowLevelVoiceCount() != 0)
        {
            NN_ATK_WARNING("Unreleased Voice exists.");
        }
    }
#endif

    DisposeInstances();

    // メモリプールのデタッチ
    if ( m_IsMemoryPoolForStreamInstanceAttached )
    {
        nn::atk::SoundSystem::DetachMemoryPool( &m_MemoryPoolForStreamInstance );
        m_IsMemoryPoolForStreamInstanceAttached = false;
    }
    if ( m_IsMemoryPoolForPlayerHeapAttached )
    {
        nn::atk::SoundSystem::DetachMemoryPool( &m_MemoryPoolForPlayerHeap );
        m_IsMemoryPoolForPlayerHeapAttached = false;
    }

    m_pDefaultOutputReceiver = nullptr;
    m_IsInitialized = false;
}


/*--------------------------------------------------------------------------------*
  Name:         GetRequiredMemSize

  Description:  サウンドアーカイブプレイヤーで使用するメモリサイズを取得する

  Arguments:    arc - サウンドアーカイブ

  Returns:      無し
 *--------------------------------------------------------------------------------*/
size_t SoundArchivePlayer::GetRequiredMemSize(
        const SoundArchive* arc ) NN_NOEXCEPT
{
    return GetRequiredMemSize(arc, 0, 0);
}

size_t SoundArchivePlayer::GetRequiredMemSize(
        const SoundArchive* arc, size_t userParamSizePerSound ) NN_NOEXCEPT
{
    return GetRequiredMemSize(arc, userParamSizePerSound, 0);
}

size_t SoundArchivePlayer::GetRequiredMemSize(
        const SoundArchive* arc, size_t userParamSizePerSound, int addonSoundArchiveCount ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( arc );

    size_t size = 0;

    // SoundPlayer
    {
        detail::SoundInstanceConfig config = SoundSystem::GetSoundInstanceConfig();
        SetupSoundInstanceConfigForSoundPlayer( config );
        // PlayerHeap と StreamSoundRuntime のアライメントを揃えるため、確保するサイズを nn::audio::MemoryPoolType::SizeGranularity に統一
        uint32_t playerCount = arc->GetPlayerCount();
        size += nn::util::align_up( playerCount * GetSoundPlayerInstanceSize( config ), nn::audio::MemoryPoolType::SizeGranularity );
        NN_ATK_INIT_LOG("[SoundPlayer] instance(%d) x playerCount(%d), ALIGN(%d) => %d\n",
            GetSoundPlayerInstanceSize( config ), playerCount, nn::audio::MemoryPoolType::SizeGranularity, size);

        for ( uint32_t playerIndex = 0; playerIndex < playerCount; ++playerIndex )
        {
            SoundArchive::PlayerInfo playerInfo;
            if ( ! arc->ReadPlayerInfo( &playerInfo, detail::Util::GetMaskedItemId(
                playerIndex, detail::ItemType_Player ) ) )
            {
                continue;
            }

            if ( playerInfo.playerHeapSize > 0 )
            {
                for ( int i = 0; i < playerInfo.playableSoundMax ; i++ )
                {
                    size += nn::util::align_up( sizeof(detail::PlayerHeap), nn::audio::MemoryPoolType::SizeGranularity );
                    NN_ATK_INIT_LOG("[%d] [PlayerHeap:%d/%d] instance(%8d) ALIGN(%d) => %8d\n",
                        playerIndex, i, playerInfo.playableSoundMax,
                        sizeof(detail::PlayerHeap), nn::audio::MemoryPoolType::SizeGranularity, size);
                    size += nn::util::align_up( playerInfo.playerHeapSize, nn::audio::MemoryPoolType::SizeGranularity );
                    NN_ATK_INIT_LOG("[%d] [PlayerHeap:%d/%d] heapSize(%8d) ALIGN(%d) => %8d\n",
                        playerIndex, i, playerInfo.playableSoundMax,
                        playerInfo.playerHeapSize, nn::audio::MemoryPoolType::SizeGranularity, size);
                }
            }
        }
    }

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        // StreamSound
        size += detail::StreamSoundRuntime::GetRequiredMemorySize(soundArchivePlayerInfo, MemAlignSize);
        NN_ATK_INIT_LOG("[StreamSound] ALIGN(%d) => %8d\n", MemAlignSize, size);

        bool isAdvancedWaveSoundEnabled = soundArchivePlayerInfo.isAdvancedWaveSoundEnabled;

        // SequenceSound
        size += detail::SequenceSoundRuntime::GetRequiredMemorySize(soundArchivePlayerInfo, MemAlignSize);
        NN_ATK_INIT_LOG("[SequenceSound] ALIGN(%d) => %8d\n", MemAlignSize, size);

        // WaveSound
        if (isAdvancedWaveSoundEnabled)
        {
            size += detail::AdvancedWaveSoundRuntime::GetRequiredMemorySize(soundArchivePlayerInfo, MemAlignSize);
        }
        else
        {
            size += detail::WaveSoundRuntime::GetRequiredMemorySize(soundArchivePlayerInfo, MemAlignSize);
        }
        NN_ATK_INIT_LOG("[WaveSound] ALIGN(%d) => %8d\n", MemAlignSize, size);

        // SequenceTrack
        size += detail::SequenceSoundRuntime::GetRequiredSequenceTrackMemorySize(soundArchivePlayerInfo, MemAlignSize);
        NN_ATK_INIT_LOG("[SequenceSound] ALIGN(%d) => %8d\n", MemAlignSize, size);
    }

    // addonSoundArchive
    if (addonSoundArchiveCount > 0)
    {
        size_t containersSize = sizeof(detail::AddonSoundArchiveContainer) * addonSoundArchiveCount;
        size += nn::util::align_up( containersSize, MemAlignSize );
    }
    NN_ATK_INIT_LOG("[AddonSoundArchive] ALIGN(%d) => %8d\n", MemAlignSize, size);

    // userParam
    if (userParamSizePerSound > 0)
    {
        uint32_t soundCount =
            soundArchivePlayerInfo.sequenceSoundCount
          + soundArchivePlayerInfo.waveSoundCount
          + soundArchivePlayerInfo.streamSoundCount;
        size_t adjustSize = nn::util::align_up( userParamSizePerSound, SoundArchivePlayer::UserParamBoundary );
        size += (adjustSize * soundCount);
        NN_ATK_INIT_LOG("[UserParam] adjustSize(%d) x soundCount(%d) => size(%8d)\n",
                adjustSize, soundCount, size);
    }
    NN_ATK_INIT_LOG("[UserParam] ALIGN(%d) => %8d\n", MemAlignSize, size);

    return size;
}

size_t SoundArchivePlayer::GetRequiredMemSize(
        const InitializeParam& param ) NN_NOEXCEPT
{
    if( param.enablePreparingStreamInstanceBufferFromSetupBuffer )
    {
        return GetRequiredMemSize( param.pSoundArchive, param.userParamSizePerSound, param.addonSoundArchiveCount );
    }
    else
    {
        return GetRequiredSoundArchivePlayerMemSizeWithoutStreamSoundInstance( param.pSoundArchive, param.userParamSizePerSound, param.addonSoundArchiveCount );
    }
}

size_t SoundArchivePlayer::GetRequiredStreamInstanceSize(
    const nn::atk::SoundArchive* pArchive) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pArchive );
    nn::atk::SoundArchive::SoundArchivePlayerInfo info;
    if ( pArchive->ReadSoundArchivePlayerInfo( &info ) )
    {
        return detail::StreamSoundRuntime::GetRequiredStreamInstanceSize( info.streamSoundCount );
    }
    else
    {
        return 0;
    }
}

size_t SoundArchivePlayer::GetRequiredStreamBufferSize(
        const nn::atk::SoundArchive* arc ) const NN_NOEXCEPT
{
    return m_StreamSoundRuntime.GetRequiredStreamBufferSize(arc);
}

int32_t SoundArchivePlayer::GetRequiredStreamBufferTimes(
    const nn::atk::SoundArchive* arc ) const NN_NOEXCEPT
{
    return m_StreamSoundRuntime.GetRequiredStreamBufferTimes(arc);
}

size_t SoundArchivePlayer::GetRequiredStreamCacheSize(const SoundArchive* arc, size_t cacheSizePerSound) NN_NOEXCEPT
{
    return detail::StreamSoundRuntime::GetRequiredStreamCacheSize(arc, cacheSizePerSound);
}

bool SoundArchivePlayer::Initialize(const InitializeParam& param) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( param.pSoundArchive );
    NN_SDK_REQUIRES_NOT_NULL( param.pSoundDataManager );
    NN_SDK_REQUIRES_NOT_NULL( param.pSetupBuffer );
    NN_SDK_REQUIRES_ALIGNED( param.pSetupBuffer, SoundArchivePlayer::BufferAlignSize );
    NN_SDK_REQUIRES_ALIGNED( param.pStreamBuffer, nn::audio::BufferAlignSize );
    NN_SDK_REQUIRES_ALIGNED( param.pStreamInstanceBuffer, nn::audio::BufferAlignSize );
    NN_SDK_REQUIRES( SoundSystem::IsInitialized() );

    if (m_IsInitialized)
    {
        NN_ATK_WARNING("SoundArchivePlayer::Initialize failed (already initialized)");
        return false;
    }

    // 必要以上のメモリは不要・必要以下だとまずい
    NN_SDK_ASSERT( param.setupBufferSize == GetRequiredMemSize( param ) );

    // サウンドシステムの初期化が必要
    if ( ! SoundSystem::IsInitialized() )
    {
        NN_ATK_WARNING("SoundArchivePlayer::Initialize failed (soundSystem is not initialized)");
        return false;
    }

    nn::util::BytePtr setupBuffer( param.pSetupBuffer );
    size_t setupBufferSize = param.setupBufferSize;

    void* streamSoundInstanceBuffer = nullptr;
    size_t streamSoundInstanceBufferSize = 0;
    if( param.enablePreparingStreamInstanceBufferFromSetupBuffer )
    {
        // ストリームサウンドのインスタンスバッファを param.pSetupBuffer から切り出して用意します。
        NN_SDK_ASSERT( param.pStreamInstanceBuffer == NULL );

        // メモリプール管理が必要なバッファをアタッチ
        streamSoundInstanceBuffer = setupBuffer.Get();
        //NN_SDK_ASSERT_ALIGNED( streamSoundInstanceBuffer, nn::audio::MemoryPoolType::AddressAlignment ); のままだとコンパイルエラーになるため、static_cast を行います
        NN_SDK_ASSERT_ALIGNED( streamSoundInstanceBuffer, static_cast<size_t>( nn::audio::MemoryPoolType::AddressAlignment ) );
        streamSoundInstanceBufferSize = nn::util::align_up( GetRequiredStreamInstanceSize( param.pSoundArchive ), nn::audio::MemoryPoolType::SizeGranularity );

        if ( streamSoundInstanceBufferSize != 0 )
        {
            setupBuffer.Advance( streamSoundInstanceBufferSize );
            setupBufferSize -= streamSoundInstanceBufferSize;

            nn::atk::SoundSystem::AttachMemoryPool( &m_MemoryPoolForStreamInstance, streamSoundInstanceBuffer, streamSoundInstanceBufferSize );
            m_IsMemoryPoolForStreamInstanceAttached = true;
        }
    }
    else
    {
        // ストリームサウンドのインスタンスバッファはユーザから与えられています。
        NN_SDK_ASSERT_NOT_NULL( param.pStreamInstanceBuffer );
        NN_SDK_ASSERT_EQUAL( param.streamInstanceBufferSize, GetRequiredStreamInstanceSize(param.pSoundArchive) );

        streamSoundInstanceBuffer = param.pStreamInstanceBuffer;
        streamSoundInstanceBufferSize = param.streamInstanceBufferSize;
    }

    if ( ! SetupMram( param.pSoundArchive, setupBuffer.Get(), setupBufferSize, param.userParamSizePerSound, param.addonSoundArchiveCount, streamSoundInstanceBuffer, streamSoundInstanceBufferSize ) )
    {
        return false;
    }

    if ( ! m_StreamSoundRuntime.SetupStreamBuffer( param.pSoundArchive, param.pStreamBuffer, param.streamBufferSize ) )
    {
        return false;
    }

    m_SoundArchiveManager.Initialize(param.pSoundArchive, param.pSoundDataManager);
    m_AddonSoundArchiveLastAddTick = nn::os::Tick();

    m_SequenceSoundRuntime.SetSoundArchiveManager(&m_SoundArchiveManager);

    if (param.pStreamCacheBuffer)
    {
        bool setupStreamCacheBufferResult =
            m_StreamSoundRuntime.SetupStreamCacheBuffer(
                param.pSoundArchive,
                param.pStreamCacheBuffer,
                param.streamCacheSize
            );

        if (!setupStreamCacheBufferResult)
        {
            return false;
        }
    }

    m_IsInitialized = true;
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         SetupMram

  Description:  サウンドアーカイブ中のプレイヤー情報にしたがって
                プレイヤーをセットアップする

  Arguments:    arc - サウンドアーカイブ
                buffer - セットアップに使用するメモリ
                size - メモリサイズ

  Returns:      セットアップに成功したらtrue
 *--------------------------------------------------------------------------------*/
bool SoundArchivePlayer::SetupMram(
        const SoundArchive* pArc,
        void* buffer,
        size_t size,
        size_t userParamSizePerSound,
        int addonSoundArchiveCount,
        void* streamSoundInstanceBuffer,
        size_t streamSoundInstanceBufferSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_ALIGNED(buffer, SoundArchivePlayer::BufferAlignSize);
    if ( reinterpret_cast<uint64_t>(buffer) % SoundArchivePlayer::BufferAlignSize != 0 )
    {
        return false;
    }

    const void* endAddr = static_cast<char*>(buffer) + size;

    NN_ATK_INIT_LOG("[%s] end(%p) begin(%p)\n", __FUNCTION__, endAddr, buffer);

    void* curAddr = buffer;

    // メモリサイズにずれが起きないよう、アライメントが最も大きな PlayerHeap, StreamSoundRuntime から初期化する
    if ( ! SetupSoundPlayer( pArc, &curAddr, endAddr ) )
    {
        return false;
    }
    NN_ATK_INIT_LOG("[%s] cur(%p) - After SetupSoundPlayer\n", __FUNCTION__, curAddr);
    NN_ATK_INIT_LOG("size(%d)\n", static_cast<char*>(curAddr) - static_cast<char*>(buffer));

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( pArc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        if ( !m_StreamSoundRuntime.Initialize( soundArchivePlayerInfo.streamSoundCount, &curAddr, endAddr, streamSoundInstanceBuffer, streamSoundInstanceBufferSize ) )
        {
            return false;
        }
        NN_ATK_INIT_LOG("[%s] cur(%p) - After StreamSoundRuntime::Initialize\n", __FUNCTION__, curAddr);
        NN_ATK_INIT_LOG("size(%d)\n", static_cast<char*>(curAddr) - static_cast<char*>(buffer));

        m_IsAdvancedWaveSoundEnabled = soundArchivePlayerInfo.isAdvancedWaveSoundEnabled;

        if ( !m_SequenceSoundRuntime.Initialize( soundArchivePlayerInfo.sequenceSoundCount, &curAddr, endAddr ) )
        {
            return false;
        }
        NN_ATK_INIT_LOG("[%s] cur(%p) - After SequenceSoundRuntime::Initialize\n", __FUNCTION__, curAddr);
        NN_ATK_INIT_LOG("size(%d)\n", static_cast<char*>(curAddr) - static_cast<char*>(buffer));

        if ( m_IsAdvancedWaveSoundEnabled )
        {
            if ( !m_AdvancedWaveSoundRuntime.Initialize( soundArchivePlayerInfo.waveSoundCount, &curAddr, endAddr ) )
            {
                return false;
            }
            NN_ATK_INIT_LOG("[%s] cur(%p) - After AdvancedWaveSoundRuntime::Initialize\n", __FUNCTION__, curAddr);
        }
        else
        {
            if ( !m_WaveSoundRuntime.Initialize( soundArchivePlayerInfo.waveSoundCount, &curAddr, endAddr ) )
            {
                return false;
            }
            NN_ATK_INIT_LOG("[%s] cur(%p) - After WaveSoundRuntime::Initialize\n", __FUNCTION__, curAddr);
        }
        NN_ATK_INIT_LOG("size(%d)\n", static_cast<char*>(curAddr) - static_cast<char*>(buffer));

        if ( !m_SequenceSoundRuntime.SetupSequenceTrack( soundArchivePlayerInfo.sequenceTrackCount, &curAddr, endAddr ) )
        {
            return false;
        }
        NN_ATK_INIT_LOG("[%s] cur(%p) - After SequenceSoundRuntime::SetupSequenceTrack\n", __FUNCTION__, curAddr);
        NN_ATK_INIT_LOG("size(%d)\n", static_cast<char*>(curAddr) - static_cast<char*>(buffer));
    }

    if (addonSoundArchiveCount > 0)
    {
        if ( !SetupAddonSoundArchiveContainer(addonSoundArchiveCount, &curAddr, endAddr ) )
        {
            return false;
        }
        NN_ATK_INIT_LOG("[%s] cur(%p) - After SetupAddonSoundArchiveContainer\n", __FUNCTION__, curAddr);
        NN_ATK_INIT_LOG("size(%d)\n", static_cast<char*>(curAddr) - static_cast<char*>(buffer));
    }

    if (userParamSizePerSound > 0 )
    {
        // BasicSound へのユーザーパラメータ用バッファ割り当て
        if ( ! SetupUserParamForBasicSound(soundArchivePlayerInfo,
                    &curAddr, endAddr, userParamSizePerSound) )
        {
            return false;
        }
        NN_ATK_INIT_LOG("[%s] cur(%p) - After SetupUserParamForBasicSound\n", __FUNCTION__, curAddr);
        NN_ATK_INIT_LOG("size(%d)\n", static_cast<char*>(curAddr) - static_cast<char*>(buffer));
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    {
        ptrdiff_t useSize = static_cast<char*>(curAddr) - static_cast<char*>(buffer);
        size_t requiredSize = GetRequiredSoundArchivePlayerMemSizeWithoutStreamSoundInstance( pArc, userParamSizePerSound, addonSoundArchiveCount );

        NN_SDK_ASSERT(useSize == static_cast<ptrdiff_t>(requiredSize) ||
                  useSize + static_cast<ptrdiff_t>(nn::atk::detail::PpcIoBufferAlign - MemAlignSize) == static_cast<ptrdiff_t>(requiredSize));
    }
#endif

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         CreatePlayerHeap

  Description:  プレイヤーヒープの作成

  Arguments:    pOutAllocatedAddr - 確保済みのアドレスへのポインタ
                endAddr - ヒープ終端のアドレス
                heapSize - プレイヤーヒープサイズ

  Returns:      作成したプレイヤーヒープへのポインタ
 *--------------------------------------------------------------------------------*/
detail::PlayerHeap* SoundArchivePlayer::CreatePlayerHeap(
    void** pOutAllocatedAddr,
    const void* endAddr,
    size_t heapSize
) NN_NOEXCEPT
{
    // PlayerHeap のアライメントを揃えるため、確保するサイズを nn::audio::MemoryPoolType::AddressAlignment に統一
    NN_SDK_REQUIRES_ALIGNED(*pOutAllocatedAddr, nn::audio::MemoryPoolType::AddressAlignment);
    if ( reinterpret_cast<uint64_t>(*pOutAllocatedAddr) % nn::audio::MemoryPoolType::AddressAlignment != 0 )
    {
        return nullptr;
    }

    // プレイヤーヒープ構造体初期化
    void* estimateEndAddr = util::BytePtr(
            util::BytePtr( *pOutAllocatedAddr, sizeof(detail::PlayerHeap) ).Get() ).AlignUp( nn::audio::MemoryPoolType::AddressAlignment ).Get();
    if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
    {
        return nullptr;
    }
    void* curAddr = *pOutAllocatedAddr;
    *pOutAllocatedAddr = estimateEndAddr;
    NN_ATK_INIT_LOG("[%s] cur(%p) - in CreatePlayerHeap alloc instance\n", __FUNCTION__, estimateEndAddr);

    detail::PlayerHeap* playerHeap = new ( curAddr ) detail::PlayerHeap();

    // プレイヤーヒープ構築
    // (PlayerHeap インスタンスの後ろに、データロード用のバッファを用意する。)
    estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, nn::util::align_up( heapSize, nn::audio::MemoryPoolType::AddressAlignment ) ).Get();
    if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
    {
        return nullptr;
    }
    curAddr = *pOutAllocatedAddr;
    *pOutAllocatedAddr = estimateEndAddr;
    NN_ATK_INIT_LOG("[%s] cur(%p) - in CreatePlayerHeap alloc heap\n", __FUNCTION__, estimateEndAddr);

    bool result = playerHeap->Create( curAddr, heapSize );
    if ( ! result )
    {
        return nullptr;
    }

    return playerHeap;
}

bool SoundArchivePlayer::SetupSoundPlayer(
    const SoundArchive* pArc,
    void** pOutAllocatedAddr,
    const void* endAddr
) NN_NOEXCEPT
{
    // PlayerHeap のアライメントを揃えるため、確保するサイズを nn::audio::MemoryPoolType::SizeGranularity に統一
    NN_SDK_REQUIRES_ALIGNED(*pOutAllocatedAddr, nn::audio::MemoryPoolType::AddressAlignment);
    if ( reinterpret_cast<uint64_t>(*pOutAllocatedAddr) % nn::audio::MemoryPoolType::AddressAlignment != 0 )
    {
        return false;
    }

    detail::SoundInstanceConfig config = nn::atk::SoundSystem::GetSoundInstanceConfig();
    SetupSoundInstanceConfigForSoundPlayer( config );
    const size_t SoundPlayerSize = sizeof(SoundPlayer);
    const size_t AdditionalParamBufferSize = detail::OutputAdditionalParam::GetRequiredMemSize( config );
    const size_t AdditionalParamSize = AdditionalParamBufferSize > 0 ? sizeof( detail::OutputAdditionalParam ) : 0 ;

    uint32_t playerCount = pArc->GetPlayerCount();
    size_t requireSize = nn::util::align_up( playerCount * GetSoundPlayerInstanceSize( config ), nn::audio::MemoryPoolType::SizeGranularity );
    void* estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();

    if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
    {
        return false;
    }
    void* curAddr = *pOutAllocatedAddr;
    *pOutAllocatedAddr = estimateEndAddr;
    NN_ATK_INIT_LOG("[%s] cur(%p) - in SetupSoundPlayer\n", __FUNCTION__, curAddr);

    // curAddr 上に SoundPlayer インスタンスを playerCount 個並べる
    m_pSoundPlayers = reinterpret_cast<SoundPlayer*>( curAddr );
    m_SoundPlayerCount = playerCount;

    uint8_t* ptr = static_cast<uint8_t*>( curAddr );
    uint8_t* additionalParamPtr = ptr + SoundPlayerSize * playerCount;
    for ( uint32_t playerIndex = 0; playerIndex < playerCount;
            ++playerIndex, ptr += SoundPlayerSize )
    {
        // AdditionalParam 設定
        detail::OutputAdditionalParam* pAdditionalParam = nullptr;
        if ( AdditionalParamBufferSize > 0 )
        {
            new(additionalParamPtr) detail::OutputAdditionalParam();
            pAdditionalParam = reinterpret_cast<detail::OutputAdditionalParam*>(additionalParamPtr);
            additionalParamPtr += AdditionalParamSize;

            void* pAdditionalParamBuffer = additionalParamPtr;
            pAdditionalParam->Initialize(pAdditionalParamBuffer, AdditionalParamBufferSize, config);
            additionalParamPtr += AdditionalParamBufferSize;
        }

        // SoundPlayer インスタンス生成
        SoundPlayer* player = new ( ptr ) SoundPlayer( pAdditionalParam );

        SoundArchive::PlayerInfo playerInfo;
        if ( ! pArc->ReadPlayerInfo( &playerInfo, detail::Util::GetMaskedItemId(
                        playerIndex, detail::ItemType_Player ) ) )
        {
            continue;
        }

        player->SetPlayableSoundCount( playerInfo.playableSoundMax );

        // プレイヤーヒープのセットアップ
        if ( playerInfo.playerHeapSize > 0 )
        {
            for ( int i = 0; i < playerInfo.playableSoundMax ; i++ )
            {
                // プレイヤーヒープは SoundPlayer インスタンスの直後に配置する
                detail::PlayerHeap* playerHeap = CreatePlayerHeap(
                        pOutAllocatedAddr, endAddr, playerInfo.playerHeapSize );

                if ( playerHeap == nullptr )
                {
                    NN_ATK_WARNING("failed to create player heap. ( player id = %d )", playerIndex);
                    return false;
                }
                player->detail_AppendPlayerHeap( playerHeap );
            }

            // プレイヤーヒープを使用する場合、
            // 用意されたプレイヤーヒープ数を超える同時再生数は指定できない
            player->detail_SetPlayableSoundLimit( playerInfo.playableSoundMax );
        }
    }

    size_t playerHeapSizeDiff = static_cast<size_t>(util::ConstBytePtr(estimateEndAddr).Distance(*pOutAllocatedAddr));
    if(playerHeapSizeDiff != 0)
    {
        nn::atk::SoundSystem::AttachMemoryPool( &m_MemoryPoolForPlayerHeap, estimateEndAddr, playerHeapSizeDiff );
        m_IsMemoryPoolForPlayerHeapAttached = true;
    }

    return true;
}

bool SoundArchivePlayer::SetupAddonSoundArchiveContainer(
    int addonSoundArchiveCount,
    void** pOutAllocatedAddr,
    const void* endAddr
) NN_NOEXCEPT
{
    NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(*pOutAllocatedAddr), MemAlignSize) );
    if ( reinterpret_cast<uint64_t>(*pOutAllocatedAddr) % MemAlignSize != 0 )
    {
        return false;
    }

    size_t requireSize = nn::util::align_up( addonSoundArchiveCount * sizeof( detail::AddonSoundArchiveContainer ), MemAlignSize );
    void* estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();

    if ( util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
    {
        return false;
    }

    void* currentAddr = *pOutAllocatedAddr;
    *pOutAllocatedAddr = estimateEndAddr;
    NN_ATK_INIT_LOG("[%s] cur(%p) - in SetupAddonSoundArchiveContainer\n", __FUNCTION__, currentAddr);

    // curAddr 上に AddonSoundArchiveContainer インスタンスを addonSoundArchiveCount 個並べる
    m_pArchiveContainers = reinterpret_cast<detail::AddonSoundArchiveContainer*>( currentAddr );
    m_ArchiveContainerCount = addonSoundArchiveCount;

    // コンストラクタ呼び出し
    for (auto i = 0; i < m_ArchiveContainerCount; ++i)
    {
        new (reinterpret_cast<detail::AddonSoundArchiveContainer*>(&m_pArchiveContainers[i])) detail::AddonSoundArchiveContainer();
    }

    return true;
}

bool SoundArchivePlayer::SetupUserParamForBasicSound(
    const SoundArchive::SoundArchivePlayerInfo& info,
    void** pOutAllocatedAddr,
    const void* endAddr,
    size_t userParamSizePerSound) NN_NOEXCEPT
{
    NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(*pOutAllocatedAddr), MemAlignSize) );
    if (reinterpret_cast<uint64_t>(*pOutAllocatedAddr) % MemAlignSize != 0)
    {
        return false;
    }

    int soundCount = info.sequenceSoundCount + info.waveSoundCount + info.streamSoundCount;
    size_t adjustSize = nn::util::align_up( userParamSizePerSound, SoundArchivePlayer::UserParamBoundary);
    size_t requireSize = soundCount * adjustSize;
    void* estimateEndAddr = util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();
    void* curAddr = *pOutAllocatedAddr;
    *pOutAllocatedAddr = estimateEndAddr;

    // WSD 向け
    if ( m_IsAdvancedWaveSoundEnabled )
    {
        m_AdvancedWaveSoundRuntime.SetupUserParam(&curAddr, adjustSize);
    }
    else
    {
        m_WaveSoundRuntime.SetupUserParam(&curAddr, adjustSize);
    }

    // SEQ 向け
    m_SequenceSoundRuntime.SetupUserParam(&curAddr, adjustSize);

    // STRM 向け
    m_StreamSoundRuntime.SetupUserParam(&curAddr, adjustSize);

    m_SoundUserParamSize = adjustSize;

    if ( util::ConstBytePtr(endAddr).Distance(curAddr) > 0 )
    {
        return false;
    }

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         Update

  Description:  プレイヤーのフレーム処理を更新する

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundArchivePlayer::Update() NN_NOEXCEPT
{
    // 別スレッドのログをまとめて出力
    detail::Util::WarningLogger::GetInstance().SwapBuffer();
    if (m_IsEnableWarningPrint)
    {
        detail::Util::WarningLogger::GetInstance().Print();
    }

    for ( uint32_t playerIndex = 0;
            playerIndex < m_SoundPlayerCount; ++playerIndex )
    {
        GetSoundPlayer( detail::Util::GetMaskedItemId(
                    playerIndex, detail::ItemType_Player ) ).Update();
    }

    m_SequenceSoundRuntime.Update();
    if ( m_IsAdvancedWaveSoundEnabled )
    {
        m_AdvancedWaveSoundRuntime.Update();
    }
    else
    {
        m_WaveSoundRuntime.Update();
    }
    m_StreamSoundRuntime.Update();

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    cmdmgr.RecvCommandReply();
    cmdmgr.FlushCommand( false, false );
}

const SoundArchive& SoundArchivePlayer::GetSoundArchive() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_SoundArchiveManager.GetMainSoundArchive() != nullptr, "Setup is not completed." );
    return *m_SoundArchiveManager.GetMainSoundArchive();
}

const AddonSoundArchive* SoundArchivePlayer::GetAddonSoundArchive(const char* soundArchiveName) const NN_NOEXCEPT
{
    const SoundArchive* pArchive = m_SoundArchiveManager.GetAddonSoundArchive(soundArchiveName);
    if (pArchive == nullptr)
    {
        return nullptr;
    }

    return static_cast<const AddonSoundArchive*>(pArchive);
}

const AddonSoundArchive* SoundArchivePlayer::GetAddonSoundArchive(int index) const NN_NOEXCEPT
{
    auto pContainer = m_SoundArchiveManager.GetAddonSoundArchiveContainer(index);
    NN_SDK_ASSERT_NOT_NULL(pContainer);
    return static_cast<const AddonSoundArchive*>(pContainer->GetSoundArchive());
}

const char* SoundArchivePlayer::GetAddonSoundArchiveName(int index) const NN_NOEXCEPT
{
    auto pContainer = m_SoundArchiveManager.GetAddonSoundArchiveContainer(index);
    NN_SDK_ASSERT_NOT_NULL(pContainer);
    return pContainer->GetSoundArchiveName();
}

nn::os::Tick SoundArchivePlayer::GetAddonSoundArchiveAddTick(int index) const NN_NOEXCEPT
{
    auto pContainer = m_SoundArchiveManager.GetAddonSoundArchiveContainer(index);
    NN_SDK_ASSERT_NOT_NULL(pContainer);
    return pContainer->GetAddTick();
}

const SoundDataManager* SoundArchivePlayer::GetAddonSoundDataManager(const char* soundArchiveName) const NN_NOEXCEPT
{
    return m_SoundArchiveManager.GetAddonSoundDataManager(soundArchiveName);
}

SoundPlayer& SoundArchivePlayer::GetSoundPlayer( SoundArchive::ItemId playerId ) NN_NOEXCEPT
{
    return const_cast<SoundPlayer&>(
        static_cast<const SoundArchivePlayer*>( this )->GetSoundPlayer( playerId ) );
}

SoundPlayer& SoundArchivePlayer::GetSoundPlayer( const char* pStr ) NN_NOEXCEPT
{
    return const_cast<SoundPlayer&>(
        static_cast<const SoundArchivePlayer*>( this )->GetSoundPlayer( pStr ) );
}

const SoundPlayer& SoundArchivePlayer::GetSoundPlayer( SoundArchive::ItemId playerId ) const NN_NOEXCEPT
{
    uint32_t itemIndex = detail::Util::GetItemIndex( playerId );
    NN_SDK_ASSERT( itemIndex <= m_SoundPlayerCount - 1 );
    return m_pSoundPlayers[ itemIndex ];
}

const SoundPlayer& SoundArchivePlayer::GetSoundPlayer( const char* pStr ) const NN_NOEXCEPT
{
    const SoundArchive* pSoundArchive = m_SoundArchiveManager.GetMainSoundArchive();
    NN_SDK_ASSERT_NOT_NULL( pSoundArchive );

    SoundArchive::ItemId playerId = pSoundArchive->GetItemId( pStr );
    NN_SDK_ASSERT( playerId != SoundArchive::InvalidId );
    return GetSoundPlayer( playerId );
}

const void*
SoundArchivePlayer::detail_GetFileAddress( SoundArchive::FileId fileId ) const NN_NOEXCEPT
{
    const nn::atk::SoundDataManager* pSoundDataManager = m_SoundArchiveManager.GetMainSoundDataManager();

    if ( pSoundDataManager == NULL ) return NULL;

    return pSoundDataManager->detail_GetFileAddress( fileId );
}

void SoundArchivePlayer::AddAddonSoundArchive(const char* name, const AddonSoundArchive* pSoundArchive, const SoundDataManager* pSoundDataManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(m_ArchiveContainerCount, 0);
    NN_SDK_REQUIRES_NOT_NULL(m_pArchiveContainers);
    NN_SDK_REQUIRES_NOT_NULL(pSoundArchive);
    NN_SDK_REQUIRES(pSoundArchive->IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(pSoundDataManager);
    NN_SDK_REQUIRES(pSoundDataManager->IsAvailable());
    NN_SDK_REQUIRES_NOT_NULL(name);
    NN_SDK_REQUIRES_NOT_EQUAL(strnlen(name, AddonSoundArchiveNameLengthMax), static_cast<size_t>(AddonSoundArchiveNameLengthMax));

    detail::AddonSoundArchiveContainer* pContainer = nullptr;
    for (auto i = 0; i < m_ArchiveContainerCount; ++i)
    {
        if (!m_pArchiveContainers[i].IsActive())
        {
            pContainer = &m_pArchiveContainers[i];
            break;
        }
    }
    NN_SDK_ASSERT(pContainer != nullptr, "too many AddonSoundArchive added.");
    pContainer->Initialize(name, pSoundArchive, pSoundDataManager);

    AddonSoundArchive* pEditableSoundArchive = const_cast<AddonSoundArchive*>(pSoundArchive);
    NN_SDK_ASSERT_NOT_NULL(pEditableSoundArchive);
    pEditableSoundArchive->detail_SetParametersHook(m_SoundArchiveManager.GetParametersHook());

    // TODO: 同名の AddonSoundArchive に対する扱いの追加
    m_SoundArchiveManager.Add(*pContainer);

    nn::os::Tick tick = nn::os::GetSystemTick();
    pContainer->SetAddTick(tick);
    m_AddonSoundArchiveLastAddTick = tick;
}

void SoundArchivePlayer::RemoveAddonSoundArchive(const AddonSoundArchive* pSoundArchive) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER(m_ArchiveContainerCount, 0);
    NN_SDK_ASSERT_NOT_NULL(m_pArchiveContainers);

    detail::AddonSoundArchiveContainer* pContainer = nullptr;
    for (auto i = 0; i < m_ArchiveContainerCount; ++i)
    {
        if (m_pArchiveContainers[i].IsActive() && m_pArchiveContainers[i].GetSoundArchive() == pSoundArchive)
        {
            pContainer = &m_pArchiveContainers[i];
            break;
        }
    }
    if (pContainer == nullptr)
    {
        NN_ATK_WARNING("specified AddonSoundArchive is not found");
        return;
    }

    AddonSoundArchive* pEditableSoundArchive = const_cast<AddonSoundArchive*>(pSoundArchive);
    NN_SDK_ASSERT_NOT_NULL(pEditableSoundArchive);
    pEditableSoundArchive->detail_SetParametersHook(nullptr);

    pContainer->Finalize();

    m_SoundArchiveManager.Remove(*pContainer);
}

void SoundArchivePlayer::SetDefaultOutputReceiver(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
{
    m_pDefaultOutputReceiver = pOutputReceiver;
}

SoundStartable::StartResult SoundArchivePlayer::detail_SetupSound(
    SoundHandle* handle,
    uint32_t soundId,
    bool holdFlag,
    const char* soundArchiveName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    return detail_SetupSoundImpl(
        handle,
        soundId,
        NULL,
        NULL,
        holdFlag,
        soundArchiveName,
        startInfo
    );
}

/*--------------------------------------------------------------------------------*
  Name:         detail_SetupSoundImpl

  Description:  再生の実装関数

  Arguments:    handle  - サウンドハンドル
                soundId - サウンドＩＤ
                ambientArgInfo - 外部パラメータ
                actor - サウンドアクター
                holdFlag - ホールドサウンドフラグ

  Returns:      結果コード
 *--------------------------------------------------------------------------------*/
SoundStartable::StartResult SoundArchivePlayer::detail_SetupSoundImpl(
    SoundHandle* handle,
    uint32_t soundId,
    detail::BasicSound::AmbientInfo* ambientArgInfo,
    SoundActor* actor,
    bool holdFlag,
    const char* soundArchiveName,
    const StartInfo* startInfo
) NN_NOEXCEPT
{
    m_SoundArchiveManager.ChangeTargetArchive(soundArchiveName);
    const detail::SoundArchiveManager::SnapShot snapShot = m_SoundArchiveManager.GetSnapShot();
    const SoundArchive& soundArchive = snapShot.GetCurrentSoundArchive();

    NN_SDK_ASSERT_NOT_NULL( handle );

    if ( ! IsAvailable() )
    {
        SoundStartable::StartResult
            result( SoundStartable::StartResult::ResultCode_ErrorNotAvailable );
        return result;
    }

    // サウンドハンドルを古いサウンドから切断
    if ( handle->IsAttachedSound() ) {
        handle->DetachSound();
    }

    const char* soundLabel = soundArchive.GetItemLabel(soundId);
    SoundArchive::SoundType soundType = SoundArchive::SoundType_Invalid;

    // インゲーム編集のフック処理
    // フック対象でない場合は、処理途中でフック対象にならないように、フックを一時的に無効化する
    // 無効化中に別のスレッドからアクセスされることは考慮していない。
    volatile bool isHookDisabled = false;
    if(IsSoundArchiveFileHooksEnabled() && soundLabel != NULL)
    {
        NN_SDK_ASSERT_NOT_NULL(soundArchive.detail_GetParametersHook());
        NN_SDK_ASSERT(soundArchive.detail_GetParametersHook()->GetIsEnable());

        // 再生要求途中でフック状態が変更されないようにロックする。
        LockSoundArchiveFileHooks();

        if(!m_pSoundArchiveFilesHook->IsTargetItem(soundLabel))
        {
            soundArchive.detail_GetParametersHook()->SetIsEnable(false);
            m_pSoundArchiveFilesHook->SetIsEnable(false);
            isHookDisabled = true;

            soundType = soundArchive.GetSoundType( soundId );
        }
        else
        {
            soundType = soundArchive.detail_GetParametersHook()->GetSoundType( soundLabel );
            // 追加サウンドアーカイブのアイテムは、インゲーム編集時にメインサウンドアーカイブのアイテムとして扱われるので、
            // メインサウンドアーカイブに存在している方のアイテム ID を取得する
            if(soundArchiveName != nullptr)
            {
                soundId = soundArchive.detail_GetParametersHook()->GetItemId( soundLabel );
            }

            // 編集対象かどうかをチェック
            switch (soundType)
            {
            case SoundArchive::SoundType_Sequence:
                isHookDisabled = !IsSequenceSoundEdited(soundLabel);
                break;

            case SoundArchive::SoundType_Stream:
                isHookDisabled = !IsStreamSoundEdited(soundLabel);
                break;

            case SoundArchive::SoundType_Wave:
                isHookDisabled = !IsWaveSoundEdited(soundLabel);
                break;

            default:
                break;
            }

            if(isHookDisabled)
            {
                soundArchive.detail_GetParametersHook()->SetIsEnable(false);
                m_pSoundArchiveFilesHook->SetIsEnable(false);
            }
        }
    }
    else
    {
        // HACK : 無効なIDが渡されたときは、ラベルが不正とみなしてます。
        if ( soundId == SoundArchive::InvalidId )
        {
            StartResult result( StartResult::ResultCode_ErrorInvalidLabelString );
            return result;
        }

        soundType = soundArchive.GetSoundType( soundId );
    }

    // サウンドアーカイブからサウンド情報の取得
    SoundArchive::SoundInfo soundInfo;
    if ( ! soundArchive.ReadSoundInfo( &soundInfo, soundId ) )
    {
        EnableHook(&soundArchive, isHookDisabled);
        SoundStartable::StartResult
            result( SoundStartable::StartResult::ResultCode_ErrorInvalidSoundId );
        return result;
    }

    // StartInfo の読み込み
    detail::StartInfoReader startInfoReader(soundInfo);
    startInfoReader.Read(startInfo);

    // 出力先の OutputReceiver を決定
    OutputReceiver* pOutputReceiver = GetOutputReceiver( startInfoReader, m_pDefaultOutputReceiver );

    // 単一再生機能による前準備
    if ( soundInfo.singlePlayType != SinglePlayType_None )
    {
        const SoundStartable::StartResult result = PreprocessSinglePlay(
            soundInfo,
            soundId,
            GetSoundPlayer( startInfoReader.GetPlayerId() ) );

        if( !result.IsSuccess() )
        {
            EnableHook(&soundArchive, isHookDisabled);
            return result;
        }
    }

    // プライオリティの計算
    int playerPriority = startInfoReader.GetPlayerPriority();
    if ( holdFlag ) --playerPriority; // HoldSoundで再生するときは同プライオリティで先着優先

    int ambientPriority = 0;
    if ( ambientArgInfo != NULL )
    {
        ambientPriority = detail::BasicSound::GetAmbientPriority( *ambientArgInfo, soundId );
    }
    int allocPriority = playerPriority + ambientPriority;
    allocPriority = nn::atk::detail::fnd::Clamp( allocPriority, nn::atk::PlayerPriorityMin, nn::atk::PlayerPriorityMax );


    // ExternalSoundPlayer
    detail::ExternalSoundPlayer* extPlayer = NULL;
    if ( actor != NULL ) {
        extPlayer = actor->detail_GetActorPlayer( startInfoReader.GetActorPlayerId() );
        if ( extPlayer == NULL )
        {
            NN_ATK_WARNING(
                "actorPlayerId(%d) is out of range. (0-%d)",
                startInfoReader.GetActorPlayerId(),
                SoundActor::ActorPlayerCount - 1
            );
            EnableHook(&soundArchive, isHookDisabled);
            return StartResult(StartResult::ResultCode_ErrorInvalidParameter);
        }
    }

    SoundArchive::ItemId playerId = startInfoReader.GetPlayerId();

    // インゲーム編集中は、メインアーカイブから名前で playerId を取得しているため以下の処理は不要
    if ( !IsSoundArchiveFileHooksEnabled() )
    {
        if (soundArchive.IsAddon())
        {
            const char* playerName = soundArchive.GetItemLabel(soundInfo.playerId);

            playerId = snapShot.GetMainSoundArchive().GetItemId(playerName);
        }
    }

    // プレイヤーの空きをチェック
    if ( playerId == SoundArchive::InvalidId )
    {
        EnableHook(&soundArchive, isHookDisabled);
        return StartResult(StartResult::ResultCode_ErrorInvalidParameter);
    }
    SoundPlayer& player = GetSoundPlayer( playerId );
    if ( ! player.detail_CanPlaySound( allocPriority ) ) {
        EnableHook(&soundArchive, isHookDisabled);
        return StartResult(StartResult::ResultCode_ErrorLowPriority);
    }
    if ( ( extPlayer != NULL ) && ( ! extPlayer->CanPlaySound( allocPriority ) ) ) {
        EnableHook(&soundArchive, isHookDisabled);
        return StartResult(StartResult::ResultCode_ErrorLowPriority);
    }

    // サウンドのアロケート
    detail::BasicSound* sound = NULL;
    detail::SequenceSound* seqSound = NULL;
    detail::StreamSound* strmSound = NULL;
    detail::WaveSound* waveSound = NULL;
    detail::AdvancedWaveSound* advancedWaveSound = nullptr;

    switch ( soundType )
    {
    case SoundArchive::SoundType_Sequence:
        seqSound = m_SequenceSoundRuntime.AllocSound(
            soundId,
            playerPriority,
            ambientPriority,
            ambientArgInfo,
            pOutputReceiver
        );
        if ( seqSound == NULL )
        {
            if (detail::Debug_GetWarningFlag( DebugWarningFlag_NotEnoughSeqsound ))
            {
                // TODO: DriverCommand の枯渇による BasicSound 確保の失敗と、インスタンス不足による確保の失敗を区別する
                NN_ATK_WARNING(
                    "Failed to start sound (id:0x%08x) for not enough SequenceSound instance.",
                    soundId
                    );
            }
            EnableHook(&soundArchive, isHookDisabled);
            SoundStartable::StartResult
                result( SoundStartable::StartResult::ResultCode_ErrorNotEnoughInstance );
            return result;
        }
        sound = seqSound;
        break;

    case SoundArchive::SoundType_Stream:
        strmSound = m_StreamSoundRuntime.AllocSound(
            soundId,
            playerPriority,
            ambientPriority,
            ambientArgInfo,
            pOutputReceiver
        );
        if ( strmSound == NULL )
        {
            if (detail::Debug_GetWarningFlag( DebugWarningFlag_NotEnoughStrmsound ))
            {
                // TODO: DriverCommand の枯渇による BasicSound 確保の失敗と、インスタンス不足による確保の失敗を区別する
                NN_ATK_WARNING(
                    "Failed to start sound (id:0x%08x) for not enough StreamSound instance.",
                    soundId
                );
            }
            EnableHook(&soundArchive, isHookDisabled);
            SoundStartable::StartResult
                result( SoundStartable::StartResult::ResultCode_ErrorNotEnoughInstance );
            return result;
        }
        sound = strmSound;
        break;

    case SoundArchive::SoundType_Wave:
        if ( m_IsAdvancedWaveSoundEnabled )
        {
            advancedWaveSound = m_AdvancedWaveSoundRuntime.AllocSound(
                soundId,
                playerPriority,
                ambientPriority,
                ambientArgInfo,
                pOutputReceiver
            );
            if ( advancedWaveSound == NULL )
            {
                if (detail::Debug_GetWarningFlag( DebugWarningFlag_NotEnoughWavesound ))
                {
                    // TODO: DriverCommand の枯渇による BasicSound 確保の失敗と、インスタンス不足による確保の失敗を区別する
                    NN_ATK_WARNING(
                        "Failed to start sound (id:0x%08x) for not enough AdvancedWaveSound instance.",
                        soundId
                    );
                }
                EnableHook(&soundArchive, isHookDisabled);
                SoundStartable::StartResult
                    result( SoundStartable::StartResult::ResultCode_ErrorNotEnoughInstance );
                return result;
            }
            sound = advancedWaveSound;
        }
        else
        {
            waveSound = m_WaveSoundRuntime.AllocSound(
                soundId,
                playerPriority,
                ambientPriority,
                ambientArgInfo,
                pOutputReceiver
            );
            if ( waveSound == NULL )
            {
                if (detail::Debug_GetWarningFlag( DebugWarningFlag_NotEnoughWavesound ))
                {
                    // TODO: DriverCommand の枯渇による BasicSound 確保の失敗と、インスタンス不足による確保の失敗を区別する
                    NN_ATK_WARNING(
                        "Failed to start sound (id:0x%08x) for not enough WaveSound instance.",
                        soundId
                    );
                }
                EnableHook(&soundArchive, isHookDisabled);
                SoundStartable::StartResult
                    result( SoundStartable::StartResult::ResultCode_ErrorNotEnoughInstance );
                return result;
            }
            sound = waveSound;
        }
        break;

    default:
        {
            EnableHook(&soundArchive, isHookDisabled);
            SoundStartable::StartResult
                result( SoundStartable::StartResult::ResultCode_ErrorInvalidSoundId );
            return result;
        }
    }

    // サウンドプレイヤーに登録
    if ( ! player.detail_AppendSound( sound ) )
    {
        sound->Finalize();
        EnableHook(&soundArchive, isHookDisabled);
        SoundStartable::StartResult
            result( SoundStartable::StartResult::ResultCode_ErrorUnknown );
        return result;
    }

    // StartInfoの処理（BasicSoundのAlloc後に行う必要があるもの）
    if (startInfo != NULL)
    {
        if ( startInfo->enableFlag & StartInfo::EnableFlagBit_SoundStopCallback )
        {
            sound->SetSoundStopCallback( startInfo->soundStopCallback );
        }
    }

    sound->ResetOutputLine();

    // サウンドのプリペア
    switch ( soundType )
    {
    case SoundArchive::SoundType_Sequence:
    {
        StartResult result = m_SequenceSoundRuntime.PrepareImpl(
            snapShot,
            soundId,
            seqSound,
            &soundInfo,
            startInfoReader
        );
        if ( ! result.IsSuccess() )
        {
            seqSound->Finalize();
            EnableHook(&soundArchive, isHookDisabled);
            return result;
        }
        break;
    }

    case SoundArchive::SoundType_Stream:
    {
        StartResult result = m_StreamSoundRuntime.PrepareImpl(
            &soundArchive,
            &snapShot.GetCurrentSoundDataManager(),
            soundId,
            strmSound,
            &soundInfo,
            startInfoReader
        );
        if ( ! result.IsSuccess() )
        {
            strmSound->Finalize();
            EnableHook(&soundArchive, isHookDisabled);
            return result;
        }
        break;
    }

    case SoundArchive::SoundType_Wave:
    {
        if ( m_IsAdvancedWaveSoundEnabled )
        {
            StartResult result = m_AdvancedWaveSoundRuntime.PrepareImpl(
                &soundArchive,
                &snapShot.GetCurrentSoundDataManager(),
                soundId,
                advancedWaveSound,
                &soundInfo,
                startInfoReader
            );

            if ( ! result.IsSuccess() )
            {
                //advancedWaveSound->Finalize();
                EnableHook(&soundArchive, isHookDisabled);
                return result;
            }
        }
        else
        {
            StartResult result = m_WaveSoundRuntime.PrepareImpl(
                &soundArchive,
                &snapShot.GetCurrentSoundDataManager(),
                soundId,
                waveSound,
                &soundInfo,
                startInfoReader
            );

            if ( ! result.IsSuccess() )
            {
                waveSound->Finalize();
                EnableHook(&soundArchive, isHookDisabled);
                return result;
            }
        }

        break;
    }

    default:
        NN_SDK_ASSERT( false );
        sound->Finalize();
        {
            EnableHook(&soundArchive, isHookDisabled);
            return StartResult(StartResult::ResultCode_ErrorInvalidSoundId);
        }
    }

    EnableHook(&soundArchive, isHookDisabled);

    SetCommonSoundParam( sound, &soundInfo );

    // 外部プレイヤーに登録
    if ( extPlayer != NULL )
    {
        if ( ! extPlayer->AppendSound( sound ) )
        {
            sound->Finalize();
            return StartResult(StartResult::ResultCode_ErrorUnknown);
        }
    }

    // アクターに登録
    if ( actor != NULL )
    {
        sound->AttachSoundActor( actor );
    }

    // プライオリティの再設定
    if ( holdFlag )
    {
        sound->SetPlayerPriority( startInfoReader.GetPlayerPriority() );
    }

    // サウンドアーカイブの関連付け
    sound->SetSoundArchive(&soundArchive);

    // Setup を終えた Tick を記録します
    sound->SetSetupTick( nn::os::GetSystemTick() );

    // サウンドハンドルを確保したサウンドと接続
    handle->detail_AttachSound( sound );

    SoundStartable::StartResult
        result( SoundStartable::StartResult::ResultCode_Success );
    return result;
} // NOLINT(impl/function_size)

void SoundArchivePlayer::SetCommonSoundParam(
    detail::BasicSound* sound,
    const SoundArchive::SoundInfo* commonInfo
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(sound);
    NN_SDK_ASSERT_NOT_NULL(commonInfo);

    sound->SetInitialVolume( static_cast<float>( commonInfo->volume ) / 127.0f );
    sound->SetPanMode( commonInfo->panMode );
    sound->SetPanCurve( commonInfo->panCurve );
}

SoundStartable::StartResult SoundArchivePlayer::PreprocessSinglePlay(const SoundArchive::SoundInfo& info, uint32_t soundId, SoundPlayer& player) NN_NOEXCEPT
{
    const nn::os::Tick currentTick = nn::os::GetSystemTick();
    nn::TimeSpan effectiveDuration = nn::TimeSpan::FromMilliSeconds( info.singlePlayEffectiveDuration );

    switch( info.singlePlayType )
    {
        case SinglePlayType_None:
            // 何もしません
            break;

        case SinglePlayType_PrioritizeOldest:
            // 効果時間が指定されていない場合は、十分長い効果時間を設定して
            // SinglePlayType_PrioritizeOldestWithDuration と同じ処理をします
            effectiveDuration = nn::TimeSpan::FromDays( 365 );
            NN_FALL_THROUGH;

        case SinglePlayType_PrioritizeOldestWithDuration:
        {
            // プレイヤが再生中のサウンドについて、サウンド ID が同じものがあるかを探します
            bool isFound = false;
            nn::os::Tick setupTick;

            player.ForEachSound(
                [=,&isFound,&setupTick](SoundHandle& handle)
                {
                    if( handle.GetId() == soundId )
                    {
                        isFound = true;
                        setupTick = handle.detail_GetAttachedSound()->GetSetupTick();
                    }
                }
            );

            // サウンド ID が同じものがあった場合、効果時間内であるかどうかを判定します
            if( isFound && ( currentTick - setupTick ).ToTimeSpan() < effectiveDuration )
            {
                return SoundStartable::StartResult( SoundStartable::StartResult::ResultCode_CanceledBySinglePlay );
            }
            break;
        }

        case SinglePlayType_PrioritizeNewest:
            // 効果時間が指定されていない場合は、十分長い効果時間を設定して
            // SinglePlayType_PrioritizeNewestWithDuration と同じ処理をします
            effectiveDuration = nn::TimeSpan::FromDays( 365 );
            NN_FALL_THROUGH;

        case SinglePlayType_PrioritizeNewestWithDuration:
        {
            // プレイヤが再生中のサウンドについて、サウンド ID が同じならば止めます
            player.ForEachSound(
                [=](SoundHandle& handle)
                {
                    // サウンド ID が同じものがあった場合、効果時間内であるかどうかを判定します
                    if( handle.GetId() == soundId && ( currentTick - handle.detail_GetAttachedSound()->GetSetupTick() ).ToTimeSpan() < effectiveDuration )
                    {
                        handle.Stop( 0 );
                    }
                }
            );
            break;
        }

        default:
            NN_UNEXPECTED_DEFAULT;
    }

    return SoundStartable::StartResult( SoundStartable::StartResult::ResultCode_Success );
}

void SoundArchivePlayer::EnableHook(const SoundArchive* pSoundArchive, bool isHookDisabled) NN_NOEXCEPT
{
    if(isHookDisabled)
    {
        pSoundArchive->detail_GetParametersHook()->SetIsEnable(true);
        m_pSoundArchiveFilesHook->SetIsEnable(true);
    }
    UnlockSoundArchiveFileHooks();
}

/*--------------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsSoundArchiveFileHooksEnabled() const NN_NOEXCEPT
{
    return m_pSoundArchiveFilesHook != NULL && m_pSoundArchiveFilesHook->GetIsEnable();
}

/*--------------------------------------------------------------------------------*/
void SoundArchivePlayer::LockSoundArchiveFileHooks() NN_NOEXCEPT
{
    if(IsSoundArchiveFileHooksEnabled())
    {
        m_pSoundArchiveFilesHook->Lock();
    }
}

/*--------------------------------------------------------------------------------*/
void SoundArchivePlayer::UnlockSoundArchiveFileHooks() NN_NOEXCEPT
{
    if(IsSoundArchiveFileHooksEnabled())
    {
        m_pSoundArchiveFilesHook->Unlock();
    }
}

/*--------------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsSequenceSoundEdited(const char* soundName) const NN_NOEXCEPT
{
    if(soundName == NULL)
    {
        return false;
    }

    return m_pSoundArchiveFilesHook->GetFileAddress(
        soundName,
        detail::SoundArchiveFilesHook::ItemTypeSequenceSound,
        detail::SoundArchiveFilesHook::FileTypeSequenceBinary) != NULL;
}

/*--------------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsStreamSoundEdited(const char* soundName) const NN_NOEXCEPT
{
    if(soundName == NULL)
    {
        return false;
    }

    return true;
}

/*--------------------------------------------------------------------------------*/
bool SoundArchivePlayer::IsWaveSoundEdited(const char* soundName) const NN_NOEXCEPT
{
    if(soundName == NULL)
    {
        return false;
    }

    return m_pSoundArchiveFilesHook->GetFileAddress(
        soundName,
        detail::SoundArchiveFilesHook::ItemTypeWaveSound,
        detail::SoundArchiveFilesHook::FileTypeWaveSoundBinary) != NULL;
}

/*--------------------------------------------------------------------------------*
  Name:         SetSequenceUserProcCallback

  Description:  シーケンスコマンド'userproc'で呼び出されるコールバックを登録する

  Arguments:    callback - コールバック
                arg - コールバック引数

  Returns:
 *--------------------------------------------------------------------------------*/
void SoundArchivePlayer::SetSequenceUserProcCallback( SequenceUserProcCallback callback, void* callbackArg ) NN_NOEXCEPT
{
    m_SequenceSoundRuntime.SetSequenceUserProcCallback(callback, callbackArg);
}

void SoundArchivePlayer::SetSequenceSkipIntervalTick( int intervalTick ) NN_NOEXCEPT
{
    detail::SequenceSoundRuntime::SetSequenceSkipIntervalTick( intervalTick );
}

int SoundArchivePlayer::GetSequenceSkipIntervalTick() NN_NOEXCEPT
{
    return detail::SequenceSoundRuntime::GetSequenceSkipIntervalTick();
}

nn::Result SoundArchivePlayer::ReadWaveSoundDataInfo(WaveSoundDataInfo* pOutInfo, SoundArchive::ItemId waveSoundId, const SoundArchive* pSoundArchive, const SoundDataManager* pSoundDataManager) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutInfo );
    NN_SDK_REQUIRES_NOT_NULL( pSoundArchive );
    NN_SDK_REQUIRES_NOT_NULL( pSoundDataManager );

    // サウンドアーカイブからサウンド情報の取得
    SoundArchive::SoundInfo soundInfo;
    if ( !pSoundArchive->ReadSoundInfo( &soundInfo, waveSoundId ) )
    {
        return ResultSoundArchivePlayerInvalidId();
    }

    // ウェーブサウンド情報の取得
    SoundArchive::WaveSoundInfo waveSoundInfo;
    if ( !pSoundArchive->detail_ReadWaveSoundInfo( waveSoundId, &waveSoundInfo ) )
    {
        return ResultSoundArchivePlayerInvalidId();
    }

    const void* pWsdFile = pSoundDataManager->detail_GetFileAddress( soundInfo.fileId );
    if ( pWsdFile == NULL )
    {
        return ResultSoundArchivePlayerSoundNotLoaded();
    }

    const void* pWaveFile =
        detail::Util::GetWaveFileOfWaveSound(
            pWsdFile,
            waveSoundInfo.index,
            *pSoundArchive,
            *pSoundDataManager
        );

    if ( pWaveFile == NULL )
    {
        return ResultSoundArchivePlayerSoundNotLoaded();
    }

    detail::WaveFileReader waveFileReader( pWaveFile );
    detail::WaveInfo waveInfo;
    if ( !waveFileReader.ReadWaveInfo( &waveInfo ) )
    {
        return ResultSoundArchivePlayerInvalidFormat();
    }

    pOutInfo->loopFlag = waveInfo.loopFlag;
    pOutInfo->sampleRate = static_cast<int>( waveInfo.sampleRate );
    pOutInfo->loopStart = waveInfo.originalLoopStartFrame;
    pOutInfo->loopEnd = waveInfo.loopEndFrame - (waveInfo.loopStartFrame - waveInfo.originalLoopStartFrame);
    pOutInfo->compatibleLoopStart = waveInfo.loopStartFrame;
    pOutInfo->compatibleLoopEnd = waveInfo.loopEndFrame;
    pOutInfo->channelCount = std::min( waveInfo.channelCount, 2 );

    return nn::ResultSuccess();
}

nn::Result SoundArchivePlayer::ReadWaveSoundDataInfo(WaveSoundDataInfo* pOutInfo, SoundArchive::ItemId waveSoundId, const char* soundArchiveName) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutInfo );
    NN_SDK_REQUIRES( m_IsInitialized );

    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );
    const SoundDataManager* pSoundDataManager = GetTargetSoundDataManager( m_SoundArchiveManager, soundArchiveName );

    return ReadWaveSoundDataInfo(pOutInfo, waveSoundId, pSoundArchive, pSoundDataManager);
}

nn::Result SoundArchivePlayer::ReadWaveSoundDataInfo(WaveSoundDataInfo* pOutInfo, const char* waveSoundName, const char* soundArchiveName) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutInfo );
    NN_SDK_REQUIRES( m_IsInitialized );

    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );
    const SoundDataManager* pSoundDataManager = GetTargetSoundDataManager( m_SoundArchiveManager, soundArchiveName );

    const SoundArchive::ItemId id = pSoundArchive->GetItemId( waveSoundName );
    if ( id == SoundArchive::InvalidId )
    {
        return ResultSoundArchivePlayerInvalidName();
    }

    return ReadWaveSoundDataInfo( pOutInfo, id, pSoundArchive, pSoundDataManager );
}

nn::Result SoundArchivePlayer::ReadWaveSoundDataInfo(WaveSoundDataInfo* pOutInfo, SoundArchive::ItemId waveSoundId) const NN_NOEXCEPT
{
    return ReadWaveSoundDataInfo(pOutInfo, waveSoundId, nullptr);
}

nn::Result SoundArchivePlayer::ReadWaveSoundDataInfo(WaveSoundDataInfo* pOutInfo, const char* waveSoundName) const NN_NOEXCEPT
{
    return ReadWaveSoundDataInfo(pOutInfo, waveSoundName, nullptr);
}

nn::Result SoundArchivePlayer::ReadStreamSoundDataInfo(StreamSoundDataInfo* pOutInfo, const SoundArchive* pSoundArchive, SoundArchive::ItemId streamSoundId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_NOT_NULL( pSoundArchive );

    //  STRM が再生中でないことを確認
    bool isStreamSoundPlaying = false;
    for(uint32_t i = 0; i < m_SoundPlayerCount; i++)
    {
        m_pSoundPlayers[i].ForEachSound( [=,&isStreamSoundPlaying](const SoundHandle& handle){
            if( handle.GetId() == streamSoundId )
            {
                isStreamSoundPlaying = true;
            }
        } );
    }
    if( isStreamSoundPlaying )
    {
        NN_ATK_WARNING( "Can't read a StreamSoundDataInfo when the stream sound is playing." );
        return ResultSoundArchivePlayerStreamSoundIsPlaying();
    }

    //  ストリームファイルをオープン
    char filePath[nn::fs::EntryNameLengthMax + 1];    // + 1 は終端文字分
    if ( !pSoundArchive->ReadStreamSoundFilePath( filePath, sizeof(filePath), streamSoundId ) )
    {
        return ResultSoundArchivePlayerReadStreamSoundFilePathFailed();
    }

    detail::fnd::FileStreamImpl fileStream;
    detail::fnd::FndResult result = fileStream.Open( filePath, detail::fnd::FileStream::AccessMode_Read );
    if( result.IsFailed() )
    {
        fileStream.Close();
        return ResultSoundArchivePlayerFileSystemError();
    }

    //  ファイルヘッダのロード
    const size_t HeaderSize = NN_ATK_ROUND_UP_32B( sizeof( detail::StreamSoundFile::FileHeader ) );
    char NN_ALIGNAS(64) headerBuffer[HeaderSize];
    size_t readSize = fileStream.Read( headerBuffer, HeaderSize );
    if( readSize != HeaderSize )
    {
        fileStream.Close();
        return ResultSoundArchivePlayerFileSystemError();
    }

    const detail::StreamSoundFile::FileHeader* header = reinterpret_cast<detail::StreamSoundFile::FileHeader*>( headerBuffer );
    if( !detail::StreamSoundFileReader::IsValidFileHeader( header ) )
    {
        fileStream.Close();
        return ResultSoundArchivePlayerInvalidFormat();
    }

    //  InfoBlock のロード
    detail::StreamSoundFile::InfoBlock infoBlock;
    fileStream.Seek( header->GetInfoBlockOffset(), detail::fnd::Stream::SeekOrigin_Begin );
    readSize = fileStream.Read( &infoBlock, sizeof(infoBlock) );
    if( readSize != sizeof(infoBlock) )
    {
        fileStream.Close();
        return ResultSoundArchivePlayerFileSystemError();
    }

    //  StreamSoundInfo のロード
    detail::StreamSoundFile::StreamSoundInfo strmInfo;
    fileStream.Seek( header->GetInfoBlockOffset() + sizeof(infoBlock.header) + infoBlock.body.toStreamSoundInfo.offset, detail::fnd::Stream::SeekOrigin_Begin );
    readSize = fileStream.Read( &strmInfo, sizeof(strmInfo) );
    fileStream.Close();
    if ( readSize != sizeof(strmInfo) )
    {
        return ResultSoundArchivePlayerFileSystemError();
    }

    pOutInfo->channelCount = strmInfo.channelCount;
    pOutInfo->sampleRate = static_cast<int>( strmInfo.sampleRate );
    pOutInfo->loopFlag = strmInfo.isLoop;
    pOutInfo->compatibleLoopStart = static_cast<int64_t>( strmInfo.loopStart );
    pOutInfo->compatibleLoopEnd = static_cast<int64_t>( strmInfo.frameCount );

    if( detail::StreamSoundFileReader::IsOriginalLoopAvailableImpl( header ) )
    {
        pOutInfo->loopStart = static_cast<int64_t>( strmInfo.originalLoopStart );
        pOutInfo->loopEnd = static_cast<int64_t>( strmInfo.originalLoopEnd );
    }
    else
    {
        pOutInfo->loopStart = static_cast<int64_t>( strmInfo.loopStart );
        pOutInfo->loopEnd = static_cast<int64_t>( strmInfo.frameCount );
    }

    return nn::ResultSuccess();
}

nn::Result SoundArchivePlayer::ReadStreamSoundDataInfo(StreamSoundDataInfo* pOutInfo, SoundArchive::ItemId streamSoundId, const char* soundArchiveName) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutInfo );
    NN_SDK_REQUIRES( m_IsInitialized );

    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );

    return ReadStreamSoundDataInfo(pOutInfo, pSoundArchive, streamSoundId);
}

nn::Result SoundArchivePlayer::ReadStreamSoundDataInfo(StreamSoundDataInfo* pOutInfo, const char* streamSoundName, const char* soundArchiveName) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutInfo );
    NN_SDK_REQUIRES( m_IsInitialized );

    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );

    const SoundArchive::ItemId id = pSoundArchive->GetItemId( streamSoundName );
    if ( id == SoundArchive::InvalidId )
    {
        return ResultSoundArchivePlayerInvalidName();
    }

    return ReadStreamSoundDataInfo( pOutInfo, pSoundArchive, id );
}

nn::Result SoundArchivePlayer::ReadStreamSoundDataInfo(StreamSoundDataInfo* pOutInfo, SoundArchive::ItemId streamSoundId) const NN_NOEXCEPT
{
    return ReadStreamSoundDataInfo(pOutInfo, streamSoundId, nullptr);
}

nn::Result SoundArchivePlayer::ReadStreamSoundDataInfo(StreamSoundDataInfo* pOutInfo, const char* streamSoundName) const NN_NOEXCEPT
{
    return ReadStreamSoundDataInfo(pOutInfo, streamSoundName, nullptr);
}

size_t SoundArchivePlayer::GetRequiredWorkBufferSizeToReadStreamSoundHeader() NN_NOEXCEPT
{
    return detail::driver::StreamSoundLoader::LoadBufferSize + FileLoadAlignSize;
}

nn::Result SoundArchivePlayer::ReadStreamSoundRegionDataInfo(StreamSoundRegionDataInfo* pOutInfo, SoundArchive::ItemId streamSoundId, const char* const* pRegionName, int infoCount, const SoundArchive* pSoundArchive, void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_REQUIRES_NOT_NULL( pOutInfo );
    NN_SDK_REQUIRES_NOT_NULL( pRegionName );
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_SDK_REQUIRES_GREATER_EQUAL( bufferSize, GetRequiredWorkBufferSizeToReadStreamSoundHeader() );
    NN_SDK_REQUIRES_GREATER( infoCount, 0 );
    NN_SDK_REQUIRES_NOT_NULL( pSoundArchive );
    NN_UNUSED( bufferSize );

    //  STRM が再生中でないことを確認
    bool isStreamSoundPlaying = false;
    for(uint32_t i = 0; i < m_SoundPlayerCount; i++)
    {
        m_pSoundPlayers[i].ForEachSound( [=,&isStreamSoundPlaying](const SoundHandle& handle){
            if( handle.GetId() == streamSoundId )
            {
                isStreamSoundPlaying = true;
            }
        } );
    }
    if( isStreamSoundPlaying )
    {
        NN_ATK_WARNING( "Can't read a StreamSoundDataInfo when the stream sound is playing." );
        return ResultSoundArchivePlayerStreamSoundIsPlaying();
    }

    //  ストリームファイルをオープン
    char filePath[nn::fs::EntryNameLengthMax + 1];    // + 1 は終端文字分
    if( !pSoundArchive->ReadStreamSoundFilePath( filePath, sizeof(filePath), streamSoundId ) )
    {
        return ResultSoundArchivePlayerReadStreamSoundFilePathFailed();
    }

    detail::fnd::FileStreamImpl fileStream;
    detail::fnd::FndResult result = fileStream.Open( filePath, detail::fnd::FileStream::AccessMode_Read );
    if( result.IsFailed() )
    {
        fileStream.Close();
        return ResultSoundArchivePlayerFileSystemError();
    }

    detail::StreamSoundFileLoader fileLoader;
    fileLoader.Initialize( &fileStream );

    //  ファイルヘッダのロード
    const size_t requiredSize = GetRequiredWorkBufferSizeToReadStreamSoundHeader();
    void* const alignedBuffer = nn::util::BytePtr( buffer ).AlignUp( FileLoadAlignSize ).Get();
    detail::StreamSoundFileReader reader;
    if( !fileLoader.LoadFileHeader( &reader, alignedBuffer, static_cast<unsigned long>( requiredSize ) ) )
    {
        fileStream.Close();
        return ResultSoundArchivePlayerLoadFileHeaderFailed();
    }

    //  ストリーム情報の読み取り
    detail::StreamSoundFile::StreamSoundInfo streamInfo;
    if( !reader.ReadStreamSoundInfo( &streamInfo ) )
    {
        fileStream.Close();
        return ResultUnknown();   // 現時点では失敗しない実装になっている
    }

    if ( streamInfo.regionCount <= 0 )
    {
        fileStream.Close();
        return ResultSoundArchivePlayerStreamSoundHasNoRegion();
    }

    //  リージョン情報の読み取り
    detail::StreamSoundFile::RegionInfo regionInfo;
    for(int k = 0; k < infoCount; k++)
    {
        pOutInfo[k].regionNo = InvalidStreamJumpRegionIndex; //  一時的に不正なリージョン番号を代入します
    }

    int readCount = 0;
    for(int i = 0; i < streamInfo.regionCount; i++)
    {
        if( readCount == infoCount )
        {
            //  必要分読み取れたらループを終えます
            break;
        }

        if( fileLoader.ReadRegionInfo( &regionInfo, i ) )
        {
            for(int k = 0; k < infoCount; k++)
            {
                if( pOutInfo[k].regionNo == InvalidStreamJumpRegionIndex &&
                    nn::util::Strncmp( regionInfo.regionName, pRegionName[k], RegionNameLengthMax + 1 )== 0 )
                {
                    pOutInfo[k].startSamplePosition = regionInfo.start;
                    pOutInfo[k].endSamplePosition = regionInfo.end;
                    pOutInfo[k].regionNo = i;
                    nn::util::Strlcpy( pOutInfo[k].regionName, regionInfo.regionName, RegionNameLengthMax + 1 );
                    readCount++;
                }
            }
        }
    }

    fileStream.Close();
    if( readCount == infoCount )
    {
        return nn::ResultSuccess();
    }
    else
    {
        if ( readCount >= 0 && readCount < infoCount )
        {
            return ResultSoundArchivePlayerStreamSoundRegionNotFound();
        }
        else
        {
            return ResultUnknown();
        }
    }
} // NOLINT(impl/function_size)

nn::Result SoundArchivePlayer::ReadStreamSoundRegionDataInfo(StreamSoundRegionDataInfo* pOutInfo, SoundArchive::ItemId streamSoundId, const char* regionName, void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    return ReadStreamSoundRegionDataInfo( pOutInfo, streamSoundId, &regionName, 1, buffer, bufferSize, nullptr );
}

nn::Result SoundArchivePlayer::ReadStreamSoundRegionDataInfo(StreamSoundRegionDataInfo* pOutInfo, const char* streamSoundName, const char* regionName, void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    return ReadStreamSoundRegionDataInfo( pOutInfo, streamSoundName, &regionName, 1, buffer, bufferSize, nullptr );
}

nn::Result SoundArchivePlayer::ReadStreamSoundRegionDataInfo(StreamSoundRegionDataInfo* pOutInfo, SoundArchive::ItemId streamSoundId, const char* const* pRegionName, int infoCount, void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    return ReadStreamSoundRegionDataInfo( pOutInfo, streamSoundId, pRegionName, infoCount, buffer, bufferSize, nullptr );
}

nn::Result SoundArchivePlayer::ReadStreamSoundRegionDataInfo(StreamSoundRegionDataInfo* pOutInfo, const char* streamSoundName, const char* const* pRegionName, int infoCount, void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    return ReadStreamSoundRegionDataInfo( pOutInfo, streamSoundName, pRegionName, infoCount, buffer, bufferSize, nullptr );
}

nn::Result SoundArchivePlayer::ReadStreamSoundRegionDataInfo(StreamSoundRegionDataInfo* pOutInfo, const char* streamSoundName, const char* regionName, void* buffer, size_t bufferSize, const char* soundArchiveName) const NN_NOEXCEPT
{
    return ReadStreamSoundRegionDataInfo( pOutInfo, streamSoundName, &regionName, 1, buffer, bufferSize, soundArchiveName );
}

nn::Result SoundArchivePlayer::ReadStreamSoundRegionDataInfo(StreamSoundRegionDataInfo* pOutInfo, SoundArchive::ItemId streamSoundId, const char* regionName, void* buffer, size_t bufferSize, const char* soundArchiveName) const NN_NOEXCEPT
{
    return ReadStreamSoundRegionDataInfo( pOutInfo, streamSoundId, &regionName, 1, buffer, bufferSize, soundArchiveName );
}

nn::Result SoundArchivePlayer::ReadStreamSoundRegionDataInfo(StreamSoundRegionDataInfo* pOutInfo, const char* streamSoundName, const char* const* pRegionName, int infoCount, void* buffer, size_t bufferSize, const char* soundArchiveName) const NN_NOEXCEPT
{
    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );
    const SoundArchive::ItemId id = pSoundArchive->GetItemId( streamSoundName );
    if( id == SoundArchive::InvalidId )
    {
        return ResultSoundArchivePlayerInvalidName();
    }
    return ReadStreamSoundRegionDataInfo( pOutInfo, id, pRegionName, infoCount, pSoundArchive, buffer, bufferSize );
}

nn::Result SoundArchivePlayer::ReadStreamSoundRegionDataInfo(StreamSoundRegionDataInfo* pOutInfo, SoundArchive::ItemId streamSoundId, const char* const* pRegionName, int infoCount, void* buffer, size_t bufferSize, const char* soundArchiveName) const NN_NOEXCEPT
{
    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );
    return ReadStreamSoundRegionDataInfo( pOutInfo, streamSoundId, pRegionName, infoCount, pSoundArchive, buffer, bufferSize );
}

nn::Result SoundArchivePlayer::ReadMarkerInfoArray(StreamSoundMarkerInfo* pOutInfoArray, int* pOutActualInfoCount, int markerInfoCountMax, SoundArchive::ItemId streamSoundId) NN_NOEXCEPT
{
    return ReadMarkerInfoArray(pOutInfoArray, pOutActualInfoCount, markerInfoCountMax, streamSoundId, nullptr);
}

nn::Result SoundArchivePlayer::ReadMarkerInfoArray(StreamSoundMarkerInfo* pOutInfoArray, int* pOutActualInfoCount, int markerInfoCountMax, const char* streamSoundName) NN_NOEXCEPT
{
    return ReadMarkerInfoArray(pOutInfoArray, pOutActualInfoCount, markerInfoCountMax, streamSoundName, nullptr);
}

nn::Result SoundArchivePlayer::ReadMarkerInfoArray(StreamSoundMarkerInfo* pOutInfoArray, int* pOutActualInfoCount, int markerInfoCountMax, SoundArchive::ItemId streamSoundId, const char* soundArchiveName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );

    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );
    return ReadMarkerInfoArrayImpl(pOutInfoArray, pOutActualInfoCount, markerInfoCountMax, streamSoundId, pSoundArchive);
}

nn::Result SoundArchivePlayer::ReadMarkerInfoArray(StreamSoundMarkerInfo* pOutInfoArray, int* pOutActualInfoCount, int markerInfoCountMax, const char* streamSoundName, const char* soundArchiveName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );

    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );
    const SoundArchive::ItemId id = pSoundArchive->GetItemId( streamSoundName );
    if( id == SoundArchive::InvalidId )
    {
        return ResultSoundArchivePlayerInvalidName();
    }

    return ReadMarkerInfoArrayImpl(pOutInfoArray, pOutActualInfoCount, markerInfoCountMax, id, pSoundArchive);
}

void SoundArchivePlayer::DumpMemory() const NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
    if (!m_IsInitialized)
    {
        NN_DETAIL_ATK_INFO("  soundArchivePlayer is not initialized.\n");
        return;
    }

    const SoundArchive* pSoundArchive = m_SoundArchiveManager.GetMainSoundArchive();

    uint32_t playerCount = pSoundArchive->GetPlayerCount();
    size_t playerHeapInstanceTotalSize = 0;
    size_t playerHeapBufferTotalSize = 0;
    for ( uint32_t playerIndex = 0; playerIndex < playerCount; ++playerIndex )
    {
        SoundArchive::PlayerInfo playerInfo;
        if ( ! pSoundArchive->ReadPlayerInfo( &playerInfo, detail::Util::GetMaskedItemId(
            playerIndex, detail::ItemType_Player ) ) )
        {
            continue;
        }

        if ( playerInfo.playerHeapSize > 0 )
        {
            for ( int i = 0; i < playerInfo.playableSoundMax ; i++ )
            {
                playerHeapInstanceTotalSize += sizeof(detail::PlayerHeap);
                playerHeapBufferTotalSize += playerInfo.playerHeapSize;
            }
        }
    }
    NN_DETAIL_ATK_INFO("  soundArchivePlayer\n"
               "    soundPlayerInstance      %8zu bytes\n"
               "    playerHeapInstance       %8zu bytes\n"
               "    playerHeap               %8zu bytes\n"
               , playerCount * sizeof( SoundPlayer ), playerHeapInstanceTotalSize, playerHeapBufferTotalSize );

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( pSoundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        size_t waveSoundSize     = m_WaveSoundRuntime.GetRequiredMemorySize(soundArchivePlayerInfo, MemAlignSize);
        size_t advancedWaveSoundSize = m_AdvancedWaveSoundRuntime.GetRequiredMemorySize(soundArchivePlayerInfo, MemAlignSize);
        size_t streamSoundSize   = m_StreamSoundRuntime.GetRequiredMemorySize(soundArchivePlayerInfo, MemAlignSize);
        size_t sequenceSoundSize = m_SequenceSoundRuntime.GetRequiredMemorySize(soundArchivePlayerInfo, MemAlignSize);
        size_t sequenceTrackSize = m_SequenceSoundRuntime.GetRequiredSequenceTrackMemorySize(soundArchivePlayerInfo, MemAlignSize);
        NN_DETAIL_ATK_INFO("    waveSound                %8zu bytes\n"
                   "    advancedWaveSound        %8zu bytes\n"
                   "    streamSound              %8zu bytes\n"
                   "    sequenceSound            %8zu bytes\n"
                   "    sequenceTrack            %8zu bytes\n"
                   , waveSoundSize, advancedWaveSoundSize, streamSoundSize, sequenceSoundSize, sequenceTrackSize );
        m_WaveSoundRuntime.DumpMemory(pSoundArchive);
        m_StreamSoundRuntime.DumpMemory(pSoundArchive);
        m_SequenceSoundRuntime.DumpMemory(pSoundArchive);
    }
#endif
}

bool SoundArchivePlayer::ReadStreamSoundInstanceState(StreamSoundInstanceState* pOutInfo) const NN_NOEXCEPT
{
    if (pOutInfo == nullptr)
    {
        return false;
    }

    pOutInfo->activeStreamSoundInstanceCount = m_StreamSoundRuntime.GetActiveCount();
    pOutInfo->activeStreamChannelCount = m_StreamSoundRuntime.GetActiveChannelCount();
    pOutInfo->activeStreamTrackCount = m_StreamSoundRuntime.GetActiveTrackCount();

    return true;
}

nn::Result SoundArchivePlayer::CheckStreamSoundFileExisting(SoundArchive::ItemId streamSoundId) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    return CheckStreamSoundFileExisting( streamSoundId, nullptr );
}

nn::Result SoundArchivePlayer::CheckStreamSoundFileExisting(const char* streamSoundName) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    return CheckStreamSoundFileExisting( streamSoundName, nullptr );
}

nn::Result SoundArchivePlayer::CheckStreamSoundFileExisting(SoundArchive::ItemId streamSoundId, const char* soundArchiveName) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );
    NN_SDK_ASSERT_NOT_NULL( pSoundArchive );
    return CheckStreamSoundFileExisting( pSoundArchive, streamSoundId );
}

nn::Result SoundArchivePlayer::CheckStreamSoundFileExisting(const char* streamSoundName, const char* soundArchiveName) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );

    const SoundArchive* pSoundArchive = GetTargetSoundArchive( m_SoundArchiveManager, soundArchiveName );
    NN_SDK_ASSERT_NOT_NULL( pSoundArchive );
    SoundArchive::ItemId id = pSoundArchive->GetItemId( streamSoundName );
    if ( id == SoundArchive::InvalidId )
    {
        return ResultSoundArchivePlayerInvalidName();
    }
    return CheckStreamSoundFileExisting( pSoundArchive, id );
}

void SoundArchivePlayer::StopAllSound(int fadeFrames, bool isCommandFlushed) NN_NOEXCEPT
{
    for ( SoundArchive::ItemId playerId = 0; playerId < m_SoundPlayerCount ; ++playerId )
    {
        m_pSoundPlayers[ playerId ].StopAllSound(fadeFrames);
    }

    // 停止コマンドの処理待ち
    if ( isCommandFlushed )
    {
        detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
        uint32_t tag = cmdmgr.FlushCommand( true, true );
        cmdmgr.WaitCommandReply( tag );
    }
}

void SoundArchivePlayer::DisposeInstances() NN_NOEXCEPT
{
    m_pArchiveContainers = nullptr;
    m_ArchiveContainerCount = 0;

    // インスタンスの破棄
    m_SoundPlayerCount = 0;
    m_pSoundPlayers = NULL;

    m_SoundArchiveManager.Finalize();

    m_SequenceSoundRuntime.Finalize();
    if ( m_IsAdvancedWaveSoundEnabled )
    {
        m_AdvancedWaveSoundRuntime.Finalize();
    }
    else
    {
        m_WaveSoundRuntime.Finalize();
    }
    m_StreamSoundRuntime.Finalize();
}


nn::Result SoundArchivePlayer::CheckStreamSoundFileExisting( const SoundArchive* pSoundArchive, SoundArchive::ItemId streamSoundId ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( m_IsInitialized );
    NN_SDK_ASSERT_NOT_NULL( pSoundArchive );
    char filePath[nn::fs::EntryNameLengthMax + 1]; // + 1 は終端文字分
    if ( !pSoundArchive->ReadStreamSoundFilePath( filePath, sizeof(filePath), streamSoundId ) )
    {
        return ResultSoundArchivePlayerReadStreamSoundFilePathFailed();
    }

    // 該当のファイルパスにファイルがあるか確認する
    nn::fs::DirectoryEntryType entryType;
    return nn::fs::GetEntryType( &entryType, filePath );
}

nn::Result SoundArchivePlayer::ReadMarkerInfoArrayImpl(StreamSoundMarkerInfo* pOutMarkerInfoArray, int* pOutActualMarkerInfoCount, int markerInfoCountMax, SoundArchive::ItemId streamSoundId, const SoundArchive* pSoundArchive) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutMarkerInfoArray );
    NN_SDK_REQUIRES_NOT_NULL( pOutActualMarkerInfoCount );
    NN_SDK_REQUIRES_GREATER( markerInfoCountMax, 0 );
    NN_SDK_REQUIRES_NOT_NULL( pSoundArchive );

    //  STRM が再生中でないことを確認
    bool isStreamSoundPlaying = false;
    for(uint32_t i = 0; i < m_SoundPlayerCount; i++)
    {
        m_pSoundPlayers[i].ForEachSound( [=,&isStreamSoundPlaying](const SoundHandle& handle){
            if( handle.GetId() == streamSoundId )
            {
                isStreamSoundPlaying = true;
            }
        } );
    }
    if( isStreamSoundPlaying )
    {
        NN_ATK_WARNING( "Can't read a StreamSoundDataInfo when the stream sound is playing." );
        return ResultSoundArchivePlayerStreamSoundIsPlaying();
    }

    // ID からストリームサウンドバイナリのファイルパスを取得
    char filePath[nn::fs::EntryNameLengthMax + 1];    // + 1 は終端文字分
    if( !pSoundArchive->ReadStreamSoundFilePath( filePath, sizeof(filePath), streamSoundId ) )
    {
        return ResultSoundArchivePlayerReadStreamSoundFilePathFailed();
    }

    //  ストリームファイルをオープン
    detail::fnd::FileStreamImpl fileStream;
    detail::fnd::FndResult result = fileStream.Open( filePath, detail::fnd::FileStream::AccessMode_Read );
    if( result.IsFailed() )
    {
        fileStream.Close();
        return ResultSoundArchivePlayerFileSystemError();
    }

    size_t fileSize = fileStream.GetSize();

    // bfstm ヘッダ情報の読み込み
    detail::StreamSoundFile::FileHeader fileHeader;
    size_t fileHeaderSize = sizeof(detail::StreamSoundFile::FileHeader);
    void* fileHeaderPtr = &fileHeader;
    if (fileSize < fileHeaderSize)
    {
        fileStream.Close();
        return ResultSoundArchivePlayerFileSystemError();
    }
    fileStream.Read(fileHeaderPtr, fileHeaderSize);

    // マーカーブロックが含まれてないバイナリの場合
    if (!fileHeader.HasMarkerBlock())
    {
        fileStream.Close();
        return ResultSoundArchivePlayerInvalidFormat();
    }

    // bfstm のマーカーブロックまでシーク
    uint32_t offset = fileHeader.GetMarkerBlockOffset() + sizeof(detail::BinaryBlockHeader);
    if (fileSize < offset + sizeof(int32_t))   // マーカー数の分まで合わせてチェック
    {
        fileStream.Close();
        return ResultSoundArchivePlayerFileSystemError();
    }
    fileStream.Seek(offset, detail::fnd::Stream::SeekOrigin::SeekOrigin_Begin);

    // 埋め込まれているマーカー情報の数を取得
    int32_t markerInfoCountBin = 0;
    fileStream.Read(&markerInfoCountBin, sizeof(int32_t));
    size_t markerInfoSize = sizeof(detail::StreamSoundFile::MarkerInfo);
    int markerInfoCount = static_cast<int>(fileHeader.GetMarkerBlockSize() / markerInfoSize);
    NN_SDK_ASSERT_EQUAL(markerInfoCount, markerInfoCountBin);

    *pOutActualMarkerInfoCount = markerInfoCount;

    // 渡された配列より多くのマーカー情報がバイナリにある場合
    // マーカー情報を先頭から渡された配列分だけをコピーする
    if (markerInfoCount > markerInfoCountMax )
    {
        markerInfoCount = markerInfoCountMax;
    }
    if (fileSize < offset + sizeof(int32_t) + markerInfoSize * markerInfoCount)
    {
        fileStream.Close();
        return ResultSoundArchivePlayerFileSystemError();
    }

    for (auto i = 0; i < markerInfoCount; ++i)
    {
        detail::StreamSoundFile::MarkerInfo markerInfo;
        void* markerInfoPtr = &markerInfo;
        fileStream.Read(markerInfoPtr, markerInfoSize);

        pOutMarkerInfoArray[i].position = markerInfo.position;
        nn::util::Strlcpy(pOutMarkerInfoArray[i].name, markerInfo.name, MarkerNameLengthMax + 1);
    }
    fileStream.Close();

    return nn::ResultSuccess();
}

} // namespace nn::atk
} // namespace nn

