﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundArchive.h>

#include <cstring>      // strlen, etc...
#include <nn/util/util_StringUtil.h>
#include <nn/atk/atk_SoundArchiveFileReader.h>
#include <nn/atk/atk_SoundArchiveParametersHook.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/string/atkfnd_String.h>

namespace nn {
namespace atk {

/* ========================================================================
        static variables
   ======================================================================== */

NN_DEFINE_STATIC_CONSTANT( const SoundArchive::ItemId SoundArchive::InvalidId );
NN_DEFINE_STATIC_CONSTANT( const int SoundArchive::UserParamIndexMax );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchive::ResultInvalidSoundId );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchive::InvalidUserParam );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchive::SequenceBankMax );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchive::StreamTrackCount );
NN_DEFINE_STATIC_CONSTANT( const int SoundArchive::FilePathMax );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchive::FileInfo::InvalidOffset );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundArchive::FileInfo::InvalidSize );

void SoundArchive::StreamSoundInfo::Setup() NN_NOEXCEPT
{
    // グローバルチャンネルインデックスをチャンネルカウントから自動的に設定
    int8_t index = 0;
    uint16_t bitMask = allocateTrackFlags;
    for ( uint32_t i = 0; i < StreamTrackCount; ++i )
    {
        if ( bitMask == 0 )
        {
            break;
        }

        if ( bitMask & 0x1 )
        {
            uint32_t count = std::min(static_cast<uint32_t>(trackInfo[i].channelCount), WaveChannelMax);
            for ( uint32_t j = 0; j < count; j++ )
            {
                trackInfo[i].globalChannelIndex[j] = index;
                ++index;
            }
        }
        bitMask >>= 1;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         SoundArchive

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
SoundArchive::SoundArchive() NN_NOEXCEPT
: m_pFileReader( NULL )
, m_pParametersHook( NULL )
{
    m_ExtFileRoot[0] = '/';
    m_ExtFileRoot[1] = '\0';
}

/*--------------------------------------------------------------------------------*
  Name:         ~SoundArchive

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
SoundArchive::~SoundArchive() NN_NOEXCEPT
{
}

bool SoundArchive::IsAvailable() const NN_NOEXCEPT
{
    if ( m_pFileReader == NULL ) return false;
    return true;
}

void SoundArchive::Initialize( detail::SoundArchiveFileReader* fileReader ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( fileReader );
    m_pFileReader = fileReader;
    m_FileBlockOffset = m_pFileReader->GetFileBlockOffset();
}

/*--------------------------------------------------------------------------------*
  Name:         Shutdown

  Description:  シャットダウン

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SoundArchive::Finalize() NN_NOEXCEPT
{
    if ( m_pFileReader != NULL )
    {
        m_pFileReader->Finalize();
        m_pFileReader = NULL;
    }
    m_ExtFileRoot[0] = '/';
    m_ExtFileRoot[1] = '\0';
}

uint32_t SoundArchive::GetSoundCount() const NN_NOEXCEPT
{
    return m_pFileReader->GetSoundCount();
}
uint32_t SoundArchive::GetPlayerCount() const NN_NOEXCEPT
{
    return m_pFileReader->GetPlayerCount();
}
uint32_t SoundArchive::GetSoundGroupCount() const NN_NOEXCEPT
{
    return m_pFileReader->GetSoundGroupCount();
}
uint32_t SoundArchive::GetGroupCount() const NN_NOEXCEPT
{
    return m_pFileReader->GetGroupCount();
}
uint32_t SoundArchive::GetBankCount() const NN_NOEXCEPT
{
    return m_pFileReader->GetBankCount();
}
uint32_t SoundArchive::GetWaveArchiveCount() const NN_NOEXCEPT
{
    return m_pFileReader->GetWaveArchiveCount();
}
uint32_t SoundArchive::detail_GetFileCount() const NN_NOEXCEPT
{
    return m_pFileReader->GetFileCount();
}

const char* SoundArchive::GetItemLabel( ItemId id ) const NN_NOEXCEPT
{
    const char* result = NULL;

    if ( m_pParametersHook != NULL )
    {
        result = m_pParametersHook->GetItemLabel( id );

        if ( result != NULL )
        {
            return result;
        }
    }

    return m_pFileReader->GetItemLabel( id );
}
SoundArchive::ItemId SoundArchive::GetItemId( const char* pStr ) const NN_NOEXCEPT
{
    SoundArchive::ItemId result = InvalidId;

    if ( m_pParametersHook != NULL )
    {
        result = m_pParametersHook->GetItemId( pStr );

        if ( result != InvalidId )
        {
            return result;
        }
    }

    return m_pFileReader->GetItemId( pStr );
}
SoundArchive::FileId SoundArchive::GetItemFileId( ItemId id ) const NN_NOEXCEPT
{
    return m_pFileReader->GetItemFileId( id );
}
SoundArchive::FileId SoundArchive::GetItemPrefetchFileId( ItemId id ) const NN_NOEXCEPT
{
    return m_pFileReader->GetItemPrefetchFileId( id );
}

uint32_t SoundArchive::GetSoundUserParam( ItemId soundId ) const NN_NOEXCEPT
{
    uint32_t userParam = m_pFileReader->GetSoundUserParam( soundId );

    if ( m_pParametersHook == NULL )
    {
        return userParam;
    }

    return m_pParametersHook->GetSoundUserParam( soundId, userParam );
}

bool SoundArchive::ReadSoundUserParam( uint32_t* pOutValue, ItemId soundId, int index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    NN_SDK_ASSERT( index >= 0 && index <= UserParamIndexMax );
    bool result = m_pFileReader->ReadSoundUserParam( pOutValue, soundId, index );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSoundUserParam( pOutValue, soundId, index );
    }

    return result;
}

SoundArchive::SoundType SoundArchive::GetSoundType( ItemId soundId ) const NN_NOEXCEPT
{
    if ( m_pParametersHook != NULL )
    {
        const char* label = m_pParametersHook->GetItemLabel( soundId );
        if(label != nullptr)
        {
            SoundType type = m_pParametersHook->GetSoundType( label );
            if( type != SoundType_Invalid )
            {
                return type;
            }
        }
    }

    return m_pFileReader->GetSoundType( soundId );
}
bool SoundArchive::ReadSoundInfo( SoundInfo* pOutValue, ItemId soundId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    bool result = m_pFileReader->ReadSoundInfo( soundId, pOutValue );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSoundInfo( soundId, pOutValue );
    }

    return result;
}
bool SoundArchive::ReadSequenceSoundInfo( SequenceSoundInfo* pOutValue, ItemId soundId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    bool result = m_pFileReader->ReadSequenceSoundInfo( soundId, pOutValue );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSequenceSoundInfo( soundId, pOutValue );
    }

    return result;
}
bool SoundArchive::ReadStreamSoundInfo( StreamSoundInfo* pOutValue, ItemId soundId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    bool result = m_pFileReader->ReadStreamSoundInfo( soundId, pOutValue );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadStreamSoundInfo( soundId, pOutValue );
    }

    return result;
}
bool SoundArchive::detail_ReadStreamSoundInfo2( ItemId soundId, StreamSoundInfo2* info ) const NN_NOEXCEPT
{
    bool result = m_pFileReader->ReadStreamSoundInfo2( soundId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadStreamSoundInfo2( soundId, info );
    }

    return result;
}
bool SoundArchive::detail_ReadWaveSoundInfo( ItemId soundId, WaveSoundInfo* info ) const NN_NOEXCEPT
{
    bool result = m_pFileReader->ReadWaveSoundInfo( soundId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadWaveSoundInfo( soundId, info );
    }

    return result;
}
bool SoundArchive::detail_ReadAdvancedWaveSoundInfo( ItemId soundId, AdvancedWaveSoundInfo* info ) const NN_NOEXCEPT
{
    bool result = m_pFileReader->ReadAdvancedWaveSoundInfo( soundId, info );

    /* TODO: インゲーム編集対応
    if ( m_pParametersHook != NULL )
    {
        |= m_pParametersHook->ReadAdvancedWaveSoundInfo( soundId, info );
    }
    */

    return result;
}
bool SoundArchive::ReadPlayerInfo( PlayerInfo* pOutValue, ItemId playerId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    bool result = m_pFileReader->ReadPlayerInfo( playerId, pOutValue );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadPlayerInfo( playerId, pOutValue );
    }

    return result;
}
bool SoundArchive::ReadSoundArchivePlayerInfo( SoundArchivePlayerInfo* pOutValue ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    bool result = m_pFileReader->ReadSoundArchivePlayerInfo( pOutValue );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSoundArchivePlayerInfo( pOutValue );
    }

    return result;
}
bool SoundArchive::ReadSound3DInfo( Sound3DInfo* pOutValue, ItemId soundId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    bool result = m_pFileReader->ReadSound3DInfo( soundId, pOutValue );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSound3DInfo( soundId, pOutValue );
    }

    return result;
}
bool SoundArchive::ReadBankInfo( BankInfo* pOutValue, ItemId bankId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    bool result = m_pFileReader->ReadBankInfo( bankId, pOutValue );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadBankInfo( bankId, pOutValue );
    }

    return result;
}
bool SoundArchive::ReadWaveArchiveInfo( ItemId warcId, WaveArchiveInfo* info ) const NN_NOEXCEPT
{
    bool result = m_pFileReader->ReadWaveArchiveInfo( warcId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadWaveArchiveInfo( warcId, info );
    }

    return result;
}
bool SoundArchive::detail_ReadSoundGroupInfo( ItemId soundGroupId, SoundGroupInfo* info ) const NN_NOEXCEPT
{
    bool result = m_pFileReader->ReadSoundGroupInfo( soundGroupId, info );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadSoundGroupInfo( soundGroupId, info );
    }

    return result;
}
bool SoundArchive::ReadGroupInfo( GroupInfo* pOutValue, ItemId groupId ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    bool result = m_pFileReader->ReadGroupInfo( groupId, pOutValue );

    if ( m_pParametersHook != NULL )
    {
        result |= m_pParametersHook->ReadGroupInfo( groupId, pOutValue );
    }

    return result;
}
#if 0
bool SoundArchive::detail_ReadGroupItemInfo( GroupId groupId, unsigned long index, GroupItemInfo* info ) const NN_NOEXCEPT
{
    bool result = m_pFileReader->ReadGroupItemInfo( groupId, index, info );

    if (!result || m_pParametersHook == NULL )
    {
        return result;
    }

    m_pParametersHook->ReadGroupItemInfo( groupId, index, info );
    return true;
}
#endif
bool SoundArchive::detail_ReadFileInfo( FileId fileId, FileInfo* info ) const NN_NOEXCEPT
{
    bool result = m_pFileReader->ReadFileInfo( fileId, info );

    if (!result || m_pParametersHook == NULL )
    {
        return result;
    }

    m_pParametersHook->ReadFileInfo( fileId, info );
    return true;
}
#if 0
bool SoundArchive::detail_ReadFilePos( FileId fileId, unsigned long index, FilePos* info ) const NN_NOEXCEPT
{
    bool result = m_pFileReader->ReadFilePos( fileId, index, info );

    if (!result || m_pParametersHook == NULL )
    {
        return result;
    }

    m_pParametersHook->ReadFilePos( fileId, index, info );
    return true;
}
#endif

const detail::Util::Table<uint32_t>*
SoundArchive::detail_GetWaveArchiveIdTable( ItemId id ) const NN_NOEXCEPT
{
    // id にはウェーブサウンドの ID か、バンクの ID しか入らない
    return m_pFileReader->GetWaveArchiveIdTable( id );
}

detail::fnd::FileStream* SoundArchive::detail_OpenFileStream(
        FileId fileId, void* buffer, size_t size,
        void* cacheBuffer, size_t cacheSize ) const NN_NOEXCEPT
{
    // ファイル情報取得
    FileInfo fileInfo;
    if ( ! detail_ReadFileInfo( fileId, &fileInfo ) )
    {
        return NULL;
    }

    // 外部参照ファイル
    if ( fileInfo.externalFilePath != NULL )
    {
        detail::fnd::FileStream* stream =
            OpenExtStreamImpl( buffer, size, fileInfo.externalFilePath, cacheBuffer, cacheSize );
        return stream;
    }

    // ファイルサイズ、またはオフセットが無効値
    if ( fileInfo.fileSize == FileInfo::InvalidSize ||
        fileInfo.offsetFromFileBlockHead == FileInfo::InvalidOffset )
    {
        return NULL;
    }

    // サウンドアーカイブ内ファイル
    detail::fnd::FileStream* stream = OpenStream(
        buffer,
        size,
        fileInfo.offsetFromFileBlockHead + m_FileBlockOffset,
        fileInfo.fileSize
    );
    return stream;
}

detail::fnd::FileStream* SoundArchive::OpenExtStreamImpl(
    void* buffer,
    size_t size,
    const char* externalFilePath,
    void* cacheBuffer,
    size_t cacheSize
) const NN_NOEXCEPT
{
    const char* fullPath;
    char pathBuffer[ FilePathMax ];

    if ( externalFilePath[0] == '/' )
    {
        fullPath = externalFilePath;
    }
    else
    {
        std::size_t fileLen = std::strlen( externalFilePath );
        std::size_t dirLen = std::strlen( m_ExtFileRoot );
        if ( fileLen + dirLen >= sizeof( pathBuffer ) / sizeof( pathBuffer[0] ) )
        {
            NN_ATK_WARNING("Too long file path \"%s/%s\"", m_ExtFileRoot, externalFilePath);
            return NULL;
        }
        util::Strlcpy( pathBuffer, m_ExtFileRoot, std::min(FilePathMax, static_cast<int>(dirLen + 1)) );
        detail::fnd::strncat( pathBuffer, FilePathMax, externalFilePath, fileLen + 1 );
        fullPath = pathBuffer;
    }

    return OpenExtStream( buffer, size, fullPath, cacheBuffer, cacheSize );
}

const char* SoundArchive::detail_GetExternalFileFullPath(
        const char* externalFilePath, char* pathBuffer, size_t bufSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(externalFilePath);
    if (externalFilePath[0] == '/')
    {
        return externalFilePath;
    }

    std::size_t fileLen = std::strlen( externalFilePath );
    std::size_t dirLen = std::strlen( m_ExtFileRoot );
    if ( fileLen + dirLen >= bufSize )
    {
        NN_ATK_WARNING("filepath buffer size is not enough. expect size is %zu, but actual size is %zu.(%s/%s)",
                        fileLen + dirLen + 1, bufSize, m_ExtFileRoot, externalFilePath);
        return NULL;
    }
    util::Strlcpy( pathBuffer, m_ExtFileRoot, static_cast<int>(std::min(bufSize, dirLen + 1)) );
    detail::fnd::strncat( pathBuffer, bufSize, externalFilePath, fileLen + 1 );
    return pathBuffer;
}

void SoundArchive::SetExternalFileRoot( const char* extFileRoot ) NN_NOEXCEPT
{
    std::size_t len = std::strlen( extFileRoot );
    std::size_t nullPos = len;

    util::Strlcpy( m_ExtFileRoot, extFileRoot, std::min(FilePathMax, static_cast<int>(len + 1)) );

    // 末尾の "/" の有無を吸収
    if ( extFileRoot[ len - 1 ] != '/' )
    {
        m_ExtFileRoot[ len ] = '/';
        nullPos++;
    }

    NN_SDK_ASSERT( nullPos < FilePathMax );
    m_ExtFileRoot[ nullPos ] = '\0';
}

bool SoundArchive::ReadStreamSoundFilePath( char* outFilePathBuffer, size_t filePathBufferSize, ItemId soundId ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( outFilePathBuffer );
    NN_SDK_REQUIRES_GREATER_EQUAL( filePathBufferSize, 0u );

    if ( GetSoundType( soundId ) != nn::atk::SoundArchive::SoundType_Stream )
    {
        NN_ATK_WARNING("0x%08x is not StreamSound ID.", soundId);
        return false;
    }

    nn::atk::SoundArchive::SoundInfo soundInfo;
    if ( !ReadSoundInfo( &soundInfo, soundId ) )
    {
        NN_ATK_WARNING("ReadSoundInfo is failed.");
        return false;
    }

    nn::atk::SoundArchive::FileInfo fileInfo;
    if ( !detail_ReadFileInfo( soundInfo.fileId, &fileInfo ) )
    {
        NN_ATK_WARNING("detail_ReadFileInfo is failed.");
        return false;
    }

    if ( fileInfo.externalFilePath == NULL )
    {
        NN_ATK_WARNING("fileInfo.externalFilePath is NULL.");
        return false;
    }

    const char* result = detail_GetExternalFileFullPath(
        fileInfo.externalFilePath, outFilePathBuffer, filePathBufferSize );

    if ( result == NULL )
    {
        return false;
    }

    return true;
}

const detail::Util::Table<uint32_t>* SoundArchive::detail_GetAttachedGroupTable(
    FileId fileId ) const NN_NOEXCEPT
{
    return m_pFileReader->GetAttachedGroupTable( fileId );
}

} // namespace nn::atk
} // namespace nn

