﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Sound3DManager.h>

#include <nn/atk/atk_Sound3DEngine.h>
#include <nn/atk/atk_SoundArchive.h>

namespace nn {
namespace atk {

namespace {

Sound3DEngine sSound3DEngine;

}

NN_DEFINE_STATIC_CONSTANT( const size_t Sound3DManager::NnUtilMathTypeAlignSize );
NN_DEFINE_STATIC_CONSTANT( const size_t Sound3DManager::BufferAlignSize );

/* ------------------------------------------------------------------------
        member function
   ------------------------------------------------------------------------ */

/*--------------------------------------------------------------------------------*
  Name:         Sound3DManager

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
Sound3DManager::Sound3DManager() NN_NOEXCEPT
: m_pSound3DEngine( &sSound3DEngine )
, m_MaxPriorityReduction( 32 )
, m_PanRange( 0.9f )
, m_SonicVelocity( 0.0f )
, m_BiquadFilterType( BiquadFilterType_Inherit )
, m_FreeMemSizeAfterChecking( 0 )
, m_IsInitialized( false )
{
}

/*--------------------------------------------------------------------------------*
  Name:         GetRequiredMemSize

  Description:  3Dサウンドに必要なメモリサイズを取得する

  Arguments:    arc - サウンドアーカイブ

  Returns:      必要となるメモリサイズ
 *--------------------------------------------------------------------------------*/
size_t Sound3DManager::GetRequiredMemSize( const SoundArchive* arc ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( arc );

    int32_t soundCount = 0;

    SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
    if ( arc->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
    {
        soundCount += soundArchivePlayerInfo.sequenceSoundCount;
        soundCount += soundArchivePlayerInfo.streamSoundCount;
        soundCount += soundArchivePlayerInfo.waveSoundCount;
    }

    return static_cast<size_t>( soundCount ) * nn::util::align_up( sizeof( Sound3DParam ), Sound3DManager::BufferAlignSize );
}

/*--------------------------------------------------------------------------------*
  Name:         Initialize

  Description:  3Dサウンドを使用するメモリの初期化を行う

  Arguments:    arc - サウンドアーカイブ
                buffer - バッファのアドレス
                size - バッファサイズ

  Returns:      成功したら true
 *--------------------------------------------------------------------------------*/
bool Sound3DManager::Initialize(
    const SoundArchive* arc,
    void* buffer,
    size_t size
) NN_NOEXCEPT
{
    if ( m_IsInitialized == true )
    {
        return false;
    }
    NN_UNUSED( arc );

    NN_SDK_REQUIRES_NOT_NULL( arc );
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_SDK_REQUIRES_ALIGNED( buffer, Sound3DManager::BufferAlignSize );
    size_t memSize = GetRequiredMemSize( arc );
    NN_SDK_REQUIRES_GREATER_EQUAL( size, memSize );

    // ActorParams Create
    m_ParamPool.Create( buffer, size, Sound3DManager::BufferAlignSize ); // Sound3DParam の nn::util::Vector3fType 型メンバのために 16 バイトアラインが必要
    m_FreeMemSizeAfterChecking = size - memSize;
    m_IsInitialized = true;

    return true;
}

bool Sound3DManager::InitializeWithMoreSoundArchive( const SoundArchive* arc ) NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return false;
    }

    NN_UNUSED( arc );
    NN_SDK_ASSERT_NOT_NULL( arc );
    size_t memSize = GetRequiredMemSize( arc );

    NN_SDK_ASSERT( m_FreeMemSizeAfterChecking >= memSize );
    m_FreeMemSizeAfterChecking -= memSize;
    return true;
}

bool Sound3DManager::Finalize() NN_NOEXCEPT
{
    if ( m_IsInitialized == false )
    {
        return false;
    }

    // 初期値に戻す
    m_pSound3DEngine = &sSound3DEngine;
    m_MaxPriorityReduction = 32;
    m_PanRange = 0.9f;
    m_SonicVelocity = 0.0f;
    m_BiquadFilterType = BiquadFilterType_Inherit;

    // リスナーを外す
    while ( !m_ListenerList.empty() )
    {
        Sound3DListener& listener = m_ListenerList.back();
        m_ListenerList.erase( m_ListenerList.iterator_to( listener ) );
    }

    // パラメータプールを破棄する
    m_ParamPool.Destroy();
    m_FreeMemSizeAfterChecking = 0;

    m_IsInitialized = false;
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         SetEngine

  Description:  3D エンジンを登録する

  Arguments:    engine - 3D エンジン

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void Sound3DManager::SetEngine( Sound3DEngine* engine ) NN_NOEXCEPT
{
    m_pSound3DEngine = engine;
}

/*--------------------------------------------------------------------------------*
  Name:         detail_UpdateAmbientParam

  Description:  3Dサウンドのパラメータ更新

  Arguments:    arg   - AmbientArgのポインタ
                soundId - サウンドＩＤ
                param - 3Dサウンドの計算結果を格納するパラメータ構造体

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Sound3DManager::detail_UpdateAmbientParam(
    const void* arg,
    uint32_t soundId,
    SoundAmbientParam* param
) NN_NOEXCEPT
{
    const Sound3DParam* actorParam = static_cast<const Sound3DParam*>( arg );

    if ( m_pSound3DEngine != NULL )
    {
        m_pSound3DEngine->detail_UpdateAmbientParam(
            this,
            actorParam,
            soundId,
            param
        );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         detail_GetAmbientPriority

  Description:  アンビエントプライオリティの計算

  Arguments:    arg   - AmbientArgのポインタ
                soundId - サウンドＩＤ

  Returns:      None.
  *--------------------------------------------------------------------------------*/
int Sound3DManager::detail_GetAmbientPriority(
    const void* arg,
    uint32_t soundId
) NN_NOEXCEPT
{
    const Sound3DParam* actorParam = static_cast<const Sound3DParam*>( arg );

    int priority = 0;

    if ( m_pSound3DEngine != NULL )
    {
        priority = m_pSound3DEngine->GetAmbientPriority(
            this,
            actorParam,
            soundId
        );
    }

    return priority;
}

/*--------------------------------------------------------------------------------*
  Name:         AllocAmbientArg

  Description:  AmbientArgを確保します

  Arguments:    argSize - AmbientArgのサイズ

  Returns:      AmbientArgへのポインタ
 *--------------------------------------------------------------------------------*/
void* Sound3DManager::detail_AllocAmbientArg( size_t argSize ) NN_NOEXCEPT
{
    if ( argSize != sizeof( Sound3DParam ) ) return NULL;
    return m_ParamPool.Alloc();
}

/*--------------------------------------------------------------------------------*
  Name:         FreeAmbientArg

  Description:  AmbientArgを解放します

  Arguments:    arg - 解放するAmbientArg
                sound - サウンドのインスタンス

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Sound3DManager::detail_FreeAmbientArg( void* arg, const detail::BasicSound* sound ) NN_NOEXCEPT
{
    (void)sound;

    Sound3DParam* actorParam = static_cast<Sound3DParam*>( arg );
    m_ParamPool.Free( actorParam );
}

/*--------------------------------------------------------------------------------*
  Name:         SetBiquadFilterType

  Description:  Biquad フィルタの種類を指定します

  Arguments:    type - Biquad フィルタの種類

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void Sound3DManager::SetBiquadFilterType( int type ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( type >= BiquadFilterType_Min && type <= BiquadFilterType_Max );
    m_BiquadFilterType = type;
}

Sound3DParam::Sound3DParam() NN_NOEXCEPT
: controlFlag( 0 )
, actorUserParam( 0 )
, soundUserParam( 0 )
, decayRatio( 0.5f )
, decayCurve( SoundArchive::Sound3DInfo::DecayCurve_Log )
, dopplerFactor( 0 )
{
    NN_ABORT_UNLESS_ALIGNED( &position, Sound3DManager::NnUtilMathTypeAlignSize );
    NN_ABORT_UNLESS_ALIGNED( &velocity, Sound3DManager::NnUtilMathTypeAlignSize );
}

} // namespace nn::atk
} // namespace nn

