﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/atk2/detail/atk2_AudioThread.h>
#include <nn/atk2/detail/atk2_AudioFrameUpdater.h>
#include <nn/atk2/detail/atk2_RendererManager.h>
#include <nn/atk2/detail/atk2_RendererVoiceManager.h>
#include <nn/atk2/detail/atk2_VirtualVoiceManager.h>
#include <nn/atk2/detail/atk2_VoiceCommandManager.h>
#include <nn/atk2/detail/atk2_VoiceReplyCommandManager.h>

namespace nn { namespace atk2 {

    // @brief オーディオ処理全体を管理するクラスです。
    class AudioEngine
    {
    public:
        // @brief 関数の処理結果を表す列挙体です
        enum Result
        {
            Result_Success = 0,        //!< 成功です
            Result_FailedToInitialize, //!< 初期化に失敗しました
            Result_ErrorUnknown        //!< 不明なエラーが発生しました
        };

    public:
        // @brief 初期化に使用するパラメータです
        struct InitArg
        {
            int renderingSampleRate;    //!< レンダリングサンプルレートです
            int voiceCount;             //!< ボイスの数です
            int effectCount;            //!< エフェクトの数です

            // オーディオスレッド
            bool isAudioThreadEnabled;            //!< オーディオスレッドを有効にするフラグです。無効の場合はオーディオフレーム更新を明示的に行う必要があります。
            int audioThreadCoreNumber;            //!< オーディオスレッドの動作コア番号です
            int audioThreadPriority;              //!< オーディオスレッドのスレッド優先度です
            int audioThreadAffinityMask;          //!< オーディオスレッドのアフィニティマスクです
            int audioThreadUpdateIntervalMSec;    //!< オーディオスレッドの更新の間隔です
            size_t audioThreadStackSize;          //!< オーディオスレッドのスタックサイズです
            size_t audioThreadMessageBufferSize;  //!< オーディオスレッドのメッセージバッファサイズです
            bool isManualAudioFrameUpdateEnabled; //!< オーディオスレッド無効時に、オーディオフレーム更新の直接呼び出しを有効にするフラグです。無効の場合はオーディオエンジンの更新時に更新されます。

            void* workBuffer;           //!< ワークバッファです
            size_t workBufferSize;      //!< ワークバッファのサイズです

            // 内部変数
            // 初期化パラメータ
            detail::RendererManager::InitArg _rendererInitArg;
            detail::RendererVoiceManager::InitArg _voiceInitArg;
            detail::VoiceCommandManager::InitArg _voiceCommandInitArg;
            detail::VoiceReplyCommandManager::InitArg _voiceReplyCommandInitArg;
            detail::AudioThread::InitArg _audioThreadInitArg;

            // ワークバッファ
            void* _audioThreadWorkBuffer;
            size_t _audioThreadWorkBufferSize;
        };

        // @brief 初期化に使用するプラットフォーム固有のパラメータです
        struct PlatformInitArg
        {
            int mixBufferCount;              //!< ミックスバッファの数です
            int subMixCount;                 //!< サブミックスの数です
            int sinkCount;                   //!< シンクの数です
            int performanceFrameCount;       //!< パフォーマンスフレーム数です
            int waveBufferPacketCount;       //!< ウェーブバッファパケットの数です
            int voiceCommandBufferSize;      //!< ボイスコマンドのバッファのサイズです

            void* workBuffer;                   //!< ワークバッファです
            size_t workBufferSize;              //!< ワークバッファのサイズです
            void* workBufferForMemoryPool;      //!< メモリプールにアタッチされたワークバッファです
            size_t workBufferSizeForMemoryPool; //!< メモリプールにアタッチされたワークバッファのサイズです

            // 内部変数
            // 初期化パラメータ
            detail::RendererManager::PlatformInitArg _rendererInitArg;
            detail::RendererVoiceManager::PlatformInitArg _voiceInitArg;
            detail::VoiceCommandManager::PlatformInitArg _voiceCommandInitArg;
            detail::VoiceReplyCommandManager::PlatformInitArg _voiceReplyCommandInitArg;
            detail::AudioThread::PlatformInitArg _audioThreadInitArg;

            // ワークバッファ
            void* _rendererWorkBuffer;
            size_t _rendererWorkBufferSize;
            void* _voiceWorkBuffer;
            size_t _voiceWorkBufferSize;
            void* _voiceCommandWorkBuffer;
            size_t _voiceCommandWorkBufferSize;
            void* _voiceReplyCommandWorkBuffer;
            size_t _voiceReplyCommandWorkBufferSize;
            void* _audioThreadWorkBuffer;
            size_t _audioThreadWorkBufferSize;

            // メモリプール用バッファ
            void* _voiceWorkBufferForMemoryPool;
            size_t _voiceWorkBufferSizeForMemoryPool;
            void* _voiceCommandWorkBufferForMemoryPool;
            size_t _voiceCommandWorkBufferSizeForMemoryPool;
        };

    public:
        // @brief コンストラクタです
        AudioEngine() NN_NOEXCEPT;
        // @brief 初期化パラメータの初期値を取得します
        void GetDefaultInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        // @brief 必要なバッファサイズを取得します
        size_t GetRequiredBufferSize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        // @brief 初期化を行います
        Result Initialize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        // @brief 終了処理を行います
        void Finalize() NN_NOEXCEPT;
        // @brief 更新処理を行います
        Result Update() NN_NOEXCEPT;

        detail::AudioThread& GetAudioThread() NN_NOEXCEPT;
        detail::AudioFrameUpdater& GetAudioFrameUpdater() NN_NOEXCEPT;

        // プラットフォーム固有
        // @brief 必要なプラットフォーム固有のバッファサイズを取得します
        size_t GetRequiredPlatformBufferSize(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        // @brief 必要なメモリプールにアタッチされたバッファサイズを取得します
        size_t GetRequiredBufferSizeForMemoryPool(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;

        Result SetupRendererInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        Result SetupVoiceInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        Result SetupVoiceCommandInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        Result SetupVoiceReplyCommandInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;
        Result SetupAudioThreadInitArg(InitArg& arg, PlatformInitArg& platformArg) NN_NOEXCEPT;

        detail::RendererManager& GetRendererManager() NN_NOEXCEPT;
        detail::RendererVoiceManager& GetRendererVoiceManager() NN_NOEXCEPT;
        detail::VirtualVoiceManager& GetVirtualVoiceManager() NN_NOEXCEPT;
        detail::VoiceCommandManager& GetVoiceCommandManager() NN_NOEXCEPT;
        detail::VoiceReplyCommandManager& GetVoiceReplyCommandManager() NN_NOEXCEPT;

    private:
        detail::AudioFrameUpdater m_AudioFrameUpdater;
        detail::AudioThread m_AudioThread;
        bool m_IsAudioThreadEnabled;
        bool m_IsManualAudioFrameUpdateEnabled;

        // プラットフォーム固有
        detail::RendererManager m_RendererManager;                   //!< オーディオレンダラの管理を行います
        detail::RendererVoiceManager m_RendererVoiceManager;         //!< オーディオレンダラに紐づくボイスの管理を行います
        detail::VoiceCommandManager m_VoiceCommandManager;           //!< ボイスの変更をレンダラボイスに反映するためのコマンド処理の管理を行います
        detail::VoiceReplyCommandManager m_VoiceReplyCommandManager; //!< レンダラボイスの変更をボイスに反映するためのコマンド処理の管理を行います
        detail::VirtualVoiceManager m_VirtualVoiceManager;           //!< ボイスコマンド処理で使用する仮想ボイスの管理を行います
    };

}}

