﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;

using LOG = Common.Logger;
using LOG_LEVEL = Common.Logger.Level;

namespace BackupSaveData
{
    public class DoExportImport
    {
        const string ConfigFileName = "config.xml";
        const string ExportDefaultDirectoryName = "BackupSaveData";
        string ExportDefaultBasePath = GetExportDefaultBasePath();

        public DoExportImport()
        {
        }

        public bool DoExport(string target, string dirPath, string exportTitles, bool existAccountOnly = false, bool verbose = false)
        {
            if (!Directory.Exists(dirPath))
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Directory {0} is not found, so create it.", dirPath);
                Directory.CreateDirectory(dirPath);
            }
            string exportDirPath = Path.Combine(dirPath, ExportDefaultDirectoryName);
            if(Directory.Exists(exportDirPath))
            {
                FileUtil.DeleteDirectory(exportDirPath);
            }
            string exportBaseDirPath = Path.Combine(ExportDefaultBasePath, ExportDefaultDirectoryName);
            if(Directory.Exists(exportBaseDirPath))
            {
                FileUtil.DeleteDirectory(exportBaseDirPath);
            }

            // execute BackupSaveData.nsp
            List<string> argsList = new List<string>() { "-export" };
            if(exportTitles != null && exportTitles.Length > 0)
            {
                argsList.Add("-title");
                argsList.Add(exportTitles.Replace(" ", ""));
            }
            if(existAccountOnly == true)
            {
                argsList.Add("--exist-account-only");
            }
            string executePath = GetBackupSaveDataTargetToolPath();
            if (executePath == null)
            {
                return false;
            }
            if (!Execute.ExecuteTarget(executePath, target, argsList.ToArray(), "Save data export succeeded.", verbose))
            {
                return false;
            }

            if (!FileUtil.CopyDirecotryResursively(exportBaseDirPath, exportDirPath)) return false;
            if (Directory.Exists(exportBaseDirPath))
            {
                FileUtil.DeleteDirectory(exportBaseDirPath);
            }

            return true;
        }

        public bool DoImport(string target, string dirPath, bool followAccount = false, bool verbose = false)
        {
            string srcDirPath = AdjustImportDirectory(dirPath);
            if (srcDirPath == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Input directory is not found. ({0})", dirPath);
                return false;
            }
            string destDirPath = Path.Combine(ExportDefaultBasePath, ExportDefaultDirectoryName);
            if(Directory.Exists(destDirPath))
            {
                FileUtil.DeleteDirectory(destDirPath);
            }

            if (!ConvertData.ConvertAndCopyExportDirectory(srcDirPath, destDirPath)) return false;

            // execute BackupSaveData.nsp
            List<string> argsList = new List<string>() { "-import" };
            string executePath = GetBackupSaveDataTargetToolPath();
            if (executePath == null)
            {
                return false;
            }
            if(followAccount)
            {
                argsList.Add("--replace-account");
            }

            if (!Execute.ExecuteTarget(executePath, target, argsList.ToArray(), "Save data import succeeded.", verbose))
            {
                return false;
            }

            if (Directory.Exists(destDirPath))
            {
                FileUtil.DeleteDirectory(destDirPath);
            }

            return true;
        }

        private string AdjustImportDirectory(string dirPath)
        {
            if(!Directory.Exists(dirPath))
            {
                return null;
            }
            string configPath = Path.Combine(dirPath, ConfigFileName);
            if(File.Exists(configPath))
            {
                return dirPath;
            }
            configPath = Path.Combine(dirPath, ExportDefaultDirectoryName, ConfigFileName);
            if (File.Exists(configPath))
            {
                return Path.Combine(dirPath, ExportDefaultDirectoryName);
            }
            string directoryPath = Path.Combine(dirPath, ExportDefaultDirectoryName);
            if (Directory.Exists(directoryPath))
            {
                return Path.Combine(dirPath, ExportDefaultDirectoryName);
            }
            return null;
        }

        private string GetBackupSaveDataTargetToolPath()
        {
            string sdkRootPath = Execute.GetSdkRootPath();
            if(sdkRootPath == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "SDK root directory is not found.");
                return null;
            }

#if DEBUG
            string buildName = "Debug";
#else
            string buildName = "Release";
#endif
#if NN_BACKUPSAVEDATA_SYSTEM
            string programFilePath = Path.Combine(sdkRootPath, @"Programs\Iris\Outputs\NX-NXFP2-a64\TargetTools\BackupSaveDataSystem", buildName, "BackupSaveDataSystem.nsp");
#elif NN_BACKUPSAVEDATA_DEVICE
            string programFilePath = Path.Combine(sdkRootPath, @"Programs\Iris\Outputs\NX-NXFP2-a64\TargetTools\BackupSaveDataDevice", buildName, "BackupSaveDataDevice.nsp");
#else
            string programFilePath = Path.Combine(sdkRootPath, @"Programs\Iris\Outputs\NX-NXFP2-a64\TargetTools\BackupSaveData", buildName, "BackupSaveData.nsp");
#endif
            if(File.Exists(programFilePath))
            {
                return programFilePath;
            }
#if NN_BACKUPSAVEDATA_SYSTEM
            string toolsFilePath = Path.Combine(sdkRootPath, @"TargetTools\NX-NXFP2-a64\BackupSaveDataSystem\Release", "BackupSaveDataSystem.nsp");
#elif NN_BACKUPSAVEDATA_DEVICE
            string toolsFilePath = Path.Combine(sdkRootPath, @"TargetTools\NX-NXFP2-a64\BackupSaveDataDevice\Release", "BackupSaveDataDevice.nsp");
#else
            string toolsFilePath = Path.Combine(sdkRootPath, @"TargetTools\NX-NXFP2-a64\BackupSaveData\Release", "BackupSaveData.nsp");
#endif
            if (File.Exists(toolsFilePath))
            {
                return toolsFilePath;
            }

            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Nsp file not found.");
            return null;
        }

        private static string GetExportDefaultBasePath()
        {
            return FileUtil.GetTempDirectoryPath();
        }
    }
}
