﻿using System;
using System.Collections.Generic;
using System.IO;
using SdkEnvironmentCheckerLibrary.Properties;

namespace SdkEnvironmentCheckerLibrary.Checkers
{
    public static class NvnGraphicsDriverChecker
    {
        private const string ToolsDirectory = "Tools";
        private const string CheckerDirectory = "NvnGraphicsDriverChecker";
        private const string CheckerName = "NvnGraphicsDriverChecker.exe";
        private static readonly Lazy<FilePath> toolpath;

        public static FilePath Path => toolpath.Value;

        static NvnGraphicsDriverChecker()
        {
            var root = NintendoSdkHelper.GetSdkRoot();

            if (!root.Exists)
            {
                toolpath = new Lazy<FilePath>(() => FilePath.Empty);
                return;
            }

            toolpath = new Lazy<FilePath>(() =>
            {
                try
                {
                    foreach (var toolsDir in Directory.EnumerateDirectories(root.FullPath, ToolsDirectory, SearchOption.AllDirectories))
                    {
                        foreach (var checkerDir in Directory.EnumerateDirectories(toolsDir, CheckerDirectory, SearchOption.AllDirectories))
                        {
                            foreach (var checker in Directory.EnumerateFiles(checkerDir, CheckerName, SearchOption.AllDirectories))
                            {
                                return FilePath.Create(checker);
                            }
                        }
                    }
                }
                catch { /* ignore */ }

                return FilePath.Empty;
            });
        }

        public static CheckerResult Check()
        {
            NvnGraphicsDriverInfo driverInfo;

            try
            {
                driverInfo = GetOutput();
            }
            catch (Exception e)
            {
                return CheckerResult.ConfirmFailed("NVN API", e.Message);
            }

            var name = $"NVN API Version {driverInfo.NvnSdkApiVersion.ToString(2)}";

            if (!driverInfo.NvnGraphicsDriverInstalled) return CheckerResult.NotInstalled(name, "Driver not installed");
            if (!driverInfo.NvnGpuSupported) return CheckerResult.NotInstalled(name, "GPU not supported");
            if (!driverInfo.IsCompatible) return CheckerResult.NotInstalled(name, "Not compatible");

            return CheckerResult.Installed(name, "Compatible");
        }

        private static NvnGraphicsDriverInfo GetOutput()
        {
            var lines = GetRawOutput().Split(new[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);
            var dictionary = new Dictionary<string, string>();

            foreach (var line in lines)
            {
                var pair = line.Split(new[] { ':' }, 2);
                var key = pair[0].Trim();
                var value = pair.Length > 1 ? pair[1].Trim() : string.Empty;

                if (!dictionary.ContainsKey(key)) dictionary.Add(key, value);
            }

            return new NvnGraphicsDriverInfo(dictionary);
        }

        private static string GetRawOutput()
        {
            if (!Path.Exists) throw new InvalidOperationException(string.Format(Resources.FileNotFound, CheckerName));

            if (ProcessHelper.Run(Path.FullPath, out var output, out _) == 0) return output;

            throw new InvalidOperationException(string.Format(Resources.ExecuteFailed, Path.FileName));
        }
    }

    public class NvnGraphicsDriverInfo
    {
        internal NvnGraphicsDriverInfo(IDictionary<string, string> output)
        {
            if (output == null) throw new ArgumentNullException(nameof(output));

            {
                NvnGraphicsDriverInstalled =
                    output.TryGetValue(nameof(NvnGraphicsDriverInstalled), out var value) &&
                    bool.TryParse(value, out var result) && result;
            }

            {
                NvnGpuSupported =
                    output.TryGetValue(nameof(NvnGpuSupported), out var value) &&
                    bool.TryParse(value, out var result) && result;
            }

            {
                NvnSdkApiVersion =
                    output.TryGetValue(nameof(NvnSdkApiVersion), out var value) && Version.TryParse(value, out var version) ? version : new Version(0, 0);
            }

            {
                NvnDriverApiVersion =
                    output.TryGetValue(nameof(NvnDriverApiVersion), out var value) && Version.TryParse(value, out var version) ? version : new Version(0, 0);
            }

            {
                GL_VENDOR = output.TryGetValue(nameof(GL_VENDOR), out var value) ? value : string.Empty;
            }

            {
                GL_RENDERER = output.TryGetValue(nameof(GL_RENDERER), out var value) ? value : string.Empty;
            }

            {
                GL_VERSION = output.TryGetValue(nameof(GL_VERSION), out var value) ? value : string.Empty;
            }
        }

        public bool NvnGraphicsDriverInstalled { get; }
        public bool NvnGpuSupported { get; }
        public Version NvnSdkApiVersion { get; }
        public Version NvnDriverApiVersion { get; }
        public string GL_VENDOR { get; }
        public string GL_RENDERER { get; }
        public string GL_VERSION { get; }

        public bool IsCompatible
            => NvnDriverApiVersion.Major == NvnSdkApiVersion.Major && NvnDriverApiVersion.Minor >= NvnSdkApiVersion.Minor;
    }
}
