﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Collections.Generic;

namespace QCIT_Maker
{
    internal class CommonParams
    {
        public const string WorkDirName = "_WorkTemporary_";
        public const string TempMetaFileName = "tempMeta.nmeta";
        public const string DefaultProgramId = "0x01004b9000490000";
        public const string DefaultVersion = "0";

        public const string QuestMenuProgramId = "0x0100069000078000";

        public const string TypeApplication = "Application";
        public const string TypePatch = "Patch";
        public const string TypeAddOnContent = "AddOnContent";

        public const string DefaultIconLanguage = "AmericanEnglish";
        public const string NoDefaultIconSetString = "noicon";

        public const string NmetaRootName = "NintendoSdkMeta";

        private const string SigloRootMarkFileName = "SigloRootMark";
        private const string NintendoSdkRootMarkFileName = "NintendoSdkRootMark";

        public const string LineFeedStr = "\r\n";

        public const string LegalInfoDirectoryName = "LegalInfoFiles";
        public const string InstallContentInfoFileName = "InstallContentInfoList.txt";

        //(SIGLO-76759)空き容量指定オプション(--free-nand-size)の最大値
        public const int MaxFreeNandSize = 8 * 1024;

        /// <summary>
        /// Sigloのルートパスを取得
        /// </summary>
        /// <returns>Sigloルートパス文字列</returns>
        public static string GetSigloRoot()
        {
            // ルートパスを確定させるために "～RootMark" ファイルを検索する
            var rootPath = FindSigloRoot(Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location));
            if (rootPath == null)
            {
                // もし取得できない場合は、環境変数"NINTENDO_SDK_ROOT"を取得
                rootPath = Environment.GetEnvironmentVariable("NINTENDO_SDK_ROOT");
                if (rootPath == null)
                {
                    throw new InvalidDataException("Rootパスを見つけられませんでした");
                }
            }

            if (rootPath.EndsWith(@"\") == false)
            {
                rootPath += @"\";
            }

            return rootPath;
        }

        private static string FindSigloRoot(string path)
        {
            string startPath = path;
            string currentDirectory = startPath;
            string rootDirectory = Path.GetPathRoot(currentDirectory);

            while (rootDirectory != currentDirectory)
            {
                // "SigloRootMark"の存在確認
                var sigloRootMarkFilePath = Path.Combine(currentDirectory, SigloRootMarkFileName);
                if (File.Exists(sigloRootMarkFilePath))
                {
                    return currentDirectory;
                }

                // "NintendoSdkRootMark"の存在確認
                var nintendoSdkRootMarkFilePath = Path.Combine(currentDirectory, NintendoSdkRootMarkFileName);
                if (File.Exists(nintendoSdkRootMarkFilePath))
                {
                    return currentDirectory;
                }

                currentDirectory = Path.GetFullPath(Path.Combine(currentDirectory, ".."));
            }

            return null;
        }
    }

    internal class SettingParams
    {
        internal class IconSettingPair
        {
            public string Language { get; set; }
            public string IconFilePath { get; set; }

            public IconSettingPair()
            {
            }

            public IconSettingPair(string inLanguage, string inFilePath)
            {
                Language = inLanguage;
                IconFilePath = inFilePath;
            }
        }

        public string NmetaFilePath { get; set; }
        public string DescFilePath { get; set; }
        public string OutputFilePath { get; set; }

        public string ApplicationId { get; set; }
        public string ReleaseVersion { get; set; }
        public string PrivateVersion { get; set; }

        public string ContentDirectoryPath { get; set; }
        public string SyncListPath { get; set; }

        public string ChildWorkDirPath { get; set; }

        public string SupportedLanguage { get; set; }

        public bool IsDebugMode { get; set; }

        public string LegalInfoFilePath { get; set; }

        public List<string> RatingList { get; set; }

        public IconSettingPair IconSetting { get; set; }

        public string OriginalNspFilePath { get; set; }
        public string PreviousPatchFilePath { get; set; }

        //(SIGLO-76759)QCIT に NAND 空き容量指定オプションを追加する
        public string FreeNandSize { get; set; }

        private void InitParameter()
        {
            IconSetting = new IconSettingPair();
            this.RatingList = new List<string>();
        }

        public SettingParams()
        {
            this.InitParameter();
        }

        public SettingParams(CommandLineParams inParam)
        {
            this.InitParameter();
            this.SetMetaParam(inParam);
        }

        /// <summary>
        /// コマンドラインの入力情報からメタ情報を抽出し自身のメンバ変数に設定する
        /// </summary>
        /// <param name="inParam">コマンドラインの入力オプションパラメータ</param>
        public void SetMetaParam(CommandLineParams inCLParam)
        {
            // プログラムIDの情報が入力されていない(nullの)場合はデフォルトのプログラムIDを適用する
            this.ApplicationId = inCLParam.InputApplicationId ?? CommonParams.DefaultProgramId;
            // 何も設定されていない場合は明示的に 0 を設定しておく
            this.ReleaseVersion = inCLParam.InputReleaseVersion ?? "0";
            this.PrivateVersion = "0";

            // ひとまず AmericanEnglish 固定
            this.SupportedLanguage = CommonParams.DefaultIconLanguage;

            {
                this.IconSetting.Language = this.SupportedLanguage;
                this.IconSetting.IconFilePath = string.Empty;
            }

            this.ContentDirectoryPath = inCLParam.ContentDirectoryPath;
            this.SyncListPath = inCLParam.SyncListPath;
            this.OutputFilePath = inCLParam.OutputFilePath;

            this.IsDebugMode = inCLParam.DebugModeFlag;

            // (SIGLO-68973) nmeta へ設定するレーティング設定へ変換する
            this.ConvertRatingRegionData(inCLParam.RatingRegion);

            this.OriginalNspFilePath = inCLParam.OriginalNspPath;
            this.PreviousPatchFilePath = inCLParam.PreviousPatchPath;

            //(SIGLO-76759)QCIT に NAND 空き容量指定オプションを追加する
            this.FreeNandSize = inCLParam.FreeNandSize;
        }

        private void ConvertRatingRegionData(string[] inRatingRegion)
        {
            // レーティングに関する設定がなければ何もしない
            if (inRatingRegion == null)
            {
                return;
            }

            // 地域とレーティングの組合せ定義
            var ratingDictionary = new Dictionary<string, string[]>()
            {
                { "japan", new string[] { "CERO" } },
                { "america", new string[] { "ESRB", "ClassInd" } },
                { "europe", new string[] { "USK", "PEGI", "PEGIPortugal", "PEGIBBFC", "Russian" } },
                { "australia", new string[] { "ACB", "OFLC" } },
            };

            var hashList = new HashSet<string>();
            foreach (var r in inRatingRegion)
            {
                string[] rtgs;
                if (ratingDictionary.TryGetValue(r.ToLower(), out rtgs))
                {
                    // 同じ地域が複数設定されている場合を考慮して HashSet に一時的に入れる形とする
                    hashList.UnionWith(rtgs);
                }
            }

            if (hashList.Count > 0)
            {
                this.RatingList = hashList.ToList();
            }
        }
    }

    internal class ContentMetaInfoParam
    {
        public string NspFileName { get; set; }
        public string Id { get; set; }
        public string Type { get; set; }
        public string Version { get; set; }
        public string Digest { get; set; }
        public string Tag { get; set; }
        public string Index { get; set; }
        public string DataSize { get; set; }
        public string ApplicationId { get; set; }
    }
}
