﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.Contracts;
using Nintendo.VibrationAmFmCodec;

namespace VibrationConverterConsole.BnvibIO
{
    public static class BnvibFileReader
    {
        private const long BnvibFileSizeMin = 12;
        private const UInt32 MetaDataSizeMin = 4;
        private const float BaseFrequencyLow = 160.0f;
        private const float BaseFrequencyHigh = 320.0f;
        private const char NvibCommentChar = '#';
        private const char NvibVariableChar = '$';

        public static BnvibFile ReadBnvibFile(string inputPath)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(inputPath);

            using (var fileStream = new FileStream(inputPath, FileMode.Open))
            {
                using (var reader = new BinaryReader(fileStream))
                {
                    return ReadBnvibFile(reader);
                }
            }
        }

        public static BnvibFile ReadBnvibFile(BinaryReader reader)
        {
            Ensure.Argument.NotNull(reader);
            BnvibFile file = new BnvibFile();

            // ファイルサイズチェック
            long fileSize = reader.BaseStream.Length;
            Ensure.Argument.True(fileSize >= BnvibFileSizeMin);

            // MetaDataSize
            reader.BaseStream.Seek(0, SeekOrigin.Begin);
            UInt32 metaDataSize = reader.ReadUInt32();
            Ensure.Argument.True(metaDataSize >= MetaDataSizeMin);
            long DataRegionOffset = 4 + metaDataSize;
            Ensure.Argument.True(DataRegionOffset + 4 <= fileSize);
            file.Info.MetaDataSize = (int)metaDataSize;

            // FormatId
            UInt16 formatId = reader.ReadUInt16();
            file.Info.FormatId = (BnvibFileFormatId)formatId;

            // SamplingRate
            UInt16 samplingRate = reader.ReadUInt16();
            file.Info.SamplingRate = (int)samplingRate;

            // LoopRange
            if (file.Info.HasLoop)
            {
                UInt32 loopStart = reader.ReadUInt32();
                UInt32 loopEnd = reader.ReadUInt32();
                file.Info.LoopStart = (int)loopStart;
                file.Info.LoopEnd = (int)loopEnd;

                // LoopInterval
                if (file.Info.HasLoopInterval)
                {
                    UInt32 loopInterval = reader.ReadUInt32();
                    file.Info.LoopInterval = (int)loopInterval;
                }
            }

            // DataSize
            reader.BaseStream.Seek(DataRegionOffset, SeekOrigin.Begin);
            UInt32 dataSize = reader.ReadUInt32();
            long DataEndOffset = DataRegionOffset + 4 + dataSize;
            Ensure.Argument.True(DataEndOffset <= fileSize);
            file.Info.DataSize = (int)dataSize;

            // フォーマットに応じて振動値取り出し
            switch (file.Info.FormatId)
            {
                case BnvibFileFormatId.AmFm5bitVer1:
                case BnvibFileFormatId.AmFm5bitVer2:
                    RetrieveAmFmCodes(file, reader);
                    break;
                case BnvibFileFormatId.PcmVer1:
                    RetrievePcmVibrationValues(file, reader);
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }

            return file;
        }

        public static BnvibFile ReadNvibFile(string inputPath)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(inputPath);

            using (var fileStream = new FileStream(inputPath, FileMode.Open))
            {
                using (var reader = new StreamReader(fileStream))
                {
                    return ReadNvibFile(reader);
                }
            }
        }

        public static BnvibFile ReadNvibFile(TextReader reader)
        {
            Ensure.Argument.NotNull(reader);
            BnvibFile file = new BnvibFile();

            int sampleLength = 0;
            VibrationValue prevValue = VibrationValue.Make();

            while(reader.Peek() >= 0)
            {
                string line = reader.ReadLine();
                if(line != null && line.Length > 0 && line[0] != NvibCommentChar)
                {
                    if(line[0] == NvibVariableChar)
                    {
                        string[] textArray = line.Substring(1).Split('=');
                        if(textArray.Length < 2)
                        {
                            throw new InvalidDataException("Invalid Format: " + line);
                        }
                        else
                        {
                            ParseVariableSettingText(file, textArray[0], textArray[1]);
                        }
                    }
                    else
                    {
                        string[] textArray = line.Split(',');
                        if(textArray.Length > 0)
                        {
                            VibrationValue v = prevValue;
                            ParseFloatFromTextArray(ref v.amplitudeLow, textArray, 0);
                            ParseFloatFromTextArray(ref v.frequencyLow, textArray, 1);
                            ParseFloatFromTextArray(ref v.amplitudeHigh, textArray, 2);
                            ParseFloatFromTextArray(ref v.frequencyHigh, textArray, 3);
                            prevValue = v;

                            v = VibrationValue.FromRawData(v.ToRawData());
                            file.Data.AddVibrationValue(v);
                            sampleLength++;
                        }
                    }
                }
            }

            file.Info.SampleLength = sampleLength;
            return file;
        }

        private static void RetrieveAmFmCodes(BnvibFile file, BinaryReader reader)
        {
            int sampleLength = 0;

            // フォーマット ID に合わせて適切な AMFM デコーダを初期化
            AmFmCommandTable table;
            switch (file.Info.FormatId)
            {
                case BnvibFileFormatId.AmFm5bitVer1:
                    table = AmFmCommandTable.FiveBitVer1;
                    break;
                case BnvibFileFormatId.AmFm5bitVer2:
                    table = AmFmCommandTable.FiveBitVer2;
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }
            AmFmDecoder decoderLow = new AmFmDecoder(table, BaseFrequencyLow);
            AmFmDecoder decoderHigh = new AmFmDecoder(table, BaseFrequencyHigh);

            // PackedAmFmCodes から順次 AMFM 符号を取り出して振動値にデコードしていく
            const int PackSize = PackedAmFmCodes.RawDataSize;
            int packLength = file.Info.DataSize / PackSize;
            for (int packIdx = 0; packIdx < packLength; packIdx++)
            {
                UInt32 rawData = reader.ReadUInt32();
                PackedAmFmCodes pack = PackedAmFmCodes.FromRawData(rawData);
                AmFmCodePair[] pairArray = pack.GetAmFmCodePairArray();

                for (int codeIdx = 0; codeIdx < pairArray.Length; codeIdx++)
                {
                    decoderLow.Decode(pairArray[codeIdx].amfmCodeLow);
                    decoderHigh.Decode(pairArray[codeIdx].amfmCodeHigh);

                    VibrationValue v;
                    v.amplitudeLow = decoderLow.GetAmplitude();
                    v.frequencyLow = decoderLow.GetFrequency();
                    v.amplitudeHigh = decoderHigh.GetAmplitude();
                    v.frequencyHigh = decoderHigh.GetFrequency();

                    file.Data.AddVibrationValue(v);
                    sampleLength++;
                }
            }

            file.Info.SampleLength = sampleLength;
        }

        private static void RetrievePcmVibrationValues(BnvibFile file, BinaryReader reader)
        {
            file.Info.SampleLength = file.Info.DataSize / VibrationValue.RawDataSize;
            for (int packIdx = 0; packIdx < file.Info.SampleLength; packIdx++)
            {
                byte[] rawData = reader.ReadBytes(VibrationValue.RawDataSize);
                VibrationValue v = VibrationValue.FromRawData(rawData);
                file.Data.AddVibrationValue(v);
            }
        }

        private static void ParseVariableSettingText(BnvibFile file, string strName, string strValue)
        {
            int value = 0;
            if(!int.TryParse(strValue, out value))
            {
                throw new InvalidDataException("Invalid Format: " + strName + "=" + strValue);
            }

            // 現時点では対応する変数は LoopStart, LoopEnd, LoopInterval のみ
            switch(strName)
            {
                case "LoopStart":
                    file.Info.HasLoop = true;
                    file.Info.LoopStart = value;
                    break;
                case "LoopEnd":
                    file.Info.HasLoop = true;
                    file.Info.LoopEnd = value;
                    break;
                case "LoopInterval":
                    file.Info.HasLoopInterval = true;
                    file.Info.LoopInterval = value;
                    break;
                default:
                    throw new InvalidDataException("Invalid Format: " + strName + "=" + strValue);
            }
        }

        private static void ParseFloatFromTextArray(ref float refValue, string[] textArray, int idx)
        {
            if(textArray.Length > idx && textArray[idx].Length > 0)
            {
                float v;
                if(float.TryParse(textArray[idx],
                    System.Globalization.NumberStyles.Float,
                    System.Globalization.CultureInfo.InvariantCulture,
                    out v))
                {
                    refValue = v;
                }
            }
        }
    }
}
