﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Collections.Generic;
    using System.Reflection;
    using System.Runtime.InteropServices;

    /// <summary>
    /// マウスを管理します。
    /// </summary>
    internal sealed class MouseManager : IDisposable
    {
        private const int WH_MOUSE_LL = 14;

        private const int WM_MOUSEMOVE = 0x0200;
        private const int WM_LBUTTONDOWN = 0x0201;
        private const int WM_LBUTTONUP = 0x0202;
        private const int WM_RBUTTONDOWN = 0x0204;
        private const int WM_RBUTTONUP = 0x0205;
        private const int WM_MBUTTONDOWN = 0x0207;
        private const int WM_MBUTTONUP = 0x0208;
        private const int WM_MOUSEWHEEL = 0x020A;
        private const int WM_XBUTTONDOWN = 0x020B;
        private const int WM_XBUTTONUP = 0x020C;

        private const int XBUTTON1 = 0x0001;
        private const int XBUTTON2 = 0x0002;

        private bool disposed = false;

        private object syncObject = new object();

        private bool isSuppressionEnabled = false;

        private readonly IList<HookProc> hookProcs = new List<HookProc>();

        private readonly IntPtr hHook;

        /// <summary>
        /// MouseManager クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal MouseManager()
        {
            // ガベージコレクションから保護
            this.hookProcs.Add(new HookProc(this.MouseHookProc));

            this.hHook = SetMouseHook(this.hookProcs[0]);
        }

        ~MouseManager()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// マウス入力イベントの伝搬を隠蔽するか否かを表す値を取得または設定します。
        /// </summary>
        internal bool IsSuppressionEnabled
        {
            get
            {
                lock (this.syncObject)
                {
                    return this.isSuppressionEnabled;
                }
            }

            set
            {
                lock (this.syncObject)
                {
                    this.isSuppressionEnabled = value;
                }
            }
        }

        /// <summary>
        /// マウスカーソルが移動すると発生します。
        /// </summary>
        internal event EventHandler<MouseEventArgs> MouseMove = delegate {};

        /// <summary>
        /// マウスボタンが押されると発生します。
        /// </summary>
        internal event EventHandler<MouseEventArgs> MouseDown = delegate {};

        /// <summary>
        /// マウスボタンが離されると発生します。
        /// </summary>
        internal event EventHandler<MouseEventArgs> MouseUp = delegate {};

        /// <summary>
        /// マウスホイールが回転すると発生します。
        /// </summary>
        internal event EventHandler<MouseEventArgs> MouseWheel = delegate {};

        /// <summary>
        /// アンマネージドリソースを開放します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        [DllImport("user32.dll")]
        private static extern IntPtr SetWindowsHookEx(
            int idHook, HookProc lpfn, IntPtr hMod, int dwThreadId);

        [DllImport("user32.dll")]
        private static extern bool UnhookWindowsHookEx(IntPtr hHook);

        [DllImport("user32.dll")]
        private static extern IntPtr CallNextHookEx(
            IntPtr hHook, int nCode, IntPtr wParam, IntPtr lParam);

        private static IntPtr SetMouseHook(HookProc hookProc)
        {
            return SetWindowsHookEx(
                WH_MOUSE_LL,
                hookProc,
                Marshal.GetHINSTANCE(
                    Assembly.GetExecutingAssembly().GetModules()[0]),
                0);
        }

        private static void UnhookMouseHook(IntPtr hHook)
        {
            if (hHook != IntPtr.Zero)
            {
                UnhookWindowsHookEx(hHook);
            }
        }

        private void Dispose(bool isDisposing)
        {
            if (!this.disposed)
            {
                this.disposed = true;

                UnhookMouseHook(this.hHook);

                if (isDisposing)
                {
                    this.hookProcs.Clear();
                }
            }
        }

        private MouseEventArgs CreateMouseEventArgs(
            IntPtr wParam, IntPtr lParam)
        {
            var message = wParam.ToInt32();

            var state = (MSLLHOOKSTRUCT)
                Marshal.PtrToStructure(lParam, typeof(MSLLHOOKSTRUCT));

            switch (wParam.ToInt32())
            {
                case WM_LBUTTONDOWN:
                case WM_LBUTTONUP:
                    return new MouseEventArgs(
                        state.pt.x, state.pt.y, MouseButtons.Left);

                case WM_RBUTTONDOWN:
                case WM_RBUTTONUP:
                    return new MouseEventArgs(
                        state.pt.x, state.pt.y, MouseButtons.Right);

                case WM_MBUTTONDOWN:
                case WM_MBUTTONUP:
                    return new MouseEventArgs(
                        state.pt.x, state.pt.y, MouseButtons.Middle);

                case WM_XBUTTONDOWN:
                case WM_XBUTTONUP:
                    switch ((short)(state.mouseData >> 16))
                    {
                        case XBUTTON1:
                            return new MouseEventArgs(
                                state.pt.x, state.pt.y, MouseButtons.Back);

                        case XBUTTON2:
                            return new MouseEventArgs(
                                state.pt.x, state.pt.y, MouseButtons.Forward);

                        default:
                            break;
                    }

                    break;

                case WM_MOUSEWHEEL:
                    return new MouseEventArgs(
                        state.pt.x,
                        state.pt.y,
                        (short)(state.mouseData >> 16));

                default:
                    break;
            }

            return new MouseEventArgs(state.pt.x, state.pt.y);
        }

        private IntPtr MouseHookProc(int nCode, IntPtr wParam, IntPtr lParam)
        {
            bool isSuppressionEnabled = this.IsSuppressionEnabled;

            if (nCode == 0)
            {
                MouseEventArgs args = CreateMouseEventArgs(wParam, lParam);

                switch (wParam.ToInt32())
                {
                    case WM_MOUSEMOVE:
                        this.MouseMove(this, args);
                        break;

                    case WM_LBUTTONDOWN:
                    case WM_RBUTTONDOWN:
                    case WM_MBUTTONDOWN:
                    case WM_XBUTTONDOWN:
                        this.MouseDown(this, args);
                        break;

                    case WM_LBUTTONUP:
                    case WM_RBUTTONUP:
                    case WM_MBUTTONUP:
                    case WM_XBUTTONUP:
                        this.MouseUp(this, args);
                        break;

                    case WM_MOUSEWHEEL:
                        this.MouseWheel(this, args);
                        break;

                    default:
                        break;
                }
            }

            if (isSuppressionEnabled || this.IsSuppressionEnabled)
            {
                return (IntPtr)1;
            }
            else
            {
                return CallNextHookEx(this.hHook, nCode, wParam, lParam);
            }
        }

        private delegate IntPtr HookProc(
            int nCode, IntPtr wParam, IntPtr lParam);

        [StructLayout(LayoutKind.Sequential)]
        private struct POINT
        {
            internal int x;
            internal int y;
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct MSLLHOOKSTRUCT
        {
            internal POINT pt;
            internal uint mouseData;
            internal uint flags;
            internal uint time;
            internal IntPtr dwExtraInfo;
        }
    }
}
