﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.Linq;
    using System.Threading;
    using System.Threading.Tasks;
    using System.Windows.Forms;
    using Forms;

    /// <summary>
    /// HID 入力のホスティングを行います。
    /// </summary>
    internal sealed class InputDirector : IDisposable
    {
        private const int ScreenMargin = 5;

        private static readonly Rectangle ScreenSize = GetScreenSize();

        private static readonly TimeSpan MonitoringInterval =
            TimeSpan.FromMilliseconds(250);

        private bool isDisposed = false;

        private bool isStarted = false;

        private readonly UserConfig userConfig;

        private readonly IoPortManager ioPortManager;

        private readonly TargetManager targetManager;

        private readonly GamePadManager gamePadManager;

        private readonly KeyboardManager keyboardManager;

        private readonly KeyboardKeyHolder keyboardKeyHolder =
            new KeyboardKeyHolder();

        private readonly MouseManager mouseManager;

        private readonly MouseButtonHolder mouseButtonHolder =
            new MouseButtonHolder();

        private readonly object sessionSyncObject = new object();

        private readonly IDictionary<string, Session> sessions;

        private readonly object captureSyncObject = new object();

        private bool isCaptureEnabled = true;

        private ICaptureSession captureSession = null;

        private InfoDialog infoDialog = null;

        private readonly object autoConnectionSyncObject = new object();

        private readonly ISet<string> stoppedSessions = new HashSet<string>();

        private CancellationTokenSource tokenSource = null;

        private Task task = null;

        /// <summary>
        /// InputDirector クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal InputDirector()
        {
            this.userConfig = new UserConfig();

            this.ioPortManager = new IoPortManager();

            this.targetManager = new TargetManager();
            this.targetManager.TargetConnected += this.OnTargetConnected;
            this.targetManager.TargetDisconnected += this.OnTargetDisconnected;

            this.gamePadManager = new GamePadManager();

            this.keyboardManager = new KeyboardManager();
            this.keyboardManager.KeyDown += this.OnKeyDown;
            this.keyboardManager.KeyUp += this.OnKeyUp;

            this.mouseManager = new MouseManager();
            this.mouseManager.MouseMove += this.OnMouseMove;
            this.mouseManager.MouseDown += this.OnMouseDown;
            this.mouseManager.MouseUp += this.OnMouseUp;
            this.mouseManager.MouseWheel += this.OnMouseWheel;

            this.sessions = new Dictionary<string, Session>();
        }

        ~InputDirector()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// ユーザ設定を返します。
        /// </summary>
        internal UserConfig UserConfig
        {
            get
            {
                return this.userConfig;
            }
        }

        /// <summary>
        /// Capture サービスが有効か否かを表す値を取得または設定します。
        /// </summary>
        internal bool IsCaptureEnabled
        {
            get
            {
                lock (this.captureSyncObject)
                {
                    return this.isCaptureEnabled;
                }
            }

            set
            {
                lock (this.captureSyncObject)
                {
                    this.isCaptureEnabled = value;
                }
            }
        }

        /// アンマネージドリソースを開放します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// HID 入力のホスティングを開始します。
        /// </summary>
        internal void Start()
        {
            this.ioPortManager.Start();

            this.gamePadManager.Start();

            this.targetManager.Start();

            this.StartMonitoring();

            this.isStarted = true;
        }

        /// <summary>
        /// HID 入力のホスティングを停止します。
        /// </summary>
        internal void Stop()
        {
            this.isStarted = false;

            this.StopMonitoring();

            this.targetManager.Stop();

            this.gamePadManager.Stop();

            this.ioPortManager.Stop();
        }

        /// <summary>
        /// セッションの対向となる開発機の名前を返します。
        /// </summary>
        internal IList<string> GetSessionNames()
        {
            lock (this.sessionSyncObject)
            {
                return this.sessions.Keys.ToList();
            }
        }

        /// <summary>
        /// セッションが接続状態にあるか否かを表す値を返します。
        /// </summary>
        /// <param name="name">セッションの対向となる開発機の名前です。</param>
        internal bool IsSessionConnected(string name)
        {
            return this.GetSessions(name).Any(x => x.IsConnected);
        }

        /// <summary>
        /// セッションを開始します。
        /// </summary>
        /// <param name="name">セッションの対向となる開発機の名前です。</param>
        internal void StartSession(string name)
        {
            lock (this.autoConnectionSyncObject)
            {
                this.stoppedSessions.Remove(name);
            }

            foreach (Session session in this.GetSessions(name))
            {
                session.Start();
            }
        }

        /// <summary>
        /// セッションを停止します。
        /// </summary>
        /// <param name="name">セッションの対向となる開発機の名前です。</param>
        internal void StopSession(string name)
        {
            lock (this.autoConnectionSyncObject)
            {
                this.stoppedSessions.Add(name);
            }

            foreach (Session session in this.GetSessions(name))
            {
                session.Stop();
            }
        }

        /// <summary>
        /// ゲームパッドの情報を返します。
        /// </summary>
        internal List<GamePadInfo> GetGamePadInfos()
        {
            return this.gamePadManager.GetGamePadInfos();
        }

        private static Rectangle GetScreenSize()
        {
            return Screen.AllScreens
                .Select(x => x.Bounds)
                .Aggregate((lhs, rhs) => Rectangle.Union(lhs, rhs));
        }

        private void Dispose(bool isDisposing)
        {
            if (!this.isDisposed)
            {
                if (isDisposing)
                {
                    if (this.isStarted)
                    {
                        this.Stop();
                    }

                    this.gamePadManager.Dispose();

                    this.ioPortManager.Dispose();
                }

                this.isDisposed = true;
            }
        }

        private IList<Session> GetSessions()
        {
            lock (this.sessionSyncObject)
            {
                return this.sessions.Values.ToList();
            }
        }

        private IList<Session> GetSessions(string name)
        {
            return this.GetSessions().Where(x => x.Name == name).ToList();
        }

        private void OnTargetConnected(object sender, TargetInfo info)
        {
            this.ioPortManager.AttachPort(info.Name);

            var args = new SessionInfo()
            {
                Name = info.Name,
                IpAddress = info.IpAddress,
                Port = info.Port,
                UserConfig = this.userConfig,
                IoPortManager = this.ioPortManager,
                GamePadManager = this.gamePadManager,
                GetSessionNames = () => this.GetSessionNames(),
            };

            var session = new Session(args);

            lock (this.sessionSyncObject)
            {
                this.sessions[info.Name] = session;
            }
        }

        private void OnTargetDisconnected(object sender, TargetInfo info)
        {
            Session session = null;

            lock (this.sessionSyncObject)
            {
                if (this.sessions.ContainsKey(info.Name))
                {
                    session = this.sessions[info.Name];

                    this.sessions.Remove(info.Name);
                }
            }

            if (session != null)
            {
                session.Dispose();
            }

            this.ioPortManager.DetachPort(info.Name);
        }

        private bool IsCaptureServiceTriggered(KeyboardEventArgs args)
        {
            return args.Key == Keys.Delete &&
                this.keyboardKeyHolder.IsModifierDown(Keys.Control) &&
                this.keyboardKeyHolder.IsModifierDown(Keys.Alt);
        }

        private bool IsCaptureServiceTriggered(
            KeyboardEventArgs args, HotKeyConfig config)
        {
            return config.IsEnabled &&
                args.Key == config.Key &&
                (config.Modifier == Keys.None ||
                 this.keyboardKeyHolder.IsModifierDown(config.Modifier));
        }

        private bool IsCaptureServiceTriggered(
            MouseEventArgs args, BorderCrossingConfig config)
        {
            bool isTriggered = config.IsEnabled;

            isTriggered = isTriggered && new Func<bool>(() =>
            {
                switch (config.Border)
                {
                    case BorderType.Right:
                        return args.X + 1 > ScreenSize.Right - ScreenMargin;

                    case BorderType.Left:
                        return args.X < ScreenSize.Left + ScreenMargin;

                    case BorderType.Top:
                        return args.Y < ScreenSize.Top + ScreenMargin;

                    case BorderType.Bottom:
                        return args.Y + 1 > ScreenSize.Bottom - ScreenMargin;

                    default:
                        return false;
                }
            })();

            isTriggered = isTriggered && new Func<bool>(() =>
            {
                switch (config.Trigger)
                {
                    case BorderCrossingTriggerType.Auto:
                        return !this.mouseButtonHolder.IsAnyButtonDown;

                    case BorderCrossingTriggerType.Click:
                        return args.Button == MouseButtons.Left &&
                               !this.mouseButtonHolder.IsDoubleClicked;

                    case BorderCrossingTriggerType.DoubleClick:
                        return args.Button == MouseButtons.Left &&
                               this.mouseButtonHolder.IsDoubleClicked;

                    default:
                        return false;
                }
            })();

            return isTriggered;
        }

        private ICaptureSession GetCaptureSession()
        {
            lock (this.captureSyncObject)
            {
                return this.captureSession;
            }
        }

        private void StartCapture(
            bool isTriggerdByMouse, BorderCrossingConfig config)
        {
            lock (this.captureSyncObject)
            {
                if (this.captureSession != null)
                {
                    return;
                }

                lock (this.sessionSyncObject)
                {
                    this.captureSession = this.sessions.Values
                        .Select(x => x.GetCaptureSession())
                        .FirstOrDefault(x => x != null);
                }

                if (this.captureSession == null)
                {
                    return;
                }

                this.captureSession.OnCaptureStart(
                    isTriggerdByMouse,
                    config.Border, ScreenSize, Cursor.Position);

                Cursor.Position = new Point(0, 0);

                this.keyboardManager.IsSuppressionEnabled = true;

                this.mouseManager.IsSuppressionEnabled = true;

                this.infoDialog = new InfoDialog(this);
                this.infoDialog.Show();
            }
        }

        private void StopCapture()
        {
            lock (this.captureSyncObject)
            {
                if (this.captureSession == null)
                {
                    return;
                }

                this.infoDialog.Invoke(
                    new MethodInvoker(() => this.infoDialog.Close()));
                this.infoDialog = null;

                Point position = this.captureSession.Position;

                this.keyboardManager.IsSuppressionEnabled = false;

                this.mouseManager.IsSuppressionEnabled = false;

                bool isStoppedByMouse = this.captureSession.IsStoppedByMouse;

                this.captureSession = null;

                if (isStoppedByMouse)
                {
                    switch (this.userConfig.GetBorderCrossingConfig().Border)
                    {
                        case BorderType.Right:
                            position.X = ScreenSize.Right - (ScreenMargin + 1);
                            break;

                        case BorderType.Left:
                            position.X = ScreenSize.Left + ScreenMargin;
                            break;

                        case BorderType.Top:
                            position.Y = ScreenSize.Top + ScreenMargin;
                            break;

                        case BorderType.Bottom:
                            position.Y =
                                ScreenSize.Bottom - (ScreenMargin + 1);
                            break;

                        default:
                            break;
                    }
                }

                Cursor.Position = position;
            }
        }

        private void OnKeyDown(object sender, KeyboardEventArgs args)
        {
            this.keyboardKeyHolder.AcceptKeyDown(args);

            ICaptureSession captureSession = this.GetCaptureSession();

            if (captureSession != null)
            {
                if (captureSession.IsCompleted)
                {
                    this.StopCapture();
                }
                else
                {
                    if (this.IsCaptureServiceTriggered(args) ||
                        this.IsCaptureServiceTriggered(
                            args, this.userConfig.GetHotKeyConfig()))
                    {
                        captureSession.OnCaptureStop();

                        this.StopCapture();
                    }
                    else
                    {
                        captureSession.OnKeyDown(args);
                    }
                }
            }
            else
            {
                if (this.IsCaptureEnabled &&
                    this.IsCaptureServiceTriggered(
                        args, this.userConfig.GetHotKeyConfig()))
                {
                    this.StartCapture(
                        false, this.userConfig.GetBorderCrossingConfig());
                }
            }
        }

        private void OnKeyUp(object sender, KeyboardEventArgs args)
        {
            this.keyboardKeyHolder.AcceptKeyUp(args);

            ICaptureSession captureSession = this.GetCaptureSession();

            if (captureSession != null)
            {
                if (captureSession.IsCompleted)
                {
                    this.StopCapture();
                }
                else
                {
                    captureSession.OnKeyUp(args);
                }
            }
        }

        private void OnMouseMove(object sender, MouseEventArgs args)
        {
            ICaptureSession captureSession = this.GetCaptureSession();

            if (captureSession != null)
            {
                if (captureSession.IsCompleted)
                {
                    this.StopCapture();
                }
                else
                {
                    captureSession.OnMouseMove(args);

                    // 一定時間処理を遅延させて発行数を減らす
                    Thread.Sleep(5);
                }
            }
            else
            {
                BorderCrossingConfig config =
                    this.userConfig.GetBorderCrossingConfig();

                if (this.IsCaptureEnabled &&
                    !this.keyboardKeyHolder.IsAnyKeyDown &&
                    this.IsCaptureServiceTriggered(args, config))
                {
                    this.StartCapture(true, config);
                }
            }
        }

        private void OnMouseDown(object sender, MouseEventArgs args)
        {
            this.mouseButtonHolder.AcceptMouseDown(args);

            ICaptureSession captureSession = this.GetCaptureSession();

            if (captureSession != null)
            {
                if (captureSession.IsCompleted)
                {
                    this.StopCapture();
                }
                else
                {
                    captureSession.OnMouseDown(args);
                }
            }
            else
            {
                BorderCrossingConfig config =
                    this.userConfig.GetBorderCrossingConfig();

                if (this.IsCaptureEnabled &&
                    !this.keyboardKeyHolder.IsAnyKeyDown &&
                    this.IsCaptureServiceTriggered(args, config))
                {
                    this.StartCapture(true, config);
                }
            }
        }

        private void OnMouseUp(object sender, MouseEventArgs args)
        {
            this.mouseButtonHolder.AcceptMouseUp(args);

            ICaptureSession captureSession = this.GetCaptureSession();

            if (captureSession != null)
            {
                if (captureSession.IsCompleted)
                {
                    this.StopCapture();
                }
                else
                {
                    captureSession.OnMouseUp(args);
                }
            }
        }

        private void OnMouseWheel(object sender, MouseEventArgs args)
        {
            ICaptureSession captureSession = this.GetCaptureSession();

            if (captureSession != null)
            {
                if (captureSession.IsCompleted)
                {
                    this.StopCapture();
                }
                else
                {
                    captureSession.OnMouseWheel(args);
                }
            }
        }

        private void Monitor(CancellationToken token)
        {
            while (!token.IsCancellationRequested)
            {
                foreach (Session session in this.GetSessions())
                {
                    if (session.IsTimedOut)
                    {
                        session.Stop();
                    }
                }

                ConnectionConfig config =
                    this.userConfig.GetConnectionConfig();

                if (config.IsAutoConnectionEnabled)
                {
                    foreach (Session session in this.GetSessions())
                    {
                        if (!session.IsConnected)
                        {
                            string name = session.Name;

                            lock (this.autoConnectionSyncObject)
                            {
                                if (this.stoppedSessions.Contains(name))
                                {
                                    continue;
                                }
                            }

                            session.Start();
                        }
                    }
                }

                token.WaitHandle.WaitOne(MonitoringInterval);
            }
        }

        private void StartMonitoring()
        {
            this.tokenSource = new CancellationTokenSource();

            CancellationToken token = this.tokenSource.Token;

            this.task = Task.Run(() => this.Monitor(token));
        }

        private void StopMonitoring()
        {
            this.tokenSource.Cancel();

            this.task.Wait();
            this.task.Dispose();
            this.task = null;

            this.tokenSource.Dispose();
            this.tokenSource = null;
        }

        private class KeyboardKeyHolder
        {
            private readonly ISet<Keys> keys = new HashSet<Keys>();

            private readonly ISet<Keys> modifiers = new HashSet<Keys>();

            internal KeyboardKeyHolder()
            {
            }

            internal bool IsAnyKeyDown
            {
                get
                {
                    return this.keys.Count != 0;
                }
            }

            internal bool IsModifierDown(Keys key)
            {
                return this.modifiers.Contains(key);
            }

            internal void AcceptKeyDown(KeyboardEventArgs args)
            {
                if (IsValidKey(args.Key))
                {
                    this.keys.Add(args.Key);

                    this.UpdateModifiers();

                    if (this.IsScreenTransitionTriggered())
                    {
                        this.keys.Clear();

                        this.modifiers.Clear();
                    }
                }
            }

            internal void AcceptKeyUp(KeyboardEventArgs args)
            {
                this.keys.Remove(args.Key);

                this.UpdateModifiers();
            }

            private static bool IsValidKey(Keys key)
            {
                if ((Keys.D0 <= key && key <= Keys.D9) ||
                    (Keys.A <= key && key <= Keys.Z) ||
                    (Keys.NumPad0 <= key && key <= Keys.NumPad9) ||
                    (Keys.F1 <= key && key <= Keys.F24))
                {
                    return true;
                }

                switch (key)
                {
                    case Keys.Cancel:
                    case Keys.Back:
                    case Keys.Tab:
                    case Keys.LineFeed:
                    case Keys.Clear:
                    case Keys.Enter:
                    case Keys.Pause:
                    case Keys.CapsLock:
                    case Keys.Escape:
                    case Keys.Space:
                    case Keys.PageUp:
                    case Keys.PageDown:
                    case Keys.End:
                    case Keys.Home:
                    case Keys.Left:
                    case Keys.Up:
                    case Keys.Right:
                    case Keys.Down:
                    case Keys.Select:
                    case Keys.Print:
                    case Keys.Execute:
                    case Keys.PrintScreen:
                    case Keys.Insert:
                    case Keys.Delete:
                    case Keys.Help:
                    case Keys.LWin:
                    case Keys.RWin:
                    case Keys.Apps:
                    case Keys.Sleep:
                    case Keys.Multiply:
                    case Keys.Add:
                    case Keys.Separator:
                    case Keys.Subtract:
                    case Keys.Decimal:
                    case Keys.Divide:
                    case Keys.NumLock:
                    case Keys.Scroll:
                    case Keys.LShiftKey:
                    case Keys.RShiftKey:
                    case Keys.LControlKey:
                    case Keys.RControlKey:
                    case Keys.LMenu:
                    case Keys.RMenu:
                    case Keys.Oem1:
                    case Keys.Oemplus:
                    case Keys.Oemcomma:
                    case Keys.OemMinus:
                    case Keys.OemPeriod:
                    case Keys.Oem2:
                    case Keys.Oem3:
                    case Keys.Oem4:
                    case Keys.Oem5:
                    case Keys.Oem6:
                    case Keys.Oem7:
                    case Keys.Oem8:
                    case Keys.Oem102:
                    case Keys.ProcessKey:
                    case Keys.Attn:
                    case Keys.Crsel:
                    case Keys.Exsel:
                    case Keys.EraseEof:
                    case Keys.Play:
                    case Keys.Zoom:
                    case Keys.Pa1:
                    case Keys.OemClear:
                        return true;
                }

                return false;
            }

            private bool IsScreenTransitionTriggered()
            {
                if (this.keys.Contains(Keys.L))
                {
                    if (this.keys.Contains(Keys.LWin) ||
                        this.keys.Contains(Keys.RWin))
                    {
                        return true;
                    }
                }

                if (this.modifiers.Contains(Keys.Control) &&
                    this.modifiers.Contains(Keys.Alt))
                {
                    if (this.keys.Contains(Keys.Delete) ||
                        this.keys.Contains(Keys.End))
                    {
                        return true;
                    }
                }

                return false;
            }

            private void UpdateModifiers()
            {
                this.modifiers.Clear();

                if (this.keys.Contains(Keys.ControlKey) ||
                    this.keys.Contains(Keys.LControlKey) ||
                    this.keys.Contains(Keys.RControlKey))
                {
                    this.modifiers.Add(Keys.Control);
                }

                if (this.keys.Contains(Keys.ShiftKey) ||
                    this.keys.Contains(Keys.LShiftKey) ||
                    this.keys.Contains(Keys.RShiftKey))
                {
                    this.modifiers.Add(Keys.Shift);
                }

                if (this.keys.Contains(Keys.Menu) ||
                    this.keys.Contains(Keys.LMenu) ||
                    this.keys.Contains(Keys.RMenu))
                {
                    this.modifiers.Add(Keys.Alt);
                }
            }
        }

        private class MouseButtonHolder
        {
            private MouseButtons buttons;

            private bool isDoubleClicked;

            private MouseButtons lastButton;

            private Rectangle rectangle;

            private Stopwatch stopwatch = new Stopwatch();

            internal MouseButtonHolder()
            {
                this.Reset();
            }

            internal bool IsAnyButtonDown
            {
                get
                {
                    return (this.buttons != MouseButtons.None);
                }
            }

            internal bool IsDoubleClicked
            {
                get
                {
                    return this.isDoubleClicked;
                }
            }

            internal void Reset()
            {
                this.buttons = MouseButtons.None;

                this.isDoubleClicked = false;

                this.lastButton = MouseButtons.None;

                this.rectangle = new Rectangle();

                if (this.stopwatch.IsRunning)
                {
                    this.stopwatch.Reset();
                }
            }

            internal void AcceptMouseDown(MouseEventArgs args)
            {
                this.buttons |= args.Button;

                int milliseconds = SystemInformation.DoubleClickTime;

                if (this.lastButton == args.Button &&
                    this.rectangle.Contains(args.X, args.Y) &&
                    this.stopwatch.ElapsedMilliseconds < milliseconds)
                {
                    this.isDoubleClicked = true;

                    this.lastButton = MouseButtons.None;

                    this.stopwatch.Reset();
                }
                else
                {
                    this.isDoubleClicked = false;

                    this.lastButton = args.Button;

                    Size size = SystemInformation.DoubleClickSize;

                    this.rectangle = new Rectangle(
                        args.X - size.Width / 2,
                        args.Y - size.Height / 2,
                        size.Width,
                        size.Height);

                    if (this.stopwatch.IsRunning)
                    {
                        this.stopwatch.Restart();
                    }
                    else
                    {
                        this.stopwatch.Start();
                    }
                }
            }

            internal void AcceptMouseUp(MouseEventArgs args)
            {
                this.buttons &= ~args.Button;

                this.isDoubleClicked = false;
            }
        }
    }
}
