﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace HidShell
{
    using System;
    using Properties;

    /// <summary>
    /// 内部エラーが発生した場合にスローされる例外です。
    /// </summary>
    internal sealed class HidShellException : Exception
    {
        /// <summary>
        /// 指定したエラーメッセージを利用して、HidShellException クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="message">例外の原因を説明するエラーメッセージです。</param>
        public HidShellException(string message)
            : base(message)
        {
        }

        /// <summary>
        /// 指定したエラーメッセージと、例外の原因である内部例外への参照を利用して、HidShellException クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="message">例外の原因を説明するエラーメッセージです。</param>
        /// <param name="innerException">例外の原因である内部例外への参照です。</param>
        public HidShellException(string message, Exception innerException)
            : base(message, innerException)
        {
        }

        /// <summary>
        /// 指定した HidShell の結果コードを利用して、HidShellException クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="result">HidShell の結果コードです。</param>
        internal HidShellException(HidShellResult result)
            : base(GetHidShellResultMessage(result))
        {
        }

        private static string GetHidShellResultMessage(HidShellResult result)
        {
            switch (result)
            {
                case HidShellResult.Success:
                    return Resources.ResultSuccess;
                case HidShellResult.NullPortPointer:
                    return Resources.ResultNullPortPointer;
                case HidShellResult.InvalidPort:
                    return Resources.ResultInvalidPort;
                case HidShellResult.PortNotFound:
                    return Resources.ResultPortNotFound;
                case HidShellResult.NoAvailablePort:
                    return Resources.ResultNoAvailablePort;
                case HidShellResult.NullPortListPointer:
                    return Resources.ResultNullPortListPointer;
                case HidShellResult.InvalidPortDirection:
                    return Resources.ResultInvalidPortDirection;
                case HidShellResult.NullHandlePointer:
                    return Resources.ResultNullHandlePointer;
                case HidShellResult.InvalidHandle:
                    return Resources.ResultInvalidHandle;
                case HidShellResult.InvalidPortName:
                    return Resources.ResultInvalidPortName;
                case HidShellResult.NullStatePointer:
                    return Resources.ResultNullStatePointer;
                case HidShellResult.StateNotSet:
                    return Resources.ResultStateNotSet;
                case HidShellResult.FailedToCreateMutex:
                    return Resources.ResultFailedToCreateMutex;
                case HidShellResult.FailedToCreateSharedMemory:
                    return Resources.ResultFailedToCreateSharedMemory;
                case HidShellResult.NullAddressPointer:
                    return Resources.ResultNullAddressPointer;
                case HidShellResult.NullExistenceFlagPointer:
                    return Resources.ResultNullExistenceFlagPointer;
                case HidShellResult.InvalidCaptureButtonState:
                    return Resources.ResultInvalidCaptureButtonState;
                case HidShellResult.InvalidDebugPadState:
                    return Resources.ResultInvalidDebugPadState;
                case HidShellResult.InvalidHomeButtonState:
                    return Resources.ResultInvalidHomeButtonState;
                case HidShellResult.InvalidTouchScreenState:
                    return Resources.ResultInvalidTouchScreenState;
                case HidShellResult.InvalidBasicXpadState:
                    return Resources.ResultInvalidBasicXpadState;
                case HidShellResult.InvalidBasicXpadId:
                    return Resources.ResultInvalidBasicXpadId;
                default:
                    return Resources.ResultUnexpectedError;
            }
        }
    }
}
