﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace HidShell
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Threading;
    using System.Threading.Tasks;
    using Properties;

    /// <summary>
    /// AbstractedPad を扱うクラスです。
    /// </summary>
    internal sealed class AbstractedPad
    {
        private const int DeviceCountMax = 8;

        private static readonly object SyncObject = new object();

        private static readonly TimeSpan Interval;

        private static readonly Dictionary<uint, EventDumper> EventDumpers;

        private readonly List<string> deviceIds = new List<string>();

        private readonly List<EventSender> eventSenders;

        private uint port = PortAccessor.DefaultPort;

        private Logger logger = null;

        private string deviceType = "AbstractedPad";

        private AbstractedPadDeviceTypes deviceTypes;

        /// <summary>
        /// AbstractedPad クラスの静的メンバを初期化します。
        /// </summary>
        static AbstractedPad()
        {
            Interval = TimeSpan.FromMilliseconds(5);

            EventDumpers = new Dictionary<uint, EventDumper>();
        }

        /// <summary>
        /// AbstractedPad クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal AbstractedPad()
        {
            this.eventSenders = new List<EventSender>();

            foreach (int i in Enumerable.Range(0, DeviceCountMax))
            {
                string deviceId = i.ToString();

                this.deviceIds.Add(deviceId);

                this.eventSenders.AddRange(new[]
                {
                    new EventSender(
                        deviceId, EventType.PowerState,
                        GetAbstractedPadPowerStateEventSender(i)),
                    new EventSender(
                        deviceId, EventType.BatteryLevel,
                        GetAbstractedPadBatteryLevelEventSender(i)),
                    new EventSender(
                        deviceId, EventType.ButtonDown,
                        GetAbstractedPadButtonDownEventSender(i)),
                    new EventSender(
                        deviceId, EventType.ButtonUp,
                        GetAbstractedPadButtonUpEventSender(i)),
                    new EventSender(
                        deviceId, EventType.StickMove,
                        GetAbstractedPadStickMoveEventSender(i)),
                });
            }
        }

        private enum AbstractedPadDeviceType : byte
        {
            ProController = 0,
            Handheld = 1,
            HandheldJoyConL = 2,
            HandheldJoyConR = 3,
            JoyConL = 4,
            JoyConR = 5,
            Unknown = 31,
        }

        [Flags]
        private enum AbstractedPadDeviceTypes : uint
        {
            None = 0,
            ProController = 1u << AbstractedPadDeviceType.ProController,
            Handheld = 1u << AbstractedPadDeviceType.Handheld,
            HandheldJoyConL = 1u << AbstractedPadDeviceType.HandheldJoyConL,
            HandheldJoyConR = 1u << AbstractedPadDeviceType.HandheldJoyConR,
            JoyConL = 1u << AbstractedPadDeviceType.JoyConL,
            JoyConR = 1u << AbstractedPadDeviceType.JoyConR,
            Unknown = 1u << AbstractedPadDeviceType.Unknown,
        }

        [Flags]
        private enum AbstractedPadAttributes : uint
        {
            None = 0,
            IsConnected = 1u,
            IsSixAxisSensorEnabled = 1u << 1,
        }

        private enum AbstractedPadPowerState : byte
        {
            Disconnected,
            OnBattery,
            NoBattery,
            Charging,
            Charged,
        }

        private enum AbstractedPadInterfaceType : byte
        {
            Unknown,
            Bluetooth,
            Rail,
            Usb,
        }

        private enum AbstractedPadBatteryLevel : byte
        {
            None,
            CriticalLow,
            Low,
            Medium,
            High,
        }

        [Flags]
        private enum AbstractedPadButtons : uint
        {
            None = 0,
            A = 1u,
            B = 1u << 1,
            X = 1u << 2,
            Y = 1u << 3,
            StickL = 1u << 4,
            StickR = 1u << 5,
            L = 1u << 6,
            R = 1u << 7,
            ZL = 1u << 8,
            ZR = 1u << 9,
            Start = 1u << 10,
            Select = 1u << 11,
            Left = 1u << 12,
            Up = 1u << 13,
            Right = 1u << 14,
            Down = 1u << 15,
            SL = 1u << 16,
            SR = 1u << 17,
        }

        /// <summary>
        /// ポートを設定します。
        /// </summary>
        /// <param name="port">ポートです。</param>
        internal void SetPort(uint port)
        {
            this.port = port;
        }

        /// <summary>
        /// ロガーを設定します。
        /// </summary>
        /// <param name="logger">ロガーです。</param>
        internal void SetLogger(Logger logger)
        {
            this.logger = logger;
        }

        /// <summary>
        /// デバイス種別を設定します。
        /// </summary>
        /// <param name="deviceType">デバイス種別です。</param>
        internal void SetDeviceType(string deviceType)
        {
            this.deviceType = deviceType;
        }

        /// <summary>
        /// 操作モードを JoyConL に設定します。
        /// </summary>
        internal void SetJoyConLMode()
        {
            this.deviceTypes =
                AbstractedPadDeviceTypes.HandheldJoyConL |
                AbstractedPadDeviceTypes.JoyConL;
        }

        /// <summary>
        /// 操作モードを JoyConR に設定します。
        /// </summary>
        internal void SetJoyConRMode()
        {
            this.deviceTypes =
                AbstractedPadDeviceTypes.HandheldJoyConR |
                AbstractedPadDeviceTypes.JoyConR;
        }

        /// <summary>
        /// 操作モードを ProController に設定します。
        /// </summary>
        internal void SetProControllerMode()
        {
            this.deviceTypes = AbstractedPadDeviceTypes.ProController;
        }

        /// <summary>
        /// デバイス識別子で指定されたデバイスについてイベントをダンプします。
        /// </summary>
        /// <param name="deviceId">デバイス識別子です。</param>
        internal void DumpEvent(string deviceId)
        {
            if (deviceId != null && !this.deviceIds.Contains(deviceId))
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorDeviceIdUnexpected, deviceId));
            }

            EventDumper dumper = null;

            lock (SyncObject)
            {
                if (!EventDumpers.TryGetValue(this.port, out dumper))
                {
                    dumper = new EventDumper(this.port);

                    EventDumpers[this.port] = dumper;
                }
            }

            dumper.Invoke(this, deviceId);
        }

        /// <summary>
        /// デバイス識別子で指定されたデバイスに対してイベントを発行します。
        /// </summary>
        /// <param name="deviceId">デバイス識別子です。</param>
        /// <param name="eventType">イベント種別です。</param>
        /// <param name="eventArgs">イベント引数です。</param>
        internal void SendEvent(
            string deviceId, string eventType, string[] eventArgs)
        {
            var senders =
                this.eventSenders.Where(x => x.SupportsDeviceId(deviceId));

            if (senders.Count() == 0)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorDeviceIdUnexpected, deviceId));
            }

            var sender =
                senders.FirstOrDefault(x => x.SupportsEventType(eventType));

            if (sender == null)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorEventTypeUnexpected, eventType));
            }

            sender.Invoke(this, eventArgs);
        }

        private static string EncodeCoordinate(int value)
        {
            if (value >= 0)
            {
                return string.Format("P{0:D5}", value);
            }
            else
            {
                return string.Format("M{0:D5}", -value);
            }
        }

        private static string DecodeCoordinate(string value)
        {
            if (value.StartsWith("P"))
            {
                return value.Substring(1);
            }

            if (value.StartsWith("M"))
            {
                return "-" + value.Substring(1);
            }

            return value;
        }

        private static void GetAbstractedPadState(
            IntPtr handle, out AbstractedPadState outState, uint id, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellAbstractedPadState(
                    handle, out outState, id, port, PortAccessor.Out);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static void GetAbstractedPadAutoPilotState(
            IntPtr handle, out AbstractedPadState outState, uint id, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellAbstractedPadState(
                    handle, out outState, id, port, PortAccessor.In);


            switch (result)
            {
                case HidShellResult.Success:
                    return;

                case HidShellResult.StateNotSet:
                    outState = new AbstractedPadState();
                    return;

                default:
                    throw new HidShellException(result);
            }
        }

        private static void SetAbstractedPadAutoPilotState(
            IntPtr handle, ref AbstractedPadState state, uint id, uint port)
        {
            var result = (HidShellResult)
                Native.SetHidShellAbstractedPadState(
                    handle, ref state, id, port, PortAccessor.In);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static void GetProControllerDefaultColor(
            ref AbstractedPadColor color)
        {
            color.Main = new byte[4];
            color.Main[0] = 45;
            color.Main[1] = 45;
            color.Main[2] = 45;
            color.Main[3] = 255;

            color.Sub = new byte[4];
            color.Sub[0] = 230;
            color.Sub[1] = 230;
            color.Sub[2] = 230;
            color.Sub[3] = 255;
        }

        private static void GetJoyConDefaultColor(
            ref AbstractedPadColor color)
        {
            color.Main = new byte[4];
            color.Main[0] = 130;
            color.Main[1] = 130;
            color.Main[2] = 130;
            color.Main[3] = 255;

            color.Sub = new byte[4];
            color.Sub[0] = 15;
            color.Sub[1] = 15;
            color.Sub[2] = 15;
            color.Sub[3] = 255;
        }

        private static AbstractedPadDeviceTypes GetAbstractedPadDeviceType(
            ref AbstractedPadState state)
        {
            if ((byte)AbstractedPadDeviceType.Unknown < (byte)state.DeviceType)
            {
                return AbstractedPadDeviceTypes.Unknown;
            }
            else
            {
                return (AbstractedPadDeviceTypes)(1 << (byte)state.DeviceType);
            }
        }

        private static void SetAbstractedPadDeviceType(
            ref AbstractedPadState state, AbstractedPadDeviceTypes deviceTypes)
        {
            state.DeviceType = AbstractedPadDeviceType.Unknown;

            if (deviceTypes.HasFlag(AbstractedPadDeviceTypes.ProController))
            {
                state.DeviceType = AbstractedPadDeviceType.ProController;

                state.Attributes =
                    AbstractedPadAttributes.IsConnected;

                GetProControllerDefaultColor(ref state.Color);

                if (state.PowerInfo.IsPowered == 0)
                {
                    state.InterfaceType = AbstractedPadInterfaceType.Bluetooth;

                    state.PowerInfo.IsCharging = 0;

                    if (state.PowerInfo.BatteryLevel ==
                        AbstractedPadBatteryLevel.None)
                    {
                        state.PowerInfo.BatteryLevel =
                            AbstractedPadBatteryLevel.High;
                    }
                }
                else
                {
                    state.InterfaceType = AbstractedPadInterfaceType.Usb;

                    if (state.PowerInfo.IsCharging == 0 ||
                        state.PowerInfo.BatteryLevel ==
                        AbstractedPadBatteryLevel.None)
                    {
                        state.PowerInfo.BatteryLevel =
                            AbstractedPadBatteryLevel.High;
                    }
                }

                return;
            }

            if (deviceTypes.HasFlag(AbstractedPadDeviceTypes.JoyConL) ||
                deviceTypes.HasFlag(AbstractedPadDeviceTypes.JoyConR))
            {
                state.Attributes =
                    AbstractedPadAttributes.IsConnected;

                GetJoyConDefaultColor(ref state.Color);

                if (state.PowerInfo.IsPowered == 0)
                {
                    state.DeviceType =
                        deviceTypes.HasFlag(AbstractedPadDeviceTypes.JoyConL)
                            ? AbstractedPadDeviceType.JoyConL
                            : AbstractedPadDeviceType.JoyConR;

                    state.InterfaceType = AbstractedPadInterfaceType.Bluetooth;

                    state.PowerInfo.IsCharging = 0;

                    if (state.PowerInfo.BatteryLevel ==
                        AbstractedPadBatteryLevel.None)
                    {
                        state.PowerInfo.BatteryLevel =
                            AbstractedPadBatteryLevel.High;
                    }
                }
                else
                {
                    state.DeviceType =
                        deviceTypes.HasFlag(AbstractedPadDeviceTypes.JoyConL)
                            ? AbstractedPadDeviceType.HandheldJoyConL
                            : AbstractedPadDeviceType.HandheldJoyConR;

                    state.InterfaceType = AbstractedPadInterfaceType.Rail;

                    if (state.PowerInfo.IsCharging == 0 ||
                        state.PowerInfo.BatteryLevel ==
                        AbstractedPadBatteryLevel.None)
                    {
                        state.PowerInfo.BatteryLevel =
                            AbstractedPadBatteryLevel.High;
                    }
                }

                return;
            }
        }

        private static AbstractedPadPowerState GetAbstractedPadPowerState(
            ref AbstractedPadState state)
        {
            if (!state.Attributes.HasFlag(AbstractedPadAttributes.IsConnected))
            {
                return AbstractedPadPowerState.Disconnected;
            }

            if (state.PowerInfo.IsCharging != 0)
            {
                return AbstractedPadPowerState.Charging;
            }

            if (state.PowerInfo.BatteryLevel == AbstractedPadBatteryLevel.None)
            {
                return AbstractedPadPowerState.NoBattery;
            }
            else
            {
                if (state.PowerInfo.IsPowered != 0)
                {
                    return AbstractedPadPowerState.Charged;
                }
                else
                {
                    return AbstractedPadPowerState.OnBattery;
                }
            }
        }

        private static void SetAbstractedPadWireless(
            ref AbstractedPadState state)
        {
            state.PowerInfo.IsPowered = 0;

            state.PowerInfo.IsCharging = 0;

            switch (state.DeviceType)
            {
                case AbstractedPadDeviceType.ProController:
                    state.InterfaceType = AbstractedPadInterfaceType.Bluetooth;
                    break;

                case AbstractedPadDeviceType.HandheldJoyConL:
                    state.DeviceType = AbstractedPadDeviceType.JoyConL;
                    state.InterfaceType = AbstractedPadInterfaceType.Bluetooth;
                    break;

                case AbstractedPadDeviceType.HandheldJoyConR:
                    state.DeviceType = AbstractedPadDeviceType.JoyConR;
                    state.InterfaceType = AbstractedPadInterfaceType.Bluetooth;
                    break;
            }
        }

        private static void SetAbstractedPadWired(ref AbstractedPadState state)
        {
            state.PowerInfo.IsPowered = 1;

            switch (state.DeviceType)
            {
                case AbstractedPadDeviceType.ProController:
                    state.InterfaceType = AbstractedPadInterfaceType.Usb;
                    break;

                case AbstractedPadDeviceType.JoyConL:
                    state.DeviceType = AbstractedPadDeviceType.HandheldJoyConL;
                    state.InterfaceType = AbstractedPadInterfaceType.Rail;
                    break;

                case AbstractedPadDeviceType.JoyConR:
                    state.DeviceType = AbstractedPadDeviceType.HandheldJoyConR;
                    state.InterfaceType = AbstractedPadInterfaceType.Rail;
                    break;
            }
        }

        private static void SetAbstractedPadPowerState(
            ref AbstractedPadState state, AbstractedPadPowerState powerState)
        {
            switch (powerState)
            {
                case AbstractedPadPowerState.Disconnected:
                    state = new AbstractedPadState();
                    return;

                case AbstractedPadPowerState.OnBattery:
                    SetAbstractedPadWireless(ref state);
                    return;

                case AbstractedPadPowerState.NoBattery:
                    SetAbstractedPadWired(ref state);
                    state.PowerInfo.IsCharging = 0;
                    state.PowerInfo.BatteryLevel =
                        AbstractedPadBatteryLevel.None;
                    return;

                case AbstractedPadPowerState.Charging:
                    SetAbstractedPadWired(ref state);
                    state.PowerInfo.IsCharging = 1;
                    return;

                case AbstractedPadPowerState.Charged:
                    SetAbstractedPadWired(ref state);
                    state.PowerInfo.IsCharging = 0;
                    state.PowerInfo.BatteryLevel =
                        AbstractedPadBatteryLevel.High;
                    return;
            }
        }

        private static void SetAbstractedPadBatteryLevel(
            ref AbstractedPadState state, AbstractedPadBatteryLevel level)
        {
            switch (level)
            {
                case AbstractedPadBatteryLevel.None:
                    SetAbstractedPadWired(ref state);
                    state.PowerInfo.IsCharging = 0;
                    break;

                case AbstractedPadBatteryLevel.CriticalLow:
                case AbstractedPadBatteryLevel.Low:
                case AbstractedPadBatteryLevel.Medium:
                    if (state.PowerInfo.IsPowered != 0)
                    {
                        state.PowerInfo.IsCharging = 1;
                    }

                    break;
            }

            state.PowerInfo.BatteryLevel = level;
        }

        private static bool IsDeviceTypeChanged(
            ref AbstractedPadState nextState, ref AbstractedPadState lastState)
        {
            var nextDeviceType = nextState.DeviceType;

            var lastDeviceType = lastState.DeviceType;

            if (nextDeviceType == lastDeviceType)
            {
                return false;
            }

            if (nextDeviceType == AbstractedPadDeviceType.HandheldJoyConL &&
                lastDeviceType == AbstractedPadDeviceType.JoyConL)
            {
                return false;
            }

            if (nextDeviceType == AbstractedPadDeviceType.JoyConL &&
                lastDeviceType == AbstractedPadDeviceType.HandheldJoyConL)
            {
                return false;
            }

            if (nextDeviceType == AbstractedPadDeviceType.HandheldJoyConR &&
                lastDeviceType == AbstractedPadDeviceType.JoyConR)
            {
                return false;
            }

            if (nextDeviceType == AbstractedPadDeviceType.JoyConR &&
                lastDeviceType == AbstractedPadDeviceType.HandheldJoyConR)
            {
                return false;
            }

            return true;
        }

        private static void DumpAbstractedPadEvent(
            uint port, EventDumper dumper)
        {
            var stopwatch = new Stopwatch();

            var nextStates = new AbstractedPadState[DeviceCountMax];
            var lastStates = new AbstractedPadState[DeviceCountMax];

            using (var accessor = new PortAccessor())
            {
                for (int i = 0; i < lastStates.Length; ++i)
                {
                    GetAbstractedPadState(
                        accessor.Handle, out lastStates[i], (uint)i, port);
                }
            }

            while (true)
            {
                stopwatch.Start();

                using (var accessor = new PortAccessor())
                {
                    for (int i = 0; i < lastStates.Length; ++i)
                    {
                        GetAbstractedPadState(
                            accessor.Handle, out nextStates[i], (uint)i, port);
                    }
                }

                foreach (int i in Enumerable.Range(0, DeviceCountMax))
                {
                    var id = i.ToString();

                    DumpAbstractedPadPowerEvent(
                        dumper, id, ref nextStates[i], ref lastStates[i]);

                    DumpAbstractedPadBatteryEvent(
                        dumper, id, ref nextStates[i], ref lastStates[i]);

                    DumpAbstractedPadButtonEvent(
                        dumper, id, ref nextStates[i], ref lastStates[i]);

                    DumpAbstractedPadStickEvent(
                        dumper, id, ref nextStates[i], ref lastStates[i]);
                }

                nextStates.CopyTo(lastStates, 0);

                stopwatch.Stop();

                if (stopwatch.Elapsed < Interval)
                {
                    Thread.Sleep(Interval - stopwatch.Elapsed);
                }

                stopwatch.Reset();
            }
        }

        private static void DumpAbstractedPadPowerEvent(
            EventDumper dumper, string deviceId,
            ref AbstractedPadState nextState, ref AbstractedPadState lastState)
        {
            var lastPowerState = GetAbstractedPadPowerState(ref lastState);

            var nextPowerState = GetAbstractedPadPowerState(ref nextState);


            if (lastPowerState == AbstractedPadPowerState.Disconnected)
            {
                if (nextPowerState != AbstractedPadPowerState.Disconnected)
                {
                    dumper.WriteLine(
                        "{0,-16}{1,-16}{2} {3}",
                        GetAbstractedPadDeviceType(ref nextState), deviceId,
                        EventType.PowerState, nextPowerState);
                }
            }
            else
            {
                if (nextPowerState == AbstractedPadPowerState.Disconnected)
                {
                    dumper.WriteLine(
                        "{0,-16}{1,-16}{2} {3}",
                        GetAbstractedPadDeviceType(ref lastState), deviceId,
                        EventType.PowerState, nextPowerState);
                }
                else
                {
                    if (IsDeviceTypeChanged(ref nextState, ref lastState))
                    {
                        dumper.WriteLine(
                            "{0,-16}{1,-16}{2} {3}",
                            GetAbstractedPadDeviceType(ref lastState),
                            deviceId, EventType.PowerState,
                            AbstractedPadPowerState.Disconnected);

                        dumper.WriteLine(
                            "{0,-16}{1,-16}{2} {3}",
                            GetAbstractedPadDeviceType(ref nextState),
                            deviceId, EventType.PowerState, nextPowerState);
                    }
                    else
                    {
                        if (lastPowerState != nextPowerState)
                        {
                            dumper.WriteLine(
                                "{0,-16}{1,-16}{2} {3}",
                                GetAbstractedPadDeviceType(ref nextState),
                                deviceId,
                                EventType.PowerState, nextPowerState);
                        }
                    }
                }
            }
        }

        private static void DumpAbstractedPadBatteryEvent(
            EventDumper dumper, string deviceId,
            ref AbstractedPadState nextState, ref AbstractedPadState lastState)
        {
            var lastPowerState = GetAbstractedPadPowerState(ref lastState);

            var nextPowerState = GetAbstractedPadPowerState(ref nextState);

            if (nextPowerState == AbstractedPadPowerState.Disconnected ||
                nextPowerState == AbstractedPadPowerState.NoBattery)
            {
                return;
            }

            if (lastPowerState == AbstractedPadPowerState.Disconnected ||
                IsDeviceTypeChanged(ref nextState, ref lastState) ||
                lastState.PowerInfo.BatteryLevel !=
                nextState.PowerInfo.BatteryLevel)
            {
                dumper.WriteLine(
                    "{0,-16}{1,-16}{2} {3}",
                    GetAbstractedPadDeviceType(ref nextState), deviceId,
                    EventType.BatteryLevel, nextState.PowerInfo.BatteryLevel);
            }
        }

        private static void DumpAbstractedPadButtonEvent(
            EventDumper dumper, string deviceId,
            ref AbstractedPadState nextState, ref AbstractedPadState lastState)
        {
            var buttons = Enum.GetValues(typeof(AbstractedPadButtons))
                .Cast<AbstractedPadButtons>();

            foreach (var button in buttons)
            {
                if ((button & lastState.Buttons) == 0 &&
                    (button & nextState.Buttons) != 0)
                {
                    dumper.WriteLine(
                        "{0,-16}{1,-16}{2} {3}",
                        GetAbstractedPadDeviceType(ref nextState), deviceId,
                        EventType.ButtonDown, button);
                }

                if ((button & lastState.Buttons) != 0 &&
                    (button & nextState.Buttons) == 0)
                {
                    dumper.WriteLine(
                        "{0,-16}{1,-16}{2} {3}",
                        GetAbstractedPadDeviceType(ref lastState), deviceId,
                        EventType.ButtonUp, button);
                }
            }
        }

        private static void DumpAbstractedPadStickEvent(
            EventDumper dumper, string deviceId,
            ref AbstractedPadState nextState, ref AbstractedPadState lastState)
        {
            if (nextState.StickL.X != lastState.StickL.X ||
                nextState.StickL.Y != lastState.StickL.Y)
            {
                dumper.WriteLine(
                    "{0,-16}{1,-16}{2} {3} {4} {5}",
                    GetAbstractedPadDeviceType(ref nextState), deviceId,
                    EventType.StickMove, StickType.Left,
                    EncodeCoordinate(nextState.StickL.X),
                    EncodeCoordinate(nextState.StickL.Y));
            }

            if (nextState.StickR.X != lastState.StickR.X ||
                nextState.StickR.Y != lastState.StickR.Y)
            {
                dumper.WriteLine(
                    "{0,-16}{1,-16}{2} {3} {4} {5}",
                    GetAbstractedPadDeviceType(ref nextState), deviceId,
                    EventType.StickMove, StickType.Right,
                    EncodeCoordinate(nextState.StickR.X),
                    EncodeCoordinate(nextState.StickR.Y));
            }
        }

        private void WriteLine(
            string format, AbstractedPadDeviceTypes deviceType,
            string deviceId, params object[] args)
        {
            if (this.deviceTypes.HasFlag(deviceType))
            {
                var arguments = new List<object>();

                arguments.Add(this.deviceType);

                arguments.Add(deviceId);

                arguments.AddRange(args);

                this.logger.WriteLine(format, arguments.ToArray());
            }
        }

        private static Action<AbstractedPad, string[]>
            GetAbstractedPadPowerStateEventSender(int index)
        {
            string[] powerStateNames =
                Enum.GetNames(typeof(AbstractedPadPowerState));

            var id = (uint)index;

            var state = new AbstractedPadState();

            return (AbstractedPad that, string[] args) =>
            {
                if (args.Count() != 1)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.PowerState));
                }

                var powerState = Array.IndexOf(powerStateNames, args[0]);

                if (powerState < 0)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorPowerUnexpectedState, args[0]));
                }

                using (var accessor = new PortAccessor())
                {
                    GetAbstractedPadAutoPilotState(
                        accessor.Handle, out state, id, that.port);

                    SetAbstractedPadDeviceType(ref state, that.deviceTypes);

                    if (state.Attributes.HasFlag(
                            AbstractedPadAttributes.IsConnected))
                    {
                        SetAbstractedPadPowerState(
                            ref state, (AbstractedPadPowerState)powerState);

                        SetAbstractedPadAutoPilotState(
                            accessor.Handle, ref state, id, that.port);
                    }
                }
            };
        }

        private static Action<AbstractedPad, string[]>
            GetAbstractedPadBatteryLevelEventSender(int index)
        {
            string[] levelNames =
                Enum.GetNames(typeof(AbstractedPadBatteryLevel));

            var id = (uint)index;

            var state = new AbstractedPadState();

            return (AbstractedPad that, string[] args) =>
            {
                if (args.Count() != 1)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.BatteryLevel));
                }

                var batteryLevel = Array.IndexOf(levelNames, args[0]);

                if (batteryLevel < 0)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorBatteryUnexpectedLevel, args[0]));
                }

                using (var accessor = new PortAccessor())
                {
                    GetAbstractedPadAutoPilotState(
                        accessor.Handle, out state, id, that.port);

                    SetAbstractedPadDeviceType(ref state, that.deviceTypes);

                    if (state.Attributes.HasFlag(
                            AbstractedPadAttributes.IsConnected))
                    {
                        SetAbstractedPadBatteryLevel(
                            ref state,
                            (AbstractedPadBatteryLevel)batteryLevel);

                        SetAbstractedPadAutoPilotState(
                            accessor.Handle, ref state, id, that.port);
                    }
                }
            };
        }

        private static Action<AbstractedPad, string[]>
            GetAbstractedPadButtonDownEventSender(int index)
        {
            string[] buttonNames = Enum.GetNames(typeof(AbstractedPadButtons));

            var buttons = Enum.GetValues(typeof(AbstractedPadButtons))
                .Cast<AbstractedPadButtons>().ToArray();

            var id = (uint)index;

            var state = new AbstractedPadState();

            return (AbstractedPad that, string[] args) =>
            {
                if (args.Count() != 1)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.ButtonDown));
                }

                var button = Array.IndexOf(buttonNames, args[0]);

                if (button <= 0)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorButtonUnexpectedName, args[0]));
                }

                using (var accessor = new PortAccessor())
                {
                    GetAbstractedPadAutoPilotState(
                        accessor.Handle, out state, id, that.port);

                    SetAbstractedPadDeviceType(ref state, that.deviceTypes);

                    if (state.Attributes.HasFlag(
                            AbstractedPadAttributes.IsConnected))
                    {
                        state.Buttons |= buttons[button];

                        SetAbstractedPadAutoPilotState(
                            accessor.Handle, ref state, id, that.port);
                    }
                }
            };
        }

        private static Action<AbstractedPad, string[]>
            GetAbstractedPadButtonUpEventSender(int index)
        {
            string[] buttonNames = Enum.GetNames(typeof(AbstractedPadButtons));

            var buttons = Enum.GetValues(typeof(AbstractedPadButtons))
                .Cast<AbstractedPadButtons>().ToArray();

            var id = (uint)index;

            var state = new AbstractedPadState();

            return (AbstractedPad that, string[] args) =>
            {
                if (args.Count() != 1)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.ButtonUp));
                }

                var button = Array.IndexOf(buttonNames, args[0]);

                if (button <= 0)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorButtonUnexpectedName, args[0]));
                }

                using (var accessor = new PortAccessor())
                {
                    GetAbstractedPadAutoPilotState(
                        accessor.Handle, out state, id, that.port);

                    SetAbstractedPadDeviceType(ref state, that.deviceTypes);

                    if (state.Attributes.HasFlag(
                            AbstractedPadAttributes.IsConnected))
                    {
                        state.Buttons &= ~buttons[button];

                        SetAbstractedPadAutoPilotState(
                            accessor.Handle, ref state, id, that.port);
                    }
                }
            };
        }

        private static Action<AbstractedPad, string[]>
            GetAbstractedPadStickMoveEventSender(int index)
        {
            var id = (uint)index;

            var state = new AbstractedPadState();

            return (AbstractedPad that, string[] args) =>
            {
                if (args.Count() != 3)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.StickMove));
                }

                var stickType = args[0];

                if (stickType != StickType.Left &&
                    stickType != StickType.Right)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorStickUnexpectedType, stickType));
                }

                var x = 0;

                if (!int.TryParse(DecodeCoordinate(args[1]), out x))
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorCoordinateInvalidX, args[1]));
                }

                var y = 0;

                if (!int.TryParse(DecodeCoordinate(args[2]), out y))
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorCoordinateInvalidY, args[2]));
                }

                using (var accessor = new PortAccessor())
                {
                    GetAbstractedPadAutoPilotState(
                        accessor.Handle, out state, id, that.port);

                    SetAbstractedPadDeviceType(ref state, that.deviceTypes);

                    if (state.Attributes.HasFlag(
                            AbstractedPadAttributes.IsConnected))
                    {
                        if (stickType == StickType.Left)
                        {
                            state.StickL.X = x;
                            state.StickL.Y = y;
                        }
                        else if (stickType == StickType.Right)
                        {
                            state.StickR.X = x;
                            state.StickR.Y = y;
                        }

                        SetAbstractedPadAutoPilotState(
                            accessor.Handle, ref state, id, that.port);
                    }
                }
            };
        }

        private struct AbstractedPadColor
        {
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
            internal byte[] Main;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
            internal byte[] Sub;
        }

        private struct PowerInfo
        {
            internal byte IsPowered;
            internal byte IsCharging;
            internal byte Padding;
            internal AbstractedPadBatteryLevel BatteryLevel;
        }

        private struct AbstractedPadStickState
        {
            internal int X;
            internal int Y;
        }

        private struct AbstractedPadState
        {
            internal AbstractedPadDeviceType DeviceType;
            internal AbstractedPadAttributes Attributes;
            internal AbstractedPadColor Color;
            internal AbstractedPadInterfaceType InterfaceType;
            internal PowerInfo PowerInfo;
            internal AbstractedPadButtons Buttons;
            internal AbstractedPadStickState StickL;
            internal AbstractedPadStickState StickR;
        }

        private static class Native
        {
            [DllImport("HidShellLibrary.dll")]
            internal static extern int GetHidShellAbstractedPadState(
                IntPtr handle, out AbstractedPadState outState, uint id,
                uint port, uint direction);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int SetHidShellAbstractedPadState(
                IntPtr handle, ref AbstractedPadState state, uint id,
                uint port, uint direction);
        }

        private static class StickType
        {
            internal const string Left = "L";

            internal const string Right = "R";
        }

        private static class EventType
        {
            internal const string BatteryLevel = "BatteryLevel";

            internal const string ButtonDown = "ButtonDown";

            internal const string ButtonUp = "ButtonUp";

            internal const string StickMove = "StickMove";

            internal const string PowerState = "PowerState";
        }

        private sealed class EventDumper
        {
            private object syncObject = new object();

            private Task task = null;

            private List<Tuple<AbstractedPad, string>> entires;

            internal EventDumper(uint port)
            {
                this.task = Task.Run(() => DumpAbstractedPadEvent(port, this));

                this.entires = new List<Tuple<AbstractedPad, string>>();
            }

            internal void Invoke(AbstractedPad that, string deviceId)
            {
                lock (this.syncObject)
                {
                    this.entires.Add(
                        new Tuple<AbstractedPad, string>(that, deviceId));
                }

                this.task.Wait();
            }

            internal void WriteLine(
                string format, AbstractedPadDeviceTypes deviceType,
                string deviceId, params object[] args)
            {
                var entires = new List<Tuple<AbstractedPad, string>>();

                lock (this.syncObject)
                {
                    entires.AddRange(this.entires);
                }

                foreach (var entry in entires)
                {
                    if (entry.Item2 != null && entry.Item2 != deviceId)
                    {
                        continue;
                    }

                    entry.Item1.WriteLine(format, deviceType, deviceId, args);
                }
            }
        }

        private sealed class EventSender
        {
            private string deviceId;

            private string eventType;

            private Action<AbstractedPad, string[]> sender;

            internal EventSender(
                string deviceId,
                string eventType,
                Action<AbstractedPad, string[]> sender)
            {
                this.deviceId = deviceId;

                this.eventType = eventType;

                this.sender = sender;
            }

            internal bool SupportsDeviceId(string deviceId)
            {
                return deviceId == this.deviceId;
            }

            internal bool SupportsEventType(string eventType)
            {
                return eventType == this.eventType;
            }

            internal void Invoke(AbstractedPad that, string[] eventArgs)
            {
                this.sender(that, eventArgs);
            }
        }
    }
}
