﻿// 設定ファイルを使用する
#define ENABLE_CONFIG_FILE

// 最近使ったファイルを使用する
//#define ENABLE_RECENT_FILE

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Management; // need to add System.Management to your project references.

namespace Nintendo.NX.KuinaFirmwareUpdater
{
    using PairType = KeyValuePair<string, Color>;

    public partial class MainForm : Form
    {
        private const int ProgressSmoother = 10;

        private const string ImageDataDirectory = "data";

        /// <summary>
        /// STMFlashLoader.exe のパス
        /// </summary>
        private string FlashLoaderPath { get; set; } = null;

        /// <summary>
        /// ポート一覧
        /// </summary>
        private List<Util.PortInfo> PortList { get; set; } = null;

        private List<UsbHid.KuinaDevice> HidDeviceList { get; set; } = null;

        /// <summary>
        /// FW イメージ情報
        /// </summary>
        private ImageInfo FwInfo { get; set; } = null;

        /// <summary>
        /// データ受信イベントハンドラ
        /// </summary>
        private DataReceivedEventHandler DataReceivedHandler { get; set; } = null;

        private Process WriterProcess { get; set; } = null;

        private UpdateMode FwUpdateMode { get; set; } = UpdateMode.NoDevice;

        private byte[] RequiredFwVersion { get; set; } = { 0xff, 0xff, 0xff };

        public MainForm()
        {
            InitializeComponent();
            PortList = new List<Util.PortInfo>();
            FwInfo = new ImageInfo();
            DataReceivedHandler = null;
        }

        /// <summary>
        /// STMFlashLoader のインストール先を確認
        /// </summary>
        /// <returns></returns>
        private bool CheckStmFlashLoader()
        {
            const string FlashLoaderFilename = "DfuSeCommand.exe";

            var searchPaths = new string[]
            {
                Path.Combine(Application.StartupPath, "bin"),  // ローカルに置いてある前提
            };

            foreach (var path in searchPaths)
            {
                if (!Directory.Exists(path))
                {
                    continue;
                }

                foreach (var filepath in Directory.GetFiles(path))
                {
                    var filename = Path.GetFileName(filepath);
                    if (string.Equals(filename, FlashLoaderFilename, StringComparison.CurrentCultureIgnoreCase))
                    {
                        FlashLoaderPath = filepath;
                        Console.WriteLine(FlashLoaderPath);
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// シリアルポート一覧の更新
        /// </summary>
        private void UpdatePortList()
        {
            serialPortBox.BeginUpdate();
            serialPortBox.Items.Clear();

            int updateRequiredId = -1;

            if (FwUpdateMode == UpdateMode.Grip_DFU)
            {
                serialPortBox.Items.Add("Joy-Con Charging Grip (USB)");
            }
            else if (FwUpdateMode == UpdateMode.ProCon_DFU)
            {
                serialPortBox.Items.Add("Switch Pro Controller (USB)");
            }
            else if (FwUpdateMode == UpdateMode.ManyDfuDevices)
            {
                //DFUデバイスは１個前提のため
                //ここはデバイスを抜いてもらうしかない
            }
            else if (FwUpdateMode == UpdateMode.UART)
            {
                PortList = Util.GetDetailedPortNames();

                foreach (var port in PortList)
                {
                    var portName = Util.RegexPortNum.Replace(port.Name, "");
                    var comName = string.Format(
                        string.IsNullOrEmpty(portName) ? "{0}" : "{0}: ",
                        port.Id);
                    serialPortBox.Items.Add(comName + portName);
                }

            }
            else if (FwUpdateMode == UpdateMode.HID)
            {
                HidDeviceList = UsbHid.GetDeviceList();
                foreach (var device in HidDeviceList)
                {
                    bool isUpdateRequired = true;

                    if (device.Version[0] >= RequiredFwVersion[0] &&
                        device.Version[1] >= RequiredFwVersion[1] &&
                        device.Version[2] >= RequiredFwVersion[2])
                    {
                        isUpdateRequired = false;
                    }

                    var deviceName = string.Format(
                        "{0}({1}) ver {2}.{3}.{4} ",
                        device.DeviceTypeStr,
                        device.Id,
                        device.Version[0],
                        device.Version[1],
                        device.Version[2]
                    );

                    var targetFwVersion = string.Format(
                        "{0}.{1}.{2} ",
                        RequiredFwVersion[0],
                        RequiredFwVersion[1],
                        RequiredFwVersion[2]
                    );

                    var fwVersion = isUpdateRequired ? "Need Update to " + targetFwVersion : "Already Updated";

                    if (isUpdateRequired)
                        updateRequiredId = serialPortBox.Items.Count;

                    serialPortBox.Items.Add(deviceName + fwVersion);

                }

            }
            else if (FwUpdateMode == UpdateMode.NoDevice)
            {

            }

            if (serialPortBox.Items.Count > 0)
            {
                serialPortBox.SelectedIndex = (updateRequiredId == -1) ? 0 : updateRequiredId;
            }

            serialPortBox.EndUpdate();
        }

        /// <summary>
        /// FW イメージファイル一覧の更新
        /// </summary>
        private void UpdateImageFileList()
        {
            Debug.WriteLine("UpdateImageFileList() FwUpdateMode == " + FwUpdateMode.ToString());

            string imageListPath = Path.Combine(Application.StartupPath, ImageDataDirectory);
            if (!Directory.Exists(imageListPath))
            {
                return;
            }

            string dfu = "ChargingGrip*.dfu";

            if (FwUpdateMode == UpdateMode.Grip_DFU)
                dfu = "ChargingGrip*.dfu";
            else if (FwUpdateMode == UpdateMode.ProCon_DFU)
                dfu = "ProController*.dfu";
            else if (FwUpdateMode == UpdateMode.UART)
                dfu = "ChargingGrip*.dfu";//とりあえずGripにしておく
            else if (FwUpdateMode == UpdateMode.HID)
            {
                if (HidDeviceList[serialPortBox.SelectedIndex].Type == UsbHid.DeviceType.ChagringGrip)
                    dfu = "ChargingGrip*.dfu";
                else if (HidDeviceList[serialPortBox.SelectedIndex].Type == UsbHid.DeviceType.ProController)
                    dfu = "ProController*.dfu";
            }

            comboImageFile.BeginUpdate();
            comboImageFile.Items.Clear();
            foreach (var filepath in Directory.GetFiles(imageListPath, dfu))
            {
                comboImageFile.Items.Add(Path.GetFileName(filepath));
            }
            comboImageFile.EndUpdate();

            UpdateFirmwareListVisibility();

#if ENABLE_RECENT_FILE
            var config = Config.GetInstance();
            if (config.RecentFileList.Count > 0)
            {
                comboImageFile.Text = config.RecentFileList[0];
            }
#else
            if (comboImageFile.Items.Count > 0)
            {
                comboImageFile.SelectedIndex = 0;
            }
#endif
        }

        private void UpdateFirmwareListVisibility()
        {
            bool existsSingleImageFile = comboImageFile.Items.Count == 1;

            //アップデートしないときは表示しない
            if (FwUpdateMode == UpdateMode.NoDevice || FwUpdateMode == UpdateMode.ManyDfuDevices)
                existsSingleImageFile = true;

            labelImage.Visible =
                comboImageFile.Enabled =
                comboImageFile.Visible =
                buttonBrowseImage.Enabled =
                buttonBrowseImage.Visible =
                !existsSingleImageFile;

            if (existsSingleImageFile)
            {
                if (FwUpdateMode == UpdateMode.HID)
                {
                    serialPortBox.Enabled = true;
                    buttonStart.Enabled = true;
                    //                    labelProgress.Text = "Already updated";
                    labelProgress.Text = "Press start button to update";
                }
                else if (FwUpdateMode == UpdateMode.NoDevice)
                {
                    serialPortBox.Enabled = false;
                    buttonStart.Enabled = false;
                    labelProgress.Text = "Error : No device is detected";
                }
                else if (FwUpdateMode == UpdateMode.ManyDfuDevices)
                {
                    serialPortBox.Enabled = false;
                    buttonStart.Enabled = false;
                    labelProgress.Text = "Error : Connect only one device";
                }
                else
                {
                    serialPortBox.Enabled = true;
                    buttonStart.Enabled = true;
                    labelProgress.Text = "Press start button to update";
                }

                var minimumSize = MinimumSize;
                minimumSize.Height = 160;
                this.MinimumSize = minimumSize;
#if !_DEBUG
                this.FormBorderStyle = FormBorderStyle.FixedSingle;
                this.MaximizeBox = false;
#endif
            }
            else
            {
                buttonStart.Enabled = true;
            }
        }

        /// <summary>
        /// フォームロード
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainForm_Load(object sender, EventArgs e)
        {
            if (!CheckStmFlashLoader())
            {
                MessageBox.Show(
                    "Could not find \"DfuSeCommand\".",
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                Close();
                return;
            }

            WriterProcess = new Process();

#if ENABLE_CONFIG_FILE
            //設定値の読み込み
            {
                var config = Config.GetInstance();
                config.Load();
                RequiredFwVersion[0] = config.FwMajorVersion;
                RequiredFwVersion[1] = config.FwMinorVersion;
                RequiredFwVersion[2] = config.FwBuildVersion;
                Debug.WriteLine("RequiredFwVersion {0}.{1}.{2}", RequiredFwVersion[0], RequiredFwVersion[1], RequiredFwVersion[2]);
            }
#endif

            FwUpdateMode = GetUpdateMode();

            UpdatePortList();

            UpdateImageFileList();

            Height = MinimumSize.Height;
#if !DEBUG
            textProcessLog.Visible = false;
#endif
        }

        /// <summary>
        /// フォームクローズ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            try
            {
#if ENABLE_CONFIG_FILE
                //                Config.GetInstance().Save();
#endif
                WriterProcess.Kill();
            }
            catch
            {
                // 失敗しても無視
            }
        }

        #region keep
        /// <summary>
        /// ファイルドラッグ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainForm_DragOver(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                e.Effect = DragDropEffects.Copy;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        /// <summary>
        /// ファイルドロップ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainForm_DragDrop(object sender, DragEventArgs e)
        {
            if (!e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                return;
            }

            var files = (string[])e.Data.GetData(DataFormats.FileDrop);
            comboImageFile.Text = files[0];
        }

        /// <summary>
        /// ポート選択
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void serialPortBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            var index = ((ComboBox)sender).SelectedIndex;
            Debug.WriteLine("PortList {0} is seletced", index);

            //if (FwUpdateMode == UpdateMode.HID)
            //    UpdateImageFileList();
        }

        /// <summary>
        /// イメージファイル選択
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void buttonBrowseImage_Click(object sender, EventArgs e)
        {
            if (openImageDialog.ShowDialog(this) != DialogResult.OK)
            {
                return;
            }

            comboImageFile.Text = openImageDialog.FileName;
        }

        /// <summary>
        /// Verify 有無変更
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void checkVerify_CheckedChanged(object sender, EventArgs e)
        {
            var control = (CheckBox)sender;
        }

        /// <summary>
        /// アップデートを続けるか判定
        /// </summary>
        /// <param name="log"></param>
        /// <returns></returns>
        private bool IsContinueUpdate(string log)
        {
            //Target 00: Error Code: Bad state machine in firmware
            // エラー系のログが出た場合は中断
            var killRegex = new Regex(@"^\s*Error Code:");
            var failRegex = new Regex(@"\[KO\]");
            if (killRegex.IsMatch(log) || failRegex.IsMatch(log))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Erase 開始ログの確認
        /// </summary>
        /// <param name="log"></param>
        private void CheckEraseLog(string log)
        {
            var eraseRegex = new Regex(@"^\s*Erase Phase\s*\.+");
            if (eraseRegex.IsMatch(log))
            {
                BeginInvoke(new Action(() => labelProgress.Text = "Erasing flash..."));
            }
        }

        /// <summary>
        /// Download (Write) の進捗確認
        /// </summary>
        /// <param name="log"></param>
        private void CheckDownloadLog(string log)
        {
            FwInfo.Sectors = 100;

            //Target 00: Upgrading - Erase Phase(0)...
            {
                var downloadRegex = new Regex(@"Erase Phase \((?<sector>\d+)\)...");

                var match = downloadRegex.Match(log);
                if (match.Success)
                {
                    var sector = int.Parse(match.Groups["sector"].Value);
                    BeginInvoke(new Action(() =>
                    {
                        progressWrite.Value = sector;// * ProgressSmoother;
                        if (sector <= FwInfo.Sectors)
                        {
                            labelProgress.Text = string.Format("Erase... ({0}/{1})", sector, FwInfo.Sectors);
                        }
                    }));
                }
            }

            //Target 00: Upgrading - Download Phase (0)...
            {
                var downloadRegex = new Regex(@"Download Phase \((?<sector>\d+)\)...");

                var match = downloadRegex.Match(log);
                if (match.Success)
                {
                    var sector = int.Parse(match.Groups["sector"].Value);
                    BeginInvoke(new Action(() =>
                    {
                        progressWrite.Value = sector;// * ProgressSmoother;
                        if (sector <= FwInfo.Sectors)
                        {
                            labelProgress.Text = string.Format("Writing... ({0}/{1})", sector, FwInfo.Sectors);
                        }
                    }));
                }
            }
        }

        /// <summary>
        /// Verify の進捗確認
        /// </summary>
        /// <param name="log"></param>
        private void CheckVerifyLog(string log)
        {
            {
                var verifyStartStr = new Regex(@"Verify successful !");
                if (verifyStartStr.IsMatch(log))
                {
                    BeginInvoke(new Action(() =>
                    {
                        labelProgress.Text = string.Format("Verify successful");
                    }));

                    Debug.WriteLine("Verify successful");
                }
            }

            {
                var verifyStartStr = new Regex(@"Upgrade successful !");
                if (verifyStartStr.IsMatch(log))
                {
                    BeginInvoke(new Action(() =>
                    {
                        labelProgress.Text = string.Format("Upgrade successful");
                    }));

                    Debug.WriteLine("Upgrade successful");
                }
            }
        }

        /// <summary>
        /// STMFlashLoader のログを表示
        /// </summary>
        /// <param name="log"></param>
        private void PrintLog(string log)
        {
            BeginInvoke(new Action(() =>
            {
                textProcessLog.Focus();

                Debug.WriteLine(log);
                textProcessLog.AppendText(log);
                textProcessLog.AppendText(Environment.NewLine);

                var hilightList = new List<PairType>()
                    {
                        new PairType("[OK]", Color.DodgerBlue),
                        new PairType("[KO]", Color.Crimson)
                    };
                int currentSelectionStart = textProcessLog.SelectionStart;
                int currentSelectionLength = textProcessLog.SelectionLength;
                int pos = currentSelectionStart;
                while (pos >= 0)
                {
                    bool isFound = false;
                    foreach (var pair in hilightList)
                    {
                        int newPos = textProcessLog.Find(pair.Key, pos, RichTextBoxFinds.None);
                        if (newPos >= 0)
                        {
                            textProcessLog.SelectionColor = pair.Value;
                            isFound = true;
                            pos = newPos + 1;
                            break;
                        }
                    }

                    if (!isFound)
                    {
                        pos = -1;
                    }
                }
                textProcessLog.Select(currentSelectionStart, currentSelectionLength);
            }));
        }

        /// <summary>
        /// プロセスからの stdout/stderr 受信ハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Process_DataReceived(object sender, DataReceivedEventArgs e)
        {
            var log = e.Data;

            if (Disposing || IsDisposed || log == null)
            {
                return;
            }


            if (!IsContinueUpdate(log))
            {
                var process = (Process)sender;
                if (!process.HasExited)
                {
                    process.Kill();
                }
                Debug.WriteLine("KILLED");
                return;
            }

            CheckEraseLog(log);
            CheckDownloadLog(log);
            CheckVerifyLog(log);

            PrintLog(log);

        }

        /// <summary>
        /// プロセス起動情報の設定
        /// </summary>
        /// <param name="process">設定対象のプロセス</param>
        private void SetupProcessInfo(Process process, string imagePath)
        {
            if (DataReceivedHandler == null)
            {
                DataReceivedHandler = Process_DataReceived;
            };

            FwInfo.Filename = imagePath;
            FwInfo.NeedVerify = true;//checkVerify.Checked;

            var startInfo = process.StartInfo;
            startInfo.FileName = FlashLoaderPath;
            startInfo.Arguments = string.Format(
                "-c -d {0} --fn  \"{1}\" -l",
                FwInfo.NeedVerify ? "--v" : "",
                FwInfo.Filename);

            Debug.WriteLine("Start " + startInfo.FileName + " " + startInfo.Arguments);

            // コンソール出力をリダイレクトするために必要な設定
            startInfo.CreateNoWindow = true;
            startInfo.UseShellExecute = false;
            //startInfo.RedirectStandardError = true;
            startInfo.RedirectStandardOutput = true;

            process.OutputDataReceived += DataReceivedHandler;
            //process.ErrorDataReceived += DataReceivedHandler;

            textProcessLog.Clear();
        }

        /// <summary>
        /// コントロールの有効状態を設定
        /// </summary>
        /// <param name="enable"></param>
        private void SetControlEnabled(bool enable)
        {
            serialPortBox.Enabled = enable;
            buttonBrowseImage.Enabled = enable;
            buttonStart.Enabled = enable;
            checkVerify.Enabled = enable;
            comboImageFile.Enabled = enable;

            waitIcon.Visible = !enable;
        }

        private string GetActualImagePath(string filepath)
        {
            if (File.Exists(filepath))
            {
                return filepath;
            }

            var pathInData = Path.Combine(ImageDataDirectory, filepath);
            if (File.Exists(pathInData))
            {
                return pathInData;
            }

            return null;
        }

        #endregion
        /// <summary>
        /// 開始ボタン
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private async void buttonStart_Click(object sender, EventArgs e)
        {
            if (serialPortBox.SelectedIndex < 0)
            {
                MessageBox.Show(
                    "COM port is not selected.",
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                return;
            }

            SetControlEnabled(false);

            //UART版ならば
            if (FwUpdateMode == UpdateMode.UART)
            {
                //DFUモードに入れる
                SendUpdateComPort(PortList[serialPortBox.SelectedIndex].Id);
            }
            else if (FwUpdateMode == UpdateMode.HID)
            {
                HidDeviceList[serialPortBox.SelectedIndex].GotoDfu();

                Debug.WriteLine("DFU CHECK START {0}", DateTime.Now.ToString("F"));

                //DFUに入るのを6秒は待つ
                const int wait = 100;
                for (int i = 0; i < wait; i++)
                {
                    Thread.Sleep(70);
                    progressWrite.Value = (i + 1);
                    labelProgress.Text = "Preparing...(" + (i + 1) + "/" + wait + ")";
                    Application.DoEvents();
                }
            }

            if (FwUpdateMode == UpdateMode.UART || FwUpdateMode == UpdateMode.HID)
            {
                Debug.WriteLine("DFU CHECK LOOP {0}", DateTime.Now.ToString("F"));
                FwUpdateMode = GetUpdateMode(true);

                while ((FwUpdateMode == UpdateMode.Grip_DFU || FwUpdateMode == UpdateMode.ProCon_DFU) == false)
                {
                    Debug.WriteLine("DFU CHECK LOOP {0}", DateTime.Now.ToString("F"));
                    Thread.Sleep(500);
                    FwUpdateMode = GetUpdateMode(true);
                    Application.DoEvents();
                }

                Debug.WriteLine("DFU CHECK DONE {0}", DateTime.Now.ToString("F"));
                //一回DFUモードにいれた後に再度USBの検索を行う
                UpdatePortList();
                UpdateImageFileList();
            }

            SetControlEnabled(false);

            var imagePath = GetActualImagePath(comboImageFile.Text);
            if (!File.Exists(imagePath))
            {
                MessageBox.Show(
                    "Specified image is not found.",
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                return;
            }


            progressWrite.Value = 0;
            labelProgress.Text = "Preparing...";

            Application.DoEvents();

#if true
            Debug.WriteLine("START UPDATE {0}", DateTime.Now.ToString("F"));
            SetupProcessInfo(WriterProcess, imagePath);
            await Task.Run(() =>
            {
                WriterProcess.Start();

                // コンソール出力のリダイレクト開始
                WriterProcess.BeginOutputReadLine();
                //WriterProcess.BeginErrorReadLine();

                WriterProcess.WaitForExit();

                // リダイレクト終了
                if (WriterProcess != null)
                {
                    //WriterProcess.CancelErrorRead();
                    WriterProcess.CancelOutputRead();
                    WriterProcess.OutputDataReceived -= DataReceivedHandler;
                    //WriterProcess.ErrorDataReceived -= DataReceivedHandler;
                }
            });

            if (WriterProcess != null)
                WriterProcess.Close();

            Debug.WriteLine("END UPDATE {0}", DateTime.Now.ToString("F"));

#endif
            progressWrite.Maximum = 100;
            if (progressWrite.Value < progressWrite.Maximum)
            {
                labelProgress.Text = "FAILED...";
                labelProgress.ForeColor = Color.Red;
            }
            else
            {
                labelProgress.Text = "Finished!!";
            }
            progressWrite.Value = progressWrite.Maximum;

            //SetControlEnabled(true);
            waitIcon.Visible = false;
            buttonStart.Text = "Exit";
            buttonStart.Image = null;
            buttonStart.Click -= buttonStart_Click;
            buttonStart.Click += buttonStart_ClickForExit;
            buttonStart.Enabled = true;
        }

        private void buttonStart_ClickForExit(object sender, EventArgs e)
        {
            Close();
        }

        /// <summary>
        /// COM ポートにUpdateコマンドを送る
        /// </summary>
        /// <param name="portName">対象の COM ポート</param>
        private void SendUpdateComPort(string portName)
        {
            using (var comPort = new System.IO.Ports.SerialPort())
            {
                comPort.PortName = portName;
                comPort.BaudRate = 115200;

                try
                {
                    comPort.Open();

                    // 送受信バッファ内のデータを全削除
                    comPort.DiscardInBuffer();
                    comPort.DiscardOutBuffer();

#if false//DEBUG
                    comPort.Write("version\n");
                    var result = comPort.ReadLine();
                    result = comPort.ReadLine();
                    Debug.WriteLine(result);
#endif
                    comPort.Write("update\n");

                    //再起動を1秒間待つ
                    const int wait = 10;
                    for (int i = 0; i < wait; i++)
                    {
                        Thread.Sleep(100);
                        labelProgress.Text = "Preparing...(" + (i + 1) + "/" + wait + ")";
                        Application.DoEvents();
                    }

                    comPort.Close();
                }
                catch (System.IO.IOException) { }
            }
        }


        enum UpdateMode
        {
            NoDevice = 0,
            ManyDfuDevices,
            UART,
            Grip_DFU,
            ProCon_DFU,
            HID
        }

        private UpdateMode GetUpdateMode(bool isOnlyDfu = false)
        {
            var usbDevices = GetUSBDevices(isOnlyDfu);

            if (usbDevices.Count() == 0)
                return UpdateMode.NoDevice;
            else if (usbDevices.NumOfDfuDevice > 1)
                return UpdateMode.ManyDfuDevices;
            else if (usbDevices.NumOfGripDfuDevice == 1)
                return UpdateMode.Grip_DFU;
            else if (usbDevices.NumOfProConDfuDevice == 1)
                return UpdateMode.ProCon_DFU;
            else if (usbDevices.NumOfVComDevice > 0)
                return UpdateMode.UART;
            else if (usbDevices.NumOfHidDevice > 0)
                return UpdateMode.HID;
            else
                return UpdateMode.ManyDfuDevices;
        }

        private DetectedUsbDeviceList GetUSBDevices(bool isOnlyDfu = false)
        {
            var list = new DetectedUsbDeviceList();
#if DEBUG
            var devices = new List<USBDeviceInfo>();
#endif

            string[] allQueryString = {
                @"SELECT * FROM Win32_PnPEntity where DeviceID Like ""USB\\VID_057E&PID_2008%""",
                @"SELECT * FROM Win32_PnPEntity where DeviceID Like ""USB\\VID_057E&PID_200F%""",
                @"SELECT * FROM Win32_PnPEntity where DeviceID Like ""USB\\VID_057E&PID_200E%""",
                @"SELECT * FROM Win32_PnPEntity where DeviceID Like ""USB\\VID_057E&PID_2009%""",
                @"SELECT * FROM Win32_PnPEntity where DeviceID Like ""USB\\VID_0483&PID_5740%""",
            };

            string[] dfuQueryString = {
                @"SELECT * FROM Win32_PnPEntity where DeviceID Like ""USB\\VID_057E&PID_2008%""",
                @"SELECT * FROM Win32_PnPEntity where DeviceID Like ""USB\\VID_057E&PID_200F%""",
            };

            var queryString = isOnlyDfu ? dfuQueryString : allQueryString;


            //using (var searcher = new ManagementObjectSearcher(@"SELECT * FROM Win32_PnPEntity where DeviceID Like ""USB%"""))
            for (int i = 0; i < queryString.Length; i++)
            {
                using (var searcher = new ManagementObjectSearcher(queryString[i]))
                {

                    var collection = searcher.Get();

                    if (collection.Count > 0)
                    {
                        list.Add(new DetectedUsbDevice((UsbDeviceType)i, collection.Count));
#if DEBUG
                    Console.WriteLine(collection.Count + " " + ((UsbDeviceType)i).ToString() + " is detected");
                    foreach (var device in collection)
                    {
                        devices.Add(new USBDeviceInfo(
                        (string)device.GetPropertyValue("DeviceID"),
                        (string)device.GetPropertyValue("PNPDeviceID"),
                        (string)device.GetPropertyValue("Description"),
                        (string)device.GetPropertyValue("Manufacturer")
                        ));
                    }
#endif
                    }
                    collection.Dispose();

                }
            }

#if DEBUG
            foreach (var usbDevice in devices)
            {
                Console.WriteLine("Device ID: {0}, PNP Device ID: {1}, Description: {2}, Manufacturer: {3}",
                    usbDevice.DeviceID, usbDevice.PnpDeviceID, usbDevice.Description, usbDevice.Manufacturer);
            }
#endif
            return list;
        }
    }

    class DetectedUsbDevice
    {
        public DetectedUsbDevice(UsbDeviceType deviceType, Int32 numOfDevice)
        {
            DeviceType = deviceType;
            NumOfDevice = numOfDevice;
        }

        public UsbDeviceType DeviceType { get; private set; }
        public Int32 NumOfDevice { get; private set; }
    }

    class DetectedUsbDeviceList : List<DetectedUsbDevice>
    {
        public int NumOfGripDfuDevice { get; private set; } = 0;
        public int NumOfProConDfuDevice { get; private set; } = 0;
        public int NumOfDfuDevice { get; private set; } = 0;
        public int NumOfVComDevice { get; private set; } = 0;
        public int NumOfHidDevice { get; private set; } = 0;

        public new void Add(DetectedUsbDevice device)
        {
            base.Add(device);

            switch (device.DeviceType)
            {
                case UsbDeviceType.GripDfu:
                    {
                        NumOfGripDfuDevice += device.NumOfDevice;
                        NumOfDfuDevice += device.NumOfDevice;
                        break;
                    }
                case UsbDeviceType.ProConDfu:
                    {
                        NumOfProConDfuDevice += device.NumOfDevice;
                        NumOfDfuDevice += device.NumOfDevice;
                        break;
                    }
                case UsbDeviceType.STVirtualCOM:
                    {
                        NumOfVComDevice += device.NumOfDevice;
                        break;
                    }
                case UsbDeviceType.Grip:
                case UsbDeviceType.ProCon:
                    {
                        NumOfHidDevice += device.NumOfDevice;
                        break;
                    }
                default:
                    break;
            }
        }
    }

    enum UsbDeviceType
    {
        GripDfu = 0, //0x2008
        ProConDfu, //0x200F
        Grip, //0x200E
        ProCon, //0x2009
        STVirtualCOM //0x0483 0x5740
    };


#if DEBUG
    class USBDeviceInfo
    {
        public USBDeviceInfo(string deviceID, string pnpDeviceID, string description, string manufacturer)
        {
            this.DeviceID = deviceID;
            this.PnpDeviceID = pnpDeviceID;
            this.Description = description;
            this.Manufacturer = manufacturer;
        }
        public string DeviceID { get; private set; }
        public string PnpDeviceID { get; private set; }
        public string Description { get; private set; }
        public string Manufacturer { get; private set; }
    }
#endif
}

