﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

/**
* @brief タイリングフラグです。
*/
enum NXTilingFlag
{
    NXTilingFlag_Default          = 0       , //!< デフォルトのタイリング（パフォーマンス優先）です。
    NXTilingFlag_Sparse           = 0x1 << 0, //!< スパーステクスチャ向けのタイリングです。
    NXTilingFlag_OptimizeSize     = 0x1 << 1, //!< サイズ優先のタイリングです。
    NXTilingFlag_OptimizeSizeAuto = 0x1 << 2  //!< データ削減率に応じてサイズ優先のタイリングを選択します。
};

//=============================================================================
//! @brief NX テクスチャの構造体です。
//=============================================================================
struct NXTexture
{
    static const int MipCountMax = 16; //!< ミップマップのレベル数の最大値です。

    uint8_t* pData; //!< テクスチャデータへのポインタです。
    size_t dataSize; //!< テクスチャデータ全体（ベース + ミップマップ）のサイズ（バイト数）です。
    int alignment; //!< アライメントです。
    size_t levelOffsets[MipCountMax]; //!< 各ミップマップデータのテクスチャデータ先頭からのオフセット配列です。
    uint32_t textureLayout[2]; //!< テクスチャレイアウトです。
};

//-----------------------------------------------------------------------------
//! @brief タイリングを変換する関数の型です。
//!
//! @param[out] pDstTexture 変換後の NX テクスチャへのポインタです。
//! @param[in] isTiling NX 向けのタイリング変換をするなら true、タイリング変換を解除するなら false です。
//! @param[in] pSrcData 変換前のデータです。
//! @param[in] srcDataSize 変換前のデータサイズです。
//! @param[in] dimension 中間ファイルの次元文字列です。
//! @param[in] format 中間ファイルのフォーマット文字列です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです（3D 以外は 1）。
//! @param[in] arrayLength 配列長です（キューブマップと配列以外は 1）。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] pTextureLayout テクスチャレイアウトへのポインタです。
//!                           NX 向けのタイリング変換をする場合は使用されません（nullptr でも構いません）。
//!                           タイリング変換を解除する場合はタイリング変換した際の値を指定します。
//! @param[in] tilingFlags タイリングフラグです。
//!                        タイリング変換を解除する場合、Sparse 以外は処理に影響しません。
//! @param[in] sizeThreshold OptimizeSizeAuto の場合のデータ削減率のしきい値 [%] です。
//!                          タイリング変換を解除する場合は処理に影響しません。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
typedef bool (*ConvertTilingFunction)(
    NXTexture* pDstTexture,
    const bool isTiling,
    const void* pSrcData,
    const size_t srcDataSize,
    const wchar_t* dimension,
    const wchar_t* format,
    const int imageW,
    const int imageH,
    const int imageD,
    const int arrayLength,
    const int mipCount,
    const void* pTextureLayout,
    const uint32_t tilingFlags,
    const int sizeThreshold
);

//-----------------------------------------------------------------------------
//! @brief NX テクスチャを解放する関数の型です。
//!
//! @param[in,out] pTexture NX テクスチャへのポインタです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
typedef bool (*ReleaseTextureFunction)(NXTexture* pTexture);

//-----------------------------------------------------------------------------
//! @brief NX 向けのタイリング変換をした場合のデータサイズを取得する関数の型です。
//!
//! @param[in] dimension 中間ファイルの次元文字列です。
//! @param[in] format 中間ファイルのフォーマット文字列です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです（3D 以外は 1）。
//! @param[in] arrayLength 配列長です（キューブマップと配列以外は 1）。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] tilingFlags タイリングフラグです。
//! @param[in] sizeThreshold OptimizeSizeAuto の場合のデータ削減率のしきい値 [%] です。
//!
//! @return NX 向けのタイリング変換をした場合のデータサイズを返します。
//!         処理失敗なら 0 を返します。
//-----------------------------------------------------------------------------
typedef size_t (*GetDataSizeNXFunction)(
    const wchar_t* dimension,
    const wchar_t* format,
    const int imageW,
    const int imageH,
    const int imageD,
    const int arrayLength,
    const int mipCount,
    const uint32_t tilingFlags,
    const int sizeThreshold
);

//-----------------------------------------------------------------------------
//! @brief メモリプールのアライメント値を取得する関数の型です。
//!
//! @param[out] pAddressAlignment アドレスのアライメント値へのポインタです。
//! @param[out] pSizeAlignment サイズのアライメント値へのポインタです。
//-----------------------------------------------------------------------------
typedef void (*GetMemoryPoolAlignmentFunction)(int* pAddressAlignment, int* pSizeAlignment);

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

