﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include "Common.h"

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
// 定数の定義です。
//=============================================================================

//! @brief サイズ変更フィルタを表す列挙型です。
enum ResizeFilter
{
    ResizeFilter_Point = 0,     //!< ポイントサンプリングです。
    ResizeFilter_Linear,        //!< 線形補間です。
    ResizeFilter_Cubic,         //!< キュービック補間です。
    ResizeFilter_CubicSmooth,   //!< キュービック補間（滑らか）です。
    ResizeFilter_CubicSharp,    //!< キュービック補間（シャープ）です。
    ResizeFilter_Invalid = -2,  //!< 無効なサイズ変更フィルタです。
    ResizeFilter_Default = -1   //!< デフォルトのサイズ変更フィルタ（指定なし）です。
};

//=============================================================================
// フォーマット変換関連の関数です。
//=============================================================================

//-----------------------------------------------------------------------------
//! @brief float 型の値から float_16 の値を取得します。
//!
//! @param[in] valueF32 float 型の値です。
//!
//! @return float_16 の値を返します。
//-----------------------------------------------------------------------------
uint16_t RGetFloat16Value(const float valueF32);

//=============================================================================
// 画像処理関連の関数です。
//=============================================================================

//-----------------------------------------------------------------------------
//! @brief フェースを水平に反転します。
//!
//! @param[in,out] pDst 反転するビットマップデータです。
//! @param[in] curW 幅です。
//! @param[in] curH 高さです。
//! @param[in] stride 行あたりのバイト数です。
//! @param[in] pixelBytes ピクセルあたりのバイト数です。
//-----------------------------------------------------------------------------
void RFlipFaceH(
    void* pDst,
    const int dstW,
    const int dstH,
    const size_t stride,
    const size_t pixelBytes
);

//-----------------------------------------------------------------------------
//! @brief フェースを垂直に反転します。
//!
//! @param[in,out] pDst 反転するビットマップデータです。
//! @param[in] curW 幅です。
//! @param[in] curH 高さです。
//! @param[in] stride 行あたりのバイト数です。
//! @param[in] pixelBytes ピクセルあたりのバイト数です。
//-----------------------------------------------------------------------------
void RFlipFaceV(
    void* pDst,
    const int dstW,
    const int dstH,
    const size_t stride,
    const size_t pixelBytes
);

//-----------------------------------------------------------------------------
//! @brief フェースを水平垂直に反転します。
//!
//! @param[in,out] pDst 反転するビットマップデータです。
//! @param[in] curW 幅です。
//! @param[in] curH 高さです。
//! @param[in] stride 行あたりのバイト数です。
//! @param[in] pixelBytes ピクセルあたりのバイト数です。
//-----------------------------------------------------------------------------
void RFlipFaceHv(
    void* pDst,
    const int dstW,
    const int dstH,
    const size_t stride,
    const size_t pixelBytes
);

//-----------------------------------------------------------------------------
//! @brief フェースを切り抜きます。
//!
//! @param[out] pDst 切り抜き後のビットマップデータを格納します。
//! @param[in] pSrc 切り抜き前のビットマップデータです。
//! @param[in] cropRect 切り抜き矩形です。
//! @param[in] srcW 切り抜き前の幅です。
//-----------------------------------------------------------------------------
template <typename T>
void RCropFace(
    T* pDst,
    const T* pSrc,
    const RCropRect& cropRect,
    const int srcW
);

//-----------------------------------------------------------------------------
//! @brief フェースを任意の幅と高さにサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//! @param[in] resizeFilter サイズ変更フィルタです。
//! @param[in] ignoresXpaRgb 透明ピクセルの RGB 成分を補間計算から除外するなら true です。
//-----------------------------------------------------------------------------
template <typename T>
void RResizeFreeFace(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
);

//-----------------------------------------------------------------------------
//! @brief 3D テクスチャを任意のサイズにサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] dstD サイズ変更後の奥行きです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//! @param[in] srcD サイズ変更前の奥行きです。
//! @param[in] resizeFilter サイズ変更フィルタです。
//! @param[in] ignoresXpaRgb 透明ピクセルの RGB 成分を補間計算から除外するなら true です。
//-----------------------------------------------------------------------------
template <typename T>
void RResizeFree3d(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int dstD,
    const int srcW,
    const int srcH,
    const int srcD,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
);

//-----------------------------------------------------------------------------
//! @brief フェースの透明ピクセルの RGB 成分を調整します。
//!
//! @param[in,out] pDst 調整するビットマップデータです。
//! @param[in] dstW 幅です。
//! @param[in] dstH 高さです。
//-----------------------------------------------------------------------------
template <typename T>
void RAdjustTransparentRgbFace(T* pDst, const int dstW, const int dstH);

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

