﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DecodeExr OutuptExrFile

//=============================================================================
// include
//=============================================================================
#include "ImageConvert.h"

using namespace std;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//-----------------------------------------------------------------------------
//! @brief EXR ファイルのデータをデコードします。
//-----------------------------------------------------------------------------
RStatus RImage::DecodeExr(ROpenExr* pOpenExr, const uint8_t* fileBuf, const size_t fileSize)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 必要なら OpenEXR ユーティリティを初期化します。
    if (pOpenExr == NULL)
    {
        return RStatus(RStatus::FAILURE, "Cannot read the EXR file: " + m_FilePath); // RShowError
    }
    else if (!pOpenExr->IsInitialized())
    {
        if (!pOpenExr->Initialize(NULL))
        {
            return RStatus(RStatus::FAILURE, "Cannot load dll: " + pOpenExr->GetDllPath() + // RShowError
                " (" + GetWindowsLastErrorMessage() + ")");
        }
    }

    //-----------------------------------------------------------------------------
    // get info
    int channelCount;
    uint32_t flags;
    if (!pOpenExr->ParseExrInfo(&m_ImageW, &m_ImageH, &m_ImageD, &channelCount, &flags,
        fileBuf, fileSize))
    {
        return RStatus(RStatus::FAILURE, pOpenExr->GetErrorShiftJisString()); // RShowError
    }
    //cout << "exr: " << m_ImageW << " x " << m_ImageH << " x " << m_ImageD << ", " << channelCount << ", " << hex << flags << dec << endl;

    if ((flags & ROpenExr::FLAGS_CUBEMAP) != 0)
    {
        m_Dimension = FtxDimension_CubeMap;
    }

    m_HasAlpha = (channelCount >= 4);
    m_IsFloat = true;

    //-----------------------------------------------------------------------------
    // read image data
    m_ImageDataSize = static_cast<size_t>(R_RGBA_FLOAT_BYTES) * m_ImageW * m_ImageH * m_ImageD;
    m_pImageData = new uint8_t[m_ImageDataSize];
    if (!pOpenExr->ParseExrAsFloat32(m_pImageData, fileBuf, fileSize, channelCount, flags))
    {
        RFreeAndClearArray(m_pImageData);
        return RStatus(RStatus::FAILURE, pOpenExr->GetErrorShiftJisString()); // RShowError
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief EXR ファイルを出力します。
//-----------------------------------------------------------------------------
RStatus RImage::OutuptExrFile(
    ROpenExr* pOpenExr,
    const std::string& filePath,
    const bool isCubeVC
) const
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 必要なら OpenEXR ユーティリティを初期化します。
    if (pOpenExr == NULL)
    {
        return RStatus(RStatus::FAILURE, "Cannot write the EXR file: " + filePath); // RShowError
    }
    else if (!pOpenExr->IsInitialized())
    {
        if (!pOpenExr->Initialize(NULL))
        {
            return RStatus(RStatus::FAILURE, "Cannot load dll: " + pOpenExr->GetDllPath() + // RShowError
                " (" + GetWindowsLastErrorMessage() + ")");
        }
    }

    //-----------------------------------------------------------------------------
    // 1 次元または 2 次元でなければ出力用ビットマップを作成します。
    uint8_t* pBitmapF32 = m_pImageData;
    int dstW = m_ImageW;
    int dstH = m_ImageH;
    int dstD = 1;
    uint32_t flags = 0;
    if (m_Dimension != FtxDimension_1d &&
        m_Dimension != FtxDimension_2d)
    {
        //-----------------------------------------------------------------------------
        // 出力用ビットマップの幅と高さを決定します。
        if (IsCubeMap())
        {
            if (!isCubeVC)
            {
                dstW *= CUBE_HC_COUNT_H;
                dstH *= CUBE_HC_COUNT_V;
            }
            else
            {
                dstW *= CUBE_VC_COUNT_H;
                dstH *= CUBE_VC_COUNT_V;
            }
            dstH *= RMax(m_ImageD / CUBE_FACE_COUNT, 1);
        }
        else if (m_Dimension == FtxDimension_3d      ||
                 m_Dimension == FtxDimension_1dArray ||
                 m_Dimension == FtxDimension_2dArray)
        {
            dstH *= m_ImageD;
        }

        //-----------------------------------------------------------------------------
        // 出力用ビットマップを確保してクリアします。
        const size_t pixCount = dstW * dstH * dstD;
        const size_t bitmapSize = pixCount * R_RGBA_FLOAT_BYTES;
        pBitmapF32 = new uint8_t[bitmapSize];
        float* pDstClear = reinterpret_cast<float*>(pBitmapF32);
        for (size_t iPix = 0; iPix < pixCount; ++iPix)
        {
            *pDstClear++ = 1.0f;
            *pDstClear++ = 1.0f;
            *pDstClear++ = 1.0f;
            *pDstClear++ = 1.0f;
        }

        //-----------------------------------------------------------------------------
        // 出力用ビットマップを設定します。
        const float* pSrcF32 = reinterpret_cast<const float*>(m_pImageData);
        const int curW = m_ImageW;
        const int curH = m_ImageH;
        const int curD = m_ImageD;
        for (int iz = 0; iz < curD; ++iz)
        {
            int iFace = iz % CUBE_FACE_COUNT;
            int dstIx = 0;
            int dstIy = m_ImageH * iz;
            if (IsCubeMap())
            {
                // m_pImageData には +X, -X, +Y, -Y, -Z, +Z の順に格納されているので
                // -Z と +Z を入れ替えます。
                if (iFace == CUBE_FACE_PZ)
                {
                    iFace = CUBE_FACE_NZ;
                }
                else if (iFace == CUBE_FACE_NZ)
                {
                    iFace = CUBE_FACE_PZ;
                }
                const int* cubePoss = (!isCubeVC) ?
                    CubeHcPositions[iFace] : CubeVcPositions[iFace];
                dstIx = cubePoss[0] * curW;
                dstIy = cubePoss[1] * curH;
                dstIy += (iz / CUBE_FACE_COUNT) * m_ImageH * ((!isCubeVC) ? CUBE_HC_COUNT_V : CUBE_VC_COUNT_V);
            }
            float* pDstTopF32 = reinterpret_cast<float*>(pBitmapF32) + (dstIx + dstIy * dstW) * R_RGBA_COUNT;
            float* pDstF32 = pDstTopF32;
            for (int iy = 0; iy < curH; ++iy)
            {
                memcpy(pDstF32, pSrcF32, curW * R_RGBA_FLOAT_BYTES);
                pSrcF32 += curW * R_RGBA_COUNT;
                pDstF32 += dstW * R_RGBA_COUNT;
            }

            // 垂直十字キューブマップは +Z 方向を上下左右反転します。
            if (IsCubeMap() && isCubeVC && iFace == CUBE_FACE_PZ)
            {
                RFlipFaceHv(pDstTopF32, curW, curH,
                    dstW * R_RGBA_FLOAT_BYTES, R_RGBA_FLOAT_BYTES);
            }
        }
    }

    //-----------------------------------------------------------------------------
    // EXR ファイルにライトします。
    const std::wstring filePathW = RGetUnicodeFromShiftJis(filePath);
    if (!pOpenExr->WriteExrFloat32(filePathW.c_str(), pBitmapF32, dstW, dstH, dstD, flags, m_HasAlpha))
    {
        status = RStatus(RStatus::FAILURE, pOpenExr->GetErrorShiftJisString()); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 出力用ビットマップを解放します。
    if (pBitmapF32 != m_pImageData)
    {
        //cerr << "delete EXR bitmap" << endl;
        delete[] pBitmapF32;
    }

    return status;
} // NOLINT(readability/fn_size)

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

