﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <thread>
#include <exception>
#include <stdexcept>
#include <future>

#if !defined( WIN32_LEAN_AND_MEAN )
    #define WIN32_LEAN_AND_MEAN
#endif
#if !defined( NOMINMAX )
    #define NOMINMAX
#endif
#include <nn/nn_Windows.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_Compiler.h>
#include <gfxTool_ShaderCompilerContext.h>
#include <gfxTool_CompileOptionManager.h>

namespace nn {
namespace gfxTool {

void CompilerBase::Compile( CompileOutput* pOutput,
    const ShaderCompilerContext* pContext, const nngfxToolShaderCompilerCompileArg* pArg )
{
    m_CompiledCount = 0;

    auto pCompileOptionCommon = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();
    m_PrintProgressRateInv = 1.0f / pCompileOptionCommon->GetPrintProgressRate();

    m_GroupCount = GetGroupCount( pContext, pArg );

    m_MaxThreads = std::min NN_PREVENT_MACRO_FUNC (
        pCompileOptionCommon->GetMaxThreads(), m_GroupCount );

    PreCompile( pOutput, pContext, pArg );

    if( m_MaxThreads <= 1 )
    {
        for( int idxGroup = 0; idxGroup < m_GroupCount; ++idxGroup )
        {
            CompileGroup( pOutput, pContext, pArg, idxGroup );
            IncrementCompiledCount();
        }
    }
    else
    {
        Custom< std::vector< std::future< void > > >::Type futures;
        futures.reserve( m_MaxThreads );
        for( int idxThread = 0; idxThread < m_MaxThreads; ++idxThread )
        {
            futures.push_back( std::async( std::launch::async, [ &, idxThread ]
            {
                _set_se_translator( []( unsigned int, struct _EXCEPTION_POINTERS* ep )
                {
                    throw ep;
                } );
                for( int idxGroup = idxThread; idxGroup < m_GroupCount; idxGroup += m_MaxThreads )
                {
                    CompileGroup( pOutput, pContext, pArg, idxGroup );
                    IncrementCompiledCount();
                }
            } ) );
        }
        for( auto&& future : futures )
        {
            future.get();
        }
    }

    PostCompile( pOutput, pContext, pArg );
}

void CompilerBase::IncrementCompiledCount()
{
    auto count = ++m_CompiledCount;
    if( NumericCast< int >( count * m_PrintProgressRateInv / m_GroupCount ) >
        NumericCast< int >( ( count - 1 ) * m_PrintProgressRateInv / m_GroupCount ) )
    {
        PrintProgress( count );
    }
}

void CompilerBase::PrintProgress( int count )
{
    NN_GFXTOOL_PRINT( "Compiling... [ %d / %d ]\n", count, m_GroupCount );
}

}
}
