﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/util/util_Compression.h>

#include <nn/gfx/gfx_Enum.h>

#include <gfxTool_Compiler-gl-binary.h>
#include <gfxTool_GlobalWindow.h>
#include <gfxTool_ShaderCompilerContext.h>
#include <gfxTool_CompileOption-glsl.h>
#include <gfxTool_GroupSource.h>
#include <gfxTool_VariationGrouper.h>
#include <gfxTool_CompileOptionManager.h>
#include <gfxTool_ShaderSourceManager.h>
#include <gfxTool_VariationManager.h>
#include <gfxTool_CompileOutput.h>
#include <gfxTool_OptionOutput.h>
#include <gfxTool_BoostPreprocessor.h>

namespace {

struct GlTypeAndGfxType
{
    GLenum glType;
    nngfxToolShaderCompilerReflectionVariableType gfxType;
} s_TypeTable[] =
{
    { GL_BOOL, nngfxToolShaderCompilerReflectionVariableType_Bool32 },
    { GL_INT, nngfxToolShaderCompilerReflectionVariableType_Int32 },
    { GL_UNSIGNED_INT, nngfxToolShaderCompilerReflectionVariableType_Uint32 },
    { GL_FLOAT, nngfxToolShaderCompilerReflectionVariableType_Float32 },
    { GL_DOUBLE, nngfxToolShaderCompilerReflectionVariableType_Float64 },

    { GL_BOOL_VEC2, nngfxToolShaderCompilerReflectionVariableType_Bool32x2 },
    { GL_BOOL_VEC3, nngfxToolShaderCompilerReflectionVariableType_Bool32x3 },
    { GL_BOOL_VEC4, nngfxToolShaderCompilerReflectionVariableType_Bool32x4 },
    { GL_INT_VEC2, nngfxToolShaderCompilerReflectionVariableType_Int32x2 },
    { GL_INT_VEC3, nngfxToolShaderCompilerReflectionVariableType_Int32x3 },
    { GL_INT_VEC4, nngfxToolShaderCompilerReflectionVariableType_Int32x4 },
    { GL_UNSIGNED_INT_VEC2, nngfxToolShaderCompilerReflectionVariableType_Uint32x2 },
    { GL_UNSIGNED_INT_VEC3, nngfxToolShaderCompilerReflectionVariableType_Uint32x3 },
    { GL_UNSIGNED_INT_VEC4, nngfxToolShaderCompilerReflectionVariableType_Uint32x4 },
    { GL_FLOAT_VEC2, nngfxToolShaderCompilerReflectionVariableType_Float32x2 },
    { GL_FLOAT_VEC3, nngfxToolShaderCompilerReflectionVariableType_Float32x3 },
    { GL_FLOAT_VEC4, nngfxToolShaderCompilerReflectionVariableType_Float32x4 },
    { GL_DOUBLE_VEC2, nngfxToolShaderCompilerReflectionVariableType_Float64x2 },
    { GL_DOUBLE_VEC3, nngfxToolShaderCompilerReflectionVariableType_Float64x3 },
    { GL_DOUBLE_VEC4, nngfxToolShaderCompilerReflectionVariableType_Float64x4 },

    { GL_FLOAT_MAT2, nngfxToolShaderCompilerReflectionVariableType_Float32x2x2 },
    { GL_FLOAT_MAT2x3, nngfxToolShaderCompilerReflectionVariableType_Float32x2x3 },
    { GL_FLOAT_MAT2x4, nngfxToolShaderCompilerReflectionVariableType_Float32x2x4 },
    { GL_FLOAT_MAT3x2, nngfxToolShaderCompilerReflectionVariableType_Float32x3x2 },
    { GL_FLOAT_MAT3, nngfxToolShaderCompilerReflectionVariableType_Float32x3x3 },
    { GL_FLOAT_MAT3x4, nngfxToolShaderCompilerReflectionVariableType_Float32x3x4 },
    { GL_FLOAT_MAT4x2, nngfxToolShaderCompilerReflectionVariableType_Float32x4x2 },
    { GL_FLOAT_MAT4x3, nngfxToolShaderCompilerReflectionVariableType_Float32x4x3 },
    { GL_FLOAT_MAT4, nngfxToolShaderCompilerReflectionVariableType_Float32x4x4 },
    { GL_DOUBLE_MAT2, nngfxToolShaderCompilerReflectionVariableType_Float64x2x2 },
    { GL_DOUBLE_MAT2x3, nngfxToolShaderCompilerReflectionVariableType_Float64x2x3 },
    { GL_DOUBLE_MAT2x4, nngfxToolShaderCompilerReflectionVariableType_Float64x2x4 },
    { GL_DOUBLE_MAT3x2, nngfxToolShaderCompilerReflectionVariableType_Float64x3x2 },
    { GL_DOUBLE_MAT3, nngfxToolShaderCompilerReflectionVariableType_Float64x3x3 },
    { GL_DOUBLE_MAT3x4, nngfxToolShaderCompilerReflectionVariableType_Float64x3x4 },
    { GL_DOUBLE_MAT4x2, nngfxToolShaderCompilerReflectionVariableType_Float64x4x2 },
    { GL_DOUBLE_MAT4x3, nngfxToolShaderCompilerReflectionVariableType_Float64x4x3 },
    { GL_DOUBLE_MAT4, nngfxToolShaderCompilerReflectionVariableType_Float64x4x4 },

    { GL_SAMPLER_1D, nngfxToolShaderCompilerReflectionVariableType_Sampler1D },
    { GL_SAMPLER_2D, nngfxToolShaderCompilerReflectionVariableType_Sampler2D },
    { GL_SAMPLER_3D, nngfxToolShaderCompilerReflectionVariableType_Sampler3D },
    { GL_SAMPLER_CUBE, nngfxToolShaderCompilerReflectionVariableType_SamplerCube },
    { GL_SAMPLER_1D_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler1DShadow },
    { GL_SAMPLER_2D_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler2DShadow },
    { GL_SAMPLER_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Sampler1DArray },
    { GL_SAMPLER_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Sampler2DArray },
    { GL_SAMPLER_1D_ARRAY_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler1DArrayShadow },
    { GL_SAMPLER_2D_ARRAY_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler2DArrayShadow },
    { GL_SAMPLER_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_Sampler2DMultisample },
    { GL_SAMPLER_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Sampler2DMultisampleArray },
    { GL_SAMPLER_CUBE_SHADOW, nngfxToolShaderCompilerReflectionVariableType_SamplerCubeShadow },
    { GL_SAMPLER_BUFFER, nngfxToolShaderCompilerReflectionVariableType_SamplerBuffer },
    { GL_SAMPLER_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_Sampler2DRect },
    { GL_SAMPLER_2D_RECT_SHADOW, nngfxToolShaderCompilerReflectionVariableType_Sampler2DRectShadow },
    { GL_INT_SAMPLER_1D, nngfxToolShaderCompilerReflectionVariableType_IntSampler1D },
    { GL_INT_SAMPLER_2D, nngfxToolShaderCompilerReflectionVariableType_IntSampler2D },
    { GL_INT_SAMPLER_3D, nngfxToolShaderCompilerReflectionVariableType_IntSampler3D },
    { GL_INT_SAMPLER_CUBE, nngfxToolShaderCompilerReflectionVariableType_IntSamplerCube },
    { GL_INT_SAMPLER_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntSampler1DArray },
    { GL_INT_SAMPLER_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntSampler2DArray },
    { GL_INT_SAMPLER_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_IntSampler2DMultisample },
    { GL_INT_SAMPLER_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntSampler2DMultisampleArray },
    { GL_INT_SAMPLER_BUFFER, nngfxToolShaderCompilerReflectionVariableType_IntSamplerBuffer },
    { GL_INT_SAMPLER_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_IntSampler2DRect },
    { GL_UNSIGNED_INT_SAMPLER_1D, nngfxToolShaderCompilerReflectionVariableType_UintSampler1D },
    { GL_UNSIGNED_INT_SAMPLER_2D, nngfxToolShaderCompilerReflectionVariableType_UintSampler2D },
    { GL_UNSIGNED_INT_SAMPLER_3D, nngfxToolShaderCompilerReflectionVariableType_UintSampler3D },
    { GL_UNSIGNED_INT_SAMPLER_CUBE, nngfxToolShaderCompilerReflectionVariableType_UintSamplerCube },
    { GL_UNSIGNED_INT_SAMPLER_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintSampler1DArray },
    { GL_UNSIGNED_INT_SAMPLER_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintSampler2DArray },
    { GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_UintSampler2DMultisample },
    { GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintSampler2DMultisampleArray },
    { GL_UNSIGNED_INT_SAMPLER_BUFFER, nngfxToolShaderCompilerReflectionVariableType_UintSamplerBuffer },
    { GL_UNSIGNED_INT_SAMPLER_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_UintSampler2DRect },
    { GL_SAMPLER_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_SamplerCubeMapArray },
    { GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW, nngfxToolShaderCompilerReflectionVariableType_SamplerCubeMapArrayShadow },
    { GL_INT_SAMPLER_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntSamplerCubeMapArray },
    { GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintSamplerCubeMapArray },

    { GL_IMAGE_1D, nngfxToolShaderCompilerReflectionVariableType_Image1D },
    { GL_IMAGE_2D, nngfxToolShaderCompilerReflectionVariableType_Image2D },
    { GL_IMAGE_3D, nngfxToolShaderCompilerReflectionVariableType_Image3D },
    { GL_IMAGE_CUBE, nngfxToolShaderCompilerReflectionVariableType_ImageCube },
    { GL_IMAGE_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Image1DArray },
    { GL_IMAGE_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Image2DArray },
    { GL_IMAGE_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_Image2DMultisample },
    { GL_IMAGE_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_Image2DMultisampleArray },
    { GL_IMAGE_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_ImageCubeMapArray },
    { GL_IMAGE_BUFFER, nngfxToolShaderCompilerReflectionVariableType_ImageBuffer },
    { GL_IMAGE_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_Image2DRect },
    { GL_INT_IMAGE_1D, nngfxToolShaderCompilerReflectionVariableType_IntImage1D },
    { GL_INT_IMAGE_2D, nngfxToolShaderCompilerReflectionVariableType_IntImage2D },
    { GL_INT_IMAGE_3D, nngfxToolShaderCompilerReflectionVariableType_IntImage3D },
    { GL_INT_IMAGE_CUBE, nngfxToolShaderCompilerReflectionVariableType_IntImageCube },
    { GL_INT_IMAGE_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntImage1DArray },
    { GL_INT_IMAGE_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntImage2DArray },
    { GL_INT_IMAGE_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_IntImage2DMultisample },
    { GL_INT_IMAGE_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntImage2DMultisampleArray },
    { GL_INT_IMAGE_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_IntImageCubeMapArray },
    { GL_INT_IMAGE_BUFFER, nngfxToolShaderCompilerReflectionVariableType_IntImageBuffer },
    { GL_INT_IMAGE_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_IntImage2DRect },
    { GL_UNSIGNED_INT_IMAGE_1D, nngfxToolShaderCompilerReflectionVariableType_UintImage1D },
    { GL_UNSIGNED_INT_IMAGE_2D, nngfxToolShaderCompilerReflectionVariableType_UintImage2D },
    { GL_UNSIGNED_INT_IMAGE_3D, nngfxToolShaderCompilerReflectionVariableType_UintImage3D },
    { GL_UNSIGNED_INT_IMAGE_CUBE, nngfxToolShaderCompilerReflectionVariableType_UintImageCube },
    { GL_UNSIGNED_INT_IMAGE_1D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintImage1DArray },
    { GL_UNSIGNED_INT_IMAGE_2D_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintImage2DArray },
    { GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE, nngfxToolShaderCompilerReflectionVariableType_UintImage2DMultisample },
    { GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintImage2DMultisampleArray },
    { GL_UNSIGNED_INT_IMAGE_CUBE_MAP_ARRAY, nngfxToolShaderCompilerReflectionVariableType_UintImageCubeMapArray },
    { GL_UNSIGNED_INT_IMAGE_BUFFER, nngfxToolShaderCompilerReflectionVariableType_UintImageBuffer },
    { GL_UNSIGNED_INT_IMAGE_2D_RECT, nngfxToolShaderCompilerReflectionVariableType_UintImage2DRect }
};

nngfxToolShaderCompilerReflectionVariableType GlTypeToGfxType( GLenum glType )
{
    auto pEnd = s_TypeTable + sizeof( s_TypeTable ) / sizeof( *s_TypeTable );
    auto found = std::find_if( s_TypeTable, pEnd, [ glType ]( const decltype( *s_TypeTable ) element )
    {
        return element.glType == glType;
    } );
    return found == pEnd ? nngfxToolShaderCompilerReflectionVariableType_Unknown : found->gfxType;
}

}

namespace nn {
namespace gfxTool {

typedef CompilerVariation< nngfxToolShaderCompilerLowLevelApiType_Gl,
    nngfxToolShaderCompilerCodeType_Binary > Target;

void Compiler< Target >::PreCompile( CompileOutput*,
    const ShaderCompilerContext*, const nngfxToolShaderCompilerCompileArg* )
{
    if( !GlobalWindow::GetInstance().IsInitialized() )
    {
        GlobalWindow::GetInstance().Initialize();
    }

    auto threadCount = GetThreadCount();
    m_GlContexts.reset( new GlContext[ threadCount ] );
    std::for_each( m_GlContexts.get(), m_GlContexts.get() + threadCount, []( GlContext& context ) {
        if( !context.IsInitialized() )
        {
            context.Initialize( GlobalWindow::GetInstance().GetDc(), nullptr );
        }
    } );

    m_GlContexts[ 0 ].MakeCurrent();
    if( !m_GlFunction.IsInitialized() )
    {
        m_GlFunction.Initialize();
    }
    ::wglMakeCurrent( nullptr, nullptr );
}

void Compiler< Target >::CompileGroup( CompileOutput* pOutput, const ShaderCompilerContext* pContext,
    const nngfxToolShaderCompilerCompileArg* pArg, int idxGroup )
{
    auto idxVariation = idxGroup;
    auto tid = GetThreadIndex( idxGroup );
    // MakeCurrent は遅いのでそれぞれのスレッドで一度だけ行う
    // ただし idxGroup == tid の条件式は作業の割り振られ方に依存しているので注意
    if( idxGroup == tid )
    {
        std::lock_guard< decltype( m_Mutex ) > lock( m_Mutex );
        auto& context = m_GlContexts[ tid ];
        if( !context.IsInitialized() )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_FailedToCreateGlContext );
        }
        context.MakeCurrent();
    }

    auto pCommonOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();
    auto pProgramOutput = pOutput->GetVariationOutput( idxVariation )->GetBinaryOutput();

    pProgramOutput->GetInfo()->flags.SetBit( nn::gfx::ShaderInfoData::Flag_SeparationEnable,
        pCommonOption->IsSeparationEnabled() );
    pProgramOutput->GetInfo()->codeType = StaticCastAuto( nn::gfx::ShaderCodeType_Binary );
    pProgramOutput->GetInfo()->sourceFormat = 0;

    GLuint hProgram = 0;
    if( !pCommonOption->IsSeparationEnabled() )
    {
        hProgram = m_GlFunction.CreateProgram();
        m_GlFunction.ProgramParameteri( hProgram, GL_PROGRAM_BINARY_RETRIEVABLE_HINT, GL_TRUE );
    }

    std::shared_ptr< OptionOutputProgramCommon > pOptionOutputProgramCommon(
        new OptionOutputProgramCommon() );
    pOptionOutputProgramCommon->Initialize( pArg );
    pProgramOutput->AddOptionOutput(
        nngfxToolShaderCompilerOptionOutputType_ProgramCommon, pOptionOutputProgramCommon );

    StageIntermediateResult stageResults[ static_cast< int >( ShaderStage::End ) ];
    if( !pCommonOption->IsSeparationEnabled() )
    {
        stageResults[ 0 ].pInfoLog.reset( new nn::gfxTool::Custom< std::string >::Type() );
    }
    std::shared_ptr< OptionOutputReflection > pReflection;
    if( pCommonOption->IsReflectionEnabled() )
    {
        pReflection.reset( new OptionOutputReflection() );
    }
    CreateSources( stageResults, pContext, pArg, idxVariation );
    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
    {
        auto& stageResult = stageResults[ idxStage ];
        stageResult.hShader = 0;
        auto stage = static_cast< ShaderStage >( idxStage );
        if( GetStageSource( pArg, stage ).pValue == nullptr )
        {
            continue;
        }

        stageResult.pInfoLog.reset( new nn::gfxTool::Custom< std::string >::Type() );

        if( pCommonOption->IsSeparationEnabled() )
        {
            auto headVariation = pContext->GetVariationManager()->GetVariationGroup(
                stage )->VariationToHeadVaritaion( idxVariation );
            if( headVariation == idxVariation )
            {
                CompileStage( &stageResult, pContext, pArg, stage, idxVariation );
                hProgram = m_GlFunction.CreateProgram();
                m_GlFunction.ProgramParameteri( hProgram, GL_PROGRAM_SEPARABLE, GL_TRUE );
                m_GlFunction.ProgramParameteri( hProgram, GL_PROGRAM_BINARY_RETRIEVABLE_HINT, GL_TRUE );
                m_GlFunction.AttachShader( hProgram, stageResult.hShader );
                m_GlFunction.LinkProgram( hProgram );
                m_GlFunction.DetachShader( hProgram, stageResult.hShader );
                m_GlFunction.DeleteShader( stageResult.hShader );
                auto infoLog = GetProgramInfoLog( hProgram );
                if( infoLog.size() > 0 )
                {
                    stageResult.pInfoLog->append( &infoLog[ 0 ], infoLog.size() );
                }
                if( !CheckProgramStatus( hProgram ) )
                {
                    // TODO
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "%s",
                        CreateErrorSource( stage, &stageResult, pCommonOption->GetCodePage() ).c_str() );
                }

                RetrieveOutput( pProgramOutput, hProgram, stage, pCommonOption->IsCompressionEnabled() );
                if( pCommonOption->IsReflectionEnabled() )
                {
                    RetrieveReflection( pReflection.get(), hProgram, stage == ShaderStage::Compute );
                }
                m_GlFunction.DeleteProgram( hProgram );
            }
        }
        else
        {
            CompileStage( &stageResult, pContext, pArg, stage, idxVariation );
            m_GlFunction.AttachShader( hProgram, stageResult.hShader );
        }

        if( pCommonOption->IsDumpEnabled() )
        {
            DumpShaderSource( pProgramOutput, &stageResult, stage );
        }
    }

    if( !pCommonOption->IsSeparationEnabled() )
    {
        m_GlFunction.LinkProgram( hProgram );
        for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
        {
            if( stageResults[ idxStage ].hShader )
            {
                m_GlFunction.DetachShader( hProgram, stageResults[ idxStage ].hShader );
                m_GlFunction.DeleteShader( stageResults[ idxStage ].hShader );
            }
        }
        auto infoLog = GetProgramInfoLog( hProgram );
        if( !CheckProgramStatus( hProgram ) )
        {
            // TODO
            Custom< std::string >::Type source;
            source.append( "\n[Variation: " ).append( LexicalCast< Custom<
                std::string >::Type >( idxVariation ) ).append( "]\n" );
            for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
            {
                auto stage = static_cast< nn::gfxTool::ShaderStage >( idxStage );
                if( GetStageSource( pArg, stage ).pValue )
                {
                    source.append( CreateErrorSource( stage,
                        stageResults + idxStage, pCommonOption->GetCodePage() ) );
                }
            }
            source.append( "\n----Link----\n" );
            if( infoLog.size() > 0 )
            {
                source.append( &infoLog[ 0 ], infoLog.size() );
            }
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "%s", source.c_str() );
        }
        if( infoLog.size() > 0 )
        {
            stageResults[ 0 ].pInfoLog->append( &infoLog[ 0 ], infoLog.size() );
        }
        RetrieveOutput( pProgramOutput, hProgram,
            ShaderStage::Vertex, pCommonOption->IsCompressionEnabled() );
        if( pCommonOption->IsReflectionEnabled() )
        {
            RetrieveReflection( pReflection.get(), hProgram,
                GetStageSource( pArg, ShaderStage::Compute ).pValue != nullptr );
        }
        m_GlFunction.DeleteProgram( hProgram );
    }

    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
    {
        pOptionOutputProgramCommon->GetOptionOutputStageCommon(
            static_cast< ShaderStage >( idxStage ) )->SetInfoLog( stageResults[ idxStage ].pInfoLog );
    }

    if( pCommonOption->IsReflectionEnabled() )
    {
        pOptionOutputProgramCommon->SetReflection( pReflection );
    }

    // 作業の割り振られ方に依存しているので注意
    if( idxGroup + GetThreadCount() >= static_cast< int >( pArg->variationCount ) )
    {
        std::lock_guard< decltype( m_Mutex ) > lock( m_Mutex );
        ::wglMakeCurrent( nullptr, nullptr );
    }
} // NOLINT

void Compiler< Target >::PostCompile( CompileOutput* pOutput,
    const ShaderCompilerContext* pContext, const nngfxToolShaderCompilerCompileArg* pArg )
{
    auto pCommonOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();
    if( !pCommonOption->IsSeparationEnabled() )
    {
        // TODO
        return;
    }

    for( int idxVariation = 0, variationCount = NumericCastAuto( pArg->variationCount );
        idxVariation < variationCount; ++idxVariation )
    {
        auto* pProgramOutput = pOutput->GetVariationOutput( idxVariation )->GetBinaryOutput();
        for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
        {
            auto stage = static_cast< ShaderStage >( idxStage );
            if( GetStageSource( pArg, stage ).pValue == nullptr )
            {
                continue;
            }

            auto headVariation = pContext->GetVariationManager()->GetVariationGroup(
                stage )->VariationToHeadVaritaion( idxVariation );
            if( headVariation != idxVariation )
            {
                auto* pHeadOutput = pOutput->GetVariationOutput( headVariation )->GetBinaryOutput();
                *GetStageCodePtr( pProgramOutput->GetInfo(), stage ) =
                    *GetStageCodePtr( pHeadOutput->GetInfo(), stage );

                auto pStageOutput = static_cast< OptionOutputProgramCommon* >(
                    pProgramOutput->GetOptionOutput( nngfxToolShaderCompilerOptionOutputType_ProgramCommon )
                    )->GetOptionOutputStageCommon( stage );
                auto pHeadStageOutput = static_cast< OptionOutputProgramCommon* >(
                    pProgramOutput->GetOptionOutput( nngfxToolShaderCompilerOptionOutputType_ProgramCommon )
                    )->GetOptionOutputStageCommon( stage );
                *pStageOutput->GetOutput() = *pHeadStageOutput->GetOutput();
            }
        }
    }
}

void Compiler< Target >::CreateSources( StageIntermediateResult* pResults,
    const ShaderCompilerContext* pContext, const nngfxToolShaderCompilerCompileArg* pArg,
    int idxVariation )
{
    auto pCommonOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();
    auto pGlslOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Glsl >();

    BoostPreprocessor preprocessors[ static_cast<int>( ShaderStage::End ) ];
    for( int idxStage = 0; idxStage < static_cast<int>( ShaderStage::End ); ++idxStage )
    {
        auto stage = static_cast<ShaderStage>( idxStage );

        auto& stageSource = GetStageSource( pArg, stage );
        if( stageSource.pValue == nullptr )
        {
            continue;
        }

        auto& result = pResults[ idxStage ];

        auto variationConstantGroup = pContext->GetVariationManager(
            )->GetVariationConstantGroup( stage )->GetVariationToGroupTable().at( idxVariation );
        auto& variationConstantSource = pContext->GetVariationManager(
            )->GetVariationConstantSource( stage )->GetSources().at( variationConstantGroup );
        auto preprocessorDefinitionGroup = pContext->GetVariationManager(
            )->GetPreprocessorDefinitionGroup( stage )->GetVariationToGroupTable().at( idxVariation );
        auto& preprocessorDefinitionSource = pContext->GetVariationManager(
            )->GetPreprocessorDefinitionSource( stage )->GetSources().at( preprocessorDefinitionGroup );

        auto pShaderSource = pContext->GetShaderSourceManager(
            )->GetShaderSource( StaticCastAuto( stage ) );

        if( pCommonOption->IsPreprocessEnabled() )
        {
            Custom< std::string >::Type source;
            source.append( pGlslOption->GetGlslHeader()->c_str() ).append(
                pCommonOption->GetPreprocessorDefinitionSource()->c_str() ).append(
                preprocessorDefinitionSource.c_str() ).append(
                variationConstantSource.c_str() ).append(
                stageSource.pValue, stageSource.length );

            auto& preprocessor = preprocessors[ idxStage ];
            preprocessor.SetReadFileCallback( pArg->pReadIncludeFileCallback,
                pArg->pReadIncludeFileCallbackParam );
            if( pArg->pVariationDefinitionArg )
            {
                preprocessor.SetVariationDefinition( GetStageVariationDefinition(
                    pArg->pVariationDefinitionArg, stage ) );
            }
            preprocessor.SetVariationConstantEmulationEnabled( true );
            preprocessor.SetUniformRegisterToBlockEnabled(
                !pCommonOption->GetUniformRegisterBlockName().empty() );
            preprocessor.Preprocess( source.c_str(), source.length() );
        }
        else
        {
            const int sourceCount = 7;
            result.sources.reserve( sourceCount );
            result.sources.push_back( pGlslOption->GetGlslHeader()->c_str() );
            result.sources.push_back( pCommonOption->GetPreprocessorDefinitionSource()->c_str() );
            result.sources.push_back( preprocessorDefinitionSource.c_str() );
            result.sources.push_back( variationConstantSource.c_str() );
            result.sources.push_back( pShaderSource->beforeVariationBufferView.data() );
            result.sources.push_back( pShaderSource->variationBufferSource.c_str() );
            result.sources.push_back( pShaderSource->afterVariationBufferView.data() );

            result.lengths.reserve( sourceCount );
            result.lengths.push_back( NumericCastAuto( pGlslOption->GetGlslHeader()->length() ) );
            result.lengths.push_back( NumericCastAuto( pCommonOption->GetPreprocessorDefinitionSource()->length() ) );
            result.lengths.push_back( NumericCastAuto( preprocessorDefinitionSource.length() ) );
            result.lengths.push_back( NumericCastAuto( variationConstantSource.length() ) );
            result.lengths.push_back( NumericCastAuto( pShaderSource->beforeVariationBufferView.length() ) );
            result.lengths.push_back( NumericCastAuto( pShaderSource->variationBufferSource.length() ) );
            result.lengths.push_back( NumericCastAuto( pShaderSource->afterVariationBufferView.length() ) );
        }
    }

    if( pCommonOption->IsPreprocessEnabled() )
    {
        BoostPreprocessor::ResolveUniformRegisterBlock( static_cast<int>( ShaderStage::End ),
            preprocessors, pCommonOption->GetUniformRegisterBlockName().c_str() );
        for( int idxStage = 0; idxStage < static_cast<int>( ShaderStage::End ); ++idxStage )
        {
            if( GetStageSource( pArg, static_cast< ShaderStage >( idxStage ) ).pValue )
            {
                auto& result = pResults[ idxStage ];
                result.preprocessedSource = preprocessors[ idxStage ].GetResult().str();
                result.sources.push_back( result.preprocessedSource.c_str() );
                result.lengths.push_back( NumericCastAuto( result.preprocessedSource.length() ) );
            }
        }
    }
}

void Compiler< Target >::CompileStage( StageIntermediateResult* pResult,
    const ShaderCompilerContext* pContext, const nngfxToolShaderCompilerCompileArg*,
    ShaderStage stage, int idxVariation )
{
    auto pCommonOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();

    static const GLenum s_StageTable[] =
    {
        GL_VERTEX_SHADER,
        GL_TESS_CONTROL_SHADER,
        GL_TESS_EVALUATION_SHADER,
        GL_GEOMETRY_SHADER,
        GL_FRAGMENT_SHADER,
        GL_COMPUTE_SHADER
    };

    pResult->hShader = m_GlFunction.CreateShader( s_StageTable[ static_cast< int >( stage ) ] );
    m_GlFunction.ShaderSource( pResult->hShader, StaticCastAuto( pResult->sources.size() ),
        &pResult->sources[ 0 ], &pResult->lengths[ 0 ] );
    m_GlFunction.CompileShader( pResult->hShader );

    auto infoLog = GetShaderInfoLog( pResult->hShader );
    if( infoLog.size() > 0 )
    {
        pResult->pInfoLog->append( &infoLog[ 0 ], infoLog.size() );
    }

    if( !CheckShaderStatus( pResult->hShader ) )
    {
        // TODO
        Custom< std::string >::Type source;
        for( int idxSource = 0, sourceLength = static_cast< int >( pResult->sources.size() );
            idxSource < sourceLength; ++idxSource )
        {
            source.append( pResult->sources[ idxSource ], pResult->lengths[ idxSource ] );
        }
        auto result = ConvertEncoding( AddLineNumber( nn::util::string_view(
            source.c_str(), source.length() ) ), pCommonOption->GetCodePage(), 0 );
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError,
            "\n[Variation: %d]\n%s\n----Error Log----\n%s",
            idxVariation, result.c_str(), pResult->pInfoLog->c_str() );
    }
}

Custom< std::string >::Type Compiler< Target >::CreateErrorSource(
    ShaderStage stage, const StageIntermediateResult* pResult, int codePage )
{
    if( pResult == nullptr )
    {
        return Custom< std::string >::Type();
    }

    static const char* s_StageNames[] =
    {
        "----Vertex Shader----",
        "----Hull Shader----",
        "----Domain Shader----",
        "----Geometry Shader----",
        "----Pixel Shader----",
        "----Compute Shader----"
    };

    Custom< std::string >::Type ret;
    ret.append( s_StageNames[ static_cast< int >( stage ) ] ).append( "\n" );
    Custom< std::string >::Type source;
    for( int idxSource = 0; idxSource < static_cast< int >( pResult->sources.size() ); ++idxSource )
    {
        source.append( pResult->sources[ idxSource ], pResult->lengths[ idxSource ] );
    }
    ret.append( ConvertEncoding( AddLineNumber( nn::util::string_view(
        source.c_str(), source.length() ) ), codePage, 0 ) );
    ret.append( "\n" ).append( *pResult->pInfoLog );
    return std::move( ret );
}

bool Compiler< Target >::CheckShaderStatus( GLuint hShader )
{
    GLint status = 0;
    m_GlFunction.GetShaderiv( hShader, GL_COMPILE_STATUS, &status );
    return status == GL_TRUE;
}

bool Compiler< Target >::CheckProgramStatus( GLuint hProgram )
{
    GLint status = 0;
    m_GlFunction.GetProgramiv( hProgram, GL_LINK_STATUS, &status );
    return status == GL_TRUE;
}

Custom< std::vector< char > >::Type Compiler< Target >::GetShaderInfoLog( GLuint hShader )
{
    Custom< std::vector< char > >::Type ret;
    GLint infoLength = 0;
    m_GlFunction.GetShaderiv( hShader, GL_INFO_LOG_LENGTH, &infoLength );
    if( infoLength > 1 )
    {
        ret.resize( infoLength );
        m_GlFunction.GetShaderInfoLog( hShader, infoLength, nullptr, &ret[ 0 ] );
    }

    return std::move( ret );
}

Custom< std::vector< char > >::Type Compiler< Target >::GetProgramInfoLog( GLuint hProgram )
{
    Custom< std::vector< char > >::Type ret;
    GLint infoLength = 0;
    m_GlFunction.GetProgramiv( hProgram, GL_INFO_LOG_LENGTH, &infoLength );
    if( infoLength > 1 )
    {
        ret.resize( infoLength );
        m_GlFunction.GetProgramInfoLog( hProgram, infoLength, nullptr, &ret[ 0 ] );
    }

    return std::move( ret );
}

void Compiler< Target >::RetrieveOutput( ProgramOutput* pOutput,
    GLuint hProgram, ShaderStage stage, bool compress )
{
    GLint binarySize = 0;
    m_GlFunction.GetProgramiv( hProgram, GL_PROGRAM_BINARY_LENGTH, &binarySize );
    std::shared_ptr< char > pBinary( new char[ binarySize ], std::default_delete< char[] >() );
    GLsizei binaryLength = 0;
    GLenum binaryFormat = 0;
    m_GlFunction.GetProgramBinary( hProgram, binarySize, &binaryLength, &binaryFormat, pBinary.get() );
    size_t decompressedCodeSize = 0;
    if( compress )
    {
        decompressedCodeSize = NumericCastAuto( binaryLength );
        Custom< std::unique_ptr< char[] > >::Type pTmpCompressdBinary( new char[ binarySize ] );
        Custom< std::unique_ptr< char[] > >::Type pWorkBuffer(
            new char[ nn::util::CompressZlibWorkBufferSizeDefault ] );
        size_t compressedSize;
        if( !nn::util::CompressZlib( &compressedSize, pTmpCompressdBinary.get(), binarySize,
            pBinary.get(), binarySize, pWorkBuffer.get(), nn::util::CompressZlibWorkBufferSizeDefault ) )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_FailedToCompress );
        }
        binaryLength = NumericCastAuto( compressedSize );
        pBinary.reset( new char[ compressedSize ], std::default_delete< char[] >() );
        memcpy( pBinary.get(), pTmpCompressdBinary.get(), compressedSize );
    }
    pOutput->GetInfo()->binaryFormat = NumericCastAuto( binaryFormat );
    pOutput->SetShaderCode( stage, pBinary,
        NumericCastAuto( binaryLength ), NumericCastAuto( decompressedCodeSize ) );
}

void Compiler< Target >::DumpShaderSource( ProgramOutput* pOutput,
    const StageIntermediateResult* pResult, ShaderStage stage )
{
    if( pResult == nullptr || pResult->sources.size() == 0 )
    {
        return;
    }

    std::shared_ptr< Custom< std::string >::Type > pDump( new Custom< std::string >::Type() );
    for( int idxSource = 0; idxSource < static_cast< int >( pResult->sources.size() ); ++idxSource )
    {
        pDump.get()->append( pResult->sources[ idxSource ], pResult->lengths[ idxSource ] );
    }
    auto pOptionOutputCommonProgram = static_cast< OptionOutputProgramCommon* >(
        pOutput->GetOptionOutput( nngfxToolShaderCompilerOptionOutputType_ProgramCommon ) );
    pOptionOutputCommonProgram->GetOptionOutputStageCommon( stage )->SetDump( pDump );
}

void Compiler< Target >::RetrieveReflection( OptionOutputReflection* pOutput,
    GLuint hProgram, bool isComputeShader )
{
    static const int s_StageReferences[] =
    {
        nngfxToolShaderCompilerReflectionStageReference_Vertex,
        nngfxToolShaderCompilerReflectionStageReference_Hull,
        nngfxToolShaderCompilerReflectionStageReference_Domain,
        nngfxToolShaderCompilerReflectionStageReference_Geometry,
        nngfxToolShaderCompilerReflectionStageReference_Pixel,
        nngfxToolShaderCompilerReflectionStageReference_Compute
    };

    static int32_t nngfxToolShaderCompilerShaderSlot::* const s_pShaderSlotStages[] =
    {
        &nngfxToolShaderCompilerShaderSlot::vertexShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::hullShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::domainShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::geometryShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::pixelShaderSlot,
        &nngfxToolShaderCompilerShaderSlot::computeShaderSlot
    };

    Custom< std::vector< char > >::Type nameBuffer;
    nameBuffer.resize( 1024 );

    auto RetrieveName = [ & ]( nngfxToolString* pDst, GLenum interfaceType, int indexResource )
    {
        GLsizei nameLength;
        m_GlFunction.GetProgramResourceName( hProgram, interfaceType,
            indexResource, StaticCastAuto( nameBuffer.size() ), &nameLength, &nameBuffer[ 0 ] );
        pDst->pValue = &nameBuffer[ 0 ];
        pDst->length = NumericCastAuto( nameLength );
    };

    auto RetrieveStageReferences = [ & ]( int32_t* pDst, GLenum interfaceType, int indexResource )
    {
        const GLenum props[ static_cast< int >( ShaderStage::End ) ] = {
            GL_REFERENCED_BY_VERTEX_SHADER, GL_REFERENCED_BY_TESS_CONTROL_SHADER,
            GL_REFERENCED_BY_TESS_EVALUATION_SHADER, GL_REFERENCED_BY_GEOMETRY_SHADER,
            GL_REFERENCED_BY_FRAGMENT_SHADER, GL_REFERENCED_BY_COMPUTE_SHADER };
        GLint values[ static_cast< int >( ShaderStage::End ) ];
        m_GlFunction.GetProgramResourceiv( hProgram, interfaceType, indexResource,
            StaticCastAuto( ShaderStage::End ), props, StaticCastAuto( ShaderStage::End ), nullptr, values );
        *pDst = 0;
        for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
        {
            *pDst |= values[ idxStage ] ? s_StageReferences[ idxStage ] : 0;
        }
    };

    auto ForEachResource = [ & ]( GLenum interfaceType, std::function< void ( int idx ) > func )
    {
        GLint count = 0;
        GLint maxNameLength = 0;
        m_GlFunction.GetProgramInterfaceiv( hProgram, interfaceType, GL_ACTIVE_RESOURCES, &count );
        m_GlFunction.GetProgramInterfaceiv( hProgram, interfaceType, GL_MAX_NAME_LENGTH, &maxNameLength );
        if( nameBuffer.size() < NumericCast< decltype( nameBuffer.size() ) >( maxNameLength ) )
        {
            nameBuffer.resize( maxNameLength );
        }
        for( int idx = 0; idx < count; ++idx )
        {
            func( idx );
        }
    };

    auto SetShaderSlot = [ & ]( nngfxToolShaderCompilerShaderSlot* pDst, int32_t& stages, GLint value )
    {
        for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
        {
            pDst->*s_pShaderSlotStages[ idxStage ] = ( stages & s_StageReferences[ idxStage ] ) ? value : -1;
        }
    };

    auto RetrieveConstantBuffer = [ & ]( int idx )
    {
        const GLenum props[] = { GL_BUFFER_BINDING, GL_BUFFER_DATA_SIZE, GL_NUM_ACTIVE_VARIABLES };
        GLint values[ 3 ];
        m_GlFunction.GetProgramResourceiv( hProgram, GL_UNIFORM_BLOCK, idx, 3, props, 3, nullptr, values );
        nngfxToolShaderCompilerConstantBuffer constantBuffer = {};
        constantBuffer.size = values[ 1 ];
        constantBuffer.activeVariableCount = values[ 2 ];
        RetrieveName( &constantBuffer.name, GL_UNIFORM_BLOCK, idx );
        RetrieveStageReferences( &constantBuffer.stages, GL_UNIFORM_BLOCK, idx );
        SetShaderSlot( &constantBuffer.shaderSlot, constantBuffer.stages, values[ 0 ] );
        pOutput->AddReflection( constantBuffer );
    };

    auto RetrieveUnorderedAccessBuffer = [ & ]( int idx )
    {
        const GLenum props[] = { GL_BUFFER_BINDING, GL_BUFFER_DATA_SIZE, GL_NUM_ACTIVE_VARIABLES };
        GLint values[ 3 ];
        m_GlFunction.GetProgramResourceiv( hProgram, GL_SHADER_STORAGE_BLOCK, idx, 3, props, 3, nullptr, values );
        nngfxToolShaderCompilerUnorderedAccessBuffer unorderedAccessBuffer = {};
        unorderedAccessBuffer.size = values[ 1 ];
        unorderedAccessBuffer.activeVariableCount = values[ 2 ];
        RetrieveName( &unorderedAccessBuffer.name, GL_SHADER_STORAGE_BLOCK, idx );
        RetrieveStageReferences( &unorderedAccessBuffer.stages, GL_SHADER_STORAGE_BLOCK, idx );
        SetShaderSlot( &unorderedAccessBuffer.shaderSlot, unorderedAccessBuffer.stages, values[ 0 ] );
        pOutput->AddReflection( unorderedAccessBuffer );
    };

    auto RetrieveShaderInput = [ & ]( int idx )
    {
        const GLenum props[] = { GL_TYPE, GL_LOCATION, GL_ARRAY_SIZE };
        GLint values[ 3 ];
        m_GlFunction.GetProgramResourceiv( hProgram, GL_PROGRAM_INPUT, idx, 3, props, 3, nullptr, values );
        nngfxToolShaderCompilerShaderInput shaderInput = {};
        shaderInput.type = GlTypeToGfxType( values[ 0 ] );
        shaderInput.shaderSlot = values[ 1 ];
        shaderInput.arrayCount = values[ 2 ];
        RetrieveName( &shaderInput.name, GL_PROGRAM_INPUT, idx );
        RetrieveStageReferences( &shaderInput.stages, GL_PROGRAM_INPUT, idx );
        pOutput->AddReflection( shaderInput );
    };

    auto RetrieveShaderOutput = [ & ]( int idx )
    {
        const GLenum props[] = { GL_TYPE, GL_LOCATION, GL_ARRAY_SIZE };
        GLint values[ 3 ];
        m_GlFunction.GetProgramResourceiv( hProgram, GL_PROGRAM_OUTPUT, idx, 3, props, 3, nullptr, values );
        nngfxToolShaderCompilerShaderOutput shaderOutput = {};
        shaderOutput.type = GlTypeToGfxType( values[ 0 ] );
        shaderOutput.shaderSlot = values[ 1 ];
        shaderOutput.arrayCount = values[ 2 ];
        RetrieveName( &shaderOutput.name, GL_PROGRAM_OUTPUT, idx );
        RetrieveStageReferences( &shaderOutput.stages, GL_PROGRAM_OUTPUT, idx );
        pOutput->AddReflection( shaderOutput );
    };

    auto RetrieveSamplerAndImageAndConstantBufferVariable = [ & ]( int idx )
    {
        const GLenum props[] = { GL_TYPE, GL_LOCATION, GL_BLOCK_INDEX, GL_OFFSET, GL_ARRAY_SIZE };
        GLint values[ 5 ];
        m_GlFunction.GetProgramResourceiv( hProgram, GL_UNIFORM, idx, 5, props, 5, nullptr, values );
        auto type = GlTypeToGfxType( values[ 0 ] );
        if( IsSamplerType( type ) )
        {
            nngfxToolShaderCompilerSampler sampler = {};
            sampler.type = type;
            RetrieveName( &sampler.name, GL_UNIFORM, idx );
            RetrieveStageReferences( &sampler.stages, GL_UNIFORM, idx );
            GLint binding;
            m_GlFunction.GetUniformiv( hProgram, values[ 1 ], &binding );
            SetShaderSlot( &sampler.shaderSlot, sampler.stages, binding );
            sampler.arrayCount = values[ 4 ];
            pOutput->AddReflection( sampler );
        }
        else if( IsImageType( type ) )
        {
            nngfxToolShaderCompilerImage image = {};
            image.type = type;
            RetrieveName( &image.name, GL_UNIFORM, idx );
            RetrieveStageReferences( &image.stages, GL_UNIFORM, idx );
            GLint binding;
            m_GlFunction.GetUniformiv( hProgram, values[ 1 ], &binding );
            SetShaderSlot( &image.shaderSlot, image.stages, binding );
            image.arrayCount = values[ 4 ];
            pOutput->AddReflection( image );
        }
        else
        {
            nngfxToolShaderCompilerConstantBufferVariable constantBufferVariable = {};
            constantBufferVariable.type = type;
            constantBufferVariable.blockIndex = values[ 2 ];
            constantBufferVariable.offset = values[ 3 ];
            constantBufferVariable.arrayCount = values[ 4 ];
            RetrieveName( &constantBufferVariable.name, GL_UNIFORM, idx );
            RetrieveStageReferences( &constantBufferVariable.stages, GL_UNIFORM, idx );
            pOutput->AddReflection( constantBufferVariable );
        }
    };

    auto RetrieveUnorderedAccessBufferVariable = [ & ]( int idx )
    {
        const GLenum props[] = { GL_TYPE, GL_BLOCK_INDEX, GL_OFFSET, GL_ARRAY_SIZE };
        GLint values[ 4 ];
        m_GlFunction.GetProgramResourceiv( hProgram, GL_BUFFER_VARIABLE, idx, 4, props, 4, nullptr, values );
        nngfxToolShaderCompilerUnorderedAccessBufferVariable unorderedAccessBufferVariable = {};
        unorderedAccessBufferVariable.type = GlTypeToGfxType( values[ 0 ] );
        unorderedAccessBufferVariable.blockIndex = values[ 1 ];
        unorderedAccessBufferVariable.offset = values[ 2 ];
        unorderedAccessBufferVariable.arrayCount = values[ 3 ];
        RetrieveName( &unorderedAccessBufferVariable.name, GL_BUFFER_VARIABLE, idx );
        RetrieveStageReferences( &unorderedAccessBufferVariable.stages, GL_BUFFER_VARIABLE, idx );
        pOutput->AddReflection( unorderedAccessBufferVariable );
    };

    ForEachResource( GL_UNIFORM_BLOCK, RetrieveConstantBuffer );
    ForEachResource( GL_SHADER_STORAGE_BLOCK, RetrieveUnorderedAccessBuffer );
    ForEachResource( GL_PROGRAM_INPUT, RetrieveShaderInput );
    ForEachResource( GL_PROGRAM_OUTPUT, RetrieveShaderOutput );
    ForEachResource( GL_UNIFORM, RetrieveSamplerAndImageAndConstantBufferVariable );
    ForEachResource( GL_BUFFER_VARIABLE, RetrieveUnorderedAccessBufferVariable );

    if( isComputeShader )
    {
        GLint workGroupSize[ 3 ];
        m_GlFunction.GetProgramiv( hProgram, GL_COMPUTE_WORK_GROUP_SIZE, workGroupSize );
        pOutput->GetOutput()->computeWorkGroupSizeX = workGroupSize[ 0 ];
        pOutput->GetOutput()->computeWorkGroupSizeY = workGroupSize[ 1 ];
        pOutput->GetOutput()->computeWorkGroupSizeZ = workGroupSize[ 2 ];
    }
} // NOLINT

}
}
