﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#pragma warning( push )
#pragma warning( disable:4100 )
#pragma warning( disable:4127 )
#pragma warning( disable:4245 )
#pragma warning( disable:4267 )
#pragma warning( disable:4458 )
#pragma warning( disable:4459 )
#pragma warning( disable:4477 )
#pragma warning( disable:4512 )
#pragma warning( disable:4702 )
#pragma warning( disable:4706 )

#include <boost/wave.hpp>

#include <boost/wave/cpplexer/cpp_lex_token.hpp>
#include <boost/wave/cpplexer/cpp_lex_iterator.hpp>

#include <boost/wave/preprocessing_hooks.hpp>

#pragma warning( pop )

#include <nn/gfxTool/gfxTool_Error.h>

#include <gfxTool_Common.h>

namespace nn {
namespace gfxTool {

class BoostPreprocessor
    : public boost::wave::context_policies::default_preprocessing_hooks
{
private:
    struct Impl;
    std::shared_ptr< Impl > m_pImpl;

public:
    BoostPreprocessor();
    ~BoostPreprocessor();

    void SetReadFileCallback( nngfxToolReadFileCallback callback, void* pCallbackParam );
    void SetVariationDefinition( const nngfxToolShaderCompilerVariationDefinition* pDefinition );
    void SetVariationConstantEmulationEnabled( bool value );
    // 有効にした場合は GetResult の前に ResolveUniformRegisterBlock を呼び出す必要があります
    void SetUniformRegisterToBlockEnabled( bool value );

    void SetInvertYEnabled( bool value );
    void SetRemapZEnabled( bool value );

    void Preprocess( const char* pSource, size_t length );
    static void ResolveUniformRegisterBlock( int preprocessorCount,
        BoostPreprocessor* pPreprocessors, const char* pUniformRegisterBlockName );

    const Custom< std::stringstream >::Type& GetResult() const;

public:
    //  This token type is one of the central types used throughout the library,
    //  because it is a template parameter to some of the public classes and
    //  instances of this type are returned from the iterators.
    typedef boost::wave::cpplexer::lex_token<> token_type;

    //  The template cpplexer::lex_iterator<> is the lexer type to
    //  to use as the token source for the preprocessing engine. It is
    //  parametrized with the token type.
    typedef boost::wave::cpplexer::lex_iterator< token_type > lex_iterator_type;

    //  This is the resulting context type to use. The first template parameter
    //  should match the iterator type to be used during construction of the
    //  corresponding context object.
    typedef boost::wave::context<
        const char*,
        lex_iterator_type,
        BoostPreprocessor,//boost::wave::iteration_context_policies::load_file_to_string,
        BoostPreprocessor >
        context_type;

private:
    void PreprocessUniform();
    bool ReadFile( void** ppOutFileData, size_t* pOutFileDataSize, const char* pFileName ) const;
    void Advance();

public:

    ///////////////////////////////////////////////////////////////////////////
    //
    //  load_file_to_string
    //
    //      Loads a file into a string and returns the iterators pointing to
    //      the beginning and the end of the loaded string.
    //
    ///////////////////////////////////////////////////////////////////////////
    template <typename IterContextT>
    class inner
    {
    public:
        template <typename PositionT>
        static void init_iterators(IterContextT &iter_ctx,
            PositionT const &, boost::wave::language_support language)
        {
            typedef typename IterContextT::iterator_type iterator_type;

            BoostPreprocessor& preprocessor = iter_ctx.ctx.get_hooks();

            void* pOutFileData;
            size_t outFileDataSize;
            if( !preprocessor.ReadFile( &pOutFileData, &outFileDataSize, iter_ctx.filename.c_str() ) )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_FailedToLoadFile, "%s", iter_ctx.filename.c_str() );
            }

            auto* pFile = static_cast< const char* >( pOutFileData );
            if( IsBom( pFile ) )
            {
                pFile += 3;
                outFileDataSize -= 3;
            }
            iter_ctx.first = iterator_type( pFile,
                pFile + outFileDataSize, PositionT( iter_ctx.filename ), language );
            iter_ctx.last = iterator_type();
        }
    };

    ///////////////////////////////////////////////////////////////////////////
    //
    //  The function 'locate_include_file' is called, whenever a #include
    //  directive was encountered. It is supposed to locate the given file and
    //  should return the full file name of the located file. This file name
    //  is expected to uniquely identify the referenced file.
    //
    //  The parameter 'ctx' is a reference to the context object used for
    //  instantiating the preprocessing iterators by the user.
    //
    //  The parameter 'file_path' contains the (expanded) file name found after
    //  the #include directive. This parameter holds the string as it is
    //  specified in the #include directive, i.e. <file> or "file" will result
    //  in a parameter value 'file'.
    //
    //  The parameter 'is_system' is set to 'true' if this call happens as a
    //  result of a #include '<file>' directive, it is 'false' otherwise, i.e.
    //  for #include "file" directives.
    //
    //  The parameter 'current_name' is only used if a #include_next directive
    //  was encountered (and BOOST_WAVE_SUPPORT_INCLUDE_NEXT was defined to be
    //  non-zero). In this case it points to unique full name of the current
    //  include file (if any). Otherwise this parameter is set to NULL.
    //
    //  The parameter 'dir_path' on return is expected to hold the directory
    //  part of the located file.
    //
    //  The parameter 'native_name' on return is expected to hold the unique
    //  full file name of the located file.
    //
    //  The return value defines whether the file was located successfully.
    //
    ///////////////////////////////////////////////////////////////////////////
    template <typename ContextT>
    bool locate_include_file(ContextT& ctx, std::string &file_path,
        bool is_system, char const *current_name, std::string &dir_path,
        std::string &native_name)
    {
        ctx.find_include_file (file_path, dir_path, is_system, current_name);

        namespace fs = boost::filesystem;

        fs::path native_path(boost::wave::util::create_path(file_path));

        // return the unique full file system path of the located file
        native_name = boost::wave::util::native_file_string(native_path);

        return true;      // include file has been located successfully
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    //  The function 'found_unknown_directive' is called, whenever an unknown
    //  preprocessor directive was encountered.
    //
    //  The parameter 'ctx' is a reference to the context object used for
    //  instantiating the preprocessing iterators by the user.
    //
    //  The parameter 'line' holds the tokens of the entire source line
    //  containing the unknown directive.
    //
    //  The parameter 'pending' may be used to push tokens back into the input
    //  stream, which are to be used as the replacement text for the whole
    //  line containing the unknown directive.
    //
    //  The return value defines, whether the given expression has been
    //  properly interpreted by the hook function or not. If this function
    //  returns 'false', the library will raise an 'ill_formed_directive'
    //  preprocess_exception. Otherwise the tokens pushed back into 'pending'
    //  are passed on to the user program.
    //
    ///////////////////////////////////////////////////////////////////////////
    template <typename ContextT, typename ContainerT>
    bool
    found_unknown_directive(ContextT const& /*ctx*/, ContainerT const& line,
        ContainerT& pending)
    {
        auto iter = line.begin();
        boost::wave::token_id id = boost::wave::util::impl::skip_whitespace(iter, line.end());

        if (id != boost::wave::T_IDENTIFIER)
        {
            return false; // nothing we could do
        }

        std::copy(line.begin(), line.end(), std::back_inserter(pending));

        return true;
    }

    template <typename ContextT, typename ContainerT>
    bool
        found_error_directive( ContextT const& /*ctx*/, ContainerT const& message )
    {
        auto iter = message.begin();
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_FailedToPreprocess,
            "%s (%d): error: %s", m_pImpl->iter->get_position().get_file().c_str(),
            m_pImpl->iter->get_position().get_line(),
            iter != message.end() ? iter->get_value().c_str() : "" );
    }

    ///////////////////////////////////////////////////////////////////////////
    //
    //  The function 'throw_exception' will be called by the library whenever a
    //  preprocessing exception occurs.
    //
    //  The parameter 'ctx' is a reference to the context object used for
    //  instantiating the preprocessing iterators by the user.
    //
    //  The parameter 'e' is the exception object containing detailed error
    //  information.
    //
    //  The default behavior is to call the function boost::throw_exception.
    //
    ///////////////////////////////////////////////////////////////////////////
    template <typename ContextT, typename ExceptionT>
    void
    throw_exception(ContextT const& /*ctx*/, ExceptionT const& e)
    {
        if (e.get_errorcode() == boost::wave::preprocess_exception::macro_redefinition)
        {
            return;
        }
        boost::throw_exception(e);
    }

    template <typename ContextT, typename TokenT, typename ContainerT>
    bool evaluated_conditional_expression( ContextT const& ctx,
        TokenT const& directive, ContainerT const& expression, bool )
    {
        if( directive.get_value() == "#if" || directive.get_value() == "#elif" )
        {
            bool isDefinedOperator = false;
            for( auto token : expression )
            {
                if( token == boost::wave::T_IDENTIFIER )
                {
                    if( token.get_value() == "defined" )
                    {
                        isDefinedOperator = true;
                        continue;
                    }
                    if( !isDefinedOperator && !ctx.is_defined_macro( token.get_value() ) )
                    {
                        NN_GFXTOOL_PRINT_WARNING(
                            "%s (%d): '%s' is not defined as a preprocessor macro, replacing with 0.\n",
                            token.get_position().get_file().c_str(), token.get_position().get_line(),
                            token.get_value().c_str() );
                    }
                    isDefinedOperator = false;
                }
            }
        }

        return false;
    }
};

}
}
