﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <Camera.h>
#include <nw/math/math_Triangular.h>


namespace nw      {
namespace eftdemo {

static void     AngleToVec_( nw::math::VEC3* lookVec, nw::math::VEC3* upVec, u32 angleH, u32 angleV );
static void     VecToAngle_( const nw::math::VEC3* vec, u32* angleH, u32* angleV, f32* distance );


//---------------------------------------------------------------------------
//! @briefprivate        sin, cos の値より角度を取得する関数
//!
//! @param[in]    sin     sin 値
//! @param[in]    cos     cos 値
//!
//! @return       2πを 0x100000000 とする値
//---------------------------------------------------------------------------
static NW_INLINE u32
GetAngle_( f32 sin, f32 cos )
{
    enum { PI_IDX = 0x80000000 };
    u32 angle;

    angle = nw::math::AsinIdx( sin );

    if ( cos < 0 )
    {
        angle = PI_IDX - angle;
    }

    return angle;
}

//---------------------------------------------------------------------------
/* ctor */
Camera::Camera()
 : m_AngleV        ( 0 ),
   m_AngleH        ( 0 ),
   m_Twist         ( 0.0f ),
   m_Distance      ( 4.0f ),
   m_RecalcFlg     ( RECALC_OBJECT ),
   m_DistanceMax   ( 1000.0f ),
   m_DistanceMin   ( 0.1f ),
 //  m_GoalFrame     ( 0 ),
 //  m_BindType      ( BINDTYPE_INDEPENDENT ),
   m_TwistReset    ( 0.0f )
{
    m_LookAtPos.x = 0.0f;
    m_LookAtPos.y = 0.0f;
    m_LookAtPos.z = 0.0f;
    m_Pos.x       = 0.0f;
    m_Pos.y       = 0.0f;
    m_Pos.z       = 1.0f;
}


//---------------------------------------------------------------------------
void
Camera::SetDirection( Camera* dirObj )
{
    dirObj->Recalc_();

    // アングルと位置を設定する
    m_AngleV     = dirObj->m_AngleV;
    m_AngleH     = dirObj->m_AngleH;
    m_Pos        = dirObj->m_Pos;
    m_LookAtPos  = dirObj->m_LookAtPos;
    m_Distance   = dirObj->m_Distance;
    m_LookVec    = dirObj->m_LookVec;
    m_UpVec      = dirObj->m_UpVec;
    m_RecalcFlg  = RECALC_NONE;
}

//---------------------------------------------------------------------------
void
Camera::SetPos( f32 x, f32 y, f32 z )
{
    RecalcIf_( RECALC_TARGET );

    m_Pos.x = x;
    m_Pos.y = y;
    m_Pos.z = z;

    m_RecalcFlg   = RECALC_ANGLE;        // アングルを再計算

    ClampDistance_();
}


//---------------------------------------------------------------------------
void
Camera::SetPos( const nw::math::VEC3& vec )
{
    RecalcIf_( RECALC_TARGET );

    m_Pos  = vec;

    m_RecalcFlg  = RECALC_ANGLE;        // アングルを再計算

    ClampDistance_();
}


//---------------------------------------------------------------------------
const nw::math::VEC3*
Camera::GetPos() const
{
    RecalcIf_( RECALC_OBJECT );

    return &m_Pos;
}


//---------------------------------------------------------------------------
void
Camera::GetPos( nw::math::VEC3* pos ) const
{
    NW_ASSERT_NOT_NULL( pos );

    *pos = *(this->GetPos());
}


//---------------------------------------------------------------------------
void
Camera::SetLookAtPos( f32 x, f32 y, f32 z )
{
    RecalcIf_( RECALC_OBJECT );

    m_LookAtPos.x = x;
    m_LookAtPos.y = y;
    m_LookAtPos.z = z;

    m_RecalcFlg   = RECALC_ANGLE;        // アングルを再計算

    ClampDistance_();
}


//---------------------------------------------------------------------------
void
Camera::SetLookAtPos( const nw::math::VEC3& vec )
{
    RecalcIf_( RECALC_OBJECT );

    m_LookAtPos = vec;

    m_RecalcFlg = RECALC_ANGLE;        // アングルを再計算

    ClampDistance_();
}


//---------------------------------------------------------------------------
const nw::math::VEC3*
Camera::GetLookAtPos() const
{
    RecalcIf_( RECALC_TARGET );

    return &m_LookAtPos;
}

//---------------------------------------------------------------------------
void
Camera::GetLookAtPos( nw::math::VEC3* pos ) const
{
    NW_ASSERT_NOT_NULL( pos );

    *pos = *(this->GetLookAtPos());
}


//---------------------------------------------------------------------------
void
Camera::SetUpVec( f32 x, f32 y, f32 z )
{
    RecalcIf_( RECALC_OBJECT );

    m_UpVec.x = x;
    m_UpVec.y = y;
    m_UpVec.z = z;

//    m_RecalcFlg   = RECALC_ANGLE;        // アングルを再計算

//    ClampDistance_();
}


//---------------------------------------------------------------------------
void
Camera::SetUpVec( const nw::math::VEC3& vec )
{
    RecalcIf_( RECALC_OBJECT );

    m_UpVec = vec;

//    m_RecalcFlg = RECALC_ANGLE;        // アングルを再計算

//    ClampDistance_();
}


//---------------------------------------------------------------------------
const nw::math::VEC3*
Camera::GetUpVec() const
{
    RecalcIf_( RECALC_OBJECT );
    RecalcIf_( RECALC_TARGET );

    return &m_UpVec;
}

//---------------------------------------------------------------------------
void
Camera::GetUpVec( nw::math::VEC3* vec ) const
{
    NW_ASSERT_NOT_NULL( vec );

    *vec = *(this->GetLookAtPos());
}

//---------------------------------------------------------------------------
f32
Camera::GetDistance() const
{
    RecalcIf_( RECALC_ANGLE );

    return m_Distance;
}


//---------------------------------------------------------------------------
void
Camera::SetDistance( f32 dist )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_TARGET );

    m_Distance = dist;
    if ( m_Distance > m_DistanceMax )
    {
        m_Distance = m_DistanceMax;
    }
    else if ( m_Distance < m_DistanceMin )
    {
        m_Distance = m_DistanceMin;
    }
    m_RecalcFlg = RECALC_OBJECT;
}


//---------------------------------------------------------------------------
void
Camera::AddDistanceByMovingObj( f32 length )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_TARGET );

    m_Distance += length;

    if ( m_Distance > m_DistanceMax )
    {
        m_Distance = m_DistanceMax;
    }
    else if ( m_Distance < m_DistanceMin )
    {
        m_Distance = m_DistanceMin;
    }

    m_RecalcFlg = RECALC_OBJECT;
}


//---------------------------------------------------------------------------
void
Camera::AddDistanceByMovingTarget( f32 length )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_OBJECT );

    m_Distance += length;

    if ( m_Distance > m_DistanceMax )
    {
        m_Distance = m_DistanceMax;
    }
    else if ( m_Distance < m_DistanceMin )
    {
        m_Distance = m_DistanceMin;
    }

    m_RecalcFlg = RECALC_TARGET;
}


//---------------------------------------------------------------------------
void
Camera::Move( f32 x, f32 y, f32 z )
{
    // NOTE:
    //   並行移動なのでアングルを再計算する必要はありません。
    //   また再計算フラグは保存されるので、
    //   座標値の加算前に再計算しておく必要もありません。
    m_Pos.x += x;
    m_Pos.y += y;
    m_Pos.z += z;

    m_LookAtPos.x += x;
    m_LookAtPos.y += y;
    m_LookAtPos.z += z;
}


//---------------------------------------------------------------------------
void
Camera::Rotate( s32 angleV, s32 angleH )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_OBJECT );

    SetAngleV_( m_AngleV + static_cast<u32>(angleV) );
    SetAngleH_( m_AngleH + static_cast<u32>(angleH) );

    m_RecalcFlg = RECALC_TARGET;
}


//---------------------------------------------------------------------------
void
Camera::Revolve( s32 angleV, s32 angleH )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_TARGET );

    SetAngleV_( m_AngleV + static_cast<u32>(angleV) );
    SetAngleH_( m_AngleH + static_cast<u32>(angleH) );

    m_RecalcFlg = RECALC_OBJECT;
}


//---------------------------------------------------------------------------
void
Camera::Rotate360( s32 angleV, s32 angleH )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_OBJECT );

    m_AngleV += static_cast<u32>(angleV);
    m_AngleH += static_cast<u32>(angleH);

    m_RecalcFlg = RECALC_TARGET;
}


//---------------------------------------------------------------------------
void
Camera::Revolve360( s32 angleV, s32 angleH )
{
    RecalcIf_( RECALC_ANGLE );
    RecalcIf_( RECALC_TARGET );

    m_AngleV += static_cast<u32>(angleV);
    m_AngleH += static_cast<u32>(angleH);

    m_RecalcFlg = RECALC_OBJECT;
}


//---------------------------------------------------------------------------
void
Camera::RecalcIf_( u32 recalcFlg ) const
{
    if ( m_RecalcFlg != recalcFlg )
    {
        return;
    }

    Recalc_();
}


//---------------------------------------------------------------------------
void
Camera::Recalc_() const
{
    if ( m_RecalcFlg == RECALC_NONE )
    {
        return;
    }

    switch ( m_RecalcFlg )
    {
    //---------------------------------------------
    // ターゲット座標と角度からカメラ座標を求める
    case RECALC_OBJECT:
        RecalcPos_();
        break;
    //---------------------------------------------
    // カメラ座標と角度からターゲット座標を求める
    case RECALC_TARGET:
        RecalcLookAtPos_();
        break;
    //---------------------------------------------
    // カメラ座標とターゲット座標から角度と距離を求める
    case RECALC_ANGLE:
        RecalcAngle_();
        break;
    }

    //---------------------------------------------
    // ツイストを再計算後の姿勢に対して適用する
    if (m_Twist != 0.0f)
    {
        nw::math::VEC3 zaxis( 0.0f, 0.0f, m_Twist );
        nw::math::MTX34 rotMtx;
        rotMtx.SetRotateXyz( zaxis );
        VEC3Transform( &m_UpVec, &rotMtx, &m_UpVec );
    }

    m_RecalcFlg = RECALC_NONE;
}


//---------------------------------------------------------------------------
void
Camera::RecalcAngle_() const
{
    m_LookVec = m_Pos - m_LookAtPos;

    VecToAngle_( &m_LookVec, &m_AngleH, &m_AngleV, &m_Distance );
    SetAngleV_( m_AngleV );
    AngleToVec_( NULL, &m_UpVec, m_AngleH, m_AngleV );
}


//---------------------------------------------------------------------------
void
Camera::RecalcPos_() const
{
    AngleToVec_( &m_LookVec, &m_UpVec, m_AngleH, m_AngleV );

    m_Pos.x = m_LookAtPos.x + m_Distance * m_LookVec.x;
    m_Pos.y = m_LookAtPos.y + m_Distance * m_LookVec.y;
    m_Pos.z = m_LookAtPos.z + m_Distance * m_LookVec.z;
}


//---------------------------------------------------------------------------
void
Camera::RecalcLookAtPos_() const
{
    AngleToVec_( &m_LookVec, &m_UpVec, m_AngleH, m_AngleV );

    m_LookAtPos.x = m_Pos.x - m_Distance * m_LookVec.x;
    m_LookAtPos.y = m_Pos.y - m_Distance * m_LookVec.y;
    m_LookAtPos.z = m_Pos.z - m_Distance * m_LookVec.z;
}


//---------------------------------------------------------------------------
void
Camera::Dump() const
{
    NW_LOG("(%f, %f, %f) -> (%f, %f, %f)\n", m_Pos.x, m_Pos.y, m_Pos.z
                                               , m_LookAtPos.x, m_LookAtPos.y, m_LookAtPos.z );
    NW_LOG("angleH = 0x%x, angleV = 0x%x\n", m_AngleH, m_AngleV );
}


//---------------------------------------------------------------------------
//! @brief        水平垂直のアングル情報からベクトル値への変換を行ないます。
//!
//! @param[out]   lookVec 視線べクトルが返されるポインタ
//! @param[out]   upVec   アップベクトルが返されるポインタ
//! @param[in]    angleH  水平角度
//! @param[in]    angleV  垂直角度
//---------------------------------------------------------------------------
static void
AngleToVec_( nw::math::VEC3* lookVec, nw::math::VEC3* upVec, u32 angleH, u32 angleV )
{
    f32 sinV, cosV, sinH, cosH;

    sinV = nw::math::SinIdx( angleV );
    cosV = nw::math::CosIdx( angleV );
    sinH = nw::math::SinIdx( angleH );
    cosH = nw::math::CosIdx( angleH );

    if ( lookVec != NULL )
    {
        lookVec->x = - cosV * sinH;
        lookVec->y = - sinV;
        lookVec->z = - cosV * cosH;
    }

    if ( upVec != NULL )
    {
        upVec->x = - sinV * sinH;
        upVec->y = cosV;
        upVec->z = - sinV * cosH;
    }
}


//---------------------------------------------------------------------------
//! @brief        ベクトルデータから水平垂直角度への変換を行ないます。
//!
//! @param[in]    vec      視線ベクトル
//! @param[out]   angleH   計算された水平角度が返されるポインタ
//! @param[out]   angleV   計算された垂直角度が返されるポインタ
//! @param[out]   distance 計算されたベクトル距離が返されるポインタ
//---------------------------------------------------------------------------
static void
VecToAngle_( const nw::math::VEC3* vec, u32* angleH, u32* angleV, f32* distance )
{
    // ベクトルの長さを取得
    *distance = vec->Length();

    // 水平方向の角度を計算
    nw::math::VEC3 srcVec, dstVec;

    srcVec.x = -vec->x;
    srcVec.y = 0;
    srcVec.z = -vec->z;

    if ( srcVec.x != 0 || srcVec.z != 0 )
    {
        dstVec.SetNormalize( srcVec );
        *angleH = GetAngle_( dstVec.x, dstVec.z );
    }

    // 垂直方向の角度を計算
    f32 hDistance = srcVec.Length();
    srcVec.x = hDistance;
    srcVec.y = -vec->y;
    srcVec.z = 0;

    if ( srcVec.x != 0 || srcVec.y != 0 )
    {
        dstVec.SetNormalize( srcVec );
        *angleV = GetAngle_( dstVec.y, dstVec.x );
    }
}


//---------------------------------------------------------------------------
void
Camera::Reset()
{
    m_Pos       = m_PosReset;
    m_LookAtPos = m_LookAtPosReset;
    m_Twist     = m_TwistReset;
    m_RecalcFlg = RECALC_ANGLE;
}

//---------------------------------------------------------------------------
void
Camera::Preset()
{
    RecalcIf_( RECALC_OBJECT );
    RecalcIf_( RECALC_TARGET );

    m_PosReset       = m_Pos;
    m_LookAtPosReset = m_LookAtPos;
    m_TwistReset     = m_Twist;
}

//---------------------------------------------------------------------------
void
Camera::PresetTwist()
{
    m_TwistReset = m_Twist;
}

//---------------------------------------------------------------------------
void
Camera::ClampDistance_()
{
    RecalcIf_( RECALC_TARGET );
    RecalcIf_( RECALC_OBJECT );

    m_LookVec = m_Pos - m_LookAtPos;

    f32 distance = m_LookVec.Length();

    if ( distance < m_DistanceMin )
    {
        if ( distance == 0.0f )
        {
            m_AngleV = 0;
            m_AngleH = 0;
        }
        else
        {
            RecalcIf_( RECALC_ANGLE );
        }
        m_Distance   = m_DistanceMin;
        m_RecalcFlg = RECALC_OBJECT;
    }
    else if ( distance > m_DistanceMax )
    {
        RecalcIf_( RECALC_ANGLE );
        m_Distance   = m_DistanceMax;
        m_RecalcFlg = RECALC_OBJECT;
    }
}

//---------------------------------------------------------------------------
void
Camera::MoveViewHorizontal( f32 right, f32 forward, const Camera& view )
{
    // カメラの水平視線ベクトルを元に移動する
    f32 viewAngleH = static_cast<f32>( view.GetAngleH() );

    f32 sinH, cosH;
    nw::math::VEC3 moveVec;

    Recalc_();

    sinH = nw::math::SinIdx( static_cast<u32>(viewAngleH) );
    cosH = nw::math::CosIdx( static_cast<u32>(viewAngleH) );

    nw::math::VEC3 vUp( 0.0f, 1.0f, 0.0f );
    nw::math::VEC3 vLook( sinH, 0.0f, cosH );

    nw::math::VEC3 vRight;
    vRight.SetCross( vUp, vLook );

    moveVec.x = right * vRight.x + forward * vLook.x;
    moveVec.y = 0;
    moveVec.z = right * vRight.z + forward * vLook.z;

    m_LookAtPos = m_LookAtPos + moveVec;
    m_Pos = m_Pos + moveVec;
}


//---------------------------------------------------------------------------
void
Camera::MoveView( f32 right, f32 forward, f32 up, const Camera& view )
{
    // カメラの水平視線ベクトルを元に移動する
    f32 viewAngleH = static_cast<f32>( view.GetAngleH() );
    f32 viewAngleV = static_cast<f32>( view.GetAngleV() );

    f32 sinH, cosH;
    f32 sinV, cosV;
    nw::math::VEC3 moveVec;

    Recalc_();

    sinH = nw::math::SinIdx( static_cast<u32>( viewAngleH ) );
    cosH = nw::math::CosIdx( static_cast<u32>( viewAngleH ) );

    sinV = nw::math::SinIdx( static_cast<u32>( viewAngleV ) );
    cosV = nw::math::CosIdx( static_cast<u32>( viewAngleV ) );

    nw::math::VEC3 vLook( sinH * cosV,  sinV, cosH * cosV  );
    nw::math::VEC3 vUp( -sinH * sinV, cosV, -cosH * sinV );

    nw::math::VEC3 vRight;
    vRight.SetCross( vUp, vLook );

    moveVec.x = right * vRight.x + forward * vLook.x + up * vUp.x;
    moveVec.y = right * vRight.y + forward * vLook.y + up * vUp.y;
    moveVec.z = right * vRight.z + forward * vLook.z + up * vUp.z;

    m_LookAtPos = m_LookAtPos + moveVec;
    m_Pos = m_Pos + moveVec;
}


//---------------------------------------------------------------------------
void
Camera::MoveByEvent( const Event& refEvent, const Camera* view /* = NULL */ )
{
    f32 distance = this->GetDistance();
    Event event = refEvent;

    if (view == NULL) { view = this; }

    // 初期位置にリセット
    if ( event.Flags() & Event::POSITION_RESET )
    {
        this->Reset();
    }

    // REVERSE_ON_VIEW_CAMERAが有効でビューカメラの場合には
    // 移動方向を反転する。
    if ( event.Flags() & Event::REVERSE_ON_VIEW_CAMERA )
    {
        if ( this == view )
        {
            event.RotateH() = - event.RotateH();
            event.RotateV() = - event.RotateV();
            event.Move()    = - event.Move();
        }
    }

    // 回転
    if ( event.RotateH() != 0.0f || event.RotateV() != 0.0f )
    {
        const s32 ROT_SPEED = 0x2600000;  // 360度が0x10000, 107フレームで一周
        s32 rotV = s32( event.RotateV() * ROT_SPEED );
        s32 rotH = s32( event.RotateH() * ROT_SPEED );

        if ( event.Flags() & Event::ROTATE_BASED_ON_TARGET )
        {
            if ( event.Flags() & Event::ROTATE_VERTICAL_360 )
            {
                // 注視点を中心に360度公転
                this->Revolve360( rotV, rotH );
            }
            else
            {
                // 注視点を中心に公転
                this->Revolve( rotV, rotH );
            }
        }
        else
        {
            if ( event.Flags() & Event::ROTATE_VERTICAL_360 )
            {
                // オブジェクト中心に360度自転
                this->Rotate360( -rotV, -rotH );
            }
            else
            {
                // オブジェクト中心に自転
                this->Rotate( -rotV, -rotH );
            }
        }
    }

    // 並行移動
    const f32 MOVE_SPEED = 60.0f / 0x800; // 体感でなんとなくこの値

    if ( event.Flags() & Event::MOVE_ALONG_AXIS )
    {
        if ( event.Move().x != 0.0f || event.Move().y != 0.0f || event.Move().z != 0.0f )
        {
            f32 moveX = - distance * event.Move().x * MOVE_SPEED;
            f32 moveY =   distance * event.Move().y * MOVE_SPEED;
            f32 moveZ = - distance * event.Move().z * MOVE_SPEED;
            this->Move( moveX, moveY, moveZ );
        }
    }
    else if ( event.Flags() & Event::MOVE_ALONG_VIEW )
    {
        if ( event.MoveForward() != 0.0f || event.MoveRight() != 0.0f || event.MoveUp() != 0.0f )
        {
            f32 moveForward = distance * event.MoveForward() * MOVE_SPEED;
            f32 moveRight   = distance * event.MoveRight()   * MOVE_SPEED;
            f32 moveUp      = distance * event.MoveUp()      * MOVE_SPEED;
            this->MoveView( moveRight, moveForward, moveUp, *view );
        }
    }
    else
    {
        if ( event.MoveForward() != 0.0f || event.MoveRight() != 0.0f )
        {
            f32 moveForward = distance * event.MoveForward() * MOVE_SPEED;
            f32 moveRight   = distance * event.MoveRight()   * MOVE_SPEED;

            this->MoveViewHorizontal( moveRight, moveForward, *view );
        }
        if ( event.Move().y != 0.0f )
        {
            f32 moveY = distance * event.Move().y * MOVE_SPEED;
            this->Move( 0, moveY, 0 );
        }
    }

    // 距離の拡縮
    if ( event.Zoom() != 0.0f )
    {
        f32 expand = distance * event.Zoom() * MOVE_SPEED;
        this->AddDistanceByMovingObj( expand );
    }

    // ツイスト
    if ( event.Twist() != 0.0f )
    {
        const f32 TWIST_SPEED = 360.0f / 100; // 100フレームで一回転程度
        f32 twist = this->GetTwist();
        twist += event.Twist() * TWIST_SPEED;
        this->SetTwist( twist );
    }
}


//---------------------------------------------------------------------------
void
Camera::GetTargetMtx_( nw::math::MTX34* mtx ) const
{
    NW_ASSERT( mtx != NULL );

    Recalc_();

    m_LookVec = m_Pos - m_LookAtPos;

    nw::math::MTX34 transMtx;

    transMtx.SetTranslate( -m_LookAtPos );

    nw::math::VEC3 lookVec;
    nw::math::VEC3 upVec;
    nw::math::VEC3 rightVec;

    if ( m_LookVec.IsZero() )
    {
        lookVec = m_LookVec;
    }
    else
    {
        lookVec.SetNormalize( m_LookVec );
    }
    rightVec.SetCross( m_UpVec, lookVec );
    rightVec.Normalize();

    upVec.SetCross( lookVec, rightVec );

    f32 tx = -rightVec.Dot( m_Pos );
    f32 ty = -upVec.Dot( m_Pos );
    f32 tz = -lookVec.Dot( m_Pos );

    mtx->m[0][0] = rightVec.x;
    mtx->m[0][1] = rightVec.y;
    mtx->m[0][2] = rightVec.z;
    mtx->m[0][3] = tx;

    mtx->m[1][0] = upVec.x;
    mtx->m[1][1] = upVec.y;
    mtx->m[1][2] = upVec.z;
    mtx->m[1][3] = ty;

    mtx->m[2][0] = lookVec.x;
    mtx->m[2][1] = lookVec.y;
    mtx->m[2][2] = lookVec.z;
    mtx->m[2][3] = tz;
}

//---------------------------------------------------------------------------
void Camera::MakeCameraEventClassic( Event* events, nw::dev::Pad* pad, nw::dev::Mouse* mouse, bool isMouseAvailable, bool isAltModified, f32 frameSpeed)
{
    NW_ASSERT_NOT_NULL( pad );

    f32 move_rate = frameSpeed;
    bool mouseAvailable = ( mouse && isMouseAvailable );

    const f32 PAD_STICK_RATE = 100.f;
    const f32 MOUSE_MOVE_RATE = 4.0f;
    const f32 MOUSE_ZOOM_RATE = 0.1f;
    const f32 WHEEL_ZOOM_RATE = 1.0f;

    // カメラのリセット
    if ( pad->IsHold( nw::dev::Pad::MASK_START ) )
    {
        events->Flags() |= Camera::Event::POSITION_RESET;
    }

    // メインスティック/マウス左ドラッグ
    const s8 STICK_MAX = 56;
    const s8 SUBSTICK_MAX = 44;
    f32 stickX = 0.f;
    f32 stickY = 0.f;

    if ( mouseAvailable && mouse->IsHold( nw::demo::Mouse::MASK_LBUTTON ) )
    {
        // マウス操作
        if ( isAltModified )
        {
            stickX = - mouse->GetPointerDiff().x * MOUSE_MOVE_RATE;
            stickY = - mouse->GetPointerDiff().y * MOUSE_MOVE_RATE;
        }
    }
    else
    {
        // パッド操作
        stickX =   pad->GetLeftStick().x * PAD_STICK_RATE;
        stickY = - pad->GetLeftStick().y * PAD_STICK_RATE;
    }

    if ( stickX != 0.f || stickY != 0.f )
    {
        // 注視点を中心に公転
        events->Flags() |= Camera::Event::ROTATE_BASED_ON_TARGET;
        events->RotateH() = stickX / STICK_MAX * move_rate;
        events->RotateV() = stickY / STICK_MAX * move_rate;
    }

    // サブスティック/マウス中ドラッグ
    f32 substickX = 0.f;
    f32 substickY = 0.f;

    if ( mouseAvailable && mouse->IsHold( nw::demo::Mouse::MASK_MBUTTON ) )
    {
        // マウス操作
        if ( isAltModified )
        {
            substickX = mouse->GetPointerDiff().x * MOUSE_MOVE_RATE / 3.f;
            substickY = mouse->GetPointerDiff().y * MOUSE_MOVE_RATE / 3.f;
        }
    }
    else
    {
        // パッド操作
        substickX = - pad->GetRightStick().x * PAD_STICK_RATE;
        substickY =   pad->GetRightStick().y * PAD_STICK_RATE;
    }

    if ( substickX != 0.f || substickY != 0.f )
    {
        if ( pad->IsHold( nw::dev::Pad::MASK_L )
        || ( mouse && mouse->IsHold( nw::dev::Mouse::MASK_MBUTTON ) /* && !isCtrlModified */ ) )
        {
            // カメラ平面水平移動
            events->MoveRight() = substickX / SUBSTICK_MAX * move_rate;
            events->Move().y    = substickY / SUBSTICK_MAX * move_rate;

            events->Flags() |= Camera::Event::MOVE_ALONG_VIEW;
        }
        else
        {
            // XZ平面水平移動
            events->MoveRight()   = substickX / SUBSTICK_MAX * move_rate;
            events->MoveForward() = substickY / SUBSTICK_MAX * move_rate;
        }
    }

    // ズーム
    if ( ! pad->IsHold( nw::dev::Pad::MASK_L ) )
    {
        if ( pad->IsHold( nw::dev::Pad::MASK_X ) && pad->IsHold( nw::dev::Pad::MASK_Y ) )
        {
            // 両方押されている場合は何もしない。
        }
        else if ( pad->IsHold( nw::dev::Pad::MASK_Y ) )
        {
            // パッドズームイン操作
            events->Flags() |= Camera::Event::EXPAND_BASED_ON_TARGET;
            events->Zoom() = move_rate;
        }
        else if ( pad->IsHold( nw::dev::Pad::MASK_X ) )
        {
            // パッドズームアウト操作
            events->Flags() |= Camera::Event::EXPAND_BASED_ON_TARGET;
            events->Zoom() = - move_rate;
        }
        else if ( mouseAvailable && mouse->GetWheel() != 0 )
        {
            // マウス操作（ホイール回転）
            events->Flags() |= Camera::Event::EXPAND_BASED_ON_TARGET;
            events->Zoom() = - move_rate * WHEEL_ZOOM_RATE * mouse->GetWheel();
        }
        else if ( mouseAvailable && mouse->IsHold( nw::demo::Mouse::MASK_RBUTTON ) )
        {
            // マウス操作（右ボタン）
            if ( isAltModified )
            {
                nw::math::VEC2 difference = mouse->GetPointerDiff();
                f32 distance = ( nw::math::FAbs(difference.x) - nw::math::FAbs(difference.y) >= 0 ) ? difference.x : difference.y;

                events->Flags() |= Camera::Event::EXPAND_BASED_ON_TARGET;
                events->Zoom() = - move_rate * MOUSE_ZOOM_RATE * distance;
            }
        }
    }
}

void Camera::UpdateCamera( nw::dev::Pad* pad, nw::dev::Mouse* mouse, bool isMouseAvailable, bool isAltModified )
{
    Event events;

    MakeCameraEventClassic( &events, pad, mouse, isMouseAvailable, isAltModified );
    MoveByEvent( events );
}


}       // eftdemo
}       // nw
