﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/dev.h>
#include <nw/demo.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>

#include <FrameBuffer.h>

#if defined(NW_PLATFORM_WIN32)
    #include <nw/dev/win/dev_PadWin.h>
    #include <nw/dev/win/dev_MouseWin.h>
#elif defined(NW_PLATFORM_CAFE)
    #include <nw/dev/cafe/dev_PadCafe.h>
#endif

namespace nw {
namespace eftdemo {

//---------------------------------------------------------------------------
//! @brief        フレームレート
//---------------------------------------------------------------------------
enum INIT_SETTINGS_FRAME_RATE
{
    INIT_SETTINGS_FRAME_RATE_60 = 0,
    INIT_SETTINGS_FRAME_RATE_30,
    INIT_SETTINGS_FRAME_RATE_20,
    INIT_SETTINGS_FRAME_RATE_15
};

//---------------------------------------------------------------------------
//! @brief        解像度
//---------------------------------------------------------------------------
enum INIT_SETTINGS_RESOLUTION
{
    INIT_SETTINGS_RESOLUTION_960_540 = 0,
    INIT_SETTINGS_RESOLUTION_640_480,
    INIT_SETTINGS_RESOLUTION_1280_720
};

const u32 DEFAULT_SCREEN_WIDTH = 1280;       //!< デフォルトの画面幅です。
const u32 DEFAULT_SCREEN_HEIGHT = 720;       //!< デフォルトの画面の高さです。

//---------------------------------------------------------------------------
//! @brief        デモシステムです。
//---------------------------------------------------------------------------
class System
{
protected:
#if defined(NW_PLATFORM_CAFE)
    //! @brief 画面表示切り替え用フラグです。
    enum
    {
        FLAG_IS_BLACK = 0x1,
        FLAG_CHANGE_BLACK = 0x2
    };
#endif

public:
    //! @brief コンストラクタに渡す生成パラメータです。
    struct CreateArg
    {
        nw::ut::IAllocator* allocator;              //!< デモシステムで用いるアロケータです。
        u32                 waitVBlank;             //!< VBlank 待ちする回数 (1なら60fps、2なら30fps, 3なら15fps) です。
        nw::ut::Color4u8    clearColor;             //!< 画面クリア色です。
        f32                 clearDepth;             //!< 画面クリアデプスです。
        u32                 width;                  //!< 画面の横幅です。
        u32                 height;                 //!< 画面の縦幅です。
        u8*                 fontBinary;             //!< デバッグ文字描画に用いるフォントバイナリへのポインタです。
        u32                 fontBinarySize;         //!< デバッグ文字描画に用いるフォントバイナリのサイズです。
        const char*         fontPath;               //!< デバッグ文字描画に用いるフォントのパスを指定します。（バイナリを直接設定せずファイルからロードする場合に指定します。デモファイルシステムが初期化されている必要があります。）
        bool                drawMeter;              //!< 負荷メーターを表示するか指定します。
        bool                usePointerCursor;       //!< ポインタカーソルを有効にするかを指定します。
        bool                fileDeviceManagerInitialize;        //!< デモファイルシステムを初期化するかを指定します。
#if defined(NW_PLATFORM_WIN32)
        f32                 fps;                                //!< waitVBlank が 0 か、ハード的に VBlank 待ちが行えない場合に、何 fps でタイマー駆動するかを指定します。
        bool                createDefaultFrameBuffer;           //!< フレームバッファを内部で生成するかどうかを指定します。
        bool                primitiveRendererInitialize;        //!< PrimitiveRenderer を初期化するかを指定します。
        u8*                 fontShaderBinary;                   //!< デバッグ文字描画に用いるシェーダーバイナリへのポインタです。
        u32                 fontShaderBinarySize;               //!< デバッグ文字描画に用いるシェーダーバイナリのサイズです。
#elif defined(NW_PLATFORM_CAFE)
        u8*                 fontShaderBinary;                   //!< デバッグ文字描画に用いるシェーダーバイナリへのポインタです。
        u32                 fontShaderBinarySize;               //!< デバッグ文字描画に用いるシェーダーバイナリのサイズです。
        const char*         fontShaderPath;                     //!< デバッグ文字描画に用いるシェーダーのパスを指定します。（バイナリを直接設定せずファイルからロードする場合に指定します。デモファイルシステムが初期化されている必要があります。）
        u8*                 primitiveRendererShaderBinary;      //!< PrimitiveRenderer で用いるシェーダーバイナリへのポインタです。
        u32                 primitiveRendererShaderBinarySize;  //!< PrimitiveRenderer で用いるシェーダーバイナリのサイズです。
        const char*         primitiveRendererShaderPath;        //!< PrimitiveRenderer で用いるシェーダのパスを指定します。（バイナリを直接設定せずファイルからロードする場合に指定します。デモファイルシステムが初期化されている必要があります。）
#endif

        //! @brief コンストラクタです。
        CreateArg() :
            allocator( NULL ),
            waitVBlank( 1 ),
            clearColor( nw::ut::Color4u8::GRAY ),
            clearDepth( 1.f ),
            width( DEFAULT_SCREEN_WIDTH ),
            height( DEFAULT_SCREEN_HEIGHT ),
            fontBinary( NULL ),
            fontBinarySize( 0 ),
            fontPath( NULL ),
            drawMeter( false ),
            usePointerCursor( false ),
            fileDeviceManagerInitialize( false ),
#if defined(NW_PLATFORM_WIN32)
            fps( 60.0 ),
            createDefaultFrameBuffer( true ),
            primitiveRendererInitialize( false )
#elif defined(NW_PLATFORM_CAFE)
            fontShaderBinary( NULL ),
            fontShaderBinarySize( 0 ),
            fontShaderPath( NULL ),
            primitiveRendererShaderBinary( NULL ),
            primitiveRendererShaderBinarySize( 0 ),
            primitiveRendererShaderPath( NULL )
#endif
        {}
    };

    //! @brief clear メソッドに渡すビットマスクです。
    enum ClearFlag
    {
        CLEAR_FLAG_NONE     = 0,        //!< 何も指定しない。
        CLEAR_FLAG_COLOR    = 1 << 0,   //!< 色要素をクリアする。
        CLEAR_FLAG_DEPTH    = 1 << 1,   //!< 深度要素をクリアする。
        CLEAR_FLAG_STENCIL  = 1 << 2    //!< ステンシル要素をクリアする。
    };

    //! @brief 画面表示状態です。
    enum DisplayState
    {
        HIDE,       //!< 非表示状態です。
        READY,      //!< 表示準備状態(次のVSyncで表示)です。
        SHOW        //!< 表示中です。
    };

    //! @brief SetProcessPriority に渡すパラメータです。
    enum ProcessPriority
    {
        IDLE,       //!< 低優先度です。
        NORMAL,     //!< 普通優先度です。
        HIGH,       //!< 高優先度です。
        REAL_TIME   //!< 最高優先度です。
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    System( const CreateArg& arg );

    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //---------------------------------------------------------------------------
    void Initialize();

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize();

#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief        黒画面か否かを返します。
    //!
    //! @return       黒画面の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsBlack() const { return m_Flags.IsMaskOn( FLAG_IS_BLACK ); }

    //---------------------------------------------------------------------------
    //! @brief        次のフレーム更新タイミングで、VISetBlack を行なうリクエストを設定します。
    //---------------------------------------------------------------------------
    void ReserveSetBlack(bool black)
    {
        if (black != IsBlack())
        {
            m_Flags.SetMaskOn( FLAG_CHANGE_BLACK );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        Mem1アロケータを取得します。
    //---------------------------------------------------------------------------
    nw::ut::MemoryAllocator* GetMem1Allocator() { return &m_VRAMAllocator; }
#endif

    //---------------------------------------------------------------------------
    //! @brief        入力インターフェースの初期化を行います。
    //---------------------------------------------------------------------------
    void InitializeInputInterface();

    //---------------------------------------------------------------------------
    //! @brief        入力インターフェースの更新を行います。
    //---------------------------------------------------------------------------
    void UpdateInputInterface();

    //---------------------------------------------------------------------------
    //! @brief        入力インターフェースの終了処理を行います。
    //---------------------------------------------------------------------------
    void FinalizeInputInterface();

    //---------------------------------------------------------------------------
    //! @brief        グラフィックエンジンの初期化を行います。
    //!
    //!   CreateArgをもとにして、以下の処理を行います。
    //!    - Graphicsのインスタンス生成。
    //!    - 引数に指定したサイズで仮想フレームバッファを作成。
    //!    - VBlank待ち回数の設定。
    //---------------------------------------------------------------------------
    void InitializeGraphicsSystem( int argc = 0, char** argv = NULL );

    //---------------------------------------------------------------------------
    //! @brief        グラフィックスエンジンの終了処理を行います。
    //---------------------------------------------------------------------------
    void FinalizeGraphicsSystem();

    //---------------------------------------------------------------------------
    //! @brief        VBlank 待ち回数を変更します。
    //!
    //! @param[in]    interval VBlank 待ち回数です。
    //!
    //! @return       設定に成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool SetWaitVBlankInterval( u32 interval );

#if defined(NW_PLATFORM_WIN32)
    //---------------------------------------------------------------------------
    //! @brief        fps を設定します。
    //!
    //!   VBlank 待ち設定が 0 でないときは無視されます。
    //!
    //! @param[in]    fps      設定する fps です。
    //---------------------------------------------------------------------------
    void SetFps( f32 fps );
#endif

    //---------------------------------------------------------------------------
    //! @brief        画面の横幅を取得します。
    //!
    //! @return       画面の横幅を返します。
    //---------------------------------------------------------------------------
    s32 GetWidth(){ return m_Arg.width; }

    //---------------------------------------------------------------------------
    //! @brief        画面の縦幅を取得します。
    //!
    //! @return       画面の縦幅を返します。
    //---------------------------------------------------------------------------
    s32 GetHeight(){ return m_Arg.height; }

    //---------------------------------------------------------------------------
    //! @brief        画面表示状態を取得します。
    //!
    //! @return       画面表示状態を返します。
    //---------------------------------------------------------------------------
    DisplayState GetDisplayState() const { return m_DisplayState; }

    //---------------------------------------------------------------------------
    //! @brief        終了判定を取得します。
    //!
    //! @return       終了判定フラグを返します。
    //---------------------------------------------------------------------------
    bool IsExiting() const;

    //---------------------------------------------------------------------------
    //! @brief        画面の切り替え処理を行います。
    //---------------------------------------------------------------------------
    void SwapBuffer( FrameBuffer* frameBuffer );

    //---------------------------------------------------------------------------
    //! @brief        スキャンバッファの切り替え処理を行います。
    //---------------------------------------------------------------------------
    void SetScanBuffer( bool linearScanBuffer );

    //---------------------------------------------------------------------------
    //! @brief        フレームバッファを標準に戻す。
    //---------------------------------------------------------------------------
    void WaitForFlip();

    //---------------------------------------------------------------------------
    //! @brief        フレーム開始時処理を行います。
    //!
    //!               ここでは、以下の処理を行います。
    //!               - DramMeter のフレーム開始処理
    //---------------------------------------------------------------------------
    void BeginFrame();

    //---------------------------------------------------------------------------
    //! @brief        フレーム終了時処理を行います。
    //!
    //!               ここでは、以下の処理を行います。
    //!               // - デバッグ用文字描画のための DevTextWriter のバッファのクリア
    //!               - DramMeter のフレーム終了処理
    //---------------------------------------------------------------------------
    void EndFrame();

    //---------------------------------------------------------------------------
    //! @brief        Pad 管理クラスのインスタンスへのポインタを取得します。
    //!
    //! @return       Pad 管理クラスのインスタンスへのポインタを返します。
    //---------------------------------------------------------------------------
    nw::dev::Pad* GetPad() const { return m_Pad; }

#ifdef NW_PLATFORM_WIN32
    //---------------------------------------------------------------------------
    //! @brief        マウス管理クラスのインスタンスへのポインタを取得します。
    //!
    //! @return       マウス管理クラスのインスタンスへのポインタを返します。
    //---------------------------------------------------------------------------
    nw::dev::Mouse* GetMouse() const { return m_Mouse; }
#endif

    //---------------------------------------------------------------------------
    //! @brief        1 フレームを測定する計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::CPUMeter& GetMeterFrame() { return m_MeterFrame; }

    //---------------------------------------------------------------------------
    //! @brief        Calc 計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::CPUMeter& GetMeterCalc() { return m_MeterCalc; }

    //---------------------------------------------------------------------------
    //! @brief        Draw 計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::CPUMeter& GetMeterDraw() { return m_MeterDraw; }

    //---------------------------------------------------------------------------
    //! @brief        GPU 計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::GPUMeter& GetMeterGPU() { return m_MeterGPU; }

    //---------------------------------------------------------------------------
    //! @brief        MeterDrawer を取得します。
    //!
    //! @return       MeterDrawer を返します。
    //---------------------------------------------------------------------------
    nw::dev::MeterDrawer& GetMeterDrawer() { return m_MeterDrawer; }

    //---------------------------------------------------------------------------
    //! @brief        負荷メータを描画します。
    //---------------------------------------------------------------------------
    void DrawLoadMeters();

    //---------------------------------------------------------------------------
    //! @brief        デバッグ用文字列描画に用いることができる DevTextWriter を取得します。
    //!
    //! @return       DevTextWriter を返します。
    //---------------------------------------------------------------------------
    nw::dev::DevTextWriter* GetTextWriter() { return &m_DevTextWriter; }

    //---------------------------------------------------------------------------
    //! @brief        デバッグ用文字列描画のための DevTextWriter をクリアします。
    //---------------------------------------------------------------------------
    void ClearTextWriter();

    //---------------------------------------------------------------------------
    //! @brief        設定ファイルを読み込みます。
    //---------------------------------------------------------------------------
    static void LoadConfigFile( nw::ut::MemoryAllocator *allocator, nw::eftdemo::System::CreateArg *arg );

    //---------------------------------------------------------------------------
    //! @brief        設定ファイルを書き込みます。
    //---------------------------------------------------------------------------
    static void SaveConfigFile( INIT_SETTINGS_FRAME_RATE frameRate, INIT_SETTINGS_RESOLUTION resolution,
                                f32 r, f32 g, f32 b, f32 a );

    //---------------------------------------------------------------------------
    //! @brief        生成パラメータを取得します。
    //---------------------------------------------------------------------------
    const CreateArg& GetCreateArg() const { return m_Arg; }

protected:

#if defined(NW_PLATFORM_WIN32)

    //! ウィンドウのセットアップを行います。
    void SetupWindow();

    //! メッセージプロシージャの実装です。
    _W64 long MsgProcImpl( void* hWnd, unsigned int msg, _W64 unsigned int wParam, _W64 long lParam, BOOL *handled );

    //! メッセージプロシージャの外側部分です。
    static _W64 long MsgProc( void* hWnd, unsigned int msg, _W64 unsigned wParam, _W64 long lParam, BOOL *handled );

    //! フレームバッファを作成します。
    void CreateFrameBuffer( const nw::math::VEC2& virtual_fb_size );

    static System*              sInstance;              //!< 唯一のインスタンスです。

#elif defined(NW_PLATFORM_CAFE)

    //! 黒画面に設定します。
    void SetBlack(bool is_black);

#endif

    CreateArg                   m_Arg;                      //!< 生成パラメータです。
    nw::dev::CPUMeter           m_MeterFrame;               //!< 1 フレームの処理負荷メーターです。
    nw::dev::CPUMeter           m_MeterCalc;                //!< 計算処理負荷メーターです。
    nw::dev::CPUMeter           m_MeterDraw;                //!< 描画処理負荷メーターです。
    nw::dev::GPUMeter           m_MeterGPU;                 //!< GPU 負荷メーターです。
    nw::dev::MeterDrawer        m_MeterDrawer;              //!< 負荷メーターを描画します。
    nw::math::MTX44             m_TextWriterProjMtx;        //!< DevTextWriter 用プロジェクション行列です。
    nw::math::MTX34             m_TextWriterViewMtx;        //!< DevTextWriter 用ビュー行列です。
    nw::dev::DevTextWriter      m_DevTextWriter;            //!< デバッグ用文字描画のための DevTextWriter です。
    bool                        m_TextWriterInitialized;    //!< DevTextWriter が初期化されているかを表します。
    DisplayState                m_DisplayState;              //!< 画面表示を行ってよいかを判定するステート変数です。

#if defined(NW_PLATFORM_WIN32)
    void*                       m_HWnd;                     //!< ウインドウハンドルです。
    bool                        m_Exit;                     //!< 無限ループ抜けフラグです。
    void*                       m_HGLRC;                    //!< GL コンテキストです。
    void*                       m_HDC;                      //!< ウインドウコンテキストです。
    nw::dev::PadWin*            m_Pad;                      //!< デバッグ用パッド管理クラスのインスタンスへのポインタです。
    nw::dev::MouseWin*          m_Mouse;                    //!< マウス管理クラスのインスタンスへのポインタです。
    nw::ut::Tick                m_LastUpdateTime;           //!< 前回のUpdateを行った時間です。
    nw::ut::TimeSpan            m_FrameTime;                //!< 1フレームに相当する時間を格納します。
    nw::ut::TimeSpan            m_LastDiffTime;             //!< 前回のフレーム更新にかかった時間です。
    nw::eftdemo::FrameBuffer    m_ScanBuffer;               //!< 実機スキャンバッファ想定のフレームバッファ
    nw::eftdemo::FrameBuffer    m_ScanBufferSRGB;           //!< 実機リニアスキャンバッファ想定のフレームバッファ
    nw::eftdemo::FrameBuffer*   m_CurrentScanBuffer;        //!< カレントのスキャンバッファ

#elif defined(NW_PLATFORM_CAFE)

    nw::ut::BitFlag32           m_Flags;                    //!< 設定フラグです。
public:
    nw::ut::MemoryAllocator     m_VRAMAllocator;            //!< VRAM アロケーターです。
protected:
    u32                         m_ScanSize;                 //!< スキャンバッファサイズ
    void*                       m_ScanBufferPtr;            //!< スキャンバッファへのポインタです。
    u32                         m_LinearScanSize;           //!< リニアスキャンバッファサイズ
    void*                       m_LinearScanBufferPtr;      //!< スキャンバッファへのポインタです。
    GX2ContextState*            m_ContextState;             //!< コンテキストステートへのポインタです。
    nw::dev::PadCafe*           m_Pad;                      //!< デバッグ用パッド管理クラスのインスタンスへのポインタです。
    u64*                        m_GpuCounters;              //!< GPU 計測用カウンタの配列のポインタです。

#endif
};

} // namespace eftdemo
} // namespace nw
