﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>

#include <nn/nn_Log.h>
#include "../Include/DebugFontGlyphTreeMap.h"


uint8_t GlyphNode::CalculateLineKind(uint16_t size)
{
    if (size <= 16)
    {
        return 16;
    }
    else if (size <= 18)
    {
        return 18;
    }
    else if (size <= 20)
    {
        return 20;
    }
    else if (size <= 22)
    {
        return 22;
    }
    else if (size <= 32)
    {
        return 32;
    }
    else
    {
        return 64;
    }
}

GlyphTreeMap::GlyphTreeMap()
: mRoot(NULL)
, mNodeCountMax(0)
{
}

void GlyphTreeMap::Initialize(nn::AlignedAllocateFunctionWithUserData allocateFunction, void* pUserDataForAllocateFunction, uint32_t nodeNumMax)
{
    mNodeCountMax = nodeNumMax;
    mFreeList.Initialize(allocateFunction(sizeof(GlyphNode) * nodeNumMax, 4, pUserDataForAllocateFunction), sizeof(GlyphNode), nodeNumMax);
}

void GlyphTreeMap::Finalize(nn::FreeFunctionWithUserData freeFunction, void* pUserDataForFreeFunction)
{
    freeFunction(mFreeList.GetWork(), pUserDataForFreeFunction);
    mFreeList.Cleanup();
}

GlyphNode* GlyphTreeMap::Find(uint16_t code, uint16_t size) const
{
    GlyphNode::KeyType key;
    key.detail.code = code;
    key.detail.size = size;
    return Find(mRoot, key.raw);
}

GlyphNode* GlyphTreeMap::Insert(uint16_t code, uint16_t size)
{
    if (mFreeList.IsEmpty())
    {
        return NULL;
    }
    else
    {
        GlyphNode* node = new(mFreeList.Get()) GlyphNode(code, size);
        mRoot = Insert(mRoot, node);
        mRoot->mColor = GlyphNode::Black;
        return node;
    }
}

void GlyphTreeMap::Erase(uint16_t code, uint16_t size)
{
    GlyphNode::KeyType key;
    key.detail.code = code;
    key.detail.size = size;
    mRoot = Erase(mRoot, key.raw);
    if (mRoot) mRoot->mColor = GlyphNode::Black;
}

GlyphNode* GlyphTreeMap::Find(GlyphNode* node, uint32_t key_raw) const
{
    while (node)
    {
        if (key_raw < node->mKey.raw) {
            node = node->mLeft;
        } else if (key_raw > node->mKey.raw) {
            node = node->mRight;
        } else {
            return node;
        }
    }
    return NULL;
}

GlyphNode* GlyphTreeMap::Insert(GlyphNode* h, GlyphNode* node)
{
    if (h == NULL) {
        node->mColor = GlyphNode::Red;
        node->mLeft = NULL;
        node->mRight = NULL;
        return node;
    }
    {
        if (node->mKey.raw < h->mKey.raw) {
            h->mLeft = Insert(h->mLeft, node);
        } else if (node->mKey.raw > h->mKey.raw) {
            h->mRight = Insert(h->mRight, node);
        } else {
            // 置き換わり
            if (h != node) {
                // 全く同じオブジェクトの場合はこの処理は行う必要がない
                node->mRight = h->mRight;
                node->mLeft = h->mLeft;
                node->mColor = h->mColor;
                mFreeList.Put(h);
            }
            h = node;
        }
        if (IsRed(h->mRight) && ! IsRed(h->mLeft)) h = RotateLeft(h);
        if (IsRed(h->mLeft) && IsRed(h->mLeft->mLeft)) h = RotateRight(h);
        if (IsRed(h->mLeft) && IsRed(h->mRight)) FlipColors(h);
    }
    return h;
}

GlyphNode* GlyphTreeMap::Erase(GlyphNode* h, uint32_t key_raw)
{
    if (key_raw < h->mKey.raw) {
        if ( ! IsRed(h->mLeft) && ! IsRed(h->mLeft->mLeft)) {
            h = MoveRedLeft(h);
        }
        h->mLeft = Erase(h->mLeft, key_raw);
    } else {
        if (IsRed(h->mLeft)) {
            h = RotateRight(h);
        }
        if (key_raw == h->mKey.raw && (h->mRight == NULL)) {
            mFreeList.Put(h);
            return NULL;
        }
        if ( ! IsRed(h->mRight) && ! IsRed(h->mRight->mLeft)) {
            h = MoveRedRight(h);
        }
        if (key_raw == h->mKey.raw) {
            GlyphNode* newh = Find(h->mRight, GetMin(h->mRight)->mKey.raw);
            newh->mRight = EraseMin(h->mRight);
            newh->mLeft = h->mLeft;
            newh->mColor = h->mColor;
            mFreeList.Put(h);
            h = newh;
        } else {
            h->mRight = Erase(h->mRight, key_raw);
        }
    }
    return FixUp(h);
}

void GlyphTreeMap::UpdateFlagsForCompleteTextureCacheRecursive(GlyphNode* node)
{
    uint32_t flag = node->mFlag.storage;
    uint32_t isKeep = static_cast<uint32_t>((flag & (GlyphNode::FlagMask_Requested | GlyphNode::FlagMask_Used)) != 0);
    node->mFlag.storage = static_cast<uint8_t>((isKeep << GlyphNode::FlagBit_Kept) | (flag & (GlyphNode::FlagMask_SystemReserved | GlyphNode::FlagMask_NotInFont)));
    if (node->mLeft)
    {
        UpdateFlagsForCompleteTextureCacheRecursive(node->mLeft);
    }
    if (node->mRight)
    {
        UpdateFlagsForCompleteTextureCacheRecursive(node->mRight);
    }
}

void GlyphTreeMap::GlyphFunctionRecursive(GlyphNode* node, GlyphFunc func, void *pUserData)
{
    if (node)
    {
        if (node->mLeft)
        {
            GlyphFunctionRecursive(node->mLeft, func, pUserData);
        }

        func(node, pUserData);

        if (node->mRight)
        {
            GlyphFunctionRecursive(node->mRight, func, pUserData);
        }
    }
}
