﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Threading;
using System.Net;
using System.Net.Sockets;
using System.Diagnostics;

namespace Nintendo.McsServer.PCIO2CommDevice
{
    public delegate void SocketListenerAccepted(SocketListenerResult result);

    /// <summary>
    /// ルータ側がチャンネル登録を受け付けるソケットの
    /// ポートを監視するオブジェクト。
    ///
    /// 指定したポートのソケット接続を監視して、
    /// その後のチャンネル登録メソッドをコールバック実行する。
    ///
    /// チャンネル登録までの流れは、大まかに以下のようになる。
    ///
    /// ・指定されたポートで接続待ちうけを開始する。
    /// ・Acceptが完了したタイミングで接続済みソケットを引数として
    ///   本クラスがコールバック関数を呼び出す。
    /// ・以後、ChannelManager等にてソケットとチャンネルとの関連付けを処理。
    ///
    /// </summary>
    public class SocketListener
    {
        /// <summary>
        /// EndAcceptによってSocketが得られたときに呼び出されるデリゲートを保持するメンバ。
        /// </summary>
        readonly SocketListenerAccepted _socketListenerAccepted;

        /// <summary>
        /// Start(), Stop()の間で同期を取るオブジェクト
        /// </summary>
        readonly object _syncPublic = new object();

        /// <summary>
        /// BeginAcceptを行うSocketを保持するメンバ。
        /// </summary>
        SocketParam[] _acceptSockets;

        /// <summary>
        /// 実行中なら真
        /// </summary>
        bool _bActive;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SocketListener(SocketListenerAccepted socketListenerAccepted)
        {
            _socketListenerAccepted = socketListenerAccepted;
        }

        /// <summary>
        /// PC側アプリからの接続待ち受けの開始。
        /// portNoで待ち受けるポートを指定します。
        /// </summary>
        public void Start(int portNo)
        {
            lock (_syncPublic)
            {
                if (_bActive)
                {
                    return;
                }

                IPEndPoint[] endPoints = new IPEndPoint[2];

                int socketNum = 0;
                if (Socket.OSSupportsIPv6)
                {
                    endPoints[socketNum++] = new IPEndPoint(IPAddress.IPv6Any, portNo);
                }
                if (Socket.OSSupportsIPv4)
                {
                    endPoints[socketNum++] = new IPEndPoint(IPAddress.Any, portNo);
                }

                // IPv4, IPv6 ともに動作していない場合はとりあえず例外を発生させる。
                // 実際にこのパターンに陥るのかどうかは未検証
                if (socketNum == 0)
                {
                    throw new SocketException();
                }

                _acceptSockets = new SocketParam[socketNum];

                try
                {
                    for (int i = 0; i < _acceptSockets.Length; ++i)
                    {
                        Socket socket = new Socket(endPoints[i].AddressFamily, SocketType.Stream, ProtocolType.Tcp);
                        _acceptSockets[i] = new SocketParam(socket);
                        lock (_acceptSockets[i].SyncObj)
                        {
                            socket.Bind(endPoints[i]);
                            socket.Listen((int)SocketOptionName.MaxConnections);
                            BeginAccept(_acceptSockets[i]);
                        }
                    }
                }
                catch (SocketException)
                {
                    // ソケットが作成されている可能性があるので閉じておく
                    CloseSocket();
                    throw;
                }

                _bActive = true;
            }

            //RouterLog.ServerReport("Listener start.");
        }

        /// <summary>
        /// 待ちうけの終了
        /// </summary>
        public void Stop()
        {
            lock (_syncPublic)
            {
                if (! _bActive)
                {
                    return;
                }

                // 受け付け用ソケットを閉じる
                // Socket::BeginAccept()をキャンセルしたいのだが、きれいな方法が見当たらない。
                // とりあえずSocketに対してClose()を呼べば、エラーとなりSocket::BeginAccept()で指定したコールバックが呼ばれるので、
                // コールバックの中で、EndAccept()を呼び出したときに、ObjectDisposedException例外であればこの状態であると判断する。
                CloseSocket();

                _bActive = false;
            }

            //RouterLog.ServerReport("Listener stop.");
        }

        /// <summary>
        /// 接続を待ち受けたソケットをすべて閉じます。
        /// </summary>
        void CloseSocket()
        {
            Debug.Assert(_acceptSockets != null);

            foreach (SocketParam socketParam in _acceptSockets)
            {
                if (socketParam != null)
                {
                    lock (socketParam.SyncObj)
                    {
                        socketParam.Socket.Close();
                    }
                }
            }

            _acceptSockets = null;
        }

        /// <summary>
        /// Socket.BeginAccept() に指定するデリゲート
        /// </summary>
        /// <param name="ar"></param>
        void CallbackAccept(IAsyncResult ar)
        {
            SocketParam socketParam = (SocketParam)ar.AsyncState;
            lock (socketParam.SyncObj)
            {
                Socket socket = null;
                try
                {
                    socket = socketParam.Socket.EndAccept(ar);
                }
                catch (ObjectDisposedException)     // Stop()にてSocketのClose()が呼ばれた。
                {
                    // Debug.WriteLine("Socket Listener : Disposed");
                    return;
                }
                catch (SocketException e)
                {
                    _socketListenerAccepted(new SocketListenerResult(null, e));
                    return;
                }

                Debug.WriteLine("Accept socket : Address family " + socket.AddressFamily);
                _socketListenerAccepted(new SocketListenerResult(socket, null));

                try
                {
                    BeginAccept(socketParam);  // 次の Accept処理
                }
                catch (SocketException e)
                {
                    _socketListenerAccepted(new SocketListenerResult(null, e));
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        void BeginAccept(SocketParam param)
        {
            param.Socket.BeginAccept(CallbackAccept, param);
            // Debug.WriteLine("Socket Listener : BeginAccept " + param.Socket.AddressFamily);
        }

        /// <summary>
        /// Accepty用ソケットを管理する構造
        /// </summary>
        class SocketParam
        {
            public readonly Socket Socket;
            public readonly object SyncObj = new object();

            public SocketParam(Socket socket)
            {
                Socket = socket;
            }
        }
    }

    /// <summary>
    /// ソケットAccept終了イベントの引数
    /// </summary>
    public class SocketListenerResult
    {
        public readonly Socket Socket;
        public readonly Exception Exception;

        public SocketListenerResult(Socket socket, Exception exception)
        {
            Socket = socket;
            Exception = exception;
        }
    }
}
