﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Threading;

namespace Nintendo.McsServer
{
    /// <summary>
    /// DataPacketにスペシャライズしたQueue。
    /// スレッドセーフである。
    ///
    /// Routerは、本クラスのインスタンスを保持して、
    /// ポーリング時にデータを取り出しては、通信路に送信する(実機へと転送される)。
    ///
    /// 各チャンネル(MCSサーバと通信するプログラムの通信単位)は、
    /// ソケット通信によって非同期に読み込んだデータを本キューにつめてゆく。
    /// </summary>
    public class DataPacketQueue
    {
        /// <summary>
        /// DataPacketキュー
        /// </summary>
        readonly Queue<DataPacket> _queue = new Queue<DataPacket>();
        /// <summary>
        ///
        /// </summary>
        readonly EventWaitHandle _newItemEvent = new EventWaitHandle(false, EventResetMode.ManualReset);

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DataPacketQueue()
        { }

        /// <summary>
        /// キューに登録されたDataPacketの数を取得します。
        /// </summary>
        public int Count
        {
            get
            {
                lock (((ICollection)_queue).SyncRoot)
                {
                    return _queue.Count;
                }
            }
        }

        /// <summary>
        /// Queueに1つでもアイテムがある場合はシグナル状態になるイベント
        /// </summary>
        public EventWaitHandle NewItemEvent
        {
            get { return _newItemEvent; }
        }

        /// <summary>
        /// キューに登録します。
        /// </summary>
        public void Enqueue(DataPacket channelPacket)
        {
            lock (((ICollection)_queue).SyncRoot)
            {
                _queue.Enqueue(channelPacket);
                _newItemEvent.Set();
            }
        }

        /// <summary>
        /// キューから取り出します。
        /// </summary>
        public DataPacket Dequeue()
        {
            lock (((ICollection)_queue).SyncRoot)
            {
                DataPacket channelPacket = _queue.Dequeue();
                if (_queue.Count == 0)
                {
                    _newItemEvent.Reset();
                }

                return channelPacket;
            }
        }

        /// <summary>
        /// キューの内容を破棄します。
        /// </summary>
        public void Clear()
        {
            lock (((ICollection)_queue).SyncRoot)
            {
                _queue.Clear();
                _newItemEvent.Reset();
            }
        }
    }
}
