﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Net.Sockets;
using System.Diagnostics;
using System.IO;

namespace Nintendo.McsServer
{
    /// <summary>
    /// Channelnfo の概要の説明です。
    ///
    /// 以下のように、動作状態を以降します。
    ///
    /// 登録フェーズ：DoChannelRegist()でServerCommand.ChannelRequestパケットの待ちうけ
    /// 読み取りフェーズ：DoDataRead()でデータ読みとり
    ///
    /// </summary>
    public class ChannelInfo
    {
        /// <summary>
        /// 無効なチャンネル
        /// </summary>
        public const ushort InvalidChannel = 0xFFFF;

        /// <summary>
        /// (コマンド)パケットヘッダサイズ
        /// </summary>
        const int ChannelChunkSize = 8;

        /// <summary>
        /// チャンネル番号
        /// </summary>
        ushort _channelNo;

        /// <summary>
        /// アプリとMCSサーバ間の通信に利用するソケット
        /// </summary>
        readonly Socket     _socket;

        /// <summary>
        /// 同期用オブジェクト。
        /// </summary>
        readonly object _syncObj = new object();

        /// <summary>
        /// ブロック読み込みラッパークラス
        /// </summary>
        readonly BlockingReader _blockingReader;

        /// <summary>
        /// チャンネルを管理しているルータへの参照
        /// </summary>
        readonly RouterInfo _routerInfo;

        /// <summary>
        /// データバッファ
        /// アプリからMCSサーバへの書き込みは、ソケットを通じて、
        /// 一旦このバッファに書き込まれ、
        /// その後、DataPacket の形で、_routerInfo.DataPacketQueueに書き込まれる。
        /// DataPacketQueueのデータは、MCSポーリング時に取り出され、
        /// 通信路(HIOなど)に送信される(実機へと転送される)。
        /// </summary>
        readonly byte[]     _dataReadBuf = new byte[32 * 1024];

        /// <summary>
        /// 切断処理用フラグ
        /// Shutdown()が呼ばれたら真になります。
        /// </summary>
        bool                _bShutdown = false;


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ChannelInfo(Socket socket, RouterInfo routerInfo)
        {
            _channelNo = InvalidChannel;
            _socket = socket;
            _routerInfo = routerInfo;
            _blockingReader = new BlockingReader(_socket, _syncObj);
        }

        /// <summary>
        /// チャンネル番号
        /// </summary>
        public ushort ChannelNo
        {
            get
            {
                lock (_syncObj)
                {
                    return _channelNo;
                }
            }
        }


        /// <summary>
        /// チャンネルを登録します。
        /// </summary>
        /// <returns></returns>
        public bool RegistChannel()
        {
            try
            {
                lock (_syncObj)
                {
                    DoChannelRegist();
                }
                return true;
            }
            catch (SocketException e)
            {
                Dispose(e);
                return false;
            }
        }

        /// <summary>
        /// ソケットはブロッキングモードであることが前提。
        /// ブロッキングモードのときは全て送信されたらコールバックが呼ばれる。
        /// </summary>
        public void WriteData(byte[] dataBuf, int offset, int dataBytes)
        {
            try
            {
                lock (_syncObj)
                {
                    _socket.BeginSend(dataBuf, offset, dataBytes, SocketFlags.None, new AsyncCallback(Callback_WriteData), null);
                }
            }
            catch (SocketException e)
            {
                RemoveAndShutdown(e);
            }
            catch (ObjectDisposedException)     // Reader側でSocket.Close()が呼ばれた
            {
                _routerInfo.ChannelManager.Remove(this);
            }
        }

        /// <summary>
        /// 切断します。
        /// </summary>
        public void Shutdown()
        {
            lock (_syncObj)
            {
                if (_bShutdown)
                {
                    return;
                }

                try
                {
                    _socket.Shutdown(SocketShutdown.Both);
                    RouterLog.ServerReport(GetInfoStr() + " : socket shutdown.");
                }
                catch (SocketException e)
                {
                    RouterLog.ServerReport(ServerReportType.InternalError, GetInfoStr() + " : " + e);
                }

                _bShutdown = true;
            }
        }

        /// <summary>
        /// PC側アプリからMCSサーバに対するチャンネル登録要求コマンド送信を
        /// 待ち受けます。
        ///
        /// ChannelChunkSizeを読み込むためのソケット読み込みを開始します。
        /// (_blockingReaderなので、何か受信するまでブロックします。）
        ///
        /// 読み込み完了時にCallback_ChunkHeader()をコールバックして
        /// 続きの処理を行います。
        /// </summary>
        void DoChannelRegist()
        {
            byte[] readBuf = new byte[ChannelChunkSize];
            _blockingReader.BeginRead(readBuf, Callback_ChunkHeader, readBuf);
        }

        /// <summary>
        /// チャンクヘッダ受信を処理します。
        /// DoChannelRegist()で
        /// ChannelChunkSizeのデータを受信した後に実行する処理です。
        /// </summary>
        void Callback_ChunkHeader(BlockingReaderContext context)
        {
            try
            {
                if (! context.GetResult())
                {
                    // データが最後まで読み取れなかった場合は、shutdownされたものと考える。
                    RemoveAndDispose(null);
                    return;
                }

                NetBinaryReader br = new NetBinaryReader(new MemoryStream((byte[])context.State));
                ServerCommand chunkID = (ServerCommand)br.ReadUInt32();
                uint msgBytes = br.ReadUInt32();

                if (msgBytes > 0x10000) // データが大きい場合は、socket自体を閉じてしまう。
                {
                    RemoveAndDispose(null);
                    return;
                }

                lock (_syncObj)
                {
                    byte[] cmdMsg = new byte[msgBytes];

                    // チャンネル登録要求コマンドか
                    if (chunkID == ServerCommand.ChannelRequest && msgBytes == 4 + 4)
                    {
                        _blockingReader.BeginRead(cmdMsg, Callback_Request, cmdMsg);
                    }
                    else
                    {
                        // 知らないコマンドのときは、unknownレスポンスを返す
                        _blockingReader.BeginRead(cmdMsg, Callback_UnknownData, cmdMsg);
                    }
                }
            }
            catch (SocketException e)
            {
                RemoveAndDispose(e);
            }
        }

        /// <summary>
        /// ServerCommand.ChannelRequest の読み込みが完了したときに
        /// コールバックされます。
        /// その後の、チャンネル登録処理を行います。
        /// レスポンス(ChannelRequestResult)を返しています。
        /// </summary>
        void Callback_Request(BlockingReaderContext context)
        {
            try
            {
                if (! context.GetResult())
                {
                    // データが最後まで読み取れなかった場合は、shutdownされたものと考える。
                    RemoveAndDispose(null);
                    return;
                }

                NetBinaryReader br = new NetBinaryReader(new MemoryStream((byte[])context.State));
                ushort channel = (ushort)br.ReadUInt32();
                uint flag = br.ReadUInt32();

                ChannelRequestResult result = _routerInfo.ChannelManager.RegistChannel(this, channel, flag);

                // レスポンスを返す
                const uint msgSize = 4;
                byte[] resBuf = new byte[4 + 4 + msgSize];
                NetBinaryWriter bw = new NetBinaryWriter(new MemoryStream(resBuf, true));
                bw.Write((uint)ServerCommand.ChannelRequest);
                bw.Write(msgSize);
                bw.Write((uint)result);
                bw.Flush();

                lock (_syncObj)
                {
                    // 登録に成功したら、チャンネル値を設定
                    if (result == ChannelRequestResult.Success)
                    {
                        _channelNo = channel;
                    }

                    _socket.Send(resBuf);

                    if (result == ChannelRequestResult.Success)
                    {
                        DoDataRead();    // データ受信フェーズへ
                    }
                    else
                    {
                        DoChannelRegist();     // 再度コマンド受付へ
                    }
                }
            }
            catch (SocketException e)
            {
                RemoveAndDispose(e);
            }
        }

        /// <summary>
        /// 未知のコマンド送信に対する応答です。
        /// </summary>
        void Callback_UnknownData(BlockingReaderContext context)
        {
            try
            {
                if (! context.GetResult())
                {
                    // データが最後まで読み取れなかった場合は、shutdownされたものと考える。
                    RemoveAndDispose(null);
                    return;
                }

                // レスポンスを返す
                const uint msgSize = 0;
                byte[] resBuf = new byte[4 + 4 + msgSize];
                NetBinaryWriter bw = new NetBinaryWriter(new MemoryStream(resBuf, true));
                bw.Write((uint)ServerCommand.Unknown);
                bw.Write(msgSize);
                bw.Flush();

                lock (_syncObj)
                {
                    _socket.Send(resBuf);

                    DoChannelRegist();     // 再度コマンド受付へ
                }
            }
            catch (SocketException e)
            {
                RemoveAndDispose(e);
            }
        }

        /// <summary>
        /// PC側アプリからMCSサーバへの送信データを読み込みます。
        /// データ受信フェーズです。
        ///
        /// 終了コールバックで、本関数を再度呼び出ししているので、
        /// ソケットが切断されるまで処理が繰り返されるようになっています。
        /// </summary>
        void DoDataRead()
        {
            _socket.BeginReceive(_dataReadBuf, 0, _dataReadBuf.Length, SocketFlags.None, new AsyncCallback(Callback_DataRead), null);
        }

        /// <summary>
        /// 読み込み完了時コールバック
        /// </summary>
        void Callback_DataRead(IAsyncResult ar)
        {
            // データ受信。
            ChannelInfo channelInfo = (ChannelInfo)ar.AsyncState;

            int readBytes = 0;

            try
            {
                lock (_syncObj)
                {
                    readBytes = _socket.EndReceive(ar);
                }
            }
            catch (SocketException e)
            {
                RemoveAndDispose(e);
                return;
            }

            if (readBytes == 0)     // 全て読み込んだ場合
            {
                Debug.WriteLine(GetInfoStr() + " : read all.");
                RemoveAndDispose(null);
                return;
            }

            // 読み込んだデータからDataPacketを生成。
            // routerのDataPacketQueueに登録する。
            // DataPacketQueueのデータは、MCSポーリング時に取り出され、
            // 通信路(HIOなど)に送信される(実機へと転送される)。
            lock (_syncObj)
            {
                // 切断後に古いコネクションのツールから到着した
                // メッセージを再接続した実機アプリに送信すると、
                // 実機アプリとツールとの状態が整合しなくなるので、
                // シャットダウン後に到着したメッセージは無視する。
                if (!_bShutdown)
                {
                    DataPacket dataPacket = DataPacket.CopyAndCreate(_channelNo, _dataReadBuf, readBytes);
                    _routerInfo.DataPacketQueue.Enqueue(dataPacket);
                }
            }

            try
            {
                lock (_syncObj)
                {
                    // 全て読み込んでいないので、再度読み込みを実行する。
                    DoDataRead();
                }
            }
            catch (SocketException e)
            {
                RemoveAndDispose(e);
            }
        }

        /// <summary>
        /// ソケットへの書き込み完了時に、コールバックされる関数
        /// </summary>
        /// <param name="ar"></param>
        void Callback_WriteData(IAsyncResult ar)
        {
            try
            {
                lock (_syncObj)
                {
                    _socket.EndSend(ar);
                }
            }
            catch (SocketException e)
            {
                RemoveAndShutdown(e);
            }
            catch (ObjectDisposedException)     // Reader側でSocket.Close()が呼ばれた
            {
                _routerInfo.ChannelManager.Remove(this);
            }
        }

        /// <summary>
        /// チャンネルを登録削除してシャットダウンします。
        /// </summary>
        void RemoveAndShutdown(SocketException e)
        {
            _routerInfo.ChannelManager.Remove(this);

            if (e != null)
            {
                ReportException(e);
            }

            Shutdown();
        }

        /// <summary>
        /// チャンネルを登録削除して破棄します。
        /// </summary>
        void RemoveAndDispose( SocketException e )
        {
            _routerInfo.ChannelManager.Remove(this);
            Dispose(e);
        }

        /// <summary>
        /// チャンネルを破棄します。
        /// 例外が発生していた場合は例外情報を出力します。
        /// </summary>
        void Dispose(SocketException e)
        {
            if (e != null)
            {
                ReportException(e);
            }

            DisposeCore();
        }

        /// <summary>
        /// チャンネルを破棄します。
        /// </summary>
        void DisposeCore()
        {
            lock (_syncObj)
            {
                Shutdown();

                string infoStr = GetInfoStr();  // Socket.Close()を呼ぶ前に情報を取得しておく
                try
                {
                    _socket.Close();
                    RouterLog.ServerReport(infoStr + " : socket closed.");
                }
                catch (SocketException e)
                {
                    RouterLog.ServerReport(ServerReportType.InternalError, infoStr + " : " + e);
                }
            }
        }

        /// <summary>
        /// 説明文字列を取得します。
        /// </summary>
        string GetInfoStr()
        {
            lock (_syncObj)
            {
                if (_channelNo == InvalidChannel)
                {
                    return string.Format("Socket [{0}]", _socket.RemoteEndPoint);
                }
                else
                {
                    return string.Format("Channel {0:X4}({0:d})", _channelNo);
                }
            }
        }

        /// <summary>
        /// 例外に応じたログを出力します。
        /// </summary>
        void ReportException(SocketException e)
        {
            switch (e.SocketErrorCode)
            {
            case SocketError.Shutdown:
            case SocketError.ConnectionReset:   // リモート側でSocket::Shutdown()が呼ばれた。
            case SocketError.ConnectionAborted: // Disconnect()が呼ばれた
                Debug.WriteLine(GetInfoStr() + " : " + e.SocketErrorCode);
                break;
            default:
                RouterLog.ServerReport(GetInfoStr() + " : " + e);
                break;
            }
        }
    }
}
