﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Reflection;
using System.Diagnostics;
using System.Net.Sockets;
using System.IO;
using System.Threading;

/*
    コメント
        MCS Serverに送るときは、Socket.BeginSend()を使用する。
        (Socket.Send()だと、ブロックする恐れがある。また、このタスクの場合送信結果を特に気にする必要がないため。)
        このとき、BeginSend()に渡すコールバックの中で、一応EndSend()を呼びだしておくこと。
        (そうしないと、メモリリークが発生するため。)

        MouseMoveのイベントに関しては、もしかしたら発生頻度が高い可能性がある。
        その場合は、送信パケットをまとめるなどの工夫が要りそうだ。

        接続のタイプを File I/O のようにするか、それとも通常タイプにするか。
        通常タイプだと、クライアントが接続するまで、入力できないようにする必要があり、
        切断されたら、使用不可にしなければならない。
        そもそも、現状では、接続された通知が無いので、接続を確認するには、
        チャンネル登録を頻繁に行う必要がある。

        File I/Oタイプにすると、
        ・クライアントが接続しないと送れない?
            → この場合は、クライアントで準備ができたことを示す関数がいる。
                McsHID_Start() みたいな。
        ・接続されるまで送れるとすると、クライアントで無視する必要がある。

        File I/O のように、Routerの管理にするか。
        (つまり、別スレッド)
        この場合、Formのみ、メインのスレッドから制御するようにする必要がある。
        (Form.Invoke()メソッドなど)
 */

/*
    仕様
        USBのHIDコードでは、数値パッドのEndキーと通常のEndキーは区別されるが、
        Windowsの仮想キーコードのレベルでは違いはないため区別しない(できない)。

        未対応のキー
            英数キー                ... Keys.Noneが返る
            半角/全角キー           ... Keys.Noneが返る
            カタカナ・ひらがなキー  ... Keys.Noneが返る
            ローマ字キー            ... Keys.Attnが返る
            (Alt + カタカナ・ひらがな+
            Breakキー(Ctrl + Pause)
            Escキー                 ... なぜか、Altキー押下のあと続けると、EscキーのKeyUpのみ発生する
            左のWindowsロゴキー
            右のWindowsロゴキー
            アプリケーションキー
 */

/*
 *  メモ
 *      ホイールは、カーソルの位置は取得できるが、ボタンは入ってこない。
 *      マウスDown, Upは、1つのボタンの状態のみが返る。また、クリックした位置でのポイントも返る。
 *      MouseMoveは、そのときのボタンの状態が返る。(複数OR)
 */

namespace Nintendo.McsServer
{
    public partial class CaptureForm : Form
    {
        const string _localHost = "localhost";
        int _port;

        const uint _channel = 0xFF7E;
        const int _timeout = 5000;
        SyncedTCPClient _tcpClient;

        /// <summary>
        /// リードスレッド停止用イベント。
        /// </summary>
        ManualResetEvent _doneEvent;

        KeyboardHook _keyboardHook = new KeyboardHook();
        MouseHook _mouseHook = new MouseHook();
        Point _hookPoint = Point.Empty;

        Size _doubleClickDelta = Size.Empty;
        MouseHookEventArgs.MouseButtons _firstMouseDownButton = MouseHookEventArgs.MouseButtons.None;
        DateTime _firstMouseDownTime = DateTime.MaxValue;
        bool _isDoubleClicked = false;

        McsMouseButton _mcsMouseButtons = McsMouseButton.None;
        KeyModifier _mcsKeyModifier = KeyModifier.None;

        /// <summary>
        /// MouseMoveイベント発生個数
        /// </summary>
        int _mouseMoveCounter = 0;

        bool _isCaptured = false;

        object _messageLock = new Object();
        bool _isMessageShowing = false;

        /// <summary>
        /// 例外発生状態。
        /// 例外を起こす処理を繰り返し行わないようにする。
        /// </summary>
        bool _isExcepted = false;

        public CaptureForm()
        {
            InitializeComponent();

            Text = Application.ProductName + " - " + Properties.Resources.CaptureFormTitle;

            Icon = MainForm.GetApplicationIcon();   // フォームのアイコン設定

            _keyboardHook.KeyDown += OnHookKeyDown;
            _keyboardHook.KeyUp += OnHookKeyUp;
            _keyboardHook.SystemKeyDown += OnHookKeyDown;
            _keyboardHook.SystemKeyUp += OnHookKeyUp;

            _mouseHook.MouseDown += OnHookMouseDown;
            _mouseHook.MouseUp += OnHookMouseUp;
            _mouseHook.MouseMove += OnHookMouseMove;
            _mouseHook.MouseWheel += OnHookMouseWheel;
        }

        public int Port
        {
            set { _port = value; }
        }

        private void CaptureForm_Load(object sender, EventArgs e)
        {
            if (!DesignMode)
            {
            }
        }

        private void CaptureForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            ReleaseInput();

            Debug.WriteLine("CaptureForm closed.");
        }

        /// <summary>
        /// キャプチャボタン
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnCapture_Click(object sender, EventArgs e)
        {
            this.StartCapture();
        }

        /// <summary>
        /// デアクティブ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void CaptureForm_Deactivate(object sender, EventArgs e)
        {
            ReleaseInput();
        }

        private void DoMouseDown(MouseHookEventArgs.MouseButtons mouseButton)
        {
            _isDoubleClicked = false;

            if (_firstMouseDownButton != mouseButton)
            {
                ResetDoubleClickParameters(mouseButton);
                return;
            }

            TimeSpan delta = DateTime.Now - _firstMouseDownTime;

            if (delta < TimeSpan.Zero || delta.TotalMilliseconds > SystemInformation.DoubleClickTime)
            {
                ResetDoubleClickParameters(mouseButton);
                return;
            }

            Size size = SystemInformation.DoubleClickSize;

            if (Math.Abs(_doubleClickDelta.Width) > SystemInformation.DoubleClickSize.Width)
            {
                ResetDoubleClickParameters(mouseButton);
                return;
            }

            if (Math.Abs(_doubleClickDelta.Height) > SystemInformation.DoubleClickSize.Height)
            {
                ResetDoubleClickParameters(mouseButton);
                return;
            }

            _firstMouseDownTime = DateTime.MaxValue;
            _firstMouseDownButton = MouseHookEventArgs.MouseButtons.None;

            _isDoubleClicked = true;
        }

        private void ResetDoubleClickParameters(MouseHookEventArgs.MouseButtons mouseDownButton)
        {
            _firstMouseDownButton = mouseDownButton;
            _firstMouseDownTime = DateTime.Now;
            _doubleClickDelta = Size.Empty;
        }

        private void OnHookKeyDown(object sender, KeyboardHookEventArgs e)
        {
            e.IsCanceled = true;

            if ((Keys)e.KeyCode == Keys.Escape)
            {
                ReleaseInput();
                return;
            }

            UpdateMcsKeyModifier(e.KeyCode, true);

            McsKeyEventArg eventArg = GenMcsKeyEventArg(e);
            if (eventArg == null)
            {
                OutputConsole(string.Format("Key Down  : Unhandled {0:X2} {1} ", (int)e.KeyCode, e.KeyCode));
                return;
            }

            if (WriteEventMessage(McsEventType.KeyDown, eventArg))
            {
                OutputConsole("Key Down  : " + eventArg.ToInfoStr());
            }
        }

        private void OnHookKeyUp(object sender, KeyboardHookEventArgs e)
        {
            McsKeyEventArg eventArg = GenMcsKeyEventArg(e);
            if (eventArg == null)
            {
                OutputConsole(string.Format("Key Up    : Unhandled {0:X2} {1} ", (int)e.KeyCode, e.KeyCode));
                return;
            }

            UpdateMcsKeyModifier(e.KeyCode, false);

            if (WriteEventMessage(McsEventType.KeyUp, eventArg))
            {
                OutputConsole("Key Up    : " + eventArg.ToInfoStr());
            }
        }

        private void tbxLog_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (!_isCaptured)
            {
                return;
            }

            McsCharEventArg eventArg = new McsCharEventArg();
            eventArg.keyChar = e.KeyChar;
            if (WriteEventMessage(McsEventType.KeyPress, eventArg))
            {
                OutputConsole("Char      : " + eventArg.ToInfoStr());
            }
        }

        void OnHookMouseMove(object sender, MouseHookEventArgs e)
        {
            Point dist = new Point(e.X - _hookPoint.X, e.Y - _hookPoint.Y);
            if (dist.X != 0 || dist.Y != 0)
            {
                _mouseMoveCounter++;

                // 1回目～2回目クリック間の移動量を計算する
                if (_firstMouseDownTime != DateTime.MaxValue)
                {
                    _doubleClickDelta.Width += dist.X;
                    _doubleClickDelta.Height += dist.Y;
                }

                McsMouseMoveEventArg eventArg = new McsMouseMoveEventArg();
                eventArg.button = _mcsMouseButtons;
                eventArg.point = dist;
                if (WriteEventMessage(McsEventType.MouseMove, eventArg))
                {
                    // string message = string.Format("移動値 {0,5:d} [{1,4:d},{2,4:d}]", _mouseMoveCounter, dist.X, dist.Y);
                    // lblMouseMove.Text = message;

                    // OutputConsole("Mouse Move : " + eventArg.ToInfoStr());
                }

                // マウスカーソル移動をキャンセルする
                e.IsCanceled = true;
            }
        }

        void OnHookMouseDown(object sender, MouseHookEventArgs e)
        {
            UpdateMcsMouseButtons(e.Button, true);

            DoMouseDown(e.Button);

            McsMouseButtonEventArg eventArg = MakeMouseButtonEventArg(e);

            if (WriteEventMessage(
                _isDoubleClicked ? McsEventType.MouseDoubleClick : McsEventType.MouseDown,
                eventArg))
            {
                if (_isDoubleClicked)
                {
                    OutputConsole("Mouse DoubleClick : " + eventArg.ToInfoStr());
                }
                else
                {
                    OutputConsole("Mouse Down : " + eventArg.ToInfoStr());
                }
            }

            // MouseDown をキャンセルする
            e.IsCanceled = true;
        }

        void OnHookMouseUp(object sender, MouseHookEventArgs e)
        {
            McsMouseButtonEventArg eventArg = MakeMouseButtonEventArg(e);
            if (WriteEventMessage(McsEventType.MouseUp, eventArg))
            {
                OutputConsole("Mouse Up   : " + eventArg.ToInfoStr());
            }

            UpdateMcsMouseButtons(e.Button, false);

            // MouseUp をキャンセルする
            e.IsCanceled = true;
        }

        void OnHookMouseWheel(object sender, MouseHookEventArgs e)
        {
            McsMouseWheelEventArg eventArg = new McsMouseWheelEventArg();
            int line = e.Delta * SystemInformation.MouseWheelScrollLines / 120;
            eventArg.moveLine = (short)line;

            if (WriteEventMessage(McsEventType.MouseWheel, eventArg))
            {
                OutputConsole("Mouse Wheel: " + eventArg.ToInfoStr());
            }

            // MouseWheel をキャンセルする
            e.IsCanceled = true;
        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // Ctrl + D をボタンのショートカットキーとして処理する
            if ((int)keyData == (int)Keys.Control + (int)Keys.D)
            {
                this.btnCapture.PerformClick();
                return true;
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        void OutputConsole(string str)
        {
            int MaxTextLength = 64 * 1024;

            bool bReadOnly = tbxLog.ReadOnly;
            tbxLog.ReadOnly = false;
            if (tbxLog.TextLength + str.Length > MaxTextLength)
            {
                if (str.Length > MaxTextLength)
                {
                    str.Remove(0, str.Length - MaxTextLength);
                    tbxLog.Clear();
                }
                else
                {
                    tbxLog.Select(0, tbxLog.TextLength + str.Length - MaxTextLength);
                    tbxLog.SelectedText = string.Empty;
                }
            }

            tbxLog.Select(tbxLog.TextLength, 0);
            tbxLog.SelectedText = str + "\n";
            if (tbxLog.Focused)
            {
                tbxLog.ScrollToCaret();
            }

            tbxLog.ReadOnly = bReadOnly;
        }

        /// <summary>
        /// 入力のキャプチャを開始します。
        /// </summary>
        public void StartCapture()
        {
            if (!this.IsCaputuring)
            {
                Debug.Assert(_port != 0);   // 設定されていること

                CaptureInput();
            }
        }

        /// <summary>
        /// 入力キャプチャ中かどうかを判定します。
        /// </summary>
        public bool IsCaputuring
        {
            get
            {
                return _isCaptured;
            }
        }

        /// <summary>
        /// 入力キャプチャを停止します。
        /// </summary>
        public void StopCapture()
        {
            if (this.IsCaputuring)
            {
                this.ReleaseInput();
            }
        }


        private void ReadThreadProc(SyncedTCPClient tcpClient, ManualResetEvent doneEvent)
        {
            McsUtil.McsSocket mcsSocket = new McsUtil.McsSocket(tcpClient.BaseConnection.Client, tcpClient.SyncRoot);
            mcsSocket.CancelWaitHandle = doneEvent;
            byte[] buffer = new byte[RequestPacketHeaderSize];
            BinaryReader binReader = new McsUtil.NetBinaryReader(new MemoryStream(buffer));

            RequestPacketHeader header = new RequestPacketHeader();
            try
            {
                while (true)
                {
                    if (doneEvent.WaitOne(0))
                    {
                        throw new OperationCanceledException();
                    }

                    mcsSocket.Read(buffer, 0, RequestPacketHeaderSize);
                    binReader.BaseStream.Position = 0;
                    header.request = (RequestID)binReader.ReadUInt32();
                    header.bodySize = binReader.ReadInt32();

                    bool bodyProcessed = false;
                    switch (header.request)
                    {
                        case RequestID.StopCapture:
                            this.BeginInvoke(new Action(() =>
                            {
                                this.ReleaseInput();
                            }));
                            break;
                    }

                    if (!bodyProcessed && header.bodySize > 0)
                    {
                        mcsSocket.Skip(header.bodySize);
                    }
                }

                // ここには来ません。
            }
            catch (EndOfStreamException)
            {
                // ソケットが閉じられたのでスレッド終了。
            }
            catch (OperationCanceledException)
            {
                // スレッドの終了が指示されたのでスレッド終了。
            }
            catch (Exception ex)
            {
                _isExcepted = true;

                // 通信エラーが起こったら、フォームのスレッドに処理してもらう
                this.BeginInvoke(new Action(() =>
                {
                    if (ex.InnerException != null && ex.InnerException is SocketException)
                    {
                        OutputConsole(((SocketException)ex.InnerException).SocketErrorCode.ToString());
                    }
                    else
                    {
                        OutputConsole("Exception");
                        Debug.WriteLine(ex.ToString());
                    }

                    this.ReleaseInput();
                }));
            }
            finally
            {
                // コネクションのクローズは ReleaseInput() で行われるので、
                // McsSocket の Close() および Dispose() はここでは呼び出しません。
            }
        }

        /// <summary>
        /// 入力キャプチャのスタート処理。
        /// 失敗した場合は キャプチャダイアログを終了します。
        /// </summary>
        private void CaptureInput()
        {
            if (_isCaptured)
            {
                return;
            }

            _tcpClient = CreateTCPClient();
            if (_tcpClient == null)
            {
                if (!_isMessageShowing)
                {
                    this.Close();
                }

                return;
            }

            _doneEvent = new ManualResetEvent(false);
            Thread readThread = new Thread(new ThreadStart(() =>
                {
                    this.ReadThreadProc(_tcpClient, _doneEvent);
                }));

            btnCapture.Enabled = false;

            // フォーカスをテキストボックスに移す
            tbxLog.Focus();

            _isCaptured = true;
            _isExcepted = false;
            _hookPoint = Cursor.Position;
            _mcsMouseButtons = McsMouseButton.None;

            _keyboardHook.Start();
            _mouseHook.Start();
            readThread.Start();

            this.WriteEventMessage(McsEventType.CaptureStart, new McsCaptureEventArg());
        }

        SyncedTCPClient CreateTCPClient()
        {
            TcpClient tcpClient = new TcpClient(_localHost, _port);
            tcpClient.ReceiveTimeout = _timeout;
            tcpClient.SendTimeout = _timeout;

            ChannelRequestResult result = ChannelRequestResult.NotConnect;
            try
            {
                result = ChannelRequest.Request(tcpClient.GetStream(), _channel, 0);
                if (result == ChannelRequestResult.NotConnect)
                {
                    ShowMessageBox(Properties.Resources.ErrorMCSServerNotConnect);
                }
                else if (result != ChannelRequestResult.Success)
                {
                    ShowMessageBox(Properties.Resources.ErrorMCSConnectFail);
                }
            }
            catch (IOException ex)
            {
                string message = Properties.Resources.ErrorMCSConnectFail;
                if (ex.InnerException != null && ex.InnerException is SocketException)
                {
                    SocketException sockEx = (SocketException)ex.InnerException;
                    message += "\n" + sockEx.SocketErrorCode;
                }
                else
                {
                    message += "\n" + ex.Message;
                }

                ShowMessageBox(message);
            }

            if (result != ChannelRequestResult.Success)
            {
                tcpClient.Close();

                return null;
            }

            return new SyncedTCPClient(tcpClient);
        }

        /// <summary>
        /// 入力をリリースし、キャプチャダイアログを終了します。
        /// </summary>
        private void ReleaseInput()
        {
            if (! _isCaptured)
            {
                return;
            }

            _keyboardHook.Stop();
            _mouseHook.Stop();
            _isCaptured = false;
            _hookPoint = Point.Empty;

            // キャプチャ終了メッセージの送信。
            if (!_isExcepted)
            {
                this.WriteEventMessage(McsEventType.CaptureStop, new McsCaptureEventArg());
            }

            _doneEvent.Set();
            _doneEvent = null;

            _tcpClient.Close();
            _tcpClient = null;

            Cursor.Clip = new Rectangle();

            // lblMouseMove.Text = string.Empty;
            btnCapture.Enabled = true;

            this.Close();
        }

        class AsyncWriteState
        {
            public SyncedTCPClient tcpClient;
            public CaptureForm captureForm;
        }

        bool WriteEventMessage(McsEventType eventType, McsEventArg arg)
        {
            MemoryStream ms = new MemoryStream();
            WriteEventType(ms, eventType);
            arg.ToStream(ms);

            try
            {
                AsyncWriteState asyncWriteState = new AsyncWriteState();
                asyncWriteState.tcpClient = _tcpClient;
                asyncWriteState.captureForm = this;
                _tcpClient.BeginWrite(ms.GetBuffer(), 0, (int)ms.Length, WriteAsyncCallback, asyncWriteState);
            }
            catch (IOException ex)
            {
                HandleIOException(ex);
                return false;
            }
            return true;
        }

        void WriteEventType(Stream stream, McsEventType eventType)
        {
            BinaryWriter bw = new BinaryWriter(stream);
            byte byteData = (byte)eventType;
            bw.Write(byteData);
            byteData = 0;
            bw.Write(byteData);
            bw.Write(byteData);
            bw.Write(byteData);
            bw.Flush();
        }

        /// <summary>
        /// BeginWrite()完了時にスレッドプールのスレッドにより呼ばれる。
        /// そのため、他のCaptureFormメソッドとは異なるスレッドである。
        /// </summary>
        /// <param name="ar"></param>
        static void WriteAsyncCallback(IAsyncResult ar)
        {
            AsyncWriteState state = (AsyncWriteState)ar.AsyncState;

            try
            {
                state.tcpClient.EndWrite(ar);
            }
            catch (IOException ex)
            {
                // 通信エラーが起こったら、フォームのスレッドに処理してもらう
                state.captureForm.BeginInvoke(
                    new AsyncIOExceptionDelegate(state.captureForm.EndWriteExceptionDelegate),
                    new object[] { ex }
                );
            }
        }

        delegate void AsyncIOExceptionDelegate(IOException ex);

        void EndWriteExceptionDelegate(IOException ex)
        {
            // 通信エラーが起こったら、通信を閉じる。(フォームのスレッドと異なる点に注意)
            HandleIOException(ex);
        }

        void HandleIOException(IOException ex)
        {
            _isExcepted = true;

            if (ex.InnerException != null && ex.InnerException is SocketException)
            {
                OutputConsole(((SocketException)ex.InnerException).SocketErrorCode.ToString());
            }
            else
            {
                OutputConsole("IOException");
                Debug.WriteLine(ex.ToString());
            }
            ReleaseInput();
        }

        /// <summary>
        /// HIDコードへの変換
        /// </summary>
        void UpdateMcsKeyModifier(int keyCode, bool isKeyDown)
        {
            Keys key = (Keys)keyCode;

            if (isKeyDown)
            {
                _mcsKeyModifier |= ToMcsKeyModifier(key);
            }
            else
            {
                _mcsKeyModifier &= ~ToMcsKeyModifier(key);
            }
        }

        KeyModifier ToMcsKeyModifier(Keys key)
        {
            switch (key)
            {
            case Keys.LShiftKey:
            case Keys.RShiftKey:
                return KeyModifier.Shift;

            case Keys.LControlKey:
            case Keys.RControlKey:
                return KeyModifier.Control;

            case Keys.Menu:
            case Keys.LMenu:
            case Keys.RMenu:
                return KeyModifier.Alt;
            }

            return KeyModifier.None;
        }

        private McsKeyEventArg GenMcsKeyEventArg(KeyboardHookEventArgs e)
        {
            McsKeyEventArg packet = new McsKeyEventArg();
            packet.modifier = _mcsKeyModifier;

            KeyCategory category = (KeyCategory)0;
            int indexValue = 0;

            bool bNumPad = false;

            Keys key = (Keys)e.KeyCode;
            indexValue = (int)key;

            if (Keys.A <= key && key <= Keys.Z)
            {
                category = KeyCategory.Alphabet;
            }
            else if (Keys.D0 <= key && key <= Keys.D9)
            {
                category = KeyCategory.Numeric;
            }
            else if (Keys.NumPad0 <= key && key <= Keys.NumPad9)
            {
                category = KeyCategory.Numeric;
                bNumPad = true;
            }
            else if (Keys.F1 <= key && key <= Keys.F24)
            {
                category = KeyCategory.Function;
            }
            else
            {
                switch (key)
                {
                case Keys.Oemtilde:         // '`'/'~' (106 '@'/'`' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.OemMinus:         // '-'/'_' (106 '-'/'=' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.Oemplus:          // '='/'+' (106 ';'/'+' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.OemPipe:          // '\'/'|' (106 '\'/'|' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.OemOpenBrackets:  // '['/'{' (106 '['/'{' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.OemCloseBrackets: // ']'/'}' (106 ']'/'}' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.OemSemicolon:     // ';'/':' (106 ':'/'*' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.OemQuotes:        // [']/["] (106 [^]/[~] )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.Oemcomma:         // ','/'<' (106 ','/'<' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.OemPeriod:        // '.'/'>' (106 '.'/'>' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.OemQuestion:      // '/' '?' (106 '/' '?' )
                    category = KeyCategory.Symbol;
                    break;
                case Keys.OemBackslash:     // 106 '\' '_'
                    category = KeyCategory.Symbol;
                    break;
                case Keys.Space:
                    category = KeyCategory.Symbol;
                    break;
                case Keys.Divide:           // 数値パッドの'/'
                    category = KeyCategory.Symbol;
                    break;
                case Keys.Multiply:         // 数値パッドの'*'
                    category = KeyCategory.Symbol;
                    break;
                case Keys.Subtract:         // 数値パッドの'-'
                    category = KeyCategory.Symbol;
                    break;
                case Keys.Add:              // 数値パッドの'+'
                    category = KeyCategory.Symbol;
                    break;
                case Keys.Decimal:          // 数値パッドの'.'
                    category = KeyCategory.Symbol;
                    break;

                case Keys.Escape:
                    category = KeyCategory.System;
                    break;
                case Keys.Back:
                    category = KeyCategory.System;
                    break;
                case Keys.Tab:  // Tabキーによってフォーカスを受け取れるキーが他にない場合はこのキーイベントが発生するようです。
                    category = KeyCategory.System;
                    break;
                case Keys.Enter:
                    category = KeyCategory.System;
                    break;
                case Keys.CapsLock:
                    category = KeyCategory.System;
                    break;
                case Keys.ShiftKey:
                case Keys.LShiftKey:
                case Keys.RShiftKey:
                    category = KeyCategory.System;
                    break;
                case Keys.ControlKey:
                case Keys.LControlKey:
                case Keys.RControlKey:
                    category = KeyCategory.System;
                    break;
                case Keys.Menu:
                case Keys.LMenu:
                case Keys.RMenu:
                    category = KeyCategory.System;
                    break;

                case Keys.PrintScreen:
                    category = KeyCategory.System;
                    break;
                case Keys.Scroll:
                    category = KeyCategory.System;
                    break;
                case Keys.Pause:
                    category = KeyCategory.System;
                    break;
#if false
                case Keys.Cancel:           // Ctrl + Pause
                    category = KeyCategory.System;
                    indexValue = (byte)SystemKeyType.Cancel;
                    break;
#endif
                case Keys.Insert:
                    category = KeyCategory.System;
                    break;
                case Keys.Delete:
                    category = KeyCategory.System;
                    break;
                case Keys.Home:
                    category = KeyCategory.System;
                    break;
                case Keys.End:
                    category = KeyCategory.System;
                    break;
                case Keys.PageUp:
                    category = KeyCategory.System;
                    break;
                case Keys.PageDown:
                    category = KeyCategory.System;
                    break;
                case Keys.Up:
                    category = KeyCategory.System;
                    break;
                case Keys.Down:
                    category = KeyCategory.System;
                    break;
                case Keys.Left:
                    category = KeyCategory.System;
                    break;
                case Keys.Right:
                    category = KeyCategory.System;
                    break;
                case Keys.NumLock:
                    category = KeyCategory.System;
                    break;
                case Keys.Clear:
                    category = KeyCategory.System;
                    break;

                case Keys.KanjiMode:        // 106 漢字
                    category = KeyCategory.System;
                    break;
                case Keys.IMENonconvert:    // 106 無変換
                    category = KeyCategory.System;
                    break;
                case Keys.IMEConvert:       // 106 変換
                    category = KeyCategory.System;
                    break;

                default:
                    return null;
                }
            }

            packet.category = category;
            if (bNumPad)
            {
                packet.attribute |= KeyAttribute.NumPad;
            }

            packet.indexValue = (byte)indexValue;

            return packet;
        }

        McsMouseButtonEventArg MakeMouseButtonEventArg(MouseHookEventArgs e)
        {
            McsMouseButtonEventArg eventArg = new McsMouseButtonEventArg();
            eventArg.button = _mcsMouseButtons;
            return eventArg;
        }

        void UpdateMcsMouseButtons(MouseHookEventArgs.MouseButtons mouseButton, bool isMouseDown)
        {
            if(isMouseDown)
            {
                _mcsMouseButtons |= ToMcsMouseButton(mouseButton);
            }
            else
            {
                _mcsMouseButtons &= ~ToMcsMouseButton(mouseButton);
            }
        }

        McsMouseButton ToMcsMouseButton(MouseHookEventArgs.MouseButtons mouseButton)
        {
            switch(mouseButton)
            {
            case MouseHookEventArgs.MouseButtons.Left:
                    return McsMouseButton.Left;

            case MouseHookEventArgs.MouseButtons.Right:
                return McsMouseButton.Right;

            case  MouseHookEventArgs.MouseButtons.Middle:
                return McsMouseButton.Middle;

            // TODO: XButton1, XButton2 のイベントが実際に発生するのか。
            case MouseHookEventArgs.MouseButtons.XButton1:
                return McsMouseButton.X1;

            case MouseHookEventArgs.MouseButtons.XButton2:
                return McsMouseButton.X2;
            }

            return McsMouseButton.None;
        }

        void ShowMessageBox(string message)
        {
            lock (_messageLock)
            {
                if (_isMessageShowing)
                {
                    return;
                }

                _isMessageShowing = true;
            }

            MessageBox.Show(this, message, Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Error);

            _isMessageShowing = false;
        }

        private void CaptureForm_Shown(object sender, EventArgs e)
        {
            if (_port != 0)
            {
                CaptureInput();
            }
        }

        /// <summary>
        /// ターゲットからサーバへのリクエストの識別子。
        /// </summary>
        private enum RequestID : uint
        {
            StopCapture
        }

        /// <summary>
        /// ターゲットからサーバへのリクエストパケットのヘッダ。
        /// </summary>
        private struct RequestPacketHeader
        {
            public RequestID request;
            public int bodySize;
        }

        /// <summary>
        /// ターゲットからサーバへのリクエストパケットのヘッダサイズ。
        /// </summary>
        const int RequestPacketHeaderSize = 8;
    }

    [Flags]
    public enum KeyModifier : byte
    {
        None = 0x00,
        Shift = 0x01,
        Control = 0x02,
        Alt = 0x04,
    }

    public enum KeyCategory : byte
    {
        Function,
        System,
        Alphabet,
        Numeric,
        Symbol,
    }

    public enum SymbolKeyType
    {
        Space,

        Quote,
        Plus,
        Comma,
        Minus,
        Period,
        Slash,
        Colon,
        GraveAccent,
        OpenBlacket,
        Backslash,
        CloseBlacket,
        Pipe,

        Divide,
        Multiply,
        Subtract,
        Add,
        Decimal,
    }

    public enum SystemKeyType : byte
    {
        Escape,
        Backspace,
        Tab,
        Enter,
        CapsLock,
        Shift,
        Control,
        Alt,
        PrintScreen,
        reserved1,          // SysRq
        Scroll,
        Pause,
        reserved2,          // Break
        Insert,
        Delete,
        Home,
        End,
        PageUp,
        PageDown,
        Up,
        Down,
        Left,
        Right,
        NumLock,
        Clear,

        reserved3,          // LWin(左のWindowsロゴキー)
        reserved4,          // RWin(右のWindowsロゴキー)
        reserved5,          // Apps(アプリケーションキー)

        reserved6,          // 半角/全角キー
        Kanji,
        reserved7,          // 英数
        IMENonConvert,
        IMEConvert,
        reserved8,          // カタカナ・ひらがなキー
        reserved9,          // ローマ字キー
    }

    public enum McsEventType
    {
        KeyDown,
        KeyUp,
        KeyPress,

        MouseMove,
        MouseDown,
        MouseUp,
        MouseDoubleClick,
        MouseWheel,

        CaptureStart,
        CaptureStop,
    }

    [Flags]
    public enum KeyAttribute : byte
    {
        NumPad = 0x01,  // 数値キーパッドの数字
    }

    [Flags]
    public enum McsMouseButton
    {
        None        = 0x00,
        Left        = 0x01,
        Right       = 0x02,
        Middle      = 0x04,
        X1          = 0x08,
        X2          = 0x10,
    }

    abstract class McsEventArg
    {
        public abstract void ToStream(Stream stream);

        protected static void WritePadding(NetBinaryWriter bw, int bytes)
        {
            for (int i = 0; i < bytes; ++i)
            {
                byte padding = 0;
                bw.Write(padding);
            }
        }
    }

    class McsKeyEventArg : McsEventArg
    {
        public KeyModifier modifier;
        public KeyAttribute attribute;
        public KeyCategory category;
        public byte indexValue;

        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write((byte)modifier);
            bw.Write((byte)attribute);
            bw.Write((byte)category);
            bw.Write(indexValue);
            WritePadding(bw, 4);
            bw.Flush();
        }

        public override string ToString()
        {
            return "McsKeyEventArg: " + ToInfoStr();
        }

        public string ToInfoStr()
        {
            return string.Format("{0}, {1}, index {2:X2}, {3}", attribute, category, indexValue, modifier);
        }
    }

    class McsCharEventArg : McsEventArg
    {
        public char keyChar;

        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write(keyChar);
            WritePadding(bw, 6);
            bw.Flush();
        }

        public override string ToString()
        {
            return "McsCharEventArg: " + ToInfoStr();
        }

        public string ToInfoStr()
        {
            if (0x21 <= keyChar && keyChar <= 0x7E)
            {
                return keyChar.ToString();
            }
            else
            {
                return string.Format("[0x{0:X2}]", (int)keyChar);
            }
        }
    }

    class McsMouseMoveEventArg : McsEventArg
    {
        public McsMouseButton button = McsMouseButton.None;
        public Point point;

        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write((byte)button);
            bw.Write((byte)0);
            bw.Write((short)point.X);
            bw.Write((short)point.Y);
            bw.Write((short)0);
            bw.Flush();
        }

        public override string ToString()
        {
            return "McsMouseMoveEventArg: " + ToInfoStr();
        }

        public string ToInfoStr()
        {
            return string.Format("[X, Y] [{0,4:d},{1,4:d}]", point.X, point.Y);
        }
    }

    class McsMouseButtonEventArg : McsEventArg
    {
        public McsMouseButton button = McsMouseButton.None;

        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write((byte)button);
            bw.Write((byte)0);
            bw.Write((short)0);
            bw.Write((short)0);
            bw.Write((short)0);
            bw.Flush();
        }

        public override string ToString()
        {
            return "McsMouseButtonEventArg: " + ToInfoStr();
        }

        public string ToInfoStr()
        {
            return button.ToString();
        }
    }

    class McsMouseWheelEventArg : McsEventArg
    {
        public short moveLine;

        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write((byte)0);
            bw.Write((byte)0);
            bw.Write((short)0);
            bw.Write((short)0);
            bw.Write(moveLine);
            bw.Flush();
        }

        public override string ToString()
        {
            return "McsMouseButtonEventArg: " + ToInfoStr();
        }

        public string ToInfoStr()
        {
            return moveLine.ToString();
        }
    }

    class McsCaptureEventArg : McsEventArg
    {
        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            WritePadding(bw, 8);
            bw.Flush();
        }

        public override string ToString()
        {
            return "McsCaptureEventArg";
        }

        public string ToInfoStr()
        {
            return "";
        }
    }

    class SyncedTCPClient
    {
        /// <summary>
        /// コネクションを閉じるときに非同期 Write の完了を待つ時間 [msec]。
        /// </summary>
        const int WaitWriteDoneTimeout = 1000;

        readonly TcpClient _tcpClient;
        readonly NetworkStream _netSm;

        /// <summary>
        /// _tcpClientの排他制御用オブジェクト
        /// </summary>
        readonly object _tcpLock = new object();

        /// <summary>
        /// 進行中の非同期Write処理の数
        /// </summary>
        int _nbAsyncWrite = 0;

        /// <summary>
        /// 非同期Writeの終了待ち用イベント。
        /// </summary>
        ManualResetEvent _writeDoneEvent = null;

        public TcpClient BaseConnection
        {
            get
            {
                return _tcpClient;
            }
        }

        public object SyncRoot
        {
            get
            {
                return _tcpLock;
            }
        }

        public SyncedTCPClient(TcpClient tcpClient)
        {
            _tcpClient = tcpClient;
            _netSm = _tcpClient.GetStream();
        }

        public IAsyncResult BeginWrite(
            byte[] buffer,
            int offset,
            int count,
            AsyncCallback callback,
            Object state
        )
        {
            lock (_tcpLock)
            {
                ++_nbAsyncWrite;
                return _netSm.BeginWrite(buffer, offset, count, callback, state);
            }
        }

        public void EndWrite(IAsyncResult asyncResult)
        {
            lock (_tcpLock)
            {
                if (_netSm.CanWrite)
                {
                    _netSm.EndWrite(asyncResult);
                }

                --_nbAsyncWrite;
                if (_nbAsyncWrite == 0 && _writeDoneEvent != null)
                {
                    _writeDoneEvent.Set();
                }
            }
        }

        /// <summary>
        /// 非同期 Write の終了を待ちます。
        /// </summary>
        private void WaitWriteDone(int timeout)
        {
            lock (_tcpLock)
            {
                if (_nbAsyncWrite > 0)
                {
                    _writeDoneEvent = new ManualResetEvent(false);
                }
            }

            if (_writeDoneEvent != null)
            {
                _writeDoneEvent.WaitOne(timeout);
            }
        }

        public void Close()
        {
            this.WaitWriteDone(WaitWriteDoneTimeout);

            lock (_tcpLock)
            {
                _netSm.Close();
                _tcpClient.Close();
            }
        }
    }
}
