﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Net.Sockets;

namespace Nintendo.McsServer
{
    /// <summary>
    /// ソケットをブロック読み込みするためのインタフェースを公開する、
    /// ラッパークラス。
    /// ChannelInfoクラスでのソケット読み込みに利用されます。
    /// </summary>
    public class BlockingReader
    {
        /// <summary>
        /// ソケット
        /// </summary>
        readonly Socket _socket;

        /// <summary>
        /// Socketを同期するためのオブジェクト
        /// </summary>
        readonly object _syncObj;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public BlockingReader(Socket socket, object syncObj)
        {
            _socket = socket;
            _syncObj = syncObj;
        }

        /// <summary>
        /// 読み込み開始
        /// </summary>
        public void BeginRead( byte[] buffer, BlockingReaderCallback callback, Object state )
        {
            BeginRead(buffer, 0, buffer.Length, callback, state);
        }

        /// <summary>
        /// 読み込み開始
        /// </summary>
        public void BeginRead( byte[] buffer, int offset, int count, BlockingReaderCallback callback, Object state )
        {
            BlockingReaderContext context = new BlockingReaderContext(new ArraySegment<byte>(buffer, offset, count), callback, state);
            lock (_syncObj)
            {
                context.BeginRead(_socket, CallbackReceive);
            }
        }

        /// <summary>
        ///  非同期操作の完了時に呼び出されるコールバック
        /// </summary>
        void CallbackReceive( IAsyncResult ar )
        {
            BlockingReaderContext context = (BlockingReaderContext)ar.AsyncState;
            Exception ex = null;

            try
            {
                lock (_syncObj)
                {
                    int crReadBytes = _socket.EndReceive(ar);
                    if (! context.CallbackRead(crReadBytes))    // 全て読み込んでいないときは続きを読み込む
                    {
                        context.BeginRead(_socket, CallbackReceive);
                        return;
                    }
                }
            }
            catch (SocketException crEx)
            {
                ex = crEx;
            }

            // 例外が発生したか、読み込むデータが無くなったか、全てを読み込んだとき
            context.InvokeCallback(ex);
        }

    }

    /// <summary>
    /// BlockingReaderに利用される、状態(コンテキスト)クラス。
    /// </summary>
    public class BlockingReaderContext
    {
        readonly ArraySegment<byte> _arraySegment;
        readonly BlockingReaderCallback _callback;
        readonly Object _state;
        int _readBytes = 0;
        Exception _ex;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public BlockingReaderContext( ArraySegment<byte> arraySegment, BlockingReaderCallback callback, Object state )
        {
            _arraySegment = arraySegment;
            _callback = callback;
            _state = state;
        }

        /// <summary>
        /// このメソッドはサブルーチン的に呼ばれます。
        /// </summary>
        /// <param name="socket"></param>
        /// <param name="asyncCallback"></param>
        public void BeginRead(Socket socket, AsyncCallback asyncCallback)
        {
            socket.BeginReceive(_arraySegment.Array, _arraySegment.Offset + _readBytes, _arraySegment.Count - _readBytes, SocketFlags.None, asyncCallback, this);
        }

        /// <summary>
        /// このメソッドはサブルーチン的に呼ばれます。
        /// </summary>
        /// <param name="crReadBytes"></param>
        /// <param name="ex"></param>
        /// <returns></returns>
        public bool CallbackRead(int crReadBytes)
        {
            _readBytes += crReadBytes;

            // 読み込むデータが無くなったか、全てを読み込んだとき
            if (crReadBytes == 0 || _readBytes == _arraySegment.Count)
            {
                return true;
            }

            return false;   // 全て読み込んでいない
        }

        /// <summary>
        ///
        /// </summary>
        public void InvokeCallback(Exception ex)
        {
            _ex = ex;
            _callback(this);
        }

        /// <summary>
        ///
        /// </summary>
        public int ReadBytes
        {
            get { return _readBytes; }
        }

        /// <summary>
        ///
        /// </summary>
        public Object State
        {
            get { return _state; }
        }

        /// <summary>
        /// 全てを読み込んでいたら真を返します。
        /// 例外が発生していたら、例外をスローします。
        /// </summary>
        public bool GetResult()
        {
            if (_ex != null)
            {
                throw _ex;
            }

            return _readBytes == _arraySegment.Count;
        }
    }

    public delegate void BlockingReaderCallback(BlockingReaderContext context);
}
