﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "LogAccessor.h"
#include "..\..\..\..\..\..\..\Externals\Oasis\include\tm.h"

#include <ctime>
#include <mutex>

namespace detail
{
    namespace
    {
        const size_t SerialNumberLength = 14;

        std::mutex g_Mutex;

        USHORT GetLogPort(const char* serialNumber)
        {
            if (!serialNumber || ::strnlen(serialNumber, SerialNumberLength) != SerialNumberLength)
            {
                return 0;
            }

            int32_t targetCount;
            if (nn::tm::GetTargetCount(&targetCount) != nn::tm::Error_Ok)
            {
                return 0;
            }

            int addressCount;
            if (nn::tm::GetHtcsAddressesCount(&addressCount) != nn::tm::Error_Ok)
            {
                return 0;
            }

            std::unique_ptr<nn::tm::HtcsAddress[]> addresses(
                new nn::tm::HtcsAddress[addressCount]);

            int readCount;
            if (nn::tm::GetHtcsAddresses(addresses.get(), &readCount, addressCount)
                != nn::tm::Error_Ok)
            {
                return 0;
            }

            for (int i = 0; i < readCount; ++i)
            {
                nn::tm::HtcsPeerName peerName;
                if (nn::tm::GetHtcsPeerName(&peerName, &addresses[i]) != nn::tm::Error_Ok)
                {
                    continue;
                }

                if (strncmp(serialNumber, peerName.name, sizeof(SerialNumberLength)) != 0)
                {
                    continue;
                }

                nn::tm::HtcsPortName portName;
                if (nn::tm::GetHtcsPortName(&portName, &addresses[i]) != nn::tm::Error_Ok)
                {
                    continue;
                }

                const char LogPortName[] = "@Log";
                if (strncmp(LogPortName, portName.name, sizeof(LogPortName)) == 0)
                {
                    auto str = std::string(addresses[i].ipAndPortAddress);
                    unsigned int port = 0;
                    sscanf_s(str.substr(str.find(':') + 1).c_str(), "%u", &port);
                    return static_cast<USHORT>(port);
                }
            }

            return 0;
        }

        std::string TimeToString(std::chrono::system_clock::time_point timePoint)
        {
            const size_t bufferSize = 26;
            char buffer[bufferSize];

            std::time_t time = std::chrono::system_clock::to_time_t(timePoint);
            ctime_s(buffer, bufferSize, &time);

            // 改行コードを含むので除去
            for (int i = bufferSize - 1; i >= 0; --i)
            {
                if (buffer[i] == '\n')
                {
                    buffer[i] = '\0';
                    break;
                }
            }

            return std::string(buffer);
        }
    }

    LogAccessor::LogAccessor()
        : m_Socket(-1), m_Index(0), m_Size(0)
    {
        m_LogList.clear();
        ::WSADATA wsaData;
        ::WSAStartup(MAKEWORD(2, 2), &wsaData);
    }

    LogAccessor::~LogAccessor()
    {
        ::WSACleanup();
    }

    void LogAccessor::ReadFunc()
    {
        char tmp;
        std::string str;
        str.clear();

        while (recv(m_Socket, &tmp, 1, 0) > 0)
        {
            if (tmp == '\n')
            {
                std::lock_guard<std::mutex> lock(g_Mutex);
                if (m_LogList.size() > m_Size)
                {
                    m_LogList.pop_front();
                    if (m_Index > 0)
                    {
                        m_Index--;
                    }
                }

                LogData data = { str, std::chrono::system_clock::now() };
                m_LogList.push_back(data);
                str.clear();

                continue;
            }
            str += tmp;
        }
    }

    LogReaderResult LogAccessor::Start(int size, const char* serialNumber)
    {
        if (m_Socket != -1)
        {
            return LogReaderResult_AlreadyStarted;
        }

        if (size < 0)
        {
            return LogReaderResult_InvalidArgument;
        }

        m_Size = size;

        auto port = GetLogPort(serialNumber);
        if (port == 0)
        {
            return LogReaderResult_ConnectFailed;
        }

        if (m_Size == 0)
        {
            return LogReaderResult_Success;
        }

        ::SOCKET socket;
        socket = ::socket(AF_INET, SOCK_STREAM, 0);
        ::sockaddr_in address;
        address.sin_family = AF_INET;
        address.sin_port = htons(port);
        inet_pton(AF_INET, "127.0.0.1", &address.sin_addr.S_un.S_addr);

        int result;
        result = ::connect(socket, reinterpret_cast<SOCKADDR *>(&address), sizeof(address));
        if (result == SOCKET_ERROR)
        {
            DEBUG_LOG("connect failed (%ld)\n", WSAGetLastError());
            result = closesocket(socket);
            if (result == SOCKET_ERROR)
            {
                DEBUG_LOG("connect failed (%ld)\n", WSAGetLastError());
            }
            return LogReaderResult_ConnectFailed;
        }

        m_Socket = socket;
        m_Thread = std::thread(&LogAccessor::ReadFunc, this);
        return LogReaderResult_Success;
    }

    void LogAccessor::Stop()
    {
        if (m_Socket >= 0)
        {
            ::closesocket(m_Socket);
            m_Socket = -1;
        }
        if (m_Thread.joinable())
        {
            m_Thread.join();
        }
        {
            std::lock_guard<std::mutex> lock(g_Mutex);
            m_Index = 0;
            m_LogList.clear();
        }
    }

    LogReaderResult LogAccessor::GetCurrent(std::string* pOutString, int timeoutMs)
    {
        std::lock_guard<std::mutex> lock(g_Mutex);

        if (m_LogList.empty())
        {
            return LogReaderResult_NotFound;
        }

        *pOutString = m_LogList.at(m_Index).log;
        return LogReaderResult_Success;
    }

    LogReaderResult LogAccessor::WaitForNextLine(int timeoutMs)
    {
        int restTime = timeoutMs;
        const int interval = 1000 / 60; // TODO: 値は要調整
        bool isEmpty;

        {
            std::lock_guard<std::mutex> lock(g_Mutex);

            if (!IsLast())
            {
                return LogReaderResult_Success;
            }

            isEmpty = m_LogList.empty();
        }


        while (restTime > 0)
        {
            {
                std::lock_guard<std::mutex> lock(g_Mutex);

                if (!IsLast() || (isEmpty && !m_LogList.empty()))
                {
                    return LogReaderResult_Success;
                }
            }

            if (interval < restTime)
            {
                ::Sleep(interval);
            }
            else
            {
                ::Sleep(restTime);
            }
            restTime -= interval;
        }

        return LogReaderResult_Timeout;
    }

    std::string LogAccessor::GetCurrentTimeString()
    {
        std::lock_guard<std::mutex> lock(g_Mutex);
        if (!m_LogList.empty() && m_Index < m_LogList.size())
        {
            return TimeToString(m_LogList.at(m_Index).time);
        }
        return "";
    }

    bool LogAccessor::IsFirst()
    {
        return (m_Index == 0);
    }

    bool LogAccessor::IsLast()
    {
        return (m_LogList.empty() || m_Index == (static_cast<int>(m_LogList.size()) - 1));
    }

    void LogAccessor::MoveToNewestLine()
    {
        std::lock_guard<std::mutex> lock(g_Mutex);
        if (m_LogList.empty())
        {
            m_Index = 0;
            return;
        }
        m_Index = static_cast<int>(m_LogList.size()) - 1;
    }

    LogReaderResult LogAccessor::MoveToNextLine()
    {
        std::lock_guard<std::mutex> lock(g_Mutex);

        if (m_Index < static_cast<int>(m_LogList.size()) - 1)
        {
            m_Index++;
            return LogReaderResult_Success;
        }
        return LogReaderResult_CannotMoveLine;
    }

    LogReaderResult LogAccessor::MoveToPreviousLine()
    {
        std::lock_guard<std::mutex> lock(g_Mutex);
        if (m_Index > 0)
        {
            m_Index--;
            return LogReaderResult_Success;
        }
        return LogReaderResult_CannotMoveLine;
    }

    void LogAccessor::MoveToFormerLineByTime(int timeMs)
    {
        if (m_LogList.size() == 0)
        {
            return;
        }

        auto backTime = std::chrono::milliseconds(timeMs);

        std::lock_guard<std::mutex> lock(g_Mutex);

        auto current = m_LogList.at(m_Index).time;

        while (m_Index > 0)
        {
            auto diff = std::chrono::duration_cast<std::chrono::milliseconds>(current - m_LogList.at(m_Index).time);
            if (diff > backTime)
            {
                return;
            }
            m_Index--;
        }
    }

    void LogAccessor::PrintFullLog()
    {
        std::lock_guard<std::mutex> lock(g_Mutex);
        for each (auto log in m_LogList)
        {
            DEBUG_LOG("[%s] %s\n", TimeToString(log.time).c_str(), log.log.c_str());
        }
    }
}
