﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MultiTouchManager.h"


MultiTouchManager& MultiTouchManager::GetInstance(IoPortSender *sender)
{
    static MultiTouchManager instance(sender);
    return instance;
}

MultiTouchManager::MultiTouchManager(IoPortSender *sender)
{
    m_Sender = sender;
    m_TouchInputer = new TouchScreenInputer(m_Sender);
    m_TouchAction = new TouchAction(m_TouchInputer);
}


MultiTouchManager::~MultiTouchManager()
{
    delete m_TouchInputer;
    delete m_TouchAction;
}

void MultiTouchManager::CorrectMaxTouchCount(int &touchCount)
{
    if (touchCount >= MaxTouchCount)
    {
        touchCount = MaxTouchCount;
    }

    if (touchCount < 0)
    {
        touchCount = 0;
    }
}

TouchResult MultiTouchManager::SetTouchState(int touchIdList[], int fingerIdList[], float posXList[], float posYList[], int touchCount)
{
    // fingerId のチェック
    for (int i = 0; i < touchCount; i++)
    {
        for (int j = i + 1; j < touchCount; j++)
        {
            if (fingerIdList[i] == fingerIdList[j])
            {
                return TouchResult::TouchResult_InvalidFingerId;
            }
        }
    }

    try
    {
        m_TouchAction->SetTouchState(touchIdList, fingerIdList, posXList, posYList, touchCount);
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::SetTouchState: " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::SetTouchState: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }


    return TouchResult::TouchResult_Success;
}

TouchResult MultiTouchManager::ResetTouchState()
{
    try
    {
        m_TouchInputer->ResetTouchState();
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::ResetTouchState: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }

    return TouchResult::TouchResult_Success;
}


TouchResult MultiTouchManager::TouchPressDown(int touchId, int x, int y)
{
    try
    {
        m_TouchAction->TouchPressDown(touchId, x, y);
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::TouchPressDown: " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::TouchPressDown: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }


    return TouchResult::TouchResult_Success;
}

TouchResult MultiTouchManager::TouchPressDown(int touchId, int x, int y, int fingerId)
{
    try
    {
        m_TouchAction->TouchPressDown(touchId, x, y,fingerId);
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::TouchPressDown: " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (InvalidFingerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::TouchPressDown: " ) + exception.what());
        return TouchResult::TouchResult_InvalidFingerId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::TouchPressDown: "+ UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }


    return TouchResult::TouchResult_Success;
}

TouchResult MultiTouchManager::TouchPressUp(int touchId)
{
    try
    {
        m_TouchAction->TouchPressUp(touchId);
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::TouchPressUp " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::TouchPressUp: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }


    return TouchResult::TouchResult_Success;
}

TouchResult MultiTouchManager::TouchMove(int touchId, int x, int y)
{
    try
    {
        m_TouchAction->TouchMove(touchId, x, y);
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::TouchMove " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::TouchMove: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }

    return TouchResult::TouchResult_Success;
}

TouchResult MultiTouchManager::Tap(int touchId, int x, int y, int waitPress)
{
    try
    {
        m_TouchAction->Tap(touchId, x, y, waitPress);
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::Tap " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::Tap: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }
    return TouchResult::TouchResult_Success;
}

TouchResult MultiTouchManager::DragAngle(int touchId, int startX, int startY, float angle, float length,
    int actionDuration, int waitSpan, bool releaseAfter)
{
    try
    {
        if (waitSpan <= 0)
        {
            return TouchResult::TouchResult_Unexpected;
        }

        m_TouchAction->DragAngle(touchId, startX, startY, angle, length,
            actionDuration, waitSpan, releaseAfter);
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::DragAngle " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::DragAngle: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }


    return TouchResult::TouchResult_Success;

}

TouchResult MultiTouchManager::DragPos(int touchId, int startX, int startY, int endX, int endY,
    int actionDuration, int waitSpan, bool releaseAfter)
{

    try
    {
        if (waitSpan <= 0)
        {
            return TouchResult::TouchResult_Unexpected;
        }

        if (actionDuration < waitSpan)
        {
            return TouchResult::TouchResult_Unexpected;
        }

        m_TouchAction->DragPos(touchId, startX, startY, endX, endY,
            actionDuration,  waitSpan, releaseAfter);
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::DragPos: " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::DragPos: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }
    return TouchResult::TouchResult_Success;
}



TouchResult MultiTouchManager::MultiTap(int touchIdList[], int touchCount, float touchPosXList[], float touchPosYList[],
    int waitPress)
{
    try
    {

        for (int i = 0; i < touchCount; i++)
        {
            m_TouchAction->TouchPressDown(touchIdList[i], static_cast<int>(touchPosXList[i]), static_cast<int>(touchPosYList[i]));
        }
        Sleep(waitPress);

        for (int i = 0; i < touchCount; i++)
        {
            m_TouchAction->TouchPressUp(touchIdList[i]);
        }
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::MultiTap: " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::MultiTap: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }

    return TouchResult::TouchResult_Success;
}

TouchResult MultiTouchManager::MultiTouchAnimator(int touchIdList[], int touchCount,
    int startCenterX, int startCenterY,
    float centerMoveDeg, float centerMoveLength,
    float touchRelativePosXList[], float touchRelativePosYList[],
    float coordsysStartDeg, float coordsysRotateDeg,
    float startScale, float increaseScale,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{

    try
    {
        CorrectMaxTouchCount(touchCount);

        if (waitSpan <= 0)
        {
            return TouchResult::TouchResult_Unexpected;
        }
        if (actionDuration < waitSpan)
        {
            return TouchResult::TouchResult_Unexpected;
        }


        float startRad = static_cast<float>((coordsysStartDeg * 3.14) / 180.0);
        float rotateRad = static_cast<float>((coordsysRotateDeg * 3.14) / 180.0);

        int endCenterX = static_cast<int>(static_cast<float>(startCenterX + (centerMoveLength * cos((centerMoveDeg * 3.14) / 180.0))));
        int endCenterY = static_cast<int>(static_cast<float>(startCenterY + (centerMoveLength * sin((centerMoveDeg * 3.14) / 180.0))));


        // touch down
        for (int i = 0; i < touchCount; i++)
        {
            float x = (touchRelativePosXList[i] * startScale);
            float y = (touchRelativePosYList[i] * startScale);

            m_TouchAction->TouchPressDown(touchIdList[i],
                static_cast<int>(startCenterX + (x * cos(startRad) - y * sin(startRad))),
                static_cast<int>(startCenterY + (x * sin(startRad) + y * cos(startRad))));
        }
        if (waitTouchBegin > 0)
        {
            Sleep(waitTouchBegin);
        }

        // touch move
        for( int i = actionDuration; i > 0; i -= waitSpan )
        {

            // 更新度
            float progress = (static_cast<float>(actionDuration - (i - waitSpan)) / actionDuration);
            if (progress > 1.0f)
            {
                // TIPS: waitSpan が actionDuration の約数で無い場合は、最後の progress が 100 を超えるため、移動量を補正する必要がある。
                progress = 1.0f;
            }

            // 中心座標更新
            float newCenterX = startCenterX + ((endCenterX - startCenterX) * progress);
            float newCenterY = startCenterY + ((endCenterY - startCenterY) * progress);
            // スケール更新
            float newScale = startScale + (increaseScale * progress);
            // 角度更新
            float newRad = startRad + (rotateRad * progress);

            // タッチ座標更新
            for (int id = 0; id < touchCount; id++)
            {
                float x = touchRelativePosXList[id] * newScale;
                float y = touchRelativePosYList[id] * newScale;

                m_TouchAction->TouchMove(touchIdList[id],
                    static_cast<int>(newCenterX + (x * cos(newRad) - y * sin(newRad))),
                    static_cast<int>(newCenterY + (x * sin(newRad) + y * cos(newRad))));
            }

            Sleep( min(waitSpan, i) );
        }

        if (waitTouchEnd > 0)
        {
            Sleep(waitTouchEnd);
        }

        // touch up
        if (releaseAfter)
        {
            for (int i = 0; i < touchCount; i++)
            {
                m_TouchAction->TouchPressUp(touchIdList[i]);
            }
        }
    }
    catch (InvalidTouchIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiTouchManager::MultiTouchAnimator: " ) + exception.what());
        return TouchResult::TouchResult_InvalidTouchId;
    }
    catch (...)
    {
        ERRMSG("MultiTouchManager::MultiTouchAnimator: " + UNEXPECTED_EXCEPTION_MESSAGE)
        return TouchResult::TouchResult_Unexpected;
    }

    return TouchResult::TouchResult_Success;

}



TouchResult MultiTouchManager::MultiDragPos(int touchIdList[], int touchCount,
    int startCenterX, int startCenterY,
    int endCenterX, int endCenterY,
    float touchRelativePosXList[], float touchRelativePosYList[],
    int actionDuration,  int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{
    CorrectMaxTouchCount(touchCount);

    float dx = float(endCenterX - startCenterX);
    float dy = float(endCenterY - startCenterY);

    float rad = static_cast<float>(atan2(dy, dx));
    float deg = static_cast<float>((rad * 180.0) / 3.14);
    float length = sqrtf((dy * dy) + (dx * dx));

    return MultiTouchAnimator(touchIdList, touchCount,
        startCenterX, startCenterY,
        deg, length,
        touchRelativePosXList, touchRelativePosYList,
        0.0, 0.0,        // deg
        1.0, 0.0,        // scale
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}


TouchResult MultiTouchManager::MultiDragAngle(int touchIdList[], int touchCount,
    int startCenterX, int startCenterY,
    float centerMoveDeg, float centerMoveLength,
    float touchRelativePosXList[], float touchRelativePosYList[],
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{

    CorrectMaxTouchCount(touchCount);

    return MultiTouchAnimator(touchIdList, touchCount,
        startCenterX, startCenterY,
        centerMoveDeg, centerMoveLength,
        touchRelativePosXList, touchRelativePosYList,
        0.0, 0.0,        // deg
        1.0, 0.0,        // scale
        actionDuration, waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}

TouchResult MultiTouchManager::Pinch(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[],
    float startRadius, float increaseRadius,
    int actionDuration,  int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{

    float touchBufX[MaxTouchCount];
    float touchBufY[MaxTouchCount];
    CorrectMaxTouchCount(touchCount);

    for (int i = 0; i < touchCount; i++)
    {
        float angleRad = static_cast<float>((touchDegList[i] * 3.14) / 180.0);
        touchBufX[i] = (cosf(angleRad));
        touchBufY[i] = (sinf(angleRad));
    }

    return MultiTouchAnimator(touchIdList, touchCount,
        centerX, centerY,
        0, 0,
        touchBufX, touchBufY,
        0.0, 0.0,        // deg
        startRadius, increaseRadius,        // scale
        actionDuration, waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}

TouchResult MultiTouchManager::CenterTouchPinch(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[],
    float startRadius, float increaseRadius,
    int actionDuration,  int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{


    float touchBufX[MaxTouchCount];
    float touchBufY[MaxTouchCount];
    CorrectMaxTouchCount(touchCount);

    if (touchCount == 0)
    {
        return TouchResult::TouchResult_Unexpected;
    }

    for (int i = 1; i < touchCount; i++)
    {
        float angleRad = (touchDegList[i] * 3.14f / 180.0f);
        touchBufX[i - 1] = (cosf(angleRad));
        touchBufY[i - 1] = (sinf(angleRad));
    }

    // [0] は中心点タッチに使用
    TouchResult result = TouchPressDown(touchIdList[0], centerX, centerY);
    if (result != TouchResult::TouchResult_Success)
    {
        return result;
    }

    return MultiTouchAnimator((touchIdList + 1), touchCount - 1,
        centerX, centerY,
        0, 0,
        touchBufX, touchBufY,
        0.0, 0.0,        // deg
        startRadius, increaseRadius,        // scale
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}

TouchResult MultiTouchManager::MultiRotate(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[], float radius,
    float coordsysStartDeg, float coordsysRotateDeg,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{

    float touchBufX[MaxTouchCount];
    float touchBufY[MaxTouchCount];
    CorrectMaxTouchCount(touchCount);

    for (int i = 0; i < touchCount; i++)
    {
        float angleRad = (touchDegList[i] * 3.14f / 180.0f);
        touchBufX[i] = cosf(angleRad);
        touchBufY[i] = sinf(angleRad);
    }


    return MultiTouchAnimator(touchIdList, touchCount,
        centerX, centerY,
        0, 0,
        touchBufX, touchBufY,
        coordsysStartDeg, coordsysRotateDeg,        // deg
        radius, 0.0,                            // scale
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}


TouchResult MultiTouchManager::CenterTouchMultiRotate(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[], float radius,
    float coordsysStartDeg, float coordsysRotateDeg,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{

    float touchBufX[MaxTouchCount];
    float touchBufY[MaxTouchCount];
    CorrectMaxTouchCount(touchCount);

    if (touchCount == 0)
    {
        return TouchResult::TouchResult_Unexpected;
    }

    for (int i = 1; i < touchCount; i++)
    {
        float angleRad = (touchDegList[i] * 3.14f / 180.0f);
        touchBufX[i - 1] = cosf(angleRad);
        touchBufY[i - 1] = sinf(angleRad);
    }

    // [0] は中心点タッチに使用
    TouchResult result = TouchPressDown(touchIdList[0], centerX, centerY);
    if (result != TouchResult::TouchResult_Success)
    {
        return result;
    }

    return MultiTouchAnimator(touchIdList + 1, touchCount - 1,
        centerX, centerY,
        0, 0,
        touchBufX, touchBufY,
        coordsysStartDeg, coordsysRotateDeg,            // deg
        radius, 0.0,                                // scale
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}
