﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <vector>

#include <autoTestAssistTool/HidInputerTypes.h>

#include "common.h"
#include "HidShellLibraryWrapper.h"
#include "IoPortRecorder.h"

const int           MaxTouchCount       = 10;
const int           MaxControllerCount = HidShellBasicXpadCountMax;
class IoPortSender
{
public:
    static IoPortSender &GetInstance();
    HidInputerResult SetPortNumberForTarget( std::string serialNumber );
    HidInputerResult SetPortNumberForLocal();

    // Xpad
    void SetXPadState(int32_t controllerId, ControllerButtonValueType buttons,
        const Stick stickList[], const ControllerStickValueType stickXList[], const ControllerStickValueType stickYList[], int32_t stickCount);
    void SetXpadConnection(int32_t controllerId, int32_t value);
    void SetXpadDefault(int32_t controllerId);
    void SetXpadPressButton(int32_t controllerId, int32_t value);
    void SetXpadReleaseButton(int32_t controllerId, int32_t value);
    void SetXpadStickL(int32_t controllerId, int32_t x, int32_t y);
    void SetXpadStickR(int32_t controllerId, int32_t x, int32_t y);
    // Debug Pad
    void SetDebugPadState(ControllerButtonValueType buttons,
        const Stick stickList[], const ControllerStickValueType stickXList[], const ControllerStickValueType stickYList[], int32_t stickCount);
    void SetDebugPadConnection(int32_t value);
    void SetDebugPadDefault();
    void SetDebugPadPressButton(int32_t value);
    void SetDebugPadReleaseButton(int32_t value);
    void SetDebugPadStickL(int32_t x, int32_t y);
    void SetDebugPadStickR(int32_t x, int32_t y);
    // キャプチャボタン
    void SetCaptureButton(int32_t controllerId, int32_t value);
    // ホームボタン
    void SetHomeButton(int32_t controllerId, int32_t value);
    // タッチスクリーン
    void SetTouchState(int32_t touchIdList[], int32_t fingerIdList[], int32_t posXList[], int32_t posYList[], int32_t touchCount);
    void ResetTouchState();
    void SetTouchPress(int32_t touchId, int32_t x, int32_t y);
    void SetTouchPress(int32_t touchId, int32_t x, int32_t y, int32_t fingerId);
    void SetTouchRelease(int32_t touchId);
    void SetTouchMove(int32_t touchId, int32_t x, int32_t y);
    // VirtualPad
    void SetVirtualPadMainColor(int32_t controllerId, int32_t mainColor);
    void SetVirtualPadSubColor(int32_t controllerId, int32_t subColor);
    void SetVirtualPadDeviceType(int32_t controllerId, int32_t deviceType);
    void SetVirtualPadInterfaceType(int32_t controllerId, int32_t interfaceType);
    void SetVirtualPadConnection(int32_t controllerId, int32_t value);
    void SetVirtualPadDefault(int32_t controllerId);
    void SetVirtualPadState(int32_t controllerId, ControllerButtonValueType buttons,
        const Stick stickList[], const ControllerStickValueType stickXList[], const ControllerStickValueType stickYList[], int32_t stickCount);
    void SetVirtualPadPressButton(int32_t controllerId, int32_t value);
    void SetVirtualPadReleaseButton(int32_t controllerId, int32_t value);
    void SetVirtualPadStickL(int32_t controllerId, int32_t x, int32_t y);
    void SetVirtualPadStickR(int32_t controllerId, int32_t x, int32_t y);

    // HID 操作記録
    HidInputerResult BeginHidInputRecording();
    HidInputerResult EndHidInputRecording();
    HidInputerResult ClearHidRecord();
    HidInputerResult ExportHidInputRecords( const char* exportFileName );

    // 状態取得
    void GetXpadStates(std::vector<HidShellBasicXpadState>& xpadStates);
    void GetDebugPadStates(HidShellDebugPadState& debugPadState);
    void GetVirtualPadStates(std::vector<HidShellAbstractedPadState>& virtualPadStates);

    // 操作反映
    void Flush();

private:
    IoPortSender();
    ~IoPortSender();
    IoPortSender(const IoPortSender &copy);
    IoPortSender& operator=(const IoPortSender &copy);

    // 操作タスク追加
    void SetXpadTask(int32_t controllerId, int32_t value, HidTaskType taskType);
    void SetDebugPadTask(int32_t value, HidTaskType taskType);
    void SetVirtualPadTask(int32_t controllerId, int32_t value, HidTaskType taskType);

    unsigned int                m_Port;

    int32_t                     m_NextFingerId;
    HidShellTouchScreenState    m_TouchStateValue;

    std::mutex                  m_ControllerJobMutex;
    std::mutex                  m_TouchJobMutex;
    std::mutex                  m_VirtualControllerJobMutex;
    std::vector<HidTask>        m_XpadTaskList;
    std::vector<HidTask>        m_DebugPadTaskList;
    std::vector<HidTask>        m_VirtualPadTaskList;

    IoPortRecorder              m_IoPortRecorder;
    bool                        m_IsRecordindHid;

    void UpdateXpadStatus( HidShellHandle hidShellHandle );
    void UpdateDebugPadState( HidShellHandle hidShellHandle );
    void UpdateVirtualPadStatus( HidShellHandle hidShellHandle );
    void UpdateTouchScreenStatus( HidShellHandle hidShellHandle );
};
