﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "..\..\..\..\..\..\..\Externals\Oasis\include\tm.h"
#include "Common\\autoTestAssistTool_FpsController.h"
#include "Common\\autoTestAssistTool_PathUtil.h"
#include "Common\\autoTestAssistTool_StringConverter.h"
#include "Common\\autoTestAssistTool_TargetManagerUtil.h"
#include "DevkitCapture.h"
#include "ICapture.h"

#include <chrono>
#include <regex>
#include <windows.h>

const float ICapture::PreviewFrameSpan = (float)((1000.0 / 90.0) - 1.0);

namespace
{
    const int       DefaultWaitSpan = 10;

    // テンプレートマッチング
    MatchingResult ExecuteTemplateMatching(int* pOutX, int* pOutY, cv::Mat& targetImage, cv::Mat& templateImage, double threshold, int method)
    {
        // テンプレートマッチ
        cv::Mat result;
        cv::matchTemplate(targetImage, templateImage, result, method);

        // 結果取得
        cv::Point   maxPos;
        cv::Point   minPos;

        double maxValue = 0.0;
        double minValue = 0.0;

        cv::minMaxLoc(result, &minValue, &maxValue, &minPos, &maxPos);

        // 評価手法に応じた結果を返す
        switch (method)
        {
            case CV_TM_SQDIFF:
            case CV_TM_SQDIFF_NORMED:
                // CV_TM_SQDIFF, CV_TM_SQDIFF_NORMED は、小さいほど一致度が高い
                if (minValue > threshold)
                {
                    return MatchingResult::MatchingResult_Failed;
                }
                (*pOutX) = minPos.x;
                (*pOutY) = minPos.y;
                return MatchingResult::MatchingResult_Success;
            case CV_TM_CCORR:
            case CV_TM_CCORR_NORMED:
            case CV_TM_CCOEFF:
            case CV_TM_CCOEFF_NORMED:
                if (maxValue < threshold)
                {
                    return MatchingResult::MatchingResult_Failed;
                }
                (*pOutX) = maxPos.x;
                (*pOutY) = maxPos.y;
                return MatchingResult::MatchingResult_Success;
                break;
            default:
                return MatchingResult::MatchingResult_Failed;
        }
    }

    bool ExecuteTesseractOcrRecognize(tesseract::TessBaseAPI* api, cv::Mat image, OcrLanguage language)
    {
        const std::string tessDataPath = autoTestAssistTool::util::GetSdkRootPath() + "\\Tools\\AutoTestAssistTools\\Libraries\\";

        switch (language)
        {
            case OcrLanguage::OcrLanguage_Japanese:
            {
                api->Init(tessDataPath.c_str(), "jpn");
                break;
            }
            case OcrLanguage::OcrLanguage_English:
            {
                api->Init(tessDataPath.c_str(), "eng");
                break;
            }
            default:
            {
                api->Init(tessDataPath.c_str(), "eng");
                break;
            }
        }
        api->SetImage( image.data, image.cols, image.rows, image.channels(), image.channels() * image.cols);

        if (api->Recognize(0) != 0)
        {
            return false;
        }
        return true;
    }

    MatchingResult ExecuteMatchingUntilTimeout(std::function<MatchingResult()> matchingFunction, int timeout)
    {
        std::chrono::system_clock::time_point beginTime = std::chrono::system_clock::now();
        std::chrono::system_clock::time_point nowTime;

        int durationTime = 0;
        do
        {
            MatchingResult result = matchingFunction();

            // 結果チェック
            switch (result)
            {
                case MatchingResult::MatchingResult_Success:
                    return MatchingResult::MatchingResult_Success;
                case MatchingResult::MatchingResult_Failed:
                    break;
                default:
                    // エラー値が返ってきた場合は、その値を返す。
                    return result;
            }

            // 経過時間取得
            nowTime = std::chrono::system_clock::now();
            durationTime = static_cast<unsigned int>(std::chrono::duration_cast<std::chrono::milliseconds>(nowTime - beginTime).count());

            // 待機
            if ( durationTime < timeout )
            {
                Sleep(std::min(DefaultWaitSpan, (timeout - durationTime)));
            }
        } while ( (timeout - durationTime) > 0 );

        return MatchingResult::MatchingResult_Failed;
    }

    void SeparateLevel2PageIteratorLevel(tesseract::PageIteratorLevel* pOutPageIteratorLevel, OcrSeparateLevel separateLevel)
    {
        switch (separateLevel)
        {
            case OcrSeparateLevel::OcrSeparateLevel_Line:
            {
                (*pOutPageIteratorLevel) = tesseract::RIL_TEXTLINE;
                break;
            }
            case OcrSeparateLevel::OcrSeparateLevel_Word:
            default:
            {
                (*pOutPageIteratorLevel) = tesseract::RIL_WORD;
                break;
            }
        }
    }

    unsigned int GenerateId(std::function<bool(unsigned int)> checkEnableIdFunction)
    {
        unsigned int id;

        do
        {
            id = GetTickCount();
        } while (!checkEnableIdFunction(id));

        return id;
    }

    int GetNearPow2(int number)
    {
        if (number < 0)
        {
            return 0;
        }

        if (((number - 1) & number) == 0)
        {
            return (number == 1) ? 2 : number;
        }

        int result = 1;
        int numberBuffer = number;

        while (numberBuffer > 0)
        {
            result = result << 1;
            numberBuffer = numberBuffer >> 1;
        }

        return result;
    }
}

// map キー用の比較演算子
bool operator <(const ImageIdType &left, const ImageIdType &right)
{
    return left.id < right.id;
}
bool operator <(const OcrResultHandleType &left, const OcrResultHandleType &right)
{
    return left.handle < right.handle;
}
bool operator <(const LabelingResultHandleType &left, const LabelingResultHandleType &right)
{
    return left.handle < right.handle;
}

ICapture::ICapture()
    : m_IsCapture(false)
    , m_IsPreview(false)
{
    m_PreviewThread = NULL;

    {
        std::lock_guard<std::mutex> lock(m_CaptureMutex);
        m_CaptureImage.create(cv::Size(DefaultWindowWidth, DefaultWindowHeight), CV_8UC3);
    }
    {
        std::lock_guard<std::mutex> lock(m_TesseractOcrMutex);
        m_TesseractOcrResultList.clear();
    }
    {
        std::lock_guard<std::mutex> lock(m_LabelingResultMutex);
        m_LabelingResultList.clear();
    }
}

ICapture::~ICapture()
{
    EndPreview();

    // イメージ解放
    ReleaseAllImages();
}

/*----------------------------------------------------
/ プレビュー処理
/---------------------------------------------------*/
void ICapture::StartPreview()
{
    m_IsPreview = true;
    if (m_PreviewThread != NULL)
    {
        return;
    }

    m_PreviewThread.reset(new std::thread(&ICapture::RunPreview, this));
}

void ICapture::EndPreview()
{
    m_IsPreview = false;
    if (m_PreviewThread != NULL)
    {
        m_PreviewThread->join();
        m_PreviewThread = NULL;
    }
}

void ICapture::RunPreview()
{
    static autoTestAssistTool::util::FpsController fpsController(60);
    WindowManager& windowManager = WindowManager::GetInstance();

    // プレビュー情報初期化
    std::string previewWindowName = GetPreviewName();
    {
        std::lock_guard<std::mutex> lock(m_CaptureMutex);
        cv::Mat preview = m_CaptureImage;
        windowManager.OpenImageWindow(previewWindowName.c_str(), preview, preview.cols, preview.rows, true);
    }

    fpsController.BeginLoop();

    while (m_IsPreview)
    {
        {
            std::lock_guard<std::mutex> lock(m_CaptureMutex);
            cv::Mat preview = m_CaptureImage;
            windowManager.UpdateWindowImage(previewWindowName.c_str(), preview, preview.cols, preview.rows, false);
        }
        fpsController.SleepFrameInterval();
    }

    windowManager.CloseImageWindow(previewWindowName.c_str(), false);
    fpsController.EndLoop();
}


/*----------------------------------------------------------
/ CaptureImageMap 操作
/---------------------------------------------------------*/
cv::Mat ICapture::GetCaptureImageMapData(ImageIdType id)
{
    std::lock_guard<std::mutex> lock(m_CaptureImageMapMutex);
    if (!m_CaptureImageMap.count(id))
    {
        throw InvalidImageIdException(id);
    }

    return m_CaptureImageMap[id];
}

ImageIdType ICapture::SetCaptureImageMapData(cv::Mat& image)
{

    ImageIdType id;
    id.id = GenerateId([this](unsigned int newId) -> bool{
        std::lock_guard<std::mutex> lock(m_CaptureImageMapMutex);

        ImageIdType idBuffer;
        idBuffer.id = newId;

        return (m_CaptureImageMap.count(idBuffer) == 0);
    });

    std::lock_guard<std::mutex> lock(m_CaptureImageMapMutex);
    m_CaptureImageMap[id] = image;
    return id;
}

void ICapture::DeleteCaptureImageMapData(ImageIdType id)
{
    std::lock_guard<std::mutex> lock(m_CaptureImageMapMutex);
    if (!m_CaptureImageMap.count(id))
    {
        throw InvalidImageIdException(id);
    }

    m_CaptureImageMap.erase(id);
}

/*----------------------------------------------------------
/ 画像取得・解放
/---------------------------------------------------------*/
CaptureResult ICapture::GetImageFromCaptureDevice(ImageIdType *pOutImageId)
{
    try
    {
        cv::Mat buffer;
        {
            std::lock_guard<std::mutex> lock(m_CaptureMutex);
            buffer = m_CaptureImage.clone();
        }

        (*pOutImageId) = SetCaptureImageMapData(buffer);
        return CaptureResult::CaptureResult_Success;
    }
    catch (...)
    {
        ERRMSG("ICapture::GetImageFromCaptureDevice: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
}

CaptureResult ICapture::LoadImageFile(ImageIdType *pOutImageId, const char *filepath)
{
    try
    {
        auto wstr = autoTestAssistTool::util::cp_to_wide(filepath, CP_UTF8);
        cv::Mat loadImage = cv::imread(autoTestAssistTool::util::wide_to_cp(wstr, CP_ACP));
        if (loadImage.empty())
        {
            ERRMSG("ICapture::LoadImageFile: " + LOADING_IMAGE_FAILED_MESSAGE);
            return CaptureResult::CaptureResult_Unexpected;
        }


        (*pOutImageId) = SetCaptureImageMapData(loadImage);
        return CaptureResult::CaptureResult_Success;
    }
    catch (...)
    {
        ERRMSG("ICapture::LoadImageFile: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
}

CaptureResult ICapture::ReleaseImage(ImageIdType id)
{
    try
    {
        DeleteCaptureImageMapData(id);
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::ReleaseImage:" + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::ReleaseImage: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }

    return CaptureResult::CaptureResult_Success;
}
CaptureResult ICapture::ReleaseAllImages()
{
    try
    {
        std::lock_guard<std::mutex> lock(m_CaptureImageMapMutex);
        m_CaptureImageMap.clear();
    }
    catch (...)
    {
        ERRMSG("ICapture::ReleaseAllImages: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;

    }
    return CaptureResult::CaptureResult_Success;
}

/*----------------------------------------------------------
/ 画像保存
/---------------------------------------------------------*/
CaptureResult ICapture::SaveImageFile(std::string filepath, ImageIdType id)
{
    try
    {
        cv::Mat saveImage = GetCaptureImageMapData(id);

        if (filepath.size() == 0)
        {
            return CaptureResult::CaptureResult_Unexpected;
        }

        auto wstr = autoTestAssistTool::util::cp_to_wide(filepath, CP_UTF8);
        if (!cv::imwrite(autoTestAssistTool::util::wide_to_cp(wstr, CP_ACP), saveImage))
        {
            return CaptureResult::CaptureResult_FileCreationFailed;
        }
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::SaveImageFile: " + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::SaveImageFile: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }

    return CaptureResult::CaptureResult_Success;
}

/*----------------------------------------------------------
/ 画像情報取得
/---------------------------------------------------------*/
CaptureResult ICapture::GetImageSize(int* pOutWidth, int* pOutHeight, ImageIdType id)
{
    try
    {
        cv::Mat image = GetCaptureImageMapData(id);
        (*pOutWidth) = image.cols;
        (*pOutHeight) = image.rows;
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::GetImageSize: " + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::GetImageSize: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }

    return CaptureResult::CaptureResult_Success;
}
CaptureResult ICapture::GetImageData(void* pOutColorBuffer, size_t colorBufferSize, ImageIdType id)
{

    try
    {
        cv::Mat image = GetCaptureImageMapData(id);

        size_t writeSize = std::min(colorBufferSize, static_cast<size_t>(image.rows * image.cols * image.channels()));
        memcpy(pOutColorBuffer, image.data, writeSize);
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::GetImageData: " + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::GetImageData: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }

    return CaptureResult::CaptureResult_Success;
}

/*----------------------------------------------------------
/ マッチング
/---------------------------------------------------------*/
MatchingResult ICapture::DetectObject(int* pOutX, int* pOutY, ImageIdType id, double threshold, int method)
{
    try
    {
        cv::Mat result;

        // グレースケール化（キャプチャ画像）
        cv::Mat grayScreenImage;
        {
            std::lock_guard<std::mutex> lock(m_CaptureMutex);
            cv::cvtColor(m_CaptureImage, grayScreenImage, CV_RGB2GRAY);
        }

        // グレースケール化（テンプレート画像）
        cv::Mat templateImage;
        cv::Mat grayObjectImage;
        templateImage = GetCaptureImageMapData(id);
        cv::cvtColor(templateImage, grayObjectImage, CV_RGB2GRAY);

        // テンプレートマッチング
        return ExecuteTemplateMatching(pOutX, pOutY, grayScreenImage, grayObjectImage, threshold, method);
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::DetectObject: " + exception.what());
        return MatchingResult::MatchingResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::DetectObject: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return MatchingResult::MatchingResult_Unexpected;
    }
}

MatchingResult ICapture::WaitDetectObject(int* pOutX, int* pOutY, ImageIdType id, double threshold, int timeout, int method)
{
    return ExecuteMatchingUntilTimeout(
        [=]() -> MatchingResult{
            return DetectObject(pOutX, pOutY, id, threshold, method);
        }, timeout);
}

MatchingResult ICapture::DetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, FilterIdType filterId, int method)
{
    try
    {
        cv::Mat result;

        // キャプチャ画像コピー
        cv::Mat captureImageClone;
        {
            std::lock_guard<std::mutex> lock(m_CaptureMutex);
            captureImageClone = m_CaptureImage.clone();
        }

        // テンプレート画像取得
        cv::Mat templateImage = GetCaptureImageMapData(id);

        // フィルター適用
        cv::Mat filterdCaptureImage;
        m_FilterManager.ApplyFilter(&filterdCaptureImage, captureImageClone, filterId);

        // グレースケール化
        cv::Mat grayCaptureImage;
        cv::Mat grayTemplateImage;
        cv::cvtColor(filterdCaptureImage, grayCaptureImage, CV_RGB2GRAY);
        cv::cvtColor(templateImage, grayTemplateImage, CV_RGB2GRAY);

        // テンプレートマッチ
        return ExecuteTemplateMatching(pOutX, pOutY, grayCaptureImage, grayTemplateImage, threshold, method);
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::DetectObjectWithFilter: " + exception.what());
        return MatchingResult::MatchingResult_InvalidImageId;
    }
    catch (InvalidFilterIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::DetectObjectWithFilter: " + exception.what());
        return MatchingResult::MatchingResult_InvalidFilterId;
    }
    catch (...)
    {
        ERRMSG("ICapture::DetectObjectWithFilter: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return MatchingResult::MatchingResult_Unexpected;
    }

    return MatchingResult::MatchingResult_Failed;
}

MatchingResult ICapture::WaitDetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, int timeout, FilterIdType filterId, int method)
{
    return ExecuteMatchingUntilTimeout(
        [=]() -> MatchingResult{
            return DetectObjectWithFilter(pOutX, pOutY, id, threshold, filterId, method);
        }, timeout);
}

/*----------------------------------------------------------------------
/ 色検出
/---------------------------------------------------------------------*/
CaptureResult ICapture::GetDotColor(Rgb24Color* pOutColor, ImageIdType id, int x, int y )
{
    try
    {
        cv::Mat image = GetCaptureImageMapData(id);
        if ( x < 0 || y < 0  || x >= image.cols ||y >= image.rows )
        {
            return CaptureResult::CaptureResult_Unexpected;
        }

        pOutColor->blue = image.at<cv::Vec3b>( y, x )[0];
        pOutColor->green = image.at<cv::Vec3b>( y, x )[1];
        pOutColor->red = image.at<cv::Vec3b>( y, x )[2];
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::GetDotColor: " + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::GetDotColor: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;
}

MatchingResult ICapture::DetectDotColor(int x, int y, Rgb24Color lowerColor, Rgb24Color upperColor)
{
    try
    {
        std::lock_guard<std::mutex> lock(m_CaptureMutex);

        if ( x < 0 || y < 0  || x >= m_CaptureImage.cols || y >= m_CaptureImage.rows )
        {
            return MatchingResult::MatchingResult_Unexpected;
        }

        if ( m_CaptureImage.at<cv::Vec3b>(y, x)[0] < lowerColor.blue
            || m_CaptureImage.at<cv::Vec3b>(y, x)[0] > upperColor.blue
            || m_CaptureImage.at<cv::Vec3b>(y, x)[1] < lowerColor.green
            || m_CaptureImage.at<cv::Vec3b>(y, x)[1] > upperColor.green
            || m_CaptureImage.at<cv::Vec3b>(y, x)[2] < lowerColor.red
            || m_CaptureImage.at<cv::Vec3b>(y, x)[2] > upperColor.red )
        {
            return MatchingResult::MatchingResult_Failed;
        }

        return MatchingResult::MatchingResult_Success;
    }
    catch (...)
    {
        ERRMSG("ICapture::DetectDotColor: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return MatchingResult::MatchingResult_Unexpected;
    }
    return MatchingResult::MatchingResult_Failed;
}

MatchingResult ICapture::WaitDetectDotColor(int x, int y, Rgb24Color lowerColor, Rgb24Color upperColor, int timeout)
{
    return ExecuteMatchingUntilTimeout(
        [=]() -> MatchingResult{
            return DetectDotColor(x, y, lowerColor, upperColor);
        }, timeout);
}

/*----------------------------------------------------------------------
/ ラベリング
/---------------------------------------------------------------------*/
CaptureResult ICapture::ExecuteLabeling(int *pOutResultAreaCount, LabelingResultHandleType *pOutlabelingResultHandle, ImageIdType imageId)
{
    try
    {
        cv::Mat targetImage = GetCaptureImageMapData(imageId);
        cv::Mat labels;
        cv::Mat stats;
        cv::Mat centroids;

        cv::Mat grayScreenImage;
        cv::cvtColor(targetImage, grayScreenImage, CV_RGB2GRAY);
        int resultCount = cv::connectedComponentsWithStats(grayScreenImage, labels, stats, centroids, 8, CV_32S);

        std::vector<LabelingResult> resultList;
        for (int i = 0; i < resultCount; i++)
        {
            LabelingResult result;
            int *statsValue = stats.ptr<int>(i);
            int *centroidValue = centroids.ptr<int>(i);

            // TIPS: area < 1 の場合は、不正な結果として扱う
            // このとき、他のパラメーターも opencv 内の初期化値（不正な値）となっている。
            if (statsValue[cv::ConnectedComponentsTypes::CC_STAT_AREA] < 1)
            {
                continue;
            }

            // 面積
            result.area = statsValue[cv::ConnectedComponentsTypes::CC_STAT_AREA];

            // 領域
            result.rect.left   = statsValue[cv::ConnectedComponentsTypes::CC_STAT_LEFT];
            result.rect.top    = statsValue[cv::ConnectedComponentsTypes::CC_STAT_TOP];
            result.rect.width  = statsValue[cv::ConnectedComponentsTypes::CC_STAT_WIDTH];
            result.rect.height = statsValue[cv::ConnectedComponentsTypes::CC_STAT_HEIGHT];

            // 重心
            result.centerX = static_cast<int>(centroidValue[0]);
            result.centerY = static_cast<int>(centroidValue[1]);

            resultList.push_back(result);
        }
        {
            LabelingResultHandleType handle;
            handle.handle = GenerateId([this](unsigned int newId) -> bool{
                std::lock_guard<std::mutex> lock(m_LabelingResultMutex);
                LabelingResultHandleType handleBuffer;
                handleBuffer.handle = newId;

                return (m_LabelingResultList.count(handleBuffer) == 0);
            });

            std::lock_guard<std::mutex> lock(m_LabelingResultMutex);
            (*pOutlabelingResultHandle) = handle;
            m_LabelingResultList[handle] = resultList;
        }

        (*pOutResultAreaCount) = static_cast<int>(resultList.size());
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::ExecuteLabeling: " + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::ExecuteLabeling: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::GetLabelingResults(int *pOutResultsCount, LabelingResult pOutResultArray[], LabelingResultHandleType labelingResultHandle, int arrayCount)
{
    try
    {
        std::lock_guard<std::mutex> lock(m_LabelingResultMutex);
        if (m_LabelingResultList.count(labelingResultHandle) == 0)
        {
            return CaptureResult::CaptureResult_InvalidLabelingResultHandle;
        }

        const std::vector<LabelingResult> &resultList = m_LabelingResultList[labelingResultHandle];

        int copyCount = std::min(arrayCount, static_cast<int>(resultList.size()));

        for (int i = 0; i < copyCount; i++)
        {
            pOutResultArray[i] = resultList.at(i);
        }

        (*pOutResultsCount) = copyCount;
    }
    catch (...)
    {
        ERRMSG("ICapture::GetLabelingResults: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;

    }
    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::ReleaseLabelingResult(LabelingResultHandleType labelingResultHandle)
{
    try
    {
        std::lock_guard<std::mutex> lock(m_LabelingResultMutex);
        if (m_LabelingResultList.count(labelingResultHandle) == 0)
        {
            return CaptureResult::CaptureResult_InvalidLabelingResultHandle;
        }

        m_LabelingResultList.erase(labelingResultHandle);
    }
    catch (...)
    {
        ERRMSG("ICapture::ReleaseLabelingResult: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;
}

/*----------------------------------------------------------------------
/ OCR
/---------------------------------------------------------------------*/
MatchingResult ICapture::DetectTextForImage(ImageRect *pOutRect,
    ImageIdType imageId,
    const char regexString[],
    OcrSeparateLevel separate,
    OcrLanguage language)
{
    try
    {
        std::lock_guard<std::mutex> lock(m_TesseractOcrMutex);
        cv::Mat targetImage = GetCaptureImageMapData(imageId);

        if (ExecuteTesseractOcrRecognize(&m_TesseractOcrBaseApi, targetImage, language) == false)
        {
            return MatchingResult::MatchingResult_Failed;
        }

        tesseract::ResultIterator *iterator = m_TesseractOcrBaseApi.GetIterator();
        if (iterator == NULL)
        {
            return MatchingResult::MatchingResult_Failed;
        }

        tesseract::PageIteratorLevel level;
        SeparateLevel2PageIteratorLevel(&level, separate);

        std::string str(regexString);
        std::wregex regexStr;
        regexStr.imbue(std::locale(""));
        regexStr = autoTestAssistTool::util::cp_to_wide(str, CP_UTF8);
        if (!iterator->Empty(level))
        {
            do
            {
                const char* word = iterator->GetUTF8Text(level);
                std::string wordBuffer = word;

                // TIPS: GetUTF8Text の仕様より、自前で開放する必要がある。
                delete[] word;

                if (std::regex_match(autoTestAssistTool::util::cp_to_wide(wordBuffer.c_str(), CP_UTF8), regexStr))
                {
                    ImageRect rect;
                    int right;
                    int bottom;

                    iterator->BoundingBox(level, &rect.left, &rect.top, &right, &bottom);
                    rect.width = right - rect.left;
                    rect.height = bottom - rect.top;

                    (*pOutRect) = rect;
                    return MatchingResult::MatchingResult_Success;
                }
            } while (iterator->Next(level));
        }
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::DetectTextForImage: " + exception.what());
        return MatchingResult::MatchingResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::DetectTextForImage: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return MatchingResult::MatchingResult_Failed;
    }

    return MatchingResult::MatchingResult_Failed;
}

CaptureResult ICapture::ExecuteOcr(int *pOutResultCount, OcrResultHandleType *pOutOcrResultHandle, ImageIdType id, OcrSeparateLevel separate, OcrLanguage language)
{
    try
    {
        cv::Mat targetImage = GetCaptureImageMapData(id);

        if (ExecuteTesseractOcrRecognize(&m_TesseractOcrBaseApi, targetImage, language) == false)
        {
            return CaptureResult::CaptureResult_Unexpected;
        }

        tesseract::ResultIterator *iterator = m_TesseractOcrBaseApi.GetIterator();
        if (iterator == NULL)
        {
            return CaptureResult::CaptureResult_Unexpected;
        }

        tesseract::PageIteratorLevel level;
        SeparateLevel2PageIteratorLevel(&level, separate);

        std::vector<OcrResultTypes> resultList;
        int resultCount = 0;
        if (!iterator->Empty(level))
        {
            do
            {
                char* word = iterator->GetUTF8Text(level);
                resultCount++;

                ImageRect rect;
                int right;
                int bottom;

                iterator->BoundingBox(level, &rect.left, &rect.top, &right, &bottom);
                rect.width = right - rect.left;
                rect.height = bottom - rect.top;

                OcrResultTypes result;

                result.word = word;
                result.rect = rect;

                resultList.push_back(result);

                // TIPS: GetUTF8Text の仕様より、自前で開放する必要がある。
                delete[] word;

            } while (iterator->Next(level));
        }
        {
            OcrResultHandleType handle;
            handle.handle = GenerateId(
                [this](unsigned int newId) -> bool{
                std::lock_guard<std::mutex> lock(m_TesseractOcrMutex);

                OcrResultHandleType handleBuffer;
                handleBuffer.handle = newId;

                return (m_TesseractOcrResultList.count(handleBuffer) == 0);
            });

            std::lock_guard<std::mutex> lock(m_TesseractOcrMutex);
            m_TesseractOcrResultList[handle] = resultList;

            (*pOutOcrResultHandle) = handle;
        }
        (*pOutResultCount) = resultCount;
        return CaptureResult::CaptureResult_Success;
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::ExecuteOcr: " + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::ExecuteOcr: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }

    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::GetOcrResultStringSize(int *pOutResultSize, OcrResultHandleType ocrResultHandle, int number)
{
    try
    {
        std::lock_guard<std::mutex> lock(m_TesseractOcrMutex);

        if (m_TesseractOcrResultList.count(ocrResultHandle) == 0)
        {
            return CaptureResult::CaptureResult_InvalidOcrResultHandle;
        }

        const std::vector<OcrResultTypes> &resultList = m_TesseractOcrResultList[ocrResultHandle];
        if (number >= resultList.size())
        {
            return CaptureResult::CaptureResult_Unexpected;
        }

        // 文字サイズ（NULL 文字分を含める）
        (*pOutResultSize) = static_cast<int>(
            resultList.at(number).word.size() != 0 ? resultList.at(number).word.size() + 1 : 0);

        return CaptureResult::CaptureResult_Success;
    }
    catch (...)
    {
        ERRMSG("ICapture::GetOcrResultStringSize: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }

    return CaptureResult::CaptureResult_Success;

}

CaptureResult ICapture::GetOcrResultString( ImageRect *pOutRect,  int *pOutResultSize, char pOutResultString[], int bufferSize, OcrResultHandleType ocrResultHandle, int number)
{
    try
    {
        std::lock_guard<std::mutex> lock(m_TesseractOcrMutex);

        if (m_TesseractOcrResultList.count(ocrResultHandle) == 0)
        {
            return CaptureResult::CaptureResult_InvalidOcrResultHandle;
        }

        const std::vector<OcrResultTypes> &resultList = m_TesseractOcrResultList[ocrResultHandle];
        if (number >= resultList.size())
        {
            return CaptureResult::CaptureResult_Unexpected;
        }


        int copySize = std::min(bufferSize, (static_cast<int>(resultList.at(number).word.size() + 1)));
        if (copySize > 0)
        {
            memcpy(pOutResultString, resultList.at(number).word.c_str(), copySize - 1);
            (*pOutRect) = resultList.at(number).rect;
            pOutResultString[copySize - 1] = '\0';
        }
        return CaptureResult::CaptureResult_Success;
    }
    catch (...)
    {
        ERRMSG("ICapture::GetOcrResultString: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::ReleaseOcrResult(OcrResultHandleType ocrResultHandle)
{
    try
    {
        std::lock_guard<std::mutex> lock(m_TesseractOcrMutex);

        if (m_TesseractOcrResultList.count(ocrResultHandle) == 0)
        {
            return CaptureResult::CaptureResult_InvalidOcrResultHandle;
        }

        m_TesseractOcrResultList.erase(ocrResultHandle);
    }
    catch (...)
    {
        ERRMSG("ICapture::ReleaseOcrResult: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;
}

/*----------------------------------------------------------------------
/ フィルタ
/---------------------------------------------------------------------*/
CaptureResult ICapture::CreateFilter(FilterIdType *pOutFilterId)
{
    try
    {
        m_FilterManager.CreateFilter(pOutFilterId);
    }
    catch (...)
    {
        ERRMSG("ICapture::CreateFilter: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::DeleteFilter(FilterIdType filterId)
{
    try
    {
        m_FilterManager.DeleteFilter(filterId);
    }
    catch (InvalidFilterIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::DeleteFilter: " + exception.what());
        return CaptureResult::CaptureResult_InvalidFilterId;
    }
    catch (...)
    {
        ERRMSG("ICapture::DeleteFilter: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;

}

CaptureResult ICapture::ApplyFilter(ImageIdType *pOutId, ImageIdType id, FilterIdType filterId)
{
    try
    {
        cv::Mat src = GetCaptureImageMapData(id);
        cv::Mat dest;

        m_FilterManager.ApplyFilter(&dest, src, filterId);

        (*pOutId) = SetCaptureImageMapData(dest);
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::ApplyFilter: " + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (InvalidFilterIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::ApplyFilter: " + exception.what());
        return CaptureResult::CaptureResult_InvalidFilterId;
    }
    catch (...)
    {
        ERRMSG("ICapture::ApplyFilter: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }

    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::SetBinarizationFilter(FilterIdType filterId, int binarizeThreshold)
{
    try
    {
        BinarizationFilterParam param;
        param.threshold = binarizeThreshold;

        m_FilterManager.SetBinarizationFilter(filterId, param);
    }
    catch (InvalidFilterIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::SetBinarizationFilter: " + exception.what());
        return CaptureResult::CaptureResult_InvalidFilterId;
    }
    catch (InvalidFilterParameterException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::SetBinarizationFilter: " + exception.what());
        return CaptureResult::CaptureResult_InvalidParameter;
    }
    catch (...)
    {
        ERRMSG("ICapture::SetBinarizationFilter: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::SetColorRangeFilter(FilterIdType filterId, Rgb24Color lowerColor, Rgb24Color upperColor)
{
    try
    {
        ColorRangeFilterParam param;
        param.lowerColor = lowerColor;
        param.upperColor = upperColor;

        m_FilterManager.SetColorRangeFilter(filterId, param);
    }
    catch (InvalidFilterIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::SetColorRangeFilter: " + exception.what());
        return CaptureResult::CaptureResult_InvalidFilterId;
    }
    catch (InvalidFilterParameterException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::SetColorRangeFilter: " + exception.what());
        return CaptureResult::CaptureResult_InvalidParameter;
    }
    catch (...)
    {
        ERRMSG("ICapture::SetColorRangeFilter: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::SetMorphologyFilter(FilterIdType filterId, MorphologyConversionType type, int iterations)
{
    try
    {
        MorphologyFilterParam param;
        param.type = type;
        param.iterations = iterations;
        m_FilterManager.SetMorphologyFilter(filterId, param);
    }
    catch (InvalidFilterIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::SetMorphologyFilter: " + exception.what());
        return CaptureResult::CaptureResult_InvalidFilterId;
    }
    catch (InvalidFilterParameterException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::SetMorphologyFilter: " + exception.what());
        return CaptureResult::CaptureResult_InvalidParameter;
    }
    catch (...)
    {
        ERRMSG("ICapture::SetMorphologyFilter: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::OpenImageWindow(const char* windowName, ImageIdType id)
{
    try
    {
        std::string previewWindowName = GetPreviewName();
        std::string newWindowName = windowName;
        if (previewWindowName == newWindowName)
        {
            return CaptureResult::CaptureResult_Unexpected;
        }

        cv::Mat image = GetCaptureImageMapData(id);

        int width = image.cols;
        int height = image.rows;
        int adjustedImageWidth = GetNearPow2(width);
        int adjustedImageHeight = GetNearPow2(height);

        cv::Mat resizedImage;
        if (width != adjustedImageWidth || height != adjustedImageHeight)
        {
            resizedImage = cv::Mat::zeros(adjustedImageHeight, adjustedImageWidth, CV_8UC3);
            cv::Mat resizedImagePart = resizedImage(cv::Rect(0, 0, image.cols, image.rows));
            image.copyTo(resizedImagePart);
        }
        else
        {
            resizedImage = image;
        }

        WindowManager& windowManager = WindowManager::GetInstance();
        windowManager.OpenImageWindow(windowName, resizedImage, width, height, true);
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::OpenImageWindow: " + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::OpenImageWindow: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }
    return CaptureResult::CaptureResult_Success;

}

CaptureResult ICapture::CloseImageWindow(const char* windowName)
{
    try
    {
        std::string previewWindowName = GetPreviewName();
        std::string newWindowName = windowName;
        if (previewWindowName == newWindowName)
        {
            return CaptureResult::CaptureResult_Unexpected;
        }

        WindowManager& windowManager = WindowManager::GetInstance();
        windowManager.CloseImageWindow(windowName, true);
    }
    catch (...)
    {
        ERRMSG("ICapture::CloseImageWindow: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }

    return CaptureResult::CaptureResult_Success;
}

CaptureResult ICapture::ExtractImage(ImageIdType *pOutImageId, ImageIdType id, ImageRect rect)
{
    try
    {
        cv::Mat targetImage = GetCaptureImageMapData(id);

        if ( rect.left < 0
            || rect.top < 0
            || (rect.left + rect.width) > targetImage.cols
            || (rect.top + rect.height) > targetImage.rows )
        {
            return CaptureResult::CaptureResult_InvalidParameter;
        }

        cv::Mat result(targetImage, cv::Rect(rect.left, rect.top, rect.width, rect.height));

        ImageIdType newImageId = SetCaptureImageMapData(result.clone());
        (*pOutImageId) = newImageId;
    }
    catch (InvalidImageIdException &exception)
    {
        (void)exception;
        ERRMSG("ICapture::ExtractImage: " + exception.what());
        return CaptureResult::CaptureResult_InvalidImageId;
    }
    catch (...)
    {
        ERRMSG("ICapture::ExtractImage: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return CaptureResult::CaptureResult_Unexpected;
    }

    return CaptureResult::CaptureResult_Success;
}
