﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_Aes128CbcDecryptor.h>
#include <nn/crypto/crypto_Aes128CbcEncryptor.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Verifier.h>

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>

#include <nn/oe.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>

#include <nn/spl/spl_Types.h>
#include <nn/spl/spl_Api.h>

#include "XcieDecryptor.h"

namespace
{
const size_t KeyLength = 256;
const size_t SignatureSize = 256;
const size_t LabelHashLength = 32;
const size_t AesKeyLength = 16;
const size_t AesIvLength = 16;

static const char g_BaseKey[AesKeyLength] =
{
#if defined(NN_XCIE_WRITER_XCIE_MODE) || defined(NN_XCIE_WRITER_XCIR_MODE)
#include <L:/keyForCode/basekey.txt>
#else
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
#endif
};

char g_EncCommonKey[AesKeyLength] =
{
#if defined(NN_XCIE_WRITER_XCIE_MODE)
#include <L:/keyForCode/xcie/commonKey.txt>
#elif defined(NN_XCIE_WRITER_XCIR_MODE)
#include <L:/keyForCode/xcir/commonKey.txt>
#else
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
#endif
};

char g_ModulusForSign[KeyLength] =
{
#if defined(NN_XCIE_WRITER_XCIE_MODE)
#include <L:/keyForCode/xcie/modulusForSign.txt>
#elif defined(NN_XCIE_WRITER_XCIR_MODE)
#include <L:/keyForCode/xcir/modulusForSign.txt>
#else
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
#endif
};

char g_RsaModulus[KeyLength];
char g_RsaExponent[KeyLength];
char g_RsaLabelHash[LabelHashLength];
char g_PublicExpnent[3] = {0x01, 0x00, 0x01};

void SecureMemoryZero(void* p, size_t size) NN_NOEXCEPT
{
    auto q = static_cast<volatile char*>(p);
    auto end = q + size;
    for(; q < end; ++q)
    {
        *q = 0;
    }
}
}

size_t DecryptWithRsaOaep(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength)
{
    nn::crypto::RsaOaepDecryptor<KeyLength, nn::crypto::Sha256Generator> rsaDecryptor;
    rsaDecryptor.Initialize(g_RsaModulus, KeyLength, g_RsaExponent, KeyLength);
    rsaDecryptor.SetLabelDigest(g_RsaLabelHash, sizeof(g_RsaLabelHash));
    size_t size = rsaDecryptor.Decrypt(outBuffer, outBufferLength, inBuffer, inBufferLength);
    return size;
}
// 本体固有の AES-CTR
void DecryptWithAesCtrByHwUnique(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength,
                                 char* keyIv, size_t keyIvSize)
{
    NN_ABORT_UNLESS(keyIvSize >= AesIvLength);
    nn::spl::Initialize();
    // 本体固有復号化
    nn::spl::AccessKey accessKey;
    const int KeyGeneration = 0;
    int option = 0x00000001; // 固有鍵
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateAesKek(&accessKey, g_BaseKey, AesKeyLength, KeyGeneration, option));

    char key[AesKeyLength];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateAesKey(key, AesKeyLength, accessKey, g_EncCommonKey, AesKeyLength));
    char iv[AesIvLength];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateAesKey(iv, AesIvLength, accessKey, keyIv, AesIvLength));

    nn::crypto::Aes128CtrDecryptor aes128CtrDecryptor;
    aes128CtrDecryptor.Initialize(key, AesKeyLength, iv, AesIvLength);
    aes128CtrDecryptor.Update(outBuffer, outBufferLength, inBuffer, inBufferLength);

    // 念のためメモリ上から key を削除
    SecureMemoryZero(key, sizeof(key));
    nn::spl::Finalize();
}

// AES-CTR
void DecryptWithAesCtr(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength,
                       char* keyIv, size_t keyIvSize)
{
    if(keyIvSize != 32)
    {
        NN_LOG("KeyIvSize Error %d\n", keyIvSize);
        return;
    }
    nn::crypto::Aes128CtrDecryptor aes128CtrDecryptor;
    aes128CtrDecryptor.Initialize(keyIv, 16, keyIv + 16, 16);
    aes128CtrDecryptor.Update(outBuffer, outBufferLength, inBuffer, inBufferLength);
}

nn::Result SetDecryptKeyForRsa()
{
    // システムセーブから鍵を読む。失敗時はアボートする。
    nn::Result result;
    static const size_t DecryptSize   = KeyLength * 2 + LabelHashLength;
    static const size_t IvOffset      = SignatureSize;
    static const size_t DecryptOffset = IvOffset + AesIvLength;
    static const size_t BodySize      = AesIvLength + DecryptSize;
    static const size_t DataSize      = SignatureSize + BodySize;
    char buffer[DataSize];
    memset(buffer, 0x00, sizeof(buffer));

    // セーブデータマウント
    nn::oe::Initialize();
    nn::account::Initialize();
    nn::account::Uid user = nn::account::InvalidUid;
    int userCount = 0;
    result = nn::account::ListAllUsers(&userCount, &user, 1);
    NN_ASSERT(result.IsSuccess() && userCount > 0);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveData("save", user));

    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, "save:/keyForXcie", nn::fs::OpenMode_Read));
    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, buffer, DataSize));
    nn::fs::CloseFile(handle);
    nn::fs::Unmount("save");
    // まず署名検証
    NN_ABORT_UNLESS(nn::crypto::VerifyRsa2048Pkcs1Sha256(buffer, SignatureSize,
                                                         g_ModulusForSign, KeyLength,
                                                         g_PublicExpnent, sizeof(g_PublicExpnent),
                                                         buffer + SignatureSize, BodySize));
    // 読み出したデータを AES-CTR で本体固有復号化
    DecryptWithAesCtrByHwUnique(buffer + DecryptOffset, DecryptSize, buffer + DecryptOffset, DecryptSize, buffer + IvOffset, AesIvLength);

    // 連結して入っているので分割
    memcpy(g_RsaModulus, buffer + DecryptOffset, KeyLength);
    memcpy(g_RsaExponent, buffer + DecryptOffset + KeyLength, KeyLength);
    memcpy(g_RsaLabelHash, buffer + DecryptOffset + KeyLength * 2, LabelHashLength);

    // 念のためメモリ上から buffer を削除
    SecureMemoryZero(buffer, sizeof(buffer));
    NN_RESULT_SUCCESS;
}

nn::Result DecryptXcieToXci(char* pOutBuffer, size_t outBufferSize,
                            char* encryptedBuffer, size_t encryptedBufferSize,
                            char* encryptedKeyIvBuffer, size_t encryptedKeyIvBufferSize)
{
    // Rsa復号で使用する鍵を復号化
    NN_RESULT_DO(SetDecryptKeyForRsa());
    char keyIv[32];
    // encryptedKeyIvBufferSize must be 0x100.
    DecryptWithRsaOaep(keyIv, 32, encryptedKeyIvBuffer, encryptedKeyIvBufferSize);
    DecryptWithAesCtr(pOutBuffer, outBufferSize, encryptedBuffer, encryptedBufferSize, keyIv, sizeof(keyIv));

    // 一応メモリ上にも残さない
    SecureMemoryZero(g_RsaModulus, sizeof(g_RsaModulus));
    SecureMemoryZero(g_RsaExponent, sizeof(g_RsaExponent));
    SecureMemoryZero(g_RsaLabelHash, sizeof(g_RsaLabelHash));
    SecureMemoryZero(keyIv, sizeof(keyIv));

    NN_RESULT_SUCCESS;
}
