﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/oe.h>

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataTransaction.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Directory.h>

#include <nn/util/util_BitUtil.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>

#include <nn/os/os_SystemEventApi.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Signer.h>
#include <nn/crypto/crypto_RsaPkcs1Sha256Verifier.h>

#include <nn/spl/spl_Types.h>
#include <nn/spl/spl_Api.h>

using namespace nn;
using namespace nn::fs;

static const size_t KeyLength = 256;
static const size_t SignatureSize = 256;
static const size_t LabelHashLength = 32;
static const size_t AesKeyLength = 16;
static const size_t AesIvLength = 16;

static const size_t SignatureBodyStartAddr = SignatureSize;
static const size_t SignatureBodySize      = AesIvLength + KeyLength * 2 + LabelHashLength;

static const size_t DataSize = SignatureSize + SignatureBodySize;

static const size_t EncryptBodyStartAddr   = SignatureBodyStartAddr + AesIvLength;
static const size_t EncryptBodySize        = SignatureBodySize - AesIvLength;


char g_RsaModulus[KeyLength];
char g_RsaExponent[KeyLength];
char g_RsaLabelHash[LabelHashLength];
char g_BaseKey[AesKeyLength];
char g_EncCommonKey[AesKeyLength];
char g_ModulusForSign[KeyLength];
char g_ExponentForSign[KeyLength];
char g_PublicExpnent[3] = {0x01, 0x00, 0x01};

void PrintArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    for(size_t i = 0; i < bufferLength; i++)
    {
        if(i != 0 && i % 16 == 0)
        {
            NN_LOG("\n");
        }

        NN_LOG("%02X ", buffer[i]);
    }
    NN_LOG("\n");
}

void EncryptWithAesCtr(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength, const char* pIv, const size_t ivLength, nn::spl::AccessKey& accessKey, int mode)
{
    char key[AesKeyLength];
    NN_ABORT_UNLESS(ivLength >= AesIvLength);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(key, AesKeyLength, accessKey, g_EncCommonKey, AesKeyLength));
    char iv[AesIvLength];
    NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(iv, AesIvLength, accessKey, pIv, AesIvLength));

    nn::crypto::Aes128CtrEncryptor aes128CtrEncryptor;
    aes128CtrEncryptor.Initialize(key, AesKeyLength, iv, AesIvLength);
    aes128CtrEncryptor.Update(outBuffer, outBufferLength, inBuffer, inBufferLength);
}

void DecryptWithAesCtr(char* outBuffer, const size_t outBufferLength, const char* inBuffer, const size_t inBufferLength, const char* pIv, const size_t ivLength, nn::spl::AccessKey& accessKey, int mode)
{
    char key[AesKeyLength];
    NN_ABORT_UNLESS(ivLength >= AesIvLength);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(key, AesKeyLength, accessKey, g_EncCommonKey, AesKeyLength));
    char iv[AesIvLength];
    NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(iv, AesIvLength, accessKey, pIv, AesIvLength));

    nn::crypto::Aes128CtrDecryptor aes128CtrDecryptor;
    aes128CtrDecryptor.Initialize(key, AesKeyLength, iv, AesIvLength);
    aes128CtrDecryptor.Update(outBuffer, outBufferLength, inBuffer, inBufferLength);
}

void CreateSaveData(nn::account::Uid& user, nn::spl::AccessKey& accessKey, int mode)
{
    fs::FileHandle handle;
    char buffer[DataSize];
    nn::Result result;

    // マウント名 "save" としてセーブデータをマウントします。
    result = nn::fs::MountSaveData("save", user);
    // 失敗した際は必ずアボートしてください。
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    fs::DeleteFile("save:/keyForXcie");

    result = fs::CreateFile("save:/keyForXcie", DataSize * 2);
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::CreateFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }

    // ----------- ファイル書き込み ----------------
    result = fs::OpenFile(&handle, "save:/keyForXcie", OpenMode_Write);
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::OpenFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }

    size_t offset = SignatureSize;
    nn::spl::GenerateRandomBytes(buffer + offset, AesIvLength);
    offset += AesIvLength;
    memcpy(buffer + offset, g_RsaModulus, KeyLength);
    offset += KeyLength;
    memcpy(buffer + offset, g_RsaExponent, KeyLength);
    offset += KeyLength;
    memcpy(buffer + offset, g_RsaLabelHash, LabelHashLength);

    EncryptWithAesCtr(buffer + EncryptBodyStartAddr, EncryptBodySize,
                      buffer + EncryptBodyStartAddr, EncryptBodySize,
                      buffer + SignatureBodyStartAddr, AesIvLength, accessKey, mode);
    if(!nn::crypto::SignRsa2048Pkcs1Sha256(buffer, SignatureSize,
                                           g_ModulusForSign, KeyLength,
                                           g_ExponentForSign, KeyLength,
                                           buffer + SignatureBodyStartAddr, SignatureBodySize))
    {
        NN_LOG("nn::crypto::SignRsa2048Pkcs1Sha256() is failure\n");
        return;
    }


    result = fs::WriteFile(handle, 0, buffer, DataSize, WriteOption());
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::WriteFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }
    result = fs::FlushFile(handle);
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::FlushFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }
    fs::CloseFile(handle);
    fs::CommitSaveData("save");
    fs::Unmount("save");
}

void VerifySaveData(nn::account::Uid& user, nn::spl::AccessKey& accessKey, int mode)
{
    fs::FileHandle handle;
    char buffer[DataSize];
    nn::Result result;

    // ----------- verify ----------------
    memset(buffer, 0x00, KeyLength * 2);
    result = nn::fs::MountSaveData("save", user);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = fs::OpenFile(&handle, "save:/keyForXcie", OpenMode_Read);
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::OpenFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }
    result = fs::ReadFile(handle, 0, buffer, DataSize);
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::ReadFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }

    if(!nn::crypto::VerifyRsa2048Pkcs1Sha256(buffer, SignatureSize,
                                             g_ModulusForSign, KeyLength,
                                             g_PublicExpnent, sizeof(g_PublicExpnent),
                                             buffer + SignatureBodyStartAddr, SignatureBodySize))
    {
        NN_LOG("nn::crypto::VerifyRsa2048Pkcs1Sha256() is failure\n");
        return;
    }
    DecryptWithAesCtr(buffer + EncryptBodyStartAddr, EncryptBodySize,
                      buffer + EncryptBodyStartAddr, EncryptBodySize,
                      buffer + SignatureBodyStartAddr, AesKeyLength, accessKey, mode);
    if((memcmp(buffer + EncryptBodyStartAddr, g_RsaModulus, KeyLength) != 0) ||
            (memcmp(buffer + EncryptBodyStartAddr + KeyLength, g_RsaExponent, KeyLength) != 0) ||
            (memcmp(buffer + EncryptBodyStartAddr + KeyLength * 2, g_RsaLabelHash, LabelHashLength) != 0))
    {
        NN_LOG("Write Rsa Modulus Error\n");
        NN_LOG("-- Expect Modulus--\n");
        for(int i = 0; i < KeyLength; i++)
        {
            if(i % 16 == 0)
                NN_LOG("\n");
            NN_LOG("0x%02x ", g_RsaModulus[i]);
        }
        NN_LOG("\n-- Actual Modulus--\n");
        for(int i = 0; i < KeyLength; i++)
        {
            if(i % 16 == 0)
                NN_LOG("\n");
            NN_LOG("0x%02x ", buffer[i]);
        }

        NN_LOG("\n-- Expect Exponent--\n");
        for(int i = 0; i < KeyLength; i++)
        {
            if(i % 16 == 0)
                NN_LOG("\n");
            NN_LOG("0x%02x ", g_RsaExponent[i]);
        }
        NN_LOG("\n-- Actual Exponent--\n");
        for(int i = 0; i < KeyLength; i++)
        {
            if(i % 16 == 0)
                NN_LOG("\n");
            NN_LOG("0x%02x ", buffer[i + KeyLength]);
        }

        NN_LOG("\n-- Expect LabelHash--\n");
        for(int i = 0; i < KeyLength; i++)
        {
            if(i % 16 == 0)
                NN_LOG("\n");
            NN_LOG("0x%02x ", g_RsaLabelHash[i]);
        }
        NN_LOG("\n-- Actual Exponent--\n");
        for(int i = 0; i < LabelHashLength; i++)
        {
            if(i % 16 == 0)
                NN_LOG("\n");
            NN_LOG("0x%02x ", buffer[i + KeyLength * 2]);
        }
        return;
    }

    fs::CloseFile(handle);
    fs::Unmount("save");
}

void WriteKey(bool isXcirMode)
{
    // 鍵をセーブに書き込み
    nn::Result result;
    int mode = isXcirMode ? 1 : 0;

    nn::oe::Initialize();
    nn::account::Initialize();
    nn::account::Uid user = nn::account::InvalidUid;
    int userCount = 0;
    result = nn::account::ListAllUsers(&userCount, &user, 1);
    NN_ASSERT(result.IsSuccess() && userCount > 0);
    result = nn::fs::EnsureSaveData(user);
    if(nn::fs::ResultUsableSpaceNotEnough::Includes(result))
    {
        // EnsureSaveData() について、アプリケーションは容量不足のエラーハンドリングを行う必要があります。
        // セーブデータ作成のための容量が不足している旨のメッセージは、システムが自動的にエラービューアで表示します。
        // アプリケーションは、アカウント選択操作をキャンセルして前のシーンに戻るなどの対応をする必要があります。
        NN_ABORT("Usable space not enough.\n");
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    // spl のアクセスキー生成
    nn::spl::AccessKey accessKey;
    const int KeyGeneration = 0;
    int option = 0x00000001; // 固有鍵
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateAesKek(&accessKey, g_BaseKey, AesKeyLength, KeyGeneration, option));

    CreateSaveData(user, accessKey, mode);
    VerifySaveData(user, accessKey, mode);
    NN_LOG("Write Success\n");
}

void BytesFromHexString(char *data, size_t bufferSize, char *string) {
    int len = (int)strlen(string);
    size_t counter = 0;
    for (int i = 0; i < len; i += 6)
    {
        while(*(string + i) < 0x2f)
        {
            i++;
        }
        unsigned int x;
        sscanf((char *)(string + i), "0x%02x", &x);
        if(counter < bufferSize)
        {
            data[counter] = x;
        }
        counter++;
    }
}

nn::Result SetOneKey(char* pOutBuffer, size_t bufferSize, const char* filePath)
{
    char readBuffer[128 * 1024];
    memset(readBuffer, 0x00, sizeof(readBuffer));
    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, filePath, nn::fs::OpenMode_Read));
    int64_t fileSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));
    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, readBuffer, fileSize));
    BytesFromHexString(pOutBuffer, bufferSize, readBuffer);
    nn::fs::CloseFile(handle);
    NN_RESULT_SUCCESS;
}

nn::Result SetKeys(bool isXcirMode)
{
    nn::fs::MountHostRoot();
    const char* modulusPath = isXcirMode ? "L:/keyForCode/xcir/modulus.txt" : "L:/keyForCode/xcie/modulus.txt";
    NN_RESULT_DO(SetOneKey(g_RsaModulus, sizeof(g_RsaModulus), modulusPath));

    const char* exponentPath = isXcirMode ? "L:/keyForCode/xcir/exponent.txt" : "L:/keyForCode/xcie/exponent.txt";
    NN_RESULT_DO(SetOneKey(g_RsaExponent, sizeof(g_RsaExponent), exponentPath));

    const char* labelPath = isXcirMode ? "L:/keyForCode/xcir/labelhash.txt" : "L:/keyForCode/xcie/labelhash.txt";
    NN_RESULT_DO(SetOneKey(g_RsaLabelHash, sizeof(g_RsaLabelHash), labelPath));

    const char* basePath = "L:/keyForCode/basekey.txt";
    NN_RESULT_DO(SetOneKey(g_BaseKey, sizeof(g_BaseKey), basePath));

    const char* encCommonKeyPath = isXcirMode ? "L:/keyForCode/xcir/commonKey.txt" : "L:/keyForCode/xcie/commonKey.txt";
    NN_RESULT_DO(SetOneKey(g_EncCommonKey, sizeof(g_EncCommonKey), encCommonKeyPath));

    const char* modulusForSignPath = isXcirMode ? "L:/keyForCode/xcir/modulusForSign.txt" : "L:/keyForCode/xcie/modulusForSign.txt";
    NN_RESULT_DO(SetOneKey(g_ModulusForSign, sizeof(g_ModulusForSign), modulusForSignPath));

    const char* exponentForSignPath = isXcirMode ? "L:/keyForCode/xcir/exponentForSign.txt" : "L:/keyForCode/xcie/exponentForSign.txt";
    NN_RESULT_DO(SetOneKey(g_ExponentForSign, sizeof(g_ExponentForSign), exponentForSignPath));

    nn::fs::UnmountHostRoot();
    NN_RESULT_SUCCESS;
}

//------------------------------------------------------------------------------
//  メイン
//------------------------------------------------------------------------------

extern "C" void nnMain()
{
    // 引数に xcie か xcir かを入れる
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    bool isXcirMode = true;
    if(argc != 2)
    {
        NN_LOG("[error] xcie or xcir is needed\n");
        return;
    }

    if(nn::util::Strncmp(argv[1], "xcie", 4) == 0)
    {
        NN_LOG("****** XCIE MODE ***** \n");
        isXcirMode = false;
    }
    else if(nn::util::Strncmp(argv[1], "xcir", 4) == 0)
    {
        NN_LOG("****** XCIR MODE ***** \n");
        isXcirMode = true;
    }
    else
    {
        NN_LOG("[error] xcie or xcir is needed\n");
        return;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(SetKeys(isXcirMode));

    nn::spl::Initialize();

    WriteKey(isXcirMode);

    nn::spl::Finalize();
}
