﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/ec/system/ec_DeviceAuthenticationApi.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/nim/srv/nim_HttpConnection.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "ShopMonitoringTool_ShopCommandHttpConnection.h"
#include "ShopMonitoringTool_ShopCommandJson.h"
#include "ShopMonitoringTool_ShopCommandUtil.h"

using namespace nn;

namespace
{
    //アカウントトークン用
    NN_ALIGNAS(4096) char g_AuthorizationBuffer[nn::account::RequiredBufferSizeForNintendoAccountAuthorizationRequestContext];

    typedef kvdb::BoundedString<8192> LongUrl;
    typedef kvdb::BoundedString<8192> LongHeader;
    typedef kvdb::BoundedString<8192> PostData;

    const char ContentTypeUrlEncodedHeader[] = "Content-Type: application/x-www-form-urlencoded";
    const char AcceptJsonHeader[] = "Accept: application/json";

    Result HttpGet(HttpConnection* connection, char outBuffer[], size_t outBufferSize, const char* url, const char* headerList[], int headerCount)
    {
        APPLOG_INFO("[HttpGet] %s\n", url);
        size_t readSize = 0;
        auto result = connection->Get(url,
            [&readSize, &outBuffer, outBufferSize](const void* buffer, size_t bufferSize) -> Result
        {
            NN_RESULT_THROW_UNLESS(readSize + bufferSize < outBufferSize, nim::ResultUnexpectedDataReceived());
            std::memcpy(&outBuffer[readSize], buffer, bufferSize);
            readSize += bufferSize;

            NN_RESULT_SUCCESS;
        }, headerList, headerCount);
        outBuffer[readSize] = '\0';
        NN_RESULT_DO(result);

        NN_RESULT_SUCCESS;
    }

    Result HttpPost(HttpConnection* connection, char outBuffer[], size_t outBufferSize, const char* url, const char* postBody, const char* headerList[], int headerCount)
    {
        APPLOG_INFO("[HttpPost] %s\n", url);
        size_t readSize = 0;
        auto result = connection->Post(url, postBody,
            [&readSize, &outBuffer, outBufferSize](const void* buffer, size_t bufferSize) -> Result
        {
            NN_RESULT_THROW_UNLESS(readSize + bufferSize < outBufferSize, nim::ResultUnexpectedDataReceived());
            std::memcpy(&outBuffer[readSize], buffer, bufferSize);
            readSize += bufferSize;

            NN_RESULT_SUCCESS;
        }, headerList, headerCount);
        outBuffer[readSize] = '\0';
        NN_RESULT_DO(result);

        NN_RESULT_SUCCESS;
    }

    Result HttpPostWithoutBody(HttpConnection* connection, char outBuffer[], size_t outBufferSize, const char* url, const char* headerList[], int headerCount)
    {
        return HttpPost(connection, outBuffer, outBufferSize, url, "", headerList, headerCount);
    }

    Result MakeNintendoAccountIdTokenHeader(std::unique_ptr<LongHeader>* outValue, const account::Uid& uid)
    {
        NintendoAccountIdToken idToken;
        {
            NintendoAccountAuthorizationCode authCode;
            NN_RESULT_DO(GetNintendoAccountAuthorization(&authCode, &idToken, uid));
        }

        auto header = std::unique_ptr<LongHeader>(new LongHeader());
        header->AssignFormat("Authorization: Bearer %s", idToken.data);

        *outValue = std::move(header);
        NN_RESULT_SUCCESS;
    }

    Result MakeDeviceAuthenticationTokenHeader(std::unique_ptr<LongHeader>* outValue)
    {
        ec::system::AsyncDeviceAuthenticationToken async;
        NN_RESULT_DO(ec::system::RequestDeviceAuthenticationToken(&async));
        ec::system::DeviceAuthenticationToken token;
        NN_RESULT_DO(async.Get(&token));

        auto header = std::unique_ptr<LongHeader>(new LongHeader());
        header->AssignFormat("X-DeviceAuthorization: Bearer %s", token.data);

        *outValue = std::move(header);
        NN_RESULT_SUCCESS;
    }

    enum class HttpRequestType : Bit8
    {
        Get,
        Post
    };

    Result GetHttpResponseWithNintendoAccountIdToken(const char* url, char* buffer, size_t bufferSize, const account::Uid& uid, HttpRequestType type)
    {
        APPLOG_INFO("GetHttpResponseWithNintendoAccountIdToken\n");
        HttpConnection connection;
        NN_RESULT_DO(connection.Initialize());

        std::unique_ptr<LongHeader> idTokenHeader;
        NN_RESULT_DO(MakeNintendoAccountIdTokenHeader(&idTokenHeader, uid));

        const char* headerList[] = { AcceptJsonHeader, idTokenHeader->Get() };

        switch (type)
        {
        case HttpRequestType::Get: return HttpGet(&connection, buffer, bufferSize, url, headerList, sizeof(headerList) / sizeof(headerList[0]));
        case HttpRequestType::Post: return HttpPostWithoutBody(&connection, buffer, bufferSize, url, headerList, sizeof(headerList) / sizeof(headerList[0]));
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    Result ParseVirtualAccountResponse(nn::ec::system::VirtualAccountId* outValue, char* buffer)
    {
        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(buffer).HasParseError(), nn::nim::ResultInvalidJson());

        auto idObj = document.FindMember("id");
        NN_RESULT_THROW_UNLESS(idObj != document.MemberEnd() && idObj->value.IsString(), nn::nim::ResultInvalidJson());

        char* endPtr;
        nn::ec::system::VirtualAccountId vaId = std::strtol(idObj->value.GetString(), &endPtr, 10);
        NN_RESULT_THROW_UNLESS(*endPtr == '\0', nn::nim::ResultInvalidJson());

        *outValue = vaId;
        NN_RESULT_SUCCESS;
    }

    Result ParseNsuidResponse(int64_t* outValue, char* buffer)
    {
        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(buffer).HasParseError(), nn::nim::ResultInvalidJson());

        auto idPairListObj = document.FindMember("id_pairs");
        NN_RESULT_THROW_UNLESS(idPairListObj != document.MemberEnd() && idPairListObj->value.IsArray() && idPairListObj->value.Size() > 0, nn::nim::ResultInvalidJson());
        auto& idPair = idPairListObj->value[0];
        NN_RESULT_THROW_UNLESS(idPair.IsObject(), nn::nim::ResultInvalidJson());
        auto idObj = idPair.FindMember("id");
        NN_RESULT_THROW_UNLESS(idObj != idPair.MemberEnd() && idObj->value.IsInt64(), nn::nim::ResultInvalidJson());

        *outValue = idObj->value.GetInt64();
        NN_RESULT_SUCCESS;
    }

    Result ParseShopAccountStatusResponse(Bit64* outNaId, char* buffer)
    {
        nne::rapidjson::Document document;
        NN_RESULT_THROW_UNLESS(!document.ParseInsitu(buffer).HasParseError(), nn::nim::ResultInvalidJson());

        auto naIdObject = document.FindMember("na_id");
        NN_RESULT_THROW_UNLESS(naIdObject != document.MemberEnd() && naIdObject->value.IsString(), nn::nim::ResultInvalidJson());

        char* endPtr;
        auto naId = std::strtoull(naIdObject->value.GetString(), &endPtr, 16);
        NN_RESULT_THROW_UNLESS(*endPtr == '\0', nn::nim::ResultInvalidJson());

        *outNaId = naId;
        NN_RESULT_SUCCESS;
    }
}

Result GetNintendoAccountAuthorization(NintendoAccountAuthorizationCode* outAuthCode, NintendoAccountIdToken* outIdToken, const account::Uid& uid)
{
    // 選択されたユーザーを Open
    nn::account::UserHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, uid));
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(handle);
    };

    // ニンテンドーアカウントの認可取得のためのパラメータ
    nn::account::NintendoAccountAuthorizationRequestParameters param = {
        "user openid",      // このパラメータは、アプリケーションがニンテンドーアカウントにどのような権限を要求するかを定める。
        "example_state",    // このパラメータは、認証,認可取得のセッションでの中間者攻撃を防止するために用いることができる。
        "example_nonce"     // このパラメータはIDトークンに含まれるため、通信の改ざんをサーバーサイドで検証できる。
    };

    nn::account::NintendoAccountAuthorizationRequestContext requestContext;
    while (NN_STATIC_CONDITION(true))
    {
        //ニンテンドーアカウントへの認可取得リクエストを行う
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::CreateNintendoAccountAuthorizationRequest(&requestContext, handle, param, g_AuthorizationBuffer, sizeof(g_AuthorizationBuffer)));
        nn::os::SystemEvent event;
        NN_ABORT_UNLESS_RESULT_SUCCESS(requestContext.GetSystemEvent(&event));
        event.Wait();

        // リクエストの結果を取得し、必要であれば本体システムのUIを表示してユーザーに操作を要求する。
        auto result = requestContext.GetResultWithInteractionIfRequired();
        if (!result.IsSuccess())
        {
            if (nn::account::ResultCancelledByUser::Includes(result))
            {
                // 本体システムのUI上でユーザーがキャンセル操作を行った。
                NN_LOG("NintendoAccountAuthorizationRequestContext cancelled");
                NN_RESULT_DO(result);
            }
            else if (nn::account::ResultNetworkCommunicationError::Includes(result))
            {
                // ログイン時に通信関係のエラーが発生
                NN_RESULT_DO(result);
            }
            // requestContext.Cancel() が呼ばれていると、ここで nn::account::ResultCancelled() が返る場合がある。
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        // 成功した場合、 state パラメータが最初に指定したものと一致するかを念のために確認する。
        size_t stateLength;
        char state[128];
        NN_ABORT_UNLESS_RESULT_SUCCESS(requestContext.GetState(&stateLength, state, sizeof(state)));
        NN_ABORT_UNLESS(strnlen(param.state, sizeof(param.state)) == stateLength, "Unexpected state used");
        NN_ABORT_UNLESS(std::strncmp(state, param.state, stateLength) == 0, "Unexpected state used");
        break;
    }

    size_t authCodeSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(requestContext.GetAuthorizationCode(&authCodeSize, outAuthCode->data, sizeof(outAuthCode->data)));
    outAuthCode->data[authCodeSize] = '\0';
    APPLOG_INFO("   NaAuthCode : %.*s\n", authCodeSize, outAuthCode->data);

    size_t tokenSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(requestContext.GetIdToken(&tokenSize, outIdToken->data, sizeof(outIdToken->data)));
    outIdToken->data[tokenSize] = '\0';
    APPLOG_INFO("   NaIdToken : %.*s\n", tokenSize, outIdToken->data);

    NN_RESULT_SUCCESS;
}

Result GetUidFromIndex(util::optional<account::Uid>* outValue, int userIndex)
{
    auto index = static_cast<size_t>(userIndex);

    std::vector<account::Uid> uidList(account::UserCountMax);
    int count;
    NN_RESULT_DO(account::ListAllUsers(&count, uidList.data(), static_cast<int>(uidList.size())));
    if (count == 0)
    {
        *outValue = util::nullopt;
        NN_RESULT_SUCCESS;
    }
    uidList.resize(static_cast<size_t>(count));

    if (index >= uidList.size())
    {
        *outValue = util::nullopt;
        NN_RESULT_SUCCESS;
    }

    *outValue = uidList[index];
    NN_RESULT_SUCCESS;
}

Result GetShopAccountStatusResponse(char* buffer, size_t bufferSize, const account::Uid& uid)
{
    NintendoAccountAuthorizationCode authCode;
    {
        NintendoAccountIdToken idToken;
        NN_RESULT_DO(GetNintendoAccountAuthorization(&authCode, &idToken, uid));
    }

    nsd::NasServiceSetting shopSetting;
    NN_RESULT_DO(nsd::GetNasServiceSetting(&shopSetting, nsd::NasServiceNameOfNxShop));

    char redirectUrl[256];
    std::strcpy(redirectUrl, shopSetting.redirectUri.value);
    NN_RESULT_DO(HttpConnection::Escape(redirectUrl, sizeof(redirectUrl), redirectUrl));

    std::unique_ptr<PostData> postData(new PostData());
    postData->AssignFormat("auth_code=%s&redirect_url=%s", authCode.data, redirectUrl);

    HttpConnection connection;
    NN_RESULT_DO(connection.Initialize());

    const char* headerList[] = { ContentTypeUrlEncodedHeader, AcceptJsonHeader };
    auto myAccountUrl = std::unique_ptr<LongUrl>(new LongUrl());
    myAccountUrl->AssignFormat("https://ashigaru-frontend.hac.%%.eshop.nintendo.net/shogun/v1/my/account?lang=ja",
        authCode.data, redirectUrl);

    NN_RESULT_DO(HttpPost(&connection, buffer, bufferSize, *myAccountUrl, *postData, headerList, sizeof(headerList) / sizeof(headerList[0])));

    NN_RESULT_SUCCESS;
}

Result GetDeviceLinkStatusResponse(char* buffer, size_t bufferSize, const account::Uid& uid)
{
    return GetHttpResponseWithNintendoAccountIdToken(
        "https://ashigaru-frontend.hac.%.eshop.nintendo.net/shogun/v1/my/devices?device_type_id=6&lang=ja",
        buffer, bufferSize, uid, HttpRequestType::Get);
}

Result GetDeleteAllRightsResponse(char* buffer, size_t bufferSize, const account::Uid& uid)
{
    return GetHttpResponseWithNintendoAccountIdToken(
        "https://ashigaru-frontend.hac.%.eshop.nintendo.net/shogun/v1/debug/my/tickets/delete/all?lang=ja",
        buffer, bufferSize, uid, HttpRequestType::Post);
}

Result GetPrepurchaseResponse(char* buffer, size_t bufferSize, const account::Uid& uid, int64_t nsuid, const char* purchaseType)
{
    auto prePurchaseUrl = std::unique_ptr<LongUrl>(new LongUrl());
    prePurchaseUrl->AssignFormat("https://ashigaru-frontend.hac.%%.eshop.nintendo.net/shogun/v1/%s/%lld/prepurchase?lang=ja", purchaseType, nsuid);

    return GetHttpResponseWithNintendoAccountIdToken(*prePurchaseUrl, buffer, bufferSize, uid, HttpRequestType::Get);
}

Result GetPurchaseResponse(char* buffer, size_t bufferSize, const account::Uid& uid, int64_t nsuid, int64_t priceId, int64_t amount, const char* purchaseType)
{
    auto purchaseUrl = std::unique_ptr<LongUrl>(new LongUrl());
    purchaseUrl->AssignFormat("https://ashigaru-frontend.hac.%%.eshop.nintendo.net/shogun/v1/%s/%lld/purchase?lang=ja&price_id=%lld&amount=%lld", purchaseType, nsuid, priceId, amount);

    return GetHttpResponseWithNintendoAccountIdToken(*purchaseUrl, buffer, bufferSize, uid, HttpRequestType::Post);
}

Result GetUnlinkDeviceAllResponse(char* buffer, size_t bufferSize)
{
    std::unique_ptr<LongHeader> deviceAuthHeader;
    NN_RESULT_DO(MakeDeviceAuthenticationTokenHeader(&deviceAuthHeader));

    const char* headerList[] = { ContentTypeUrlEncodedHeader, AcceptJsonHeader, deviceAuthHeader->Get() };

    auto unlinkAllUrl = std::unique_ptr<LongUrl>(new LongUrl());
    unlinkAllUrl->AssignFormat("https://beach.hac.%%.eshop.nintendo.net/v1/my/devices/hac/unlink_all");

    HttpConnection connection;
    NN_RESULT_DO(connection.Initialize());
    NN_RESULT_DO(HttpPostWithoutBody(&connection, buffer, bufferSize, *unlinkAllUrl, headerList, sizeof(headerList) / sizeof(headerList[0])));

    NN_RESULT_SUCCESS;
}

Result GetVirtualAccount(nn::ec::system::VirtualAccountId* outValue, char* buffer, size_t bufferSize, const account::Uid& uid)
{
    std::unique_ptr<LongHeader> idTokenHeader;
    NN_RESULT_DO(MakeNintendoAccountIdTokenHeader(&idTokenHeader, uid));

    const char* headerList[] = { AcceptJsonHeader, idTokenHeader->Get() };

    HttpConnection connection;
    NN_RESULT_DO(connection.Initialize());
    NN_RESULT_DO(HttpGet(&connection, buffer, bufferSize,
        "https://ashigaru-frontend.hac.%.eshop.nintendo.net/shogun/v1/my/virtual_account?lang=ja",
        headerList, sizeof(headerList) / sizeof(headerList[0])));

    NN_RESULT_DO(ParseVirtualAccountResponse(outValue, buffer));
    NN_RESULT_SUCCESS;
}

nn::Result GetSearchResponse(char* buffer, size_t bufferSize)
{
    const char* headerList[] = { AcceptJsonHeader };

    HttpConnection connection;
    NN_RESULT_DO(connection.Initialize());
    NN_RESULT_DO(HttpGet(&connection, buffer, bufferSize,
        "https://ashigaru-frontend.hac.%.eshop.nintendo.net/shogun/v1/titles"
        "?lang=ja&offset=0&limit=8192&sort=popular&country=JP&shop_id=4",
        headerList, sizeof(headerList) / sizeof(headerList[0])));

    NN_RESULT_SUCCESS;
}

nn::Result GetSearchAddOnContentResponse(char* buffer, size_t bufferSize, int64_t nsuid)
{
    const char* headerList[] = { AcceptJsonHeader };

    auto searchUrl = std::unique_ptr<LongUrl>(new LongUrl());
    searchUrl->AssignFormat(
        "https://ashigaru-frontend.hac.%%.eshop.nintendo.net/shogun/v1/titles/%lld/aocs"
        "?lang=ja&offset=0&limit=8192&country=JP&shop_id=4", nsuid);

    HttpConnection connection;
    NN_RESULT_DO(connection.Initialize());
    NN_RESULT_DO(HttpGet(&connection, buffer, bufferSize, *searchUrl, headerList, sizeof(headerList) / sizeof(headerList[0])));

    NN_RESULT_SUCCESS;
}

const char* GetShopIdTypeString(ShopIdType type)
{
    switch (type)
    {
    case ShopIdType::Demo: return "demo";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

nn::Result GetNsuidByContentMetaId(int64_t* outValue, char* buffer, size_t bufferSize, nn::Bit64 id, ShopIdType type)
{
    APPLOG_INFO("GetNsuidByContentMetaId\n");
    auto shopIdUrl = std::unique_ptr<LongUrl>(new LongUrl());
    shopIdUrl->AssignFormat(
        "https://ashigaru-frontend.hac.%%.eshop.nintendo.net/shogun/v1/contents/ids?"
        "lang=ja&country=JP&shop_id=4&type=%s&title_ids=%016llx",
        GetShopIdTypeString(type), id);

    const char* headerList[] = { AcceptJsonHeader };

    HttpConnection connection;
    NN_RESULT_DO(connection.Initialize());
    NN_RESULT_DO(HttpGet(&connection, buffer, bufferSize, *shopIdUrl,
        headerList, sizeof(headerList) / sizeof(headerList[0])));

    NN_RESULT_DO(ParseNsuidResponse(outValue, buffer));
    APPLOG_INFO("in(MetaId : %lld) out(NsuId : %lld)",id,*outValue);
    NN_RESULT_SUCCESS;
}

nn::Result GetDownloadDemoResponse(char* buffer, size_t bufferSize, const nn::account::Uid& uid, int64_t nsuid)
{
    auto downloadUrl = std::unique_ptr<LongUrl>(new LongUrl());
    downloadUrl->AssignFormat("https://ashigaru-frontend.hac.%%.eshop.nintendo.net/shogun/v1/demos/%lld/download?lang=ja&shop_id=4", nsuid);

    return GetHttpResponseWithNintendoAccountIdToken(*downloadUrl, buffer, bufferSize, uid, HttpRequestType::Post);
}

nn::Result GetNaId(Bit64* out, const nn::account::Uid& uid)
{
    char accountStatusResponse[512];
    NN_RESULT_DO(GetShopAccountStatusResponse(accountStatusResponse, sizeof(accountStatusResponse), uid));

    NN_RESULT_DO(ParseShopAccountStatusResponse(out, accountStatusResponse));

    NN_RESULT_SUCCESS;
}

nn::Result UnregisterGameCardByNaId(nn::Bit64 naId, char* buffer, size_t bufferSize)
{
    auto unregisterUrl = std::unique_ptr<LongUrl>(new LongUrl());
    unregisterUrl->AssignFormat("https://beach.hac.%%.eshop.nintendo.net/v1/rom_cards/delete_by_na");

    NN_UTIL_SCOPE_EXIT{
        NN_LOG("%s\n", buffer);
    };

    const char* headerList[] = { ContentTypeUrlEncodedHeader, AcceptJsonHeader };

    char postField[256] = {};
    util::TSNPrintf(postField, sizeof(postField), "na_user_id=%016llx", naId);

    HttpConnection connection;
    NN_RESULT_DO(connection.Initialize());
    NN_RESULT_DO(HttpPost(&connection, buffer, bufferSize, unregisterUrl->Get(), postField, headerList, sizeof(headerList) / sizeof(headerList[0])));

    NN_RESULT_SUCCESS;
}
