﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#ifdef __cplusplus

namespace mw {
namespace qre {

/** 誤り訂正レベル */
typedef enum EccLevel
{
    ECC_LEVEL_L = 0,
    ECC_LEVEL_M,
    ECC_LEVEL_Q,
    ECC_LEVEL_H
} EccLevel;

/** QRコード情報 */
typedef struct EncodeData
{
    uint32_t size;      /**< バイナリQRデータのバイト数 */
    uint8_t* data;      /**< バイナリQRデータへのポインタ */
    uint32_t version;   /**< フォーマットバージョン （0～40, 0:自動） */
    uint32_t total;     /**< 分割数 （0～16, 0:自動） */
    uint32_t cell_size; /**< セルのサイズ */
    EccLevel ecc_level; /**< 誤り訂正レベル */
} EncodeData;

/** 位置情報 */
typedef struct Rect
{
    int32_t top;        /**< 上限 */
    int32_t left;       /**< 左 */
    int32_t right;      /**< 右 */
    int32_t bottom;     /**< 下限 */
} Rect;

/** RGBイメージ情報 */
typedef struct ImageInfo
{
    uint16_t width;     /**< 幅 */
    uint16_t height;    /**< 高さ */
    uint8_t* rgbData;   /**< RGB画素データ */
    Rect imgPos;        /**< イメージ一座標 */
} ImageInfo;

/** @brief QRコード生成クラス */
class QREncoder
{
public:
    /** @brief コンストラクタ */
    QREncoder();
    /** @brief デストラクタ */
    virtual ~QREncoder();

    /**
     * @brief QRコード生成
     *
     * @param[in] qrInfo QRコード情報構造体
     * @retval  true    正常終了
     * @return  false   処理失敗
     */
    bool Encode(EncodeData* qrInfo);

    /**
     * @brief デザインQRコード生成
     *
     * @param[in] qrInfo    QRコード情報構造体
     * @param[in] imageInfo QRコードに埋め込むイメージの情報
     * @return  true        正常終了
     * @return  false       処理失敗
     */
    bool EncodeWithImage(EncodeData* qrInfo, const ImageInfo* imageInfo);

    /**
     * @brief エラー情報取得
     *
     * @param[in] pMess エラー情報文字列格納用
     * @param[in] size  メッセージ
     * @return  true    正常終了
     * @return  false   処理失敗
     */
    bool GetErrorMessage(char* pMess, int32_t size);

    /**
     * @brief 分割数取得
     *
     * @return 分割数
     */
    uint32_t GetDivCount();

    /**
     * @brief フォーマットバージョン取得
     *
     * @return フォーマットバージョン
     */
    uint32_t GetVersion();

    /**
     * @brief QRデータサイズ取得
     *
     * @param[in]  index 分割番号(0～15)
     * @param[in]  isBmp 取得するデータの種類 true: BMPファイル false: イメージ情報
     * @return BMPファイルサイズ
     */
    uint32_t GetQRSize(uint32_t index = 0, bool isBmp = false);

    /**
     * @brief BMPデータ取得
     *
     * @param[out] buff BMPデータバッファ
     * @param[in] size  バッファサイズ
     * @param[in] index 分割番号(0～15)
     * @param[out] pos  イメージの位置
     * @return   true   正常終了
     * @return   false  処理失敗
     */
    bool GetQRBMPData(uint8_t* buff, uint32_t size, uint32_t index = 0, mw::qre::Rect *pos = NULL);

    /**
     * @brief QRイメージ情報取得
     *
     * @param[out] info QRイメージ情報
     * @param[in] index 分割番号(0～15)
     * @return   true   正常終了
     * @return   false  処理失敗
     */
    bool GetQRData(ImageInfo* info, uint32_t index = 0);

    /**
     * @brief ライブラリで使用するメモリ領域の登録
     *
     * @param[in] ptr   メモリ領域
     * @param[in] size  メモリ領域のサイズ
     * @return  true    正常終了
     * @return  false   処理失敗
     */
    bool InitMemory(void* ptr, uint32_t size);

    /**
     * @brief ライブラリで使用するメモリサイズの取得
     *
     * @param[in] data_size エンコードするデータのサイズ
     * @param[in] cell_size QRコードイメージの1フォーマットの幅 (or 高さ dot)
     * @param[in] count QRコード分割数 (0 … データサイズに依存する)
     * @return  必要とするメモリサイズ
     */
    static uint32_t GetEncodeBufferSize(uint32_t data_size, uint32_t cell_size, uint32_t count = 0);

    /**
     * @brief デザインQR埋め込み可能最大イメージサイズ取得
     *
     * @param[in] version  QRバージョン
     * @param[in] cellSize 1セルのドット数
     * @return  埋め込み可能な最大イメージの一辺のサイズ
     */
    static uint32_t GetDesignQRImageSize(uint32_t version, uint32_t cellSize);
};

} // namespace qre
} // namespcae mw

#endif    // __cplusplus

#ifdef __cplusplus
# define MW_QRE_EXTERN_C extern "C"
#else
# define MW_QRE_EXTERN_C extern
#endif

/*!
 * @union mwqreQREncoder
 * @brief  QRエンコーダをあらわす C の共用体です。
 * @brief  対応するmw::qre::QREncoder を参照してください。
 */
typedef union mwqreQREncoder
{
    int8_t buf[4];
    uint32_t alignment_holder;
} mwqreQREncoder;

/** 誤り訂正レベル */
typedef enum mwqreEccLevel
{
    MW_QRE_ECC_LEVEL_L = 0,
    MW_QRE_ECC_LEVEL_M,
    MW_QRE_ECC_LEVEL_Q,
    MW_QRE_ECC_LEVEL_H
} mwqreEccLevel;

/** QRコード情報 */
typedef struct mwqreEncodeData
{
    uint32_t size;              /**< バイナリQRデータのバイト数 */
    uint8_t* data;              /**< バイナリQRデータへのポインタ */
    uint32_t version;           /**< フォーマットバージョン （0～40, 0:自動） */
    uint32_t total;             /**< 分割数 （0～16, 0:自動） */
    uint32_t cell_size;         /**< セルのサイズ */
    mwqreEccLevel ecc_level;    /**< 誤り訂正レベル  */
} mwqreEncodeData;

/** 位置情報 */
typedef struct mwqreRect
{
    int32_t top;                /**< 上限 */
    int32_t left;               /**< 左 */
    int32_t right;              /**< 右 */
    int32_t bottom;             /**< 下限 */
} mwqreRect;

/** RGBイメージ情報 */
typedef struct mwqreImageInfo
{
    uint16_t width;             /**< 幅 */
    uint16_t height;            /**< 高さ */
    uint8_t* rgbData;           /**< RGB画素データ */
    mwqreRect imgPos;           /**< イメージ一座標 */
} mwqreImageInfo;

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::Encode を参照してください。
 */
MW_QRE_EXTERN_C bool mwqreQREncoderEncode(mwqreQREncoder* this_, mwqreEncodeData* qrInfo);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::EncodeWithImage を参照してください。
 */
MW_QRE_EXTERN_C bool mwqreQREncoderEncodeWithImage(mwqreQREncoder* this_, mwqreEncodeData* qrInfo, const mwqreImageInfo* imageInfo);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::GetErrorMessage を参照してください。
 */
MW_QRE_EXTERN_C bool mwqreQREncoderGetErrorMessage(mwqreQREncoder* this_, char* pMess, int32_t size);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::GetDivCount を参照してください。
 */
MW_QRE_EXTERN_C uint32_t mwqreQREncoderGetDivCount(mwqreQREncoder* this_);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::GetVersion を参照してください。
 */
MW_QRE_EXTERN_C uint32_t mwqreQREncoderGetVersion(mwqreQREncoder* this_);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::GetQRSize を参照してください。
 */
MW_QRE_EXTERN_C uint32_t mwqreQREncoderGetQRSize(mwqreQREncoder* this_, uint32_t index, bool isBmp);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::GetQRBMPData を参照してください。
 */
MW_QRE_EXTERN_C bool mwqreQREncoderGetQRBMPData(mwqreQREncoder* this_, uint8_t* buff, uint32_t size, uint32_t index, mwqreRect* pos);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::GetQRData を参照してください。
 */
MW_QRE_EXTERN_C bool mwqreQREncoderGetQRData(mwqreQREncoder* this_, mwqreImageInfo* info, uint32_t index);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::InitMemory を参照してください。
 */
MW_QRE_EXTERN_C bool mwqreQREncoderInitMemory(mwqreQREncoder* this_, void* ptr, uint32_t size);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::GetEncodeBufferSize を参照してください。
 */
MW_QRE_EXTERN_C uint32_t mwqreQREncoderGetEncodeBufferSize(uint32_t data_size, uint32_t cell_size, uint32_t count);

/*!
 *  @brief 対応する C++ 関数 @ref mw::qre::QREncoder::GetDesignQRImageSize を参照してください。
 */
MW_QRE_EXTERN_C uint32_t mwqreQREncoderGetDesignQRImageSize(uint32_t version, uint32_t cellSize);
