﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include "common.h"
#include <glv_resources.h>

#include "QCIT_ModalView.h"

namespace qcit
{
    /**
    * @brief       QCIT のエントリポイントです
    */
    void QCITMain() NN_NOEXCEPT;

    /**
    * @brief       QCIT 用の RootSurface です
    */
    class RootSurfaceContext : public glv::Window, public glv::GLV, public glv::ApplicationLoopCallback
    {
    public:
        /**
        * @brief       コンストラクタです。
        *
        * @param[in]   width       幅
        * @param[in]   height      高
        */
        RootSurfaceContext(const unsigned width, const unsigned height) NN_NOEXCEPT;

        /**
        * @brief       デストラクタです。
        */
        virtual ~RootSurfaceContext() NN_NOEXCEPT NN_OVERRIDE;

        /**
        * @brief       ページ共通背景画像描画オブジェクト取得.
        */
        glv::Drawable* GetBackgroundDrawable() NN_NOEXCEPT;

        /**
        * @brief       ランタイムエンジンにアタッチされた際に呼ばれます。
        *
        * @param[in]   context     コンテキスト
        *
        * @see         glv::ApplicationLoopCallback::OnLoopAttached()
        */
        virtual void OnLoopAttached(glv::ApplicationLoopContext& context) NN_NOEXCEPT NN_OVERRIDE;

        /**
        * @brief       ランタイムエンジンからデタッチされた際に呼ばれます。
        *
        * @param[in]   context     コンテキスト
        *
        * @see         glv::ApplicationLoopCallback::OnLoopDetached()
        */
        virtual void OnLoopDetached(glv::ApplicationLoopContext& context) NN_NOEXCEPT NN_OVERRIDE;

        /**
        * @brief       glv シーンレンダラ前に呼ばれます。
        *
        * @param[in]   context     コンテキスト
        * @param[in]   events      イベント
        *
        * @details     glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
        *              この時点ではまだ glv コンテキストのレンダリングは開始していません。
        *              このメソッドが呼び出されるフレームは OnLoopAfterSceneRenderer() と同じです。
        *
        * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
        *
        * @see         glv::ApplicationLoopCallback::OnLoopBeforeSceneRenderer()
        */
        virtual const glv::RequiredRestoration OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE;

        /**
        * @brief       glv シーンレンダラ後に呼ばれます。
        *
        * @param[in]   context     コンテキスト
        * @param[in]   events      イベント
        *
        * @details     glv シーンレンダラのレンダリングが終わった後に呼び出されます。
        *              このメソッドが呼び出されるフレームは OnLoopBeforeSceneRenderer() と同じです。
        *
        * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
        *
        * @see         glv::ApplicationLoopCallback::OnLoopAfterSceneRenderer()
        */
        virtual const glv::RequiredRestoration OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE;

        void StartModal(ModalView* pModalView, bool isDisposable, bool isDelayExec = false, bool isAutomaticExitAtBackgroundEnabled = false) NN_NOEXCEPT
        {
            m_InvisibleWall.StartModal(pModalView, isDisposable, isDelayExec, isAutomaticExitAtBackgroundEnabled);
        }

        bool IsModalViewRunning() NN_NOEXCEPT
        {
            return m_InvisibleWall.IsModalViewRunning();
        }

        glv::Rect GetSurfacePageRect()
        {
            return m_PageSurface.rect();
        }

        View* GetActivePage() NN_NOEXCEPT;

    private:
        /**
        * @brief       ループ毎の前処理を行います。
        *
        * @param[in]   context     コンテキスト
        * @param[in]   events      イベント
        */
        void HandleLoopPre(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

        /**
        * @brief       ループ毎の後処理を行います。
        *
        * @param[in]   context     コンテキスト
        * @param[in]   events      イベント
        */
        void HandleLoopPost(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

        /**
        * @brief       HID イベントを処理します。
        *
        * @param[in]   context     コンテキスト
        * @param[in]   events      イベント
        */
        void HandleHidEvent(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

        glv::Drawable*                   m_pBackgroundDrawable;      //!< ページ共通背景画像描画オブジェクト
        glv::Table*                      m_pSurfaceTable;            //!< ルートサーフェイスレイアウト用
        glv::Group                       m_PageSurface;              //!< ページサーフェイスルート

    private:
        InvisibleWall m_InvisibleWall;
    };

}
